#!/usr/bin/env python3

"""This module represents caching. It uses sqlite database to save date of each extraction.
.. module:: cache
    :platform: linux
    :synopsis: Represents caching module
.. moduleauthor:: Martin Bazik
"""

from pathlib import Path
from datetime import datetime
from datetime import timedelta
import time
import sqlite3

class Cache:
    def __init__(self,path):
        self.path = path
        self.createConnection()
        self.createTables()
        self.filter = {}

    def setDevice(self,device=None,deviceID=None):
        """Sets the device we are going to work with
        """

        if(deviceID is None):
            self.device = device
            self.cursor.execute("SELECT id FROM device WHERE identifier = ? AND timestamp = (SELECT MAX(timestamp) FROM device)",(device,))
            self.deviceID = self.cursor.fetchone()
            if(self.deviceID):
                self.deviceID = self.deviceID[0]
        else:
            self.deviceID = deviceID


    def createConnection(self):
        """ create a database connection to a database that resides
            in the memory
        """

        dbFile = str(self.path)
        try:
            self.conn = sqlite3.connect(dbFile)
            self.cursor = self.conn.cursor()
        except sqlite3.IntegrityError as e:
            print(e)

    def setFilter(self,filtering):
        """Sets filter that can be later used to filtering data from database
        """

        # Two possibilities "time" and "name"
        # Filter is stored in a dictionary
        if(len(filtering) == 3 and filtering[0] == "time"):
            try:
                dates = []
                dates.append(datetime.strptime(filtering[1], "%d-%m-%Y-%H-%M").timestamp())
                dates.append(datetime.strptime(filtering[2], "%d-%m-%Y-%H-%M").timestamp())
                self.filter = {"time":dates}
            except ValueError as e:
                print(e)
        elif(len(filtering) == 2 and filtering[0] == "name"):
            self.filter = {"name":filtering[1]}

    def createTables(self):
        """Creates tables
        """

        query = []

        #query.append('''DROP TABLE device''') 

        query.append('''CREATE TABLE IF NOT EXISTS device(id INTEGER primary key, identifier TEXT, version text,
                       architecture text, root text, timestamp INTEGER)''')
        
        
        #query.append('''DROP TABLE package''')
 
        query.append('''CREATE TABLE IF NOT EXISTS package(id INTEGER PRIMARY KEY AUTOINCREMENT, identifier TEXT, name TEXT, 
                       activityname TEXT, lastused INTEGER, totaltime INTEGER, deviceid INTEGER, directories TEXT,
                       FOREIGN KEY (deviceid) REFERENCES device(id))''')

        #query.append('''DROP TABLE file''')

        query.append('''CREATE TABLE IF NOT EXISTS file(id INTEGER PRIMARY KEY AUTOINCREMENT, 
                       name TEXT, size INTEGER, lastmodified INTEGER, path TEXT, packageid INTEGER,
                       FOREIGN KEY (packageid) REFERENCES package(id))''')

        #query.append('''DROP TABLE log''')
 
        query.append('''CREATE TABLE IF NOT EXISTS log(id INTEGER PRIMARY KEY AUTOINCREMENT, 
                       package TEXT, logcat TEXT, batterystats TEXT, activity TEXT, procstats TEXT,
                       appops TEXT, notification TEXT, packageid INTEGER,
                       FOREIGN KEY (packageid) REFERENCES package(id))''')   
        
        for q in query:
           self.cursor.execute(q)
        self.conn.commit()

    def addDevice(self,identifier,version,architecture,root,time):
        """Saves devices
        """

        rootstr = ""

        if(root):
            rootstr = "TRUE"
        else:
            rootstr = "FALSE"
        try:
            self.cursor.execute("""INSERT into device(identifier, version, architecture, root, timestamp)
                values(?,?,?,?,?)""",(identifier,version,architecture,rootstr,time))
        except sqlite3.IntegrityError as e:
            print("Already present")
        self.conn.commit()

    def addPackage(self, identifier, name, activityname, lastused, totaltime, directories):
        """Saves packages
        """

        totaltime = totaltime.total_seconds()
        lastused = lastused.timestamp()
        try:
            self.cursor.execute("""INSERT into package(identifier, name, activityname, lastused, 
                totaltime,directories,deviceid) values(?,?,?,?,?,?,?)""",(identifier, name, 
                activityname, lastused, totaltime, directories, self.deviceID))

        except sqlite3.IntegrityError as e:
            print("Already present")


    def addFile(self, name, size, lastmodified, path, packageid):
        """Saves files
        """

        self.cursor.execute("SELECT id FROM package WHERE deviceid = ? AND identifier = ?",(self.deviceID,packageid))
        packageID = self.cursor.fetchone()
        if(packageID):
            packageID = packageID[0]
        else:
            return

        lastmodified = lastmodified.timestamp()
        try:
            self.cursor.execute("""INSERT into file (name, size, lastmodified, path, packageid)
                values(?,?,?,?,?)""",(name, size, lastmodified, path, packageID))
        except sqlite3.IntegrityError as e:
            print("Already present")


    def addLog(self, package, logcat, batterystats, activity, procstats, appops, notification,
        packageid):
        """Saves logs
        """
        
        self.cursor.execute("SELECT id FROM package WHERE deviceid = ? AND identifier = ?",(self.deviceID,packageid))
        packageID = self.cursor.fetchone()
        if(packageID):
            packageID = packageID[0]
        else:
            return

        try:
            self.cursor.execute("""INSERT into log (package, logcat, batterystats, activity, procstats, 
                appops, notification,packageid) values(?,?,?,?,?,?,?,?)""",(package, logcat, batterystats,
                activity, procstats, appops, notification, packageID))
        except sqlite3.IntegrityError as e:
            print("Already present")

    def commit(self):
        """Commiting changes
        """
        self.conn.commit()

    def getDevice(self,deviceID):
        dictionary = {}

        self.cursor.execute("SELECT * FROM device WHERE id = ?",(deviceID,))
        deviceAttrList = self.cursor.fetchall()
        if(deviceAttrList):
            deviceAttrList = deviceAttrList[0]
        if(deviceAttrList):
            dictionary["id"] = deviceAttrList[1]
            dictionary["version"] = deviceAttrList[2]
            dictionary["arch"] = deviceAttrList[3]
            dictionary["root"] = False
            if(deviceAttrList[4] == "TRUE"):
                dictionary["root"] = True
            dictionary["time"] = datetime.fromtimestamp(deviceAttrList[5])

        return dictionary

    def getDevices(self):
        if("time" in self.filter.keys()):
            self.cursor.execute("SELECT * FROM device WHERE timestamp >= ? AND timestamp <= ?",(self.filter["time"][0],self.filter["time"][1]))
        elif("name" in self.filter.keys()):
            filtering = "%"+self.filter["name"]+"%"
            self.cursor.execute("SELECT * FROM device WHERE identifier LIKE ?",(filtering,))  
        else:
            self.cursor.execute("SELECT * FROM device")
        return self.cursor.fetchall()

    def getPackages(self,deviceID=None,doFilter=True):
        """Getter for all packages
        Parameters: app (string)    app identifier
        """        

        if(deviceID is None):
            deviceID = self.deviceID

        packages = []
        if(deviceID):
            if(doFilter and "time" in self.filter.keys()):
                self.cursor.execute("SELECT * FROM package WHERE deviceid = ? AND lastused >= ? AND lastused <= ?",(deviceID,self.filter["time"][0],self.filter["time"][1]))          
            elif(doFilter and "name" in self.filter.keys()):
                filtering = "%"+self.filter["name"]+"%"
                self.cursor.execute("SELECT * FROM package WHERE deviceid = ? AND identifier LIKE ?",(deviceID,filtering))  
            else:
                self.cursor.execute("SELECT * FROM package WHERE deviceid = ?",(deviceID,))
            packages = self.cursor.fetchall()

        listOfPackages = []

        # Getting directories
        for p in packages:
            dirs = p[7].split(";")
            timestamp = 0
            try:
                timestamp = float(p[4])
            except ValueError:
                timestamp = 0
            listOfPackages.append((p[1],p[2],p[3],datetime.fromtimestamp(timestamp),timedelta(seconds=float(p[5])),dirs))

        return listOfPackages

    def getFiles(self,app,deviceID=None,doFilter=True):
        """Getter for files for app
        Parameters: app (string)    app identifier
        """

        if(deviceID is None):
            deviceID = self.deviceID

        files = []
        if(deviceID):
            if(doFilter and "time" in self.filter.keys()):
                 self.cursor.execute('SELECT f.* FROM file f, package p WHERE f.packageid = p.id AND p.identifier = ? AND p.deviceid = ? AND lastmodified >= ? AND lastmodified <= ?',(app,deviceID,self.filter["time"][0],self.filter["time"][1]))
            elif(doFilter and "name" in self.filter.keys()):
                filtering = "%"+self.filter["name"]+"%"
                self.cursor.execute("SELECT f.* FROM file f, package p WHERE f.packageid = p.id AND p.identifier = ? AND p.deviceid = ? AND f.name LIKE ?",(app,deviceID,filtering))  
            else:
                self.cursor.execute('SELECT f.* FROM file f, package p WHERE f.packageid = p.id AND p.identifier = ? AND p.deviceid = ? ',(app,deviceID))

        fileList = self.cursor.fetchall()
        for f in fileList:
            files.append({"name":f[1],"size":f[2],"time":datetime.fromtimestamp(f[3]),"fullPath":f[4]})

        return files

    def getLogs(self,app,deviceID=None):
        """Getter for logs for app
        Parameters: app (string)    app identifier
        """ 
        if(deviceID is None):
            deviceID = self.deviceID

        logs = {}
        self.cursor.execute('SELECT l.* FROM log l, package p WHERE l.packageid = p.id AND p.identifier = ? AND p.deviceid = ? ',(app,deviceID))

        logList = self.cursor.fetchone()
        if(logList):
            logs["package"] = logList[1]
            logs["logcat"] = logList[2]
            logs["batterystats"] = logList[3]
            logs["activity"] = logList[4]
            logs["procstats"] = logList[5]
            logs["appops"] = logList[6]
            logs["notification"] = logList[7]
        return logs


    def getAppName(self,app):
        """Getter for app names for app
        Parameters: app (string)    app identifier
        """        

        self.cursor.execute('SELECT name FROM package WHERE identifier = ?',(app,))
        name = self.cursor.fetchone()
        if(name):
            return name[0]
        else:
            return ""

    def getAppActivityName(self,app):
        """Getter for activity names for app
        Parameters: app (string)    app identifier
        """
        
        self.cursor.execute('SELECT activityname FROM package WHERE identifier = ?',(app,))
        name = self.cursor.fetchone()
        if(name):
            return name[0]
        else:
            return ""
    
    def printDevices(self):
        for d in self.getDevices():
            print(str(d[0])+")\tID: "+d[1]+"\tversion: "+d[2]+"\tarchitecture: "+d[3]+"\tdate: "+str(datetime.fromtimestamp(d[5])))

if __name__ == "__main__":
    path = Path("output/cache.db")
    c = Cache(path)
    print(c.getDevice(1))
    #for d in c.getDevices():
    #    print(str(d[0])+")\tID: "+d[1]+"\tversion: "+d[2]+"\tarchitecture: "+d[3]+"\tdate: "+str(datetime.fromtimestamp(d[5])))
    """
    c.addDevice("a","b","c","d")
    c.addDevice("a","d","r","r")
    c.setDevice("a")
    c.addPackage("identifier", "name", "activityname", datetime(1970,1,1), timedelta(0), "directories")
    c.addFile("name", 0, datetime(1970,1,1), "path", "identifier")
    c.addDevice("b","d","r","r")
    c.setDevice("b")
    c.addPackage("identifier", "name", "activityname", datetime(1970,1,1), timedelta(0), "directories")
    c.addFile("name", 0, datetime(1970,1,1), "path", "identifier")
    c.setDevice("B6R4NZTKAQAMRW59")
    print(c.getAppName("com.android.chrome"))
    print(c.getDevices())
    print(c.getPackages())
    print(c.getLogs("org.coursera.android"))
    for i in c.getLogs("org.coursera.android"):
        for a in i:
            print(a)
    """
