"""This module represents the communication with Android device using adb (Android Debug Bridge).
.. module:: adb
    :platform: linux
    :synopsis: Represents adb communication
.. moduleauthor:: Martin Bazik
"""

from subprocess import PIPE
import subprocess
import sys
import re

class ADB:
    """This is the main class of this module and it takes care of communication with the device.
    Attributes: err (string) text printed on the standard error output
                out (string) text printed on the standard output
                rc  (string) return code
                device (string) identifier of the device
    """
    
    def __init__(self):
        """Constructor
        Sets up object attributes 
        """

        self.err = ""
        self.out = ""
        self.rc = ""
        self.device = ""
        self.root = False
        self.production = False
        try:
            p = subprocess.Popen(["adb"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
            out_r, err_r = p.communicate(b"stdin")
        except FileNotFoundError as err:
            raise ValueError(err)
        
    def connect(self,ip):
        """This method connects the device wirelessly using IP defined in ip.
        Parameters: ip (string) ip of the device
        """
        
        p = subprocess.Popen(["adb","connect",ip], stdin=PIPE, stdout=PIPE, stderr=PIPE)
        out_r, err_r = p.communicate(b"stdin")
        self.rc = p.returncode
        self.out = out_r.decode("utf-8").strip()
        if("No such host is known" in self.out or "unable to connect" in self.out):
            raise ValueError("Unknown IP address!")

    def choose(self):
        """This method is used to choose device we want to work with.
        Input:  STDIN
        """

        # Kill server first
        #p = subprocess.Popen(["adb","kill-server"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
        #out_r, err_r = p.communicate(b"stdin")

        # Looks for connected devices
        p = subprocess.Popen(["adb","devices"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
        out_r, err_r = p.communicate(b"stdin")
        self.rc = p.returncode
        self.out = out_r.decode("utf-8")
        
        items = []
        # Separates devices and their status
        for item in self.out.splitlines():
            foundItem = re.fullmatch("(\S+)\s+(\S+)",item.strip())
            if(foundItem):
                items.append(foundItem.groups())

        if(len(items) == 0):
            print("There are no connected devices. Try to reconnect the device.")
            choice = input("Press enter afterwards: ")
            if(choice == "exit"):
                sys.exit()
            else:
                self.choose()
                return
        # counter of devices
        count = 0

        print("Choose one of the following devices:")
        
        # Lists devices
        for item in items:
            print(str(count) + ") " + item[0] + "\t" + item[1])
            count += 1
       
        # Choosing the device
        choice = input("Choose the device: ")
        if(choice == "exit"):
            sys.exit()
        else:

            # It must be corresponding number
            try:
                c = int(choice)
            except ValueError:
                print("Value must be between 0 and " + str(count-1) + "!")
                self.choose()
                return
            if(c >= count or c < 0):
                print("Value must be between 0 and " + str(count-1) + "!")
                self.choose()
                return
            else:
                self.device = items[c][0]
                print("Chosen device: " + self.device)
            
            if(not self.checkDevice()):
                print("Not a device!")
                self.choose()

    def setDevice(self,device):
        self.device = device
        return self.checkDevice()

    def listDevices(self):
        # Looks for connected devices
        p = subprocess.Popen(["adb","devices"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
        out_r, err_r = p.communicate(b"stdin")
        self.rc = p.returncode
        self.out = out_r.decode("utf-8")
        
        items = []
        # Separates devices and their status
        for item in self.out.splitlines():
            foundItem = re.fullmatch("(\S+)\s+(\S+)",item.strip())
            if(foundItem):
                items.append(foundItem.groups())

        return items

    def checkDevice(self):
        """Checks state of the given device
        """
        
        #self.query(["get-state"])
        p = subprocess.Popen(["adb","-s",self.device,"get-state"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
        out_r, err_r = p.communicate(b"stdin")
        self.out = out_r.decode("utf-8")
        if("device" in self.out):
            return True
        else:
            return False

    def getDevice(self):
        """Getter that returns identifier of the device
        """

        return self.device


    def commandExists(self,command):
        """Check existence of the given shell command on the device
        Returns: True if the command is present
        """

        self.query(["shell"] + command)
        if("not found" in self.out or "not found" in self.err):
            return False
        else:
            return True
    
    def query(self,query):
        """This method caries out the adb command on the selected device.
        Parameters: query (string)  adb command
        Output:     err (string) error output
                    out (string) standard output
                    rc  (string) return code
        """
        
        if(not self.checkDevice()):
            raise ValueError("Device is not connected!")

        # It is a shell command that has to be treated differently on the production builds
        # because adb cannot be run in root mode
        if(query[0] == "shell"):
            if(self.production == False):
                p = subprocess.Popen(["adb","-s",self.device] + query, stdin=PIPE, stdout=PIPE, stderr=PIPE)
            else:
                p = subprocess.Popen(["adb","-s",self.device,"shell","su","-c"] + query[1:], stdin=PIPE, stdout=PIPE, stderr=PIPE)
            out_r, err_r = p.communicate(b"stdin")
            self.out = out_r.decode("utf-8")
            self.err = err_r.decode("utf-8")
            if(len(self.out) >= 6):
                if(self.out[:5] == "stdin"):
                    self.out = self.out[5:]

        
        elif(query[0] == "pull" or query[0] == "push"):
            # Pull command can access root files only when it is run under adb in root mode
            
            numberOfOutput = 1
            if(query[0] == "push"):
                numberOfOutput = 2

            # /sdcard partition does not need root access
            if("/sdcard" == query[numberOfOutput][:7] or not self.production):
                p = subprocess.Popen(["adb","-s",self.device] + query, stdin=PIPE, stdout=PIPE, stderr=PIPE)
                out_r, err_r = p.communicate(b"stdin")
                self.out = out_r.decode("utf-8")
            else:
                # In the production build it is required to move the file into temporary directory
                # before it can be pulled from the file


                # Looks for temporary direcotry /sdcard/tmp
                p = subprocess.Popen(["adb","-s",self.device,"shell","ls","/sdcard/tmp"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
                out_r, err_r = p.communicate(b"stdin")
                existTmp = out_r.decode("utf-8")


                # Creates temporary direcotry /sdcard/tmp
                if("No such file or directory" in existTmp):

                    p = subprocess.Popen(["adb","-s",self.device,"shell","su","-c","mkdir","/sdcard/tmp"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
                    out_r, err_r = p.communicate(b"stdin")


                if(query[0] == "pull"):
                    # Copies file to the temporary directory
                    p = subprocess.Popen(["adb","-s",self.device,"shell","su","-c","cp","-r","\""+query[1]+"\"","/sdcard/tmp"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
                    out_r, err_r = p.communicate(b"stdin")
                    
                    # Gets name of the file that is being pulled
                    p = subprocess.Popen(["adb","-s",self.device,"shell","ls","/sdcard/tmp/"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
                    out_r, err_r = p.communicate(b"stdin")
                    fileName = out_r.decode("utf-8").strip()
                    if(fileName[:5] == "stdin"):
                        fileName = fileName[5:].strip()

                    # Pulls the file from temporary directory
                    p = subprocess.Popen(["adb","-s",self.device,"pull","/sdcard/tmp/"+fileName,query[2]], stdin=PIPE, stdout=PIPE, stderr=PIPE)
                    out_r, err_r = p.communicate(b"stdin")
                else: 
                    # Pushes the file to temporary directory
                    p = subprocess.Popen(["adb","-s",self.device,"push",query[1],"/sdcard/tmp/tmpFile"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
                    out_r, err_r = p.communicate(b"stdin")

                    # Copies file to the temporary directory
                    p = subprocess.Popen(["adb","-s",self.device,"shell","su","-c","cp","/sdcard/tmp/tmpFile","\""+query[2]+"\"",], stdin=PIPE, stdout=PIPE, stderr=PIPE)
                    out_r, err_r = p.communicate(b"stdin")
                    

                # Removes the file from the temporary directory
                p = subprocess.Popen(["adb","-s",self.device,"shell","rm","-rf","/sdcard/tmp/*"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
                out_r, err_r = p.communicate(b"stdin")

        else:
            p = subprocess.Popen(["adb","-s",self.device] + query, stdin=PIPE, stdout=PIPE, stderr=PIPE)
            out_r, err_r = p.communicate(b"stdin")
            self.out = out_r.decode("utf-8")

        self.rc = p.returncode
        self.err = err_r.decode("utf-8")


    def checkRoot(self):
        """Checks if the device has a root access and if it is a production build
        """
        self.query(["shell","getprop","ro.debuggable"])
        if("0" in self.out):
            self.production = True
            self.query(["shell","su","-c","ls"])
            if(not "Unallowed user" in self.out):
                self.root = True
            else:
                self.root = False
        else:
            self.query(["root"])
            self.production = False
            self.root = True
        return self.root

    
    def getOutput(self):
        """This method returns standard output of the last adb command
        """

        return self.out

    def getErrorOutput(self):
        """This method returns error output of the last adb command
        """

        return self.err

    def getReturnCode(self):
        """This method returns return code of the last adb command
        """

        return self.rc

    def checkEOL(self):
        self.query(["shell","ls"])
        if("\r\r\n" in self.out):
            return "\r\r\n"
        elif("\r\n" in self.out):
            return "\r\n"
        else:
            return "\n"
    
    def checkArch(self):
        self.query(["shell","getprop","ro.product.cpu.abi"])
        if "arm" in self.out:
            self.arch = "arm"
        elif "x86" in self.out:
            self.arch = "x86"
        else:
            self.arch = ""
        return self.arch

    def getArch(self):
        return self.arch
      

# Debug
if __name__ == "__main__":
    adb = ADB()
    adb.choose()
    adb.checkRoot()
    adb.checkArch()
    if(adb.getArch() == "x86"):
        adb.query(["push","aapt/aapt-x86-pie","/data/local/aapt"])
    elif(adb.getArch() == "arm"):
        adb.query(["push","aapt/aapt-arm-pie","/data/local/aapt"])
    print(adb.getOutput())
    #adb.query(['pull', '/system/plugin/FwkPlugin/FwkPlugin.apk', 'base.apk'])
    #adb.query(["shell","shell","ls"])
    #adb.query(["shell","dumpsys","procstats","--full-details","--hours","24"])
    #print("\r\n" in adb.checkEOL())
    #print(re.findall("  \* com.android.location.fused [\S\s]*?:\n([\S\s]*?)(?:\n  \*|\n\n)",adb.getOutput()))
