/*
								+---------------------------------+
								|								  |
								|	 ***   OpenGL ES 2.0   ***	  |
								|								  |
								|  Copyright   -tHE SWINe- 2011  |
								|								  |
								|			GLES20Emu.h			  |
								|								  |
								+---------------------------------+
*/

#pragma once
#ifndef __OPENGLES2_EMULATOR_INCLUDED
#define __OPENGLES2_EMULATOR_INCLUDED

/**
 *	@file gles2/GLES20Emu.h
 *	@author -tHE SWINe-
 *	@brief OpenGL ES 2.0 emulation
 *	@date 2011
 *
 *	This header contains entry points of relevant OpenGL ES 2 functions / extensions.
 *		It was prepared from GLES2/gl2.h, available from <www.opengl.org>
 *
 *	@date 2011-05-04
 *
 *	Added GL_AUTOCHECK functionality (note to self - sadly, this is unfeasible for
 *	OpenGL 4.0 as it requires parameter names to be present in the function prototypes,
 *	and they are regrettably missing very often).
 *
 *	@date 2012-02-20
 *
 *	Added the following Tegra 3 extensions:
 *
 *	GL_EXT_texture_array
 *	GL_EXT_occlusion_query_boolean
 *	GL_EXT_robustness
 *	GL_NV_read_depth_stencil
 *	GL_NV_texture_compression_s3tc_update
 *	GL_EXT_unpack_subimage
 *	GL_EXT_texture_filter_anisotropic
 *	GL_OES_standard_derivatives
 *	GL_OES_compressed_ETC1_RGB8_texture
 *	GL_OES_texture_float
 *	GL_OES_vertex_half_float
 *	GL_OES_fbo_render_mipmap
 *
 *	@date 2012-06-19
 *
 *	Added \#pragma once.
 *
 */

/*
** Copyright (c) 2007-2009 The Khronos Group Inc.
** 
** Permission is hereby granted, free of charge, to any person obtaining a
** copy of this software and/or associated documentation files (the
** "Materials"), to deal in the Materials without restriction, including
** without limitation the rights to use, copy, modify, merge, publish,
** distribute, sublicense, and/or sell copies of the Materials, and to
** permit persons to whom the Materials are furnished to do so, subject to
** the following conditions:
** 
** The above copyright notice and this permission notice shall be included
** in all copies or substantial portions of the Materials.
** 
** THE MATERIALS ARE PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
** EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
** MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
** IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
** CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
** MATERIALS OR THE USE OR OTHER DEALINGS IN THE MATERIALS.
*/

/*
** License Applicability. Except to the extent portions of this file are
** made subject to an alternative license as permitted in the SGI Free
** Software License B, Version 1.1 (the "License"), the contents of this
** file are subject only to the provisions of the License. You may not use
** this file except in compliance with the License. You may obtain a copy
** of the License at Silicon Graphics, Inc., attn: Legal Services, 1600
** Amphitheatre Parkway, Mountain View, CA 94043-1351, or at:
** 
** http://oss.sgi.com/projects/FreeB
** 
** Note that, as provided in the License, the Software is distributed on an
** "AS IS" basis, with ALL EXPRESS AND IMPLIED WARRANTIES AND CONDITIONS
** DISCLAIMED, INCLUDING, WITHOUT LIMITATION, ANY IMPLIED WARRANTIES AND
** CONDITIONS OF MERCHANTABILITY, SATISFACTORY QUALITY, FITNESS FOR A
** PARTICULAR PURPOSE, AND NON-INFRINGEMENT.
** 
** Original Code. The Original Code is: OpenGL Sample Implementation,
** Version 1.2.1, released January 26, 2000, developed by Silicon Graphics,
** Inc. The Original Code is Copyright (c) 1991-2004 Silicon Graphics, Inc.
** Copyright in any portions created by third parties is as indicated
** elsewhere herein. All Rights Reserved.
** 
** Additional Notice Provisions: This software was created using the
** OpenGL(R) version 1.2.1 Sample Implementation published by SGI, but has
** not been independently verified as being compliant with the OpenGL(R)
** version 1.2.1 Specification.
*/

#include <string>
#include <stdexcept>
#include <cstdarg>
#include <cstdio>
#include <cstdlib>

#include "../StlUtils.h"
#include "../Integer.h"

#if defined(_WIN32) || defined(_WIN64)
#define WIN32_LEAND_AND_MEAN 1
#include <windows.h>
#endif // _WIN32 || _WIN64

/*
 *	makes life a little bit easier when including gl.h by mistake
 */
#if defined(__gl2_h_) || defined(__GL_H__) || defined(GL_VERSION_1_1)
#error("gl2.h must NOT be included in files using GLES20Emu.h")
#endif // __gl2_h_ || __GL_H__ || GL_VERSION_1_1

/*
 *	makes life a little bit easier when including glext.h by mistake
 */
#if defined(__glext_h_) || defined(GL_GLEXT_VERSION)
#error("glext.h must NOT be included in files using GLES20Emu.h")
#endif // __glext_h_ || GL_GLEXT_VERSION

#define __gl2_h_
#define __gl_h_
#define __GL_H__
// does not include gl.h, defines GL up to 1.1 below, want to prevent gl.h
// header inclusion

#define __glext_h_
#define __GLEXT_H__
// glext.h is not included anymore, it does contain significant amount of
// deprecated functionality, want to prevent glext.h header inclusion

/**
 *	@def NULL_FUNCTION_POINTERS
 *	@brief if defined, initializes entry function pointers to null
 *
 *	In case NULL_FUNCTION_POINTERS is defined (default), all extension function pointers
 *		are initialized to 0, but user never needs to compare individual function values
 *		so it's posible to save some space in executable file by skipping the initialization.
 */
#define NULL_FUNCTION_POINTERS

/**
 *	@def FIX_GL_FUNCTION_LINKING_CONFLICTS
 *	@brief avoids function linking conflicts by renaming function pointers, and fixing
 *		names using \#define
 *
 *	Some compilers may contain newer OpenGL libraries than this header requires
 *		(OpenGL 1.1, as supported by Microsoft Visual studio family), so names of function
 *		pointers, such as <tt>glBlendColor</tt> are present in static library, as well as in this
 *		file, causing link-time error. This is fixed by renaming the function, here
 *		<tt>glBlendColor_ptr</tt> would be used, and fixing it by macro, such as <tt>\#define
 *		glBlendColor_ptr glBlendColor</tt>. The downside of this is, the Visual Studio edior
 *		will no longer show pop-up function parameters (intellisense), so this is best
 *		avoided if possible.
 */
#define FIX_GL_FUNCTION_LINKING_CONFLICTS

/**
 *	@def __GL_AUTOCHECK
 *	@brief automatically checks for OpenGL errors after every OpenGL function except glGetError()
 *	@note This is only available in debug mode unless FORCE_GL_CHECK is defined.
 */
#define __GL_AUTOCHECK

/**
 *	@def __GL_AUTOCHECK_USE_EXCEPTION
 *	@brief if defined, autocheck throws std::runtime_exception if an OpenGL error occurs
 */
//#define __GL_AUTOCHECK_USE_EXCEPTION

/**
 *	@def __GL_AUTOCHECK_USE_BREAKPOINT
 *	@brief if defined, autocheck calls user breakpoint if an OpenGL error occurs
 *	@note This is only available in windows / MSVC and in x86 mode (not x64).
 */
#define __GL_AUTOCHECK_USE_BREAKPOINT

/**
 *	@def __GL_AUTOCHECK_USE_ABORT
 *	@brief if defined, autocheck calls clib's abort() if an OpenGL error occurs
 */
//#define __GL_AUTOCHECK_USE_ABORT

/**
 *	@def __GL_AUTOCHECK_USE_EXIT
 *	@brief if defined, autocheck prints the error to stderr and calls
 *		exit(-1) if an OpenGL error occurs
 *	@note This is default behavior if nothing else is defined.
 */
//#define __GL_AUTOCHECK_USE_EXIT

/**
 *	@brief class for handling OpenGL extensions
 *
 *	Supplies means to get OpenGL function entry point, check extension support, or get
 *	OpenGL version. Functions with name n_Get*FuncPointers() returns number of function
 *	pointers that were not found (zero is success).
 */
class CGLES20ExtensionHandler {
public:
	/**
	 *	@brief compares OpenGL version to reference
	 *
	 *	Compares supplied version to current OpenGL implementation version and returns
	 *		true if current version is greater or equal (supplied version is supported).
	 *
	 *	@param[in] n_min_major is reference minimal major version number
	 *	@param[in] n_min_minor is reference minimal minor version number
	 *
	 *	@return Returns true in case OpenGL version	 is greater or equal to n_min_major.n_min_minor.
	 */
	static bool b_Support_OpenGL(int n_min_major = 3, int n_min_minor = 0);

	/**
	 *	@brief returns OpenGL version
	 *
	 *	Returns OpenGL version as (comparable) integer value.
	 *
	 *	@return Returns OpenGL version as major * 10 + minor (ie. for OpenGL 3.2 returns 32).
	 */
	static int n_OpenGL_Version();

	/**
	 *	@brief returns OpenGL version major
	 *
	 *	@return Returns OpenGL version as major (ie. for OpenGL 3.2 returns 3).
	 */
	static int n_OpenGL_Version_Major();

	/**
	 *	@brief returns OpenGL version minor
	 *
	 *	@return Returns OpenGL version as minor (ie. for OpenGL 3.2 returns 2).
	 */
	static int n_OpenGL_Version_Minor();

	/**
	 *	@brief returns OpenGL version as (major, minor) pair
	 *
	 *	@return Returns OpenGL version.
	 */
	static std::pair<int, int> t_OpenGL_Version();

	/**
	 *	@brief checks extension support
	 *
	 *	Checks support of a given OpenGL extension, returns true if extension is supported.
	 *
	 *	@param[in] p_s_extension_name is null-terminated string, containing name
	 *		of OpenGL extension encoded in UTF-8
	 *
	 *	@return Returns true in case p_s_extension_name is supported, otherwise false.
	 */
	static bool b_SupportedExtension(const char *p_s_extension_name);

	/**
	 *	@brief checks (platform-specific) windowing system extension support
	 *
	 *	Checks support of a given WGL (GLX) extension, returns true if extension is supported.
	 *
	 *	@param[in] p_s_extension_name is null-terminated string, containing name
	 *		of WGL (GLX) extension encoded in UTF-8
	 *
	 *	@return Returns true in case p_s_extension_name is supported, otherwise false.
	 */
	static bool b_SupportedPlatformExtension(const char *p_s_extension_name);

	/**
	 *	@brief gets extension string
	 *
	 *	OpenGL 3.0 doesn't have classic extension string, it returns names of individual
	 *		extensions instead. This is to prevent buffer overruns, people always seem
	 *		to be copying OpenGL extensions to fixed-size buffers, causing older software
	 *		from times when extension string was shorter to fail on modern hardware.
	 *		This functions returns "classic" OpenGL extension string in std::string,
	 *		encoded in UTF-8, names of individual extensions are separated by space (' ')
	 *		character.
	 *
	 *	@return Returns true on success, false on failure (not enough memory).
	 */
	static bool Get_ExtensionString(std::string &r_s_extension_list);

	/**
	 *	@brief gets entry points of all extensions
	 *
	 *	Gets pointers of function entry points for all supported extensions and OpenGL versions.
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note Note this is aimed at simple and test implementations without
	 *		much need for error checking. it is always better to call individual
	 *		functions, because in case this fails, there's no telling
	 *		which extension exactly failed to initialize.
	 */
	static int n_GetAllSupportedExtensionsFuncPointers();

	/**
	 *	@brief gets entry points of all OpenGL ES 2.0 core functions
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetGLES20FuncPointers();

	/**
	 *	@brief gets GL_ARB_vertex_array_object function pointers, and sets GLEH_ARB_vertex_array_object
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetARBVertexArrayObjectFuncPointers();

	/**
	 *	@brief gets GL_OES_texture_3D function pointers, and sets GLEH_OES_texture_3D
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetOESTexture3DFuncPointers();

	/**
	 *	@brief gets GL_OES_depth24 function pointers, and sets GLEH_OES_depth24
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESDepth24FuncPointers();

	/**
	 *	@brief gets GL_OES_depth32 function pointers, and sets GLEH_OES_depth32
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESDepth32FuncPointers();

	/**
	 *	@brief gets GL_OES_element_index_uint function pointers, and sets GLEH_OES_element_index_uint
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESElementIndexUIntFuncPointers();

	/**
	 *	@brief gets GL_OES_mapbuffer function pointers, and sets GLEH_OES_mapbuffer
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetOESMapbufferFuncPointers();

	/**
	 *	@brief gets GL_OES_rgb8_rgba8 function pointers, and sets GLEH_OES_rgb8_rgba8
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESRGB8RGBA8FuncPointers();

	/**
	 *	@brief gets GL_OES_stencil1 function pointers, and sets GLEH_OES_stencil1
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESStencil1FuncPointers();

	/**
	 *	@brief gets GL_OES_stencil4 function pointers, and sets GLEH_OES_stencil4
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESStencil4FuncPointers();

	/**
	 *	@brief gets GL_OES_stencil8 function pointers, and sets GLEH_OES_stencil8
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESStencil8FuncPointers();

	/**
	 *	@brief gets GL_OES_depth_texture function pointers, and sets GLEH_OES_depth_texture
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESDepthTextureFuncPointers();

	/**
	 *	@brief gets GL_OES_packed_depth_stencil function pointers, and sets GLEH_OES_packed_depth_stencil
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESPackedDepthStencilFuncPointers();

	/**
	 *	@brief gets GL_OES_get_program_binary function pointers, and sets GLEH_OES_get_program_binary
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetOESGetProgramBinaryFuncPointers();

	/**
	 *	@brief gets GL_OES_fbo_render_mipmap function pointers, and sets GLEH_OES_fbo_render_mipmap
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESFboRenderMipmapFuncPointers();

	/**
	 *	@brief gets GL_OES_vertex_half_float function pointers, and sets GLEH_OES_vertex_half_float
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESVertexHalfFloatFuncPointers();

	/**
	 *	@brief gets GL_OES_texture_float function pointers, and sets GLEH_OES_texture_float
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESTextureFloatFuncPointers();

	/**
	 *	@brief gets GL_OES_compressed_ETC1_RGB8_texture function pointers, and sets GLEH_OES_compressed_ETC1_RGB8_texture
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESCompressedETC1RGB8TextureFuncPointers();

	/**
	 *	@brief gets GL_OES_standard_derivatives function pointers, and sets GLEH_OES_standard_derivatives
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetOESStandardDerivativesFuncPointers();

	/**
	 *	@brief gets GL_EXT_texture_format_BGRA8888 function pointers, and sets GLEH_EXT_texture_format_BGRA8888
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetEXTTextureFormatBGRA8888FuncPointers();

	/**
	 *	@brief gets GL_NV_fence function pointers, and sets GLEH_NV_fence
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetNVFenceFuncPointers();

	/**
	 *	@brief gets GL_EXT_texture_lod_bias function pointers, and sets GLEH_EXT_texture_lod_bias
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetEXTTextureLODBiasFuncPointers();

	/**
	 *	@brief gets GL_NV_draw_buffers function pointers, and sets GLEH_NV_draw_buffers
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetNVDrawBuffersFuncPointers();

	/**
	 *	@brief gets GL_NV_fbo_color_attachments function pointers, and sets GLEH_NV_fbo_color_attachments
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetNVFBOColorAttachmentsFuncPointers();

	/**
	 *	@brief gets GL_NV_read_buffer function pointers, and sets GLEH_NV_read_buffer
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetNVReadBufferFuncPointers();

	/**
	 *	@brief gets GL_EXT_texture_filter_anisotropic function pointers, and sets GLEH_EXT_texture_filter_anisotropic
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetEXTTextureFilterAnisotropicFuncPointers();

	/**
	 *	@brief gets GL_EXT_unpack_subimage function pointers, and sets GLEH_EXT_unpack_subimage
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetEXTUnpackSubimageFuncPointers();

	/**
	 *	@brief gets GL_NV_texture_compression_s3tc_update function pointers, and sets GLEH_NV_texture_compression_s3tc_update
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetNVTextureCompressionS3tcUpdateFuncPointers();

	/**
	 *	@brief gets GL_NV_read_depth_stencil function pointers, and sets GLEH_NV_read_depth_stencil
	 *
	 *	@return Returns number of missing functions (zero means success).
	 *
	 *	@note this extension has no functions, so this always returns zero.
	 */
	static int n_GetNVReadDepthStencilFuncPointers();

	/**
	 *	@brief gets GL_EXT_robustness function pointers, and sets GLEH_EXT_robustness
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetEXTRobustnessFuncPointers();

	/**
	 *	@brief gets GL_EXT_occlusion_query_boolean function pointers, and sets GLEH_EXT_occlusion_query_boolean
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetEXTOcclusionQueryBooleanFuncPointers();

	/**
	 *	@brief gets GL_EXT_texture_array function pointers, and sets GLEH_EXT_texture_array
	 *
	 *	@return Returns number of missing functions (zero means success).
	 */
	static int n_GetEXTTextureArrayFuncPointers();
};

#ifdef FIX_GL_FUNCTION_LINKING_CONFLICTS
#define __ConCat__(a,b) a##b
#define GLFuncPtrName(f) __ConCat__(f,_ptr)
#else // FIX_GL_FUNCTION_LINKING_CONFLICTS
#define GLFuncPtrName(f) f
#endif // FIX_GL_FUNCTION_LINKING_CONFLICTS

#include <GLES2/gl2platform.h> // ...

typedef void			 GLvoid;
typedef char			 GLchar;
typedef unsigned int	 GLenum;
typedef unsigned char	 GLboolean;
typedef unsigned int	 GLbitfield;
typedef khronos_int8_t	 GLbyte;
typedef short			 GLshort;
typedef int				 GLint;
typedef int				 GLsizei;
typedef khronos_uint8_t	 GLubyte;
typedef unsigned short	 GLushort;
typedef unsigned int	 GLuint;
typedef khronos_float_t	 GLfloat;
typedef khronos_float_t	 GLclampf;
typedef khronos_int32_t	 GLfixed;

/* GL types for handling large vertex buffer objects */
typedef khronos_intptr_t GLintptr;
typedef khronos_ssize_t	 GLsizeiptr;
// base OpenGL data types

typedef ptrdiff_t GLintptrARB;
typedef ptrdiff_t GLsizeiptrARB;
// GL_ARB_vertex_buffer_object data types (GL types for handling large vertex buffer objects)

typedef char GLcharARB;
typedef unsigned int GLhandleARB;
// GL_ARB_shader_objects data types (GL types for handling shader object handles and program/shader text)

typedef unsigned short GLhalfARB;
// GL_ARB_half_float_pixel data types ("half" precision (s10e5) float data in host memory)

typedef unsigned short GLhalfNV;
// GL_NV_half_float data types ("half" precision (s10e5) float data in host memory)

typedef int64_t GLint64EXT;
typedef uint64_t GLuint64EXT;
// GL_EXT_timer_query data types

typedef int64_t GLint64;
typedef uint64_t GLuint64;
typedef struct __GLsync *GLsync;
// GL_ARB_sync data types

typedef unsigned short GLhalf;
// GL_NV_half_float data types

/**
 *	@def n_GL_TypeSize
 *	@brief determines size of OpenGL data type
 *	@param[in] n_type is OpenGL (internal) data type
 *	@return Returns size of the given type, or 0 in case n_type is not a valid internal type.
 */
#define n_GL_TypeSize(n_type) \
	((false/*(n_type) == GL_DOUBLE*/)? 8 : \
	((n_type) == GL_FLOAT || (n_type) == GL_INT || (n_type) == GL_UNSIGNED_INT)? 4 : \
	(/*(n_type) == GL_HALF_FLOAT ||*/ (n_type) == GL_SHORT || (n_type) == GL_UNSIGNED_SHORT)? 2 : \
	((n_type) == GL_BYTE || (n_type) == GL_UNSIGNED_BYTE)? 1 : 0)

#ifndef APIENTRY
#define ApiEntry
#else // !APIENTRY
#define ApiEntry APIENTRY
#endif // !APIENTRY
#ifndef GLStaticApi
#define GLStaticApi extern
#endif // !GLStaticApi
// defines for OpenGL 1.0 / OpenGL 1.1 (staticaly linked) functions

#if defined(_WIN32) || defined(_WIN64)
#define GLApi __stdcall
#else // _WIN32 || _WIN64
#define GLApi 
#endif // _WIN32 || _WIN64
// defines for newer (dynamicaly linked) functions

#include <string>
#include <stdexcept>

#if defined(__GL_AUTOCHECK) && (defined(_DEBUG) || defined(__GL_FORCE_AUTOCHECK))

/**
 *	@def __HAVE_GL_AUTOCHECK
 *	@brief defined in case GL_AUTOCHECK_BLOCK is not empty
 */
#define __HAVE_GL_AUTOCHECK

#ifndef FIX_GL_FUNCTION_LINKING_CONFLICTS
#define __ConCat__(a,b) a##b
#endif // !FIX_GL_FUNCTION_LINKING_CONFLICTS

/**
 *	@def GLAutoCheckFuncPtrName
 *	@brief creates name for a _glachwrap function
 *	@param[in] fn is a name of an OpenGL function
 */
#define GLAutoCheckFuncPtrName(fn) __ConCat__(fn,_glachwrap)

/**
 *	@def GLAutoCheckStaticFuncName
 *	@brief creates name for a _glachwrap function
 *	@param[in] fn is a name of a statically linked OpenGL function
 */
#define GLAutoCheckStaticFuncName(fn) __ConCat__(fn,_glachwrap)

/**
 *	@def GLAutoCheckExtraParams
 *	@brief additional parameters that are needed for the autocheck functions
 */
#define GLAutoCheckExtraParams , __FILE__, __LINE__

/**
 *	@def GLAutoCheckExtraParamsSA
 *	@brief standalone parameters that are needed for the autocheck functions
 */
#define GLAutoCheckExtraParamsSA __FILE__, __LINE__

/**
 *	@def GLAutoCheckFire
 *	@brief OpenGL autocheck error event handler
 *	@param[in] what is exception description
 */
#if defined(__GL_AUTOCHECK_USE_EXCEPTION)
#define GLAutoCheckFire(what) throw std::runtime_error(what)
#elif defined(__GL_AUTOCHECK_USE_BREAKPOINT)
#if defined(_WIN32) || defined(_WIN64) 
#if defined(_M_X64) || defined(_M_AMD64) || defined(_M_IA64) || defined(__x86_64) || defined(__amd64) || defined(__ia64)
#include <math.h> // otherwise get "warning C4985: 'ceil': attributes not present on previous declaration."
#include <intrin.h>
#define GLAutoCheckFire(what) do { fprintf(stderr, "GL_AUTOCHECK: %s\n", (what)); __debugbreak(); } while(0)
#else // _M_X64 || _M_AMD64 || _M_IA64 || __x86_64 || __amd64 || __ia64
#define GLAutoCheckFire(what) do { fprintf(stderr, "GL_AUTOCHECK: %s\n", (what)); __asm { int 3 } } while(0)
#endif // _M_X64 || _M_AMD64 || _M_IA64 || __x86_64 || __amd64 || __ia64
#else // _WIN32 || _WIN64
#include <signal.h>
#ifdef SIGTRAP
#define GLAutoCheckFire(what) do { fprintf(stderr, "GL_AUTOCHECK: %s\n", (what)); raise(SIGTRAP); } while(0)
#else // SIGTRAP
#define GLAutoCheckFire(what) do { fprintf(stderr, "GL_AUTOCHECK: %s\n", (what)); abort(); } while(0)
#endif // SIGTRAP
// posix compliant
#endif // _WIN32 || _WIN64
#elif defined(__GL_AUTOCHECK_USE_ABORT)
#define GLAutoCheckFire(what) do { fprintf(stderr, "GL_AUTOCHECK: %s\n", (what)); abort(); } while(0)
#else // if defined(__GL_AUTOCHECK_USE_EXIT)
#define GLAutoCheckFire(what) do { fprintf(stderr, "GL_AUTOCHECK: %s\n", (what)); exit(-1); } while(0)
#endif // __GL_AUTOCHECK_USE_EXCEPTION

/**
 *	@def GL_AUTOCHECK_BLOCK
 *
 *	@param[in] p_s_function is the OpenGL function being called
 *	@param[in] p_s_file is name of file of invocation
 *	@param[in] n_line is (one-based) index of source code line of invocation
 *
 *	@brief checks for OpenGL errors and throws std::runtime_error
 *		with error descritption in case any error occured
 *	@note This is only available in debug mode unless __GL_FORCE_AUTOCHECK is defined.
 */
#if defined(__STL_UTILS_INCLUDED)

#define GL_AUTOCHECK_BLOCK(p_s_function,p_s_file,n_line) \
	do { \
		GLenum n_error; \
		if((n_error = glGetError()) == GL_NO_ERROR) \
			break;	\
		std::string s_message; \
		switch(n_error) { \
		case GL_INVALID_ENUM: \
			stl_ut::Format(s_message, "error: file \'" "%s" "\', line %d, function %s()" \
				": OpenGL error: GL_INVALID_ENUM", p_s_file, n_line, p_s_function); \
			GLAutoCheckFire(s_message.c_str()); \
		case GL_INVALID_VALUE: \
			stl_ut::Format(s_message, "error: file \'" "%s" "\', line %d, function %s()" \
				": OpenGL error: GL_INVALID_VALUE", p_s_file, n_line, p_s_function); \
			GLAutoCheckFire(s_message.c_str()); \
		case GL_INVALID_OPERATION: \
			stl_ut::Format(s_message, "error: file \'" "%s" "\', line %d, function %s()" \
				": OpenGL error: GL_INVALID_OPERATION", p_s_file, n_line, p_s_function); \
			GLAutoCheckFire(s_message.c_str()); \
		case GL_OUT_OF_MEMORY: \
			stl_ut::Format(s_message, "error: file \'" "%s" "\', line %d, function %s()" \
				": OpenGL error: GL_OUT_OF_MEMORY", p_s_file, n_line, p_s_function); \
			GLAutoCheckFire(s_message.c_str()); \
		default: \
			stl_ut::Format(s_message, "error: file \'" "%s" "\', line %d, function %s()" \
				": OpenGL error: 0x%04x", p_s_file, n_line, p_s_function, n_error); /* this shouldn't really happen */ \
			GLAutoCheckFire(s_message.c_str()); \
		}; \
	} while(0)
// use stl utils' Format()

#elif defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER >= 1400

#define GL_AUTOCHECK_BLOCK(p_s_function,p_s_file,n_line) \
	do { \
		GLenum n_error; \
		if((n_error = glGetError()) != GL_NO_ERROR) { \
			char p_s_buffer[16]; \
			const char *p_s_error; \
			switch(n_error) { \
			case GL_INVALID_ENUM: \
				p_s_error = "GL_INVALID_ENUM"; \
				break; \
			case GL_INVALID_VALUE: \
				p_s_error = "GL_INVALID_VALUE"; \
				break; \
			case GL_INVALID_OPERATION: \
				p_s_error = "GL_INVALID_OPERATION"; \
				break; \
			case GL_OUT_OF_MEMORY: \
				p_s_error = "GL_OUT_OF_MEMORY"; \
				break; \
			default: \
				sprintf(p_s_buffer, "0x%04x", n_error); /* this shouldn't really happen */ \
				p_s_error = p_s_buffer; \
			}; \
			const char *p_s_fmt = "error: file \'%s\', line %d, " \
				"function %s(): OpenGL error: %s"; \
			std::string s_message; \
			s_message.resize(strlen(p_s_fmt) + strlen(p_s_file) + \
				16 + strlen(p_s_function) + strlen(p_s_error)); \
			_snprintf(&s_message[0], s_message.length() * sizeof(char), \
				p_s_fmt, p_s_file, n_line, p_s_function, p_s_error); \
			GLAutoCheckFire(s_message.c_str()); \
		} \
	} while(0)
// use snprintf()

#else // _MSC_VER && !__MWERKS__ && _MSC_VER >= 1400

#define GL_AUTOCHECK_BLOCK(p_s_function,p_s_file,n_line) \
	do { \
		GLenum n_error; \
		if((n_error = glGetError()) != GL_NO_ERROR) { \
			char p_s_buffer[16]; \
			const char *p_s_error; \
			switch(n_error) { \
			case GL_INVALID_ENUM: \
				p_s_error = "GL_INVALID_ENUM"; \
				break; \
			case GL_INVALID_VALUE: \
				p_s_error = "GL_INVALID_VALUE"; \
				break; \
			case GL_INVALID_OPERATION: \
				p_s_error = "GL_INVALID_OPERATION"; \
				break; \
			case GL_OUT_OF_MEMORY: \
				p_s_error = "GL_OUT_OF_MEMORY"; \
				break; \
			default: \
				sprintf(p_s_buffer, "0x%04x", n_error); /* this shouldn't really happen */ \
				p_s_error = p_s_buffer; \
			}; \
			const char *p_s_fmt = "error: file \'%s\', line %d, " \
				"function %s(): OpenGL error: %s"; \
			std::string s_message; \
			s_message.resize(strlen(p_s_fmt) + strlen(p_s_file) + \
				16 + strlen(p_s_function) + strlen(p_s_error)); \
			sprintf(&s_message[0], p_s_fmt, p_s_file, n_line, p_s_function, p_s_error); \
			GLAutoCheckFire(s_message.c_str()); \
		} \
	} while(0)
// use unsafe sprintf()

#endif // _MSC_VER && !__MWERKS__ && _MSC_VER >= 1400

#else // __GL_AUTOCHECK && (_DEBUG || __GL_FORCE_AUTOCHECK)

/**
 *	@def GLAutoCheckFuncPtrName
 *	@brief creates name for a _glachwrap function
 *	@param[in] fn is a name of an OpenGL function
 */
#define GLAutoCheckFuncPtrName(fn) GLFuncPtrName(fn)

/**
 *	@def GLAutoCheckStaticFuncName
 *	@brief creates name for a _glachwrap function
 *	@param[in] fn is a name of a statically linked OpenGL function
 */
#define GLAutoCheckStaticFuncName(fn) fn

/**
 *	@def GLAutoCheckExtraParams
 *	@brief additional parameters that are needed for the autocheck functions
 */
#define GLAutoCheckExtraParams 

/**
 *	@def GLAutoCheckExtraParamsSA
 *	@brief standalone parameters that are needed for the autocheck functions
 */
#define GLAutoCheckExtraParamsSA 

/**
 *	@def GL_AUTOCHECK_BLOCK
 *
 *	@param[in] p_s_function is the OpenGL function being called
 *	@param[in] p_s_file is name of file of invocation
 *	@param[in] n_line is (one-based) index of source code line of invocation
 *
 *	@brief has no effect (_DEBUG or __GL_FORCE_AUTOCHECK are not defined).
 */
#define GL_AUTOCHECK_BLOCK(p_s_function,p_s_file,n_line) do {} while(0)

#endif // __GL_AUTOCHECK && (_DEBUG || __GL_FORCE_AUTOCHECK)

//								--- OpenGL ES 2.0 emulator helpers ---

/**
 *	@brief enables wire-frame rendering
 *	@note This is only available in the emulator for debugging
 *		purposes, and is not an OpenGL ES function.
 */
void glesemuEnableWireframe();

/**
 *	@brief disables wire-frame rendering
 *	@note This is only available in the emulator for debugging
 *		purposes, and is not an OpenGL ES function.
 */
void glesemuDisableWireframe();

//								--- ~OpenGL ES 2.0 emulator helpers ---

//								--- OpenGL ES 2.0 ---

#ifndef GL_ES_VERSION_2_0
/* OpenGL ES core versions */
#define GL_ES_VERSION_2_0 1

extern bool GLEH_OpenGL_ES_2_0;

/* ClearBufferMask */
#define GL_DEPTH_BUFFER_BIT				  0x00000100
#define GL_STENCIL_BUFFER_BIT			  0x00000400
#define GL_COLOR_BUFFER_BIT				  0x00004000

/* Boolean */
#define GL_FALSE						  0
#define GL_TRUE							  1

/* BeginMode */
#define GL_POINTS						  0x0000
#define GL_LINES						  0x0001
#define GL_LINE_LOOP					  0x0002
#define GL_LINE_STRIP					  0x0003
#define GL_TRIANGLES					  0x0004
#define GL_TRIANGLE_STRIP				  0x0005
#define GL_TRIANGLE_FAN					  0x0006

/* AlphaFunction (not supported in ES20) */
/*		GL_NEVER */
/*		GL_LESS */
/*		GL_EQUAL */
/*		GL_LEQUAL */
/*		GL_GREATER */
/*		GL_NOTEQUAL */
/*		GL_GEQUAL */
/*		GL_ALWAYS */

/* BlendingFactorDest */
#define GL_ZERO							  0
#define GL_ONE							  1
#define GL_SRC_COLOR					  0x0300
#define GL_ONE_MINUS_SRC_COLOR			  0x0301
#define GL_SRC_ALPHA					  0x0302
#define GL_ONE_MINUS_SRC_ALPHA			  0x0303
#define GL_DST_ALPHA					  0x0304
#define GL_ONE_MINUS_DST_ALPHA			  0x0305

/* BlendingFactorSrc */
/*		GL_ZERO */
/*		GL_ONE */
#define GL_DST_COLOR					  0x0306
#define GL_ONE_MINUS_DST_COLOR			  0x0307
#define GL_SRC_ALPHA_SATURATE			  0x0308
/*		GL_SRC_ALPHA */
/*		GL_ONE_MINUS_SRC_ALPHA */
/*		GL_DST_ALPHA */
/*		GL_ONE_MINUS_DST_ALPHA */

/* BlendEquationSeparate */
#define GL_FUNC_ADD						  0x8006
#define GL_BLEND_EQUATION				  0x8009
#define GL_BLEND_EQUATION_RGB			  0x8009	/* same as BLEND_EQUATION */
#define GL_BLEND_EQUATION_ALPHA			  0x883D

/* BlendSubtract */
#define GL_FUNC_SUBTRACT				  0x800A
#define GL_FUNC_REVERSE_SUBTRACT		  0x800B

/* Separate Blend Functions */
#define GL_BLEND_DST_RGB				  0x80C8
#define GL_BLEND_SRC_RGB				  0x80C9
#define GL_BLEND_DST_ALPHA				  0x80CA
#define GL_BLEND_SRC_ALPHA				  0x80CB
#define GL_CONSTANT_COLOR				  0x8001
#define GL_ONE_MINUS_CONSTANT_COLOR		  0x8002
#define GL_CONSTANT_ALPHA				  0x8003
#define GL_ONE_MINUS_CONSTANT_ALPHA		  0x8004
#define GL_BLEND_COLOR					  0x8005

/* Buffer Objects */
#define GL_ARRAY_BUFFER					  0x8892
#define GL_ELEMENT_ARRAY_BUFFER			  0x8893
#define GL_ARRAY_BUFFER_BINDING			  0x8894
#define GL_ELEMENT_ARRAY_BUFFER_BINDING	  0x8895

#define GL_STREAM_DRAW					  0x88E0
#define GL_STATIC_DRAW					  0x88E4
#define GL_DYNAMIC_DRAW					  0x88E8

#define GL_BUFFER_SIZE					  0x8764
#define GL_BUFFER_USAGE					  0x8765

#define GL_CURRENT_VERTEX_ATTRIB		  0x8626

/* CullFaceMode */
#define GL_FRONT						  0x0404
#define GL_BACK							  0x0405
#define GL_FRONT_AND_BACK				  0x0408

/* DepthFunction */
/*		GL_NEVER */
/*		GL_LESS */
/*		GL_EQUAL */
/*		GL_LEQUAL */
/*		GL_GREATER */
/*		GL_NOTEQUAL */
/*		GL_GEQUAL */
/*		GL_ALWAYS */

/* EnableCap */
#define GL_TEXTURE_2D					  0x0DE1
#define GL_CULL_FACE					  0x0B44
#define GL_BLEND						  0x0BE2
#define GL_DITHER						  0x0BD0
#define GL_STENCIL_TEST					  0x0B90
#define GL_DEPTH_TEST					  0x0B71
#define GL_SCISSOR_TEST					  0x0C11
#define GL_POLYGON_OFFSET_FILL			  0x8037
#define GL_SAMPLE_ALPHA_TO_COVERAGE		  0x809E
#define GL_SAMPLE_COVERAGE				  0x80A0

/* ErrorCode */
#define GL_NO_ERROR						  0
#define GL_INVALID_ENUM					  0x0500
#define GL_INVALID_VALUE				  0x0501
#define GL_INVALID_OPERATION			  0x0502
#define GL_OUT_OF_MEMORY				  0x0505

/* FrontFaceDirection */
#define GL_CW							  0x0900
#define GL_CCW							  0x0901

/* GetPName */
#define GL_LINE_WIDTH					  0x0B21
#define GL_ALIASED_POINT_SIZE_RANGE		  0x846D
#define GL_ALIASED_LINE_WIDTH_RANGE		  0x846E
#define GL_CULL_FACE_MODE				  0x0B45
#define GL_FRONT_FACE					  0x0B46
#define GL_DEPTH_RANGE					  0x0B70
#define GL_DEPTH_WRITEMASK				  0x0B72
#define GL_DEPTH_CLEAR_VALUE			  0x0B73
#define GL_DEPTH_FUNC					  0x0B74
#define GL_STENCIL_CLEAR_VALUE			  0x0B91
#define GL_STENCIL_FUNC					  0x0B92
#define GL_STENCIL_FAIL					  0x0B94
#define GL_STENCIL_PASS_DEPTH_FAIL		  0x0B95
#define GL_STENCIL_PASS_DEPTH_PASS		  0x0B96
#define GL_STENCIL_REF					  0x0B97
#define GL_STENCIL_VALUE_MASK			  0x0B93
#define GL_STENCIL_WRITEMASK			  0x0B98
#define GL_STENCIL_BACK_FUNC			  0x8800
#define GL_STENCIL_BACK_FAIL			  0x8801
#define GL_STENCIL_BACK_PASS_DEPTH_FAIL	  0x8802
#define GL_STENCIL_BACK_PASS_DEPTH_PASS	  0x8803
#define GL_STENCIL_BACK_REF				  0x8CA3
#define GL_STENCIL_BACK_VALUE_MASK		  0x8CA4
#define GL_STENCIL_BACK_WRITEMASK		  0x8CA5
#define GL_VIEWPORT						  0x0BA2
#define GL_SCISSOR_BOX					  0x0C10
/*		GL_SCISSOR_TEST */
#define GL_COLOR_CLEAR_VALUE			  0x0C22
#define GL_COLOR_WRITEMASK				  0x0C23
#define GL_UNPACK_ALIGNMENT				  0x0CF5
#define GL_PACK_ALIGNMENT				  0x0D05
#define GL_MAX_TEXTURE_SIZE				  0x0D33
#define GL_MAX_VIEWPORT_DIMS			  0x0D3A
#define GL_SUBPIXEL_BITS				  0x0D50
#define GL_RED_BITS						  0x0D52
#define GL_GREEN_BITS					  0x0D53
#define GL_BLUE_BITS					  0x0D54
#define GL_ALPHA_BITS					  0x0D55
#define GL_DEPTH_BITS					  0x0D56
#define GL_STENCIL_BITS					  0x0D57
#define GL_POLYGON_OFFSET_UNITS			  0x2A00
/*		GL_POLYGON_OFFSET_FILL */
#define GL_POLYGON_OFFSET_FACTOR		  0x8038
#define GL_TEXTURE_BINDING_2D			  0x8069
#define GL_SAMPLE_BUFFERS				  0x80A8
#define GL_SAMPLES						  0x80A9
#define GL_SAMPLE_COVERAGE_VALUE		  0x80AA
#define GL_SAMPLE_COVERAGE_INVERT		  0x80AB

/* GetTextureParameter */
/*		GL_TEXTURE_MAG_FILTER */
/*		GL_TEXTURE_MIN_FILTER */
/*		GL_TEXTURE_WRAP_S */
/*		GL_TEXTURE_WRAP_T */

#define GL_NUM_COMPRESSED_TEXTURE_FORMATS 0x86A2
#define GL_COMPRESSED_TEXTURE_FORMATS	  0x86A3

/* HintMode */
#define GL_DONT_CARE					  0x1100
#define GL_FASTEST						  0x1101
#define GL_NICEST						  0x1102

/* HintTarget */
#define GL_GENERATE_MIPMAP_HINT			   0x8192

/* DataType */
#define GL_BYTE							  0x1400
#define GL_UNSIGNED_BYTE				  0x1401
#define GL_SHORT						  0x1402
#define GL_UNSIGNED_SHORT				  0x1403
#define GL_INT							  0x1404
#define GL_UNSIGNED_INT					  0x1405
#define GL_FLOAT						  0x1406
#define GL_FIXED						  0x140C

/* PixelFormat */
#define GL_DEPTH_COMPONENT				  0x1902
#define GL_ALPHA						  0x1906
#define GL_RGB							  0x1907
#define GL_RGBA							  0x1908
#define GL_LUMINANCE					  0x1909
#define GL_LUMINANCE_ALPHA				  0x190A

/* PixelType */
/*		GL_UNSIGNED_BYTE */
#define GL_UNSIGNED_SHORT_4_4_4_4		  0x8033
#define GL_UNSIGNED_SHORT_5_5_5_1		  0x8034
#define GL_UNSIGNED_SHORT_5_6_5			  0x8363

/* Shaders */
#define GL_FRAGMENT_SHADER					0x8B30
#define GL_VERTEX_SHADER					0x8B31
#define GL_MAX_VERTEX_ATTRIBS				0x8869
#define GL_MAX_VERTEX_UNIFORM_VECTORS		0x8DFB
#define GL_MAX_VARYING_VECTORS				0x8DFC
#define GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS 0x8B4D
#define GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS	0x8B4C
#define GL_MAX_TEXTURE_IMAGE_UNITS			0x8872
#define GL_MAX_FRAGMENT_UNIFORM_VECTORS		0x8DFD
#define GL_SHADER_TYPE						0x8B4F
#define GL_DELETE_STATUS					0x8B80
#define GL_LINK_STATUS						0x8B82
#define GL_VALIDATE_STATUS					0x8B83
#define GL_ATTACHED_SHADERS					0x8B85
#define GL_ACTIVE_UNIFORMS					0x8B86
#define GL_ACTIVE_UNIFORM_MAX_LENGTH		0x8B87
#define GL_ACTIVE_ATTRIBUTES				0x8B89
#define GL_ACTIVE_ATTRIBUTE_MAX_LENGTH		0x8B8A
#define GL_SHADING_LANGUAGE_VERSION			0x8B8C
#define GL_CURRENT_PROGRAM					0x8B8D

/* StencilFunction */
#define GL_NEVER						  0x0200
#define GL_LESS							  0x0201
#define GL_EQUAL						  0x0202
#define GL_LEQUAL						  0x0203
#define GL_GREATER						  0x0204
#define GL_NOTEQUAL						  0x0205
#define GL_GEQUAL						  0x0206
#define GL_ALWAYS						  0x0207

/* StencilOp */
/*		GL_ZERO */
#define GL_KEEP							  0x1E00
#define GL_REPLACE						  0x1E01
#define GL_INCR							  0x1E02
#define GL_DECR							  0x1E03
#define GL_INVERT						  0x150A
#define GL_INCR_WRAP					  0x8507
#define GL_DECR_WRAP					  0x8508

/* StringName */
#define GL_VENDOR						  0x1F00
#define GL_RENDERER						  0x1F01
#define GL_VERSION						  0x1F02
#define GL_EXTENSIONS					  0x1F03

/* TextureMagFilter */
#define GL_NEAREST						  0x2600
#define GL_LINEAR						  0x2601

/* TextureMinFilter */
/*		GL_NEAREST */
/*		GL_LINEAR */
#define GL_NEAREST_MIPMAP_NEAREST		  0x2700
#define GL_LINEAR_MIPMAP_NEAREST		  0x2701
#define GL_NEAREST_MIPMAP_LINEAR		  0x2702
#define GL_LINEAR_MIPMAP_LINEAR			  0x2703

/* TextureParameterName */
#define GL_TEXTURE_MAG_FILTER			  0x2800
#define GL_TEXTURE_MIN_FILTER			  0x2801
#define GL_TEXTURE_WRAP_S				  0x2802
#define GL_TEXTURE_WRAP_T				  0x2803

/* TextureTarget */
/*		GL_TEXTURE_2D */
#define GL_TEXTURE						  0x1702

#define GL_TEXTURE_CUBE_MAP				  0x8513
#define GL_TEXTURE_BINDING_CUBE_MAP		  0x8514
#define GL_TEXTURE_CUBE_MAP_POSITIVE_X	  0x8515
#define GL_TEXTURE_CUBE_MAP_NEGATIVE_X	  0x8516
#define GL_TEXTURE_CUBE_MAP_POSITIVE_Y	  0x8517
#define GL_TEXTURE_CUBE_MAP_NEGATIVE_Y	  0x8518
#define GL_TEXTURE_CUBE_MAP_POSITIVE_Z	  0x8519
#define GL_TEXTURE_CUBE_MAP_NEGATIVE_Z	  0x851A
#define GL_MAX_CUBE_MAP_TEXTURE_SIZE	  0x851C

/* TextureUnit */
#define GL_TEXTURE0						  0x84C0
#define GL_TEXTURE1						  0x84C1
#define GL_TEXTURE2						  0x84C2
#define GL_TEXTURE3						  0x84C3
#define GL_TEXTURE4						  0x84C4
#define GL_TEXTURE5						  0x84C5
#define GL_TEXTURE6						  0x84C6
#define GL_TEXTURE7						  0x84C7
#define GL_TEXTURE8						  0x84C8
#define GL_TEXTURE9						  0x84C9
#define GL_TEXTURE10					  0x84CA
#define GL_TEXTURE11					  0x84CB
#define GL_TEXTURE12					  0x84CC
#define GL_TEXTURE13					  0x84CD
#define GL_TEXTURE14					  0x84CE
#define GL_TEXTURE15					  0x84CF
#define GL_TEXTURE16					  0x84D0
#define GL_TEXTURE17					  0x84D1
#define GL_TEXTURE18					  0x84D2
#define GL_TEXTURE19					  0x84D3
#define GL_TEXTURE20					  0x84D4
#define GL_TEXTURE21					  0x84D5
#define GL_TEXTURE22					  0x84D6
#define GL_TEXTURE23					  0x84D7
#define GL_TEXTURE24					  0x84D8
#define GL_TEXTURE25					  0x84D9
#define GL_TEXTURE26					  0x84DA
#define GL_TEXTURE27					  0x84DB
#define GL_TEXTURE28					  0x84DC
#define GL_TEXTURE29					  0x84DD
#define GL_TEXTURE30					  0x84DE
#define GL_TEXTURE31					  0x84DF
#define GL_ACTIVE_TEXTURE				  0x84E0

/* TextureWrapMode */
#define GL_REPEAT						  0x2901
#define GL_CLAMP_TO_EDGE				  0x812F
#define GL_MIRRORED_REPEAT				  0x8370

/* Uniform Types */
#define GL_FLOAT_VEC2					  0x8B50
#define GL_FLOAT_VEC3					  0x8B51
#define GL_FLOAT_VEC4					  0x8B52
#define GL_INT_VEC2						  0x8B53
#define GL_INT_VEC3						  0x8B54
#define GL_INT_VEC4						  0x8B55
#define GL_BOOL							  0x8B56
#define GL_BOOL_VEC2					  0x8B57
#define GL_BOOL_VEC3					  0x8B58
#define GL_BOOL_VEC4					  0x8B59
#define GL_FLOAT_MAT2					  0x8B5A
#define GL_FLOAT_MAT3					  0x8B5B
#define GL_FLOAT_MAT4					  0x8B5C
#define GL_SAMPLER_2D					  0x8B5E
#define GL_SAMPLER_CUBE					  0x8B60

/* Vertex Arrays */
#define GL_VERTEX_ATTRIB_ARRAY_ENABLED		  0x8622
#define GL_VERTEX_ATTRIB_ARRAY_SIZE			  0x8623
#define GL_VERTEX_ATTRIB_ARRAY_STRIDE		  0x8624
#define GL_VERTEX_ATTRIB_ARRAY_TYPE			  0x8625
#define GL_VERTEX_ATTRIB_ARRAY_NORMALIZED	  0x886A
#define GL_VERTEX_ATTRIB_ARRAY_POINTER		  0x8645
#define GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING 0x889F

/* Read Format */
#define GL_IMPLEMENTATION_COLOR_READ_TYPE	0x8B9A
#define GL_IMPLEMENTATION_COLOR_READ_FORMAT 0x8B9B

/* Shader Source */
#define GL_COMPILE_STATUS				  0x8B81
#define GL_INFO_LOG_LENGTH				  0x8B84
#define GL_SHADER_SOURCE_LENGTH			  0x8B88
#define GL_SHADER_COMPILER				  0x8DFA

/* Shader Binary */
#define GL_SHADER_BINARY_FORMATS		  0x8DF8
#define GL_NUM_SHADER_BINARY_FORMATS	  0x8DF9

/* Shader Precision-Specified Types */
#define GL_LOW_FLOAT					  0x8DF0
#define GL_MEDIUM_FLOAT					  0x8DF1
#define GL_HIGH_FLOAT					  0x8DF2
#define GL_LOW_INT						  0x8DF3
#define GL_MEDIUM_INT					  0x8DF4
#define GL_HIGH_INT						  0x8DF5

/* Framebuffer Object. */
#define GL_FRAMEBUFFER					  0x8D40
#define GL_RENDERBUFFER					  0x8D41

#define GL_RGBA4						  0x8056
#define GL_RGB5_A1						  0x8057
#define GL_RGB565						  0x8D62
#define GL_DEPTH_COMPONENT16			  0x81A5
#define GL_STENCIL_INDEX				  0x1901
#define GL_STENCIL_INDEX8				  0x8D48

#define GL_RENDERBUFFER_WIDTH			  0x8D42
#define GL_RENDERBUFFER_HEIGHT			  0x8D43
#define GL_RENDERBUFFER_INTERNAL_FORMAT	  0x8D44
#define GL_RENDERBUFFER_RED_SIZE		  0x8D50
#define GL_RENDERBUFFER_GREEN_SIZE		  0x8D51
#define GL_RENDERBUFFER_BLUE_SIZE		  0x8D52
#define GL_RENDERBUFFER_ALPHA_SIZE		  0x8D53
#define GL_RENDERBUFFER_DEPTH_SIZE		  0x8D54
#define GL_RENDERBUFFER_STENCIL_SIZE	  0x8D55

#define GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE			0x8CD0
#define GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME			0x8CD1
#define GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL			0x8CD2
#define GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE 0x8CD3

#define GL_COLOR_ATTACHMENT0			  0x8CE0
#define GL_DEPTH_ATTACHMENT				  0x8D00
#define GL_STENCIL_ATTACHMENT			  0x8D20

#define GL_NONE							  0

#define GL_FRAMEBUFFER_COMPLETE						 0x8CD5
#define GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT		 0x8CD6
#define GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT 0x8CD7
#define GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS		 0x8CD9
#define GL_FRAMEBUFFER_UNSUPPORTED					 0x8CDD

#define GL_FRAMEBUFFER_BINDING			  0x8CA6
#define GL_RENDERBUFFER_BINDING			  0x8CA7
#define GL_MAX_RENDERBUFFER_SIZE		  0x84E8

#define GL_INVALID_FRAMEBUFFER_OPERATION  0x0506

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

/**
 *	@brief bind a named texture to a texturing target
 *	@param[in] target Specifies the target to which the texture is bound. Must be either
 *		GL_TEXTURE_1D, GL_TEXTURE_2D, GL_TEXTURE_3D, or GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, GL_TEXTURE_CUBE_MAP, GL_TEXTURE_2D_MULTISAMPLE or GL_TEXTURE_2D_MULTISAMPLE_ARRAY.
 *	@param[in] texture Specifies the name of a texture.
 */
GLStaticApi void ApiEntry glBindTexture(GLenum target, GLuint texture);

/**
 *	@brief specify pixel arithmetic
 *	@param[in] sfactor Specifies how the red, green, blue, and alpha source blending factors are
 *		computed. The initial value is GL_ONE.
 *	@param[in] dfactor Specifies how the red, green, blue, and alpha destination blending factors
 *		are computed. The following symbolic constants are accepted: GL_ZERO, GL_ONE, GL_SRC_COLOR,
 *		GL_ONE_MINUS_SRC_COLOR, GL_DST_COLOR, GL_ONE_MINUS_DST_COLOR, GL_SRC_ALPHA,
 *		GL_ONE_MINUS_SRC_ALPHA, GL_DST_ALPHA, GL_ONE_MINUS_DST_ALPHA. GL_CONSTANT_COLOR,
 *		GL_ONE_MINUS_CONSTANT_COLOR, GL_CONSTANT_ALPHA, and GL_ONE_MINUS_CONSTANT_ALPHA. The initial
 *		value is GL_ZERO.
 */
GLStaticApi void ApiEntry glBlendFunc(GLenum sfactor, GLenum dfactor);

/**
 *	@brief clear buffers to preset values
 *	@param[in] mask Bitwise OR of masks that indicate the buffers to be cleared. The three masks are
 *		GL_COLOR_BUFFER_BIT, GL_DEPTH_BUFFER_BIT, and GL_STENCIL_BUFFER_BIT.
 */
GLStaticApi void ApiEntry glClear(GLbitfield mask);

/**
 *	@brief specify clear values for the color buffers
 *	@param[in] red Specify the red, green, blue, and alpha values used when the color buffers are
 *		cleared. The initial values are all 0.
 *	@param[in] green Specify the red, green, blue, and alpha values used when the color buffers are
 *		cleared. The initial values are all 0.
 *	@param[in] blue Specify the red, green, blue, and alpha values used when the color buffers are
 *		cleared. The initial values are all 0.
 *	@param[in] alpha Specify the red, green, blue, and alpha values used when the color buffers are
 *		cleared. The initial values are all 0.
 */
GLStaticApi void ApiEntry glClearColor(GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha);

/**
 *	@brief specify the clear value for the stencil buffer
 *	@param[in] s Specifies the index used when the stencil buffer is cleared. The initial value is 0.
 */
GLStaticApi void ApiEntry glClearStencil(int s);

/**
 *	@brief enable and disable writing of frame buffer color components
 *	@param[in] red Specify whether red, green, blue, and alpha are to be written into the frame
 *		buffer. The initial values are all GL_TRUE, indicating that the color components are written.
 *	@param[in] green Specify whether red, green, blue, and alpha are to be written into the frame
 *		buffer. The initial values are all GL_TRUE, indicating that the color components are written.
 *	@param[in] blue Specify whether red, green, blue, and alpha are to be written into the frame
 *		buffer. The initial values are all GL_TRUE, indicating that the color components are written.
 *	@param[in] alpha Specify whether red, green, blue, and alpha are to be written into the frame
 *		buffer. The initial values are all GL_TRUE, indicating that the color components are written.
 */
GLStaticApi void ApiEntry glColorMask(GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha);

/**
 *	@brief copy pixels into a 2D texture image
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, or GL_TEXTURE_CUBE_MAP_NEGATIVE_Z.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] internalformat Specifies the internal format of the texture. Must be one of the
 *		following symbolic constants: GL_COMPRESSED_RED, GL_COMPRESSED_RG, GL_COMPRESSED_RGB,
 *		GL_COMPRESSED_RGBA. GL_COMPRESSED_SRGB, GL_COMPRESSED_SRGB_ALPHA. GL_DEPTH_COMPONENT,
 *		GL_DEPTH_COMPONENT16, GL_DEPTH_COMPONENT24, GL_DEPTH_COMPONENT32, GL_RED, GL_RG, GL_RGB,
 *		GL_R3_G3_B2, GL_RGB4, GL_RGB5, GL_RGB8, GL_RGB10, GL_RGB12, GL_RGB16, GL_RGBA, GL_RGBA2,
 *		GL_RGBA4, GL_RGB5_A1, GL_RGBA8, GL_RGB10_A2, GL_RGBA12, GL_RGBA16, GL_SRGB, GL_SRGB8,
 *		GL_SRGB_ALPHA, or GL_SRGB8_ALPHA8.
 *	@param[in] x Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] y Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] width Specifies the width of the texture image. Must be 0 or 2 n + 2 border for some
 *		integer n.
 *	@param[in] height Specifies the height of the texture image. Must be 0 or 2 m + 2 border for
 *		some integer m.
 *	@param[in] border Specifies the width of the border. Must be either 0 or 1.
 */
GLStaticApi void ApiEntry glCopyTexImage2D(GLenum target, int level, GLenum internalformat, int x, int y, GLsizei width, GLsizei height, int border);

/**
 *	@brief copy a two-dimensional texture subimage
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, or GL_TEXTURE_CUBE_MAP_NEGATIVE_Z.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] x Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] y Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 */
GLStaticApi void ApiEntry glCopyTexSubImage2D(GLenum target, int level, int xoffset, int yoffset, int x, int y, GLsizei width, GLsizei height);

/**
 *	@brief specify whether front- or back-facing facets can be culled
 *	@param[in] mode Specifies whether front- or back-facing facets are candidates for culling.
 *		Symbolic constants GL_FRONT, GL_BACK, and GL_FRONT_AND_BACK are accepted. The initial value
 *		is GL_BACK.
 */
GLStaticApi void ApiEntry glCullFace(GLenum mode);

/**
 *	@brief delete named textures
 *	@param[in] n Specifies the number of textures to be deleted.
 *	@param[in] textures Specifies an array of textures to be deleted.
 */
GLStaticApi void ApiEntry glDeleteTextures(GLsizei n, const GLuint *textures);

/**
 *	@brief specify the value used for depth buffer comparisons
 *	@param[in] func Specifies the depth comparison function. Symbolic constants GL_NEVER, GL_LESS,
 *		GL_EQUAL, GL_LEQUAL, GL_GREATER, GL_NOTEQUAL, GL_GEQUAL, and GL_ALWAYS are accepted. The
 *		initial value is GL_LESS.
 */
GLStaticApi void ApiEntry glDepthFunc(GLenum func);

/**
 *	@brief enable or disable writing into the depth buffer
 *	@param[in] flag Specifies whether the depth buffer is enabled for writing. If flag is GL_FALSE,
 *		depth buffer writing is disabled. Otherwise, it is enabled. Initially, depth buffer writing
 *		is enabled.
 */
GLStaticApi void ApiEntry glDepthMask(GLboolean flag);

/**
 *	@brief enable or disable server-side GL capabilities
 *	@param[in] cap Specifies a symbolic constant indicating a GL capability.
 */
GLStaticApi void ApiEntry glDisable(GLenum cap);

/**
 *	@brief render primitives from array data
 *	@param[in] mode Specifies what kind of primitives to render. Symbolic constants GL_POINTS,
 *		GL_LINE_STRIP, GL_LINE_LOOP, GL_LINES, GL_LINE_STRIP_ADJACENCY, GL_LINES_ADJACENCY,
 *		GL_TRIANGLE_STRIP, GL_TRIANGLE_FAN, GL_TRIANGLES, GL_TRIANGLE_STRIP_ADJACENCY,
 *		GL_TRIANGLES_ADJACENCY and GL_PATCHES are accepted.
 *	@param[in] first Specifies the starting index in the enabled arrays.
 *	@param[in] count Specifies the number of indices to be rendered.
 */
GLStaticApi void ApiEntry glDrawArrays(GLenum mode, int first, GLsizei count);

/**
 *	@brief render primitives from array data
 *	@param[in] mode Specifies what kind of primitives to render. Symbolic constants GL_POINTS,
 *		GL_LINE_STRIP, GL_LINE_LOOP, GL_LINES, GL_LINE_STRIP_ADJACENCY, GL_LINES_ADJACENCY,
 *		GL_TRIANGLE_STRIP, GL_TRIANGLE_FAN, GL_TRIANGLES, GL_TRIANGLE_STRIP_ADJACENCY,
 *		GL_TRIANGLES_ADJACENCY and GL_PATCHES are accepted.
 *	@param[in] count Specifies the number of elements to be rendered.
 *	@param[in] type Specifies the type of the values in indices. Must be one of GL_UNSIGNED_BYTE,
 *		GL_UNSIGNED_SHORT, or GL_UNSIGNED_INT.
 *	@param[in] indices Specifies a pointer to the location where the indices are stored.
 */
GLStaticApi void ApiEntry glDrawElements(GLenum mode, GLsizei count, GLenum type, const void *indices);

/**
 *	@brief enable or disable server-side GL capabilities
 *	@param[in] cap Specifies a symbolic constant indicating a GL capability.
 */
GLStaticApi void ApiEntry glEnable(GLenum cap);

/**
 *	@brief block until all GL execution is complete
 */
GLStaticApi void ApiEntry glFinish();

/**
 *	@brief force execution of GL commands in finite time
 */
GLStaticApi void ApiEntry glFlush();

/**
 *	@brief define front- and back-facing polygons
 *	@param[in] mode Specifies the orientation of front-facing polygons. GL_CW and GL_CCW are
 *		accepted. The initial value is GL_CCW.
 */
GLStaticApi void ApiEntry glFrontFace(GLenum mode);

/**
 *	@brief generate texture names
 *	@param[in] n Specifies the number of texture names to be generated.
 *	@param[out] textures Specifies an array in which the generated texture names are stored.
 */
GLStaticApi void ApiEntry glGenTextures(GLsizei n, GLuint *textures);

/**
 *	@brief return the value or values of a selected parameter
 *	@param[in] pname Specifies the parameter value to be returned. The symbolic constants in the
 *		list below are accepted.
 *	@param[out] params Returns the value or values of the specified parameter.
 */
GLStaticApi void ApiEntry glGetBooleanv(GLenum pname, GLboolean *params);

/**
 *	@brief return error information
 */
GLStaticApi GLenum ApiEntry glGetError();

/**
 *	@brief return the value or values of a selected parameter
 *	@param[in] pname Specifies the parameter value to be returned. The symbolic constants in the
 *		list below are accepted.
 *	@param[out] params Returns the value or values of the specified parameter.
 */
GLStaticApi void ApiEntry glGetFloatv(GLenum pname, float *params);

/**
 *	@brief return the value or values of a selected parameter
 *	@param[in] pname Specifies the parameter value to be returned. The symbolic constants in the
 *		list below are accepted.
 *	@param[out] params Returns the value or values of the specified parameter.
 */
GLStaticApi void ApiEntry glGetIntegerv(GLenum pname, int *params);

/**
 *	@brief return a string describing the current GL connection
 *	@param[in] name Specifies a symbolic constant, one of GL_VENDOR, GL_RENDERER, GL_VERSION, or
 *		GL_SHADING_LANGUAGE_VERSION. Additionally, glGetStringi accepts the GL_EXTENSIONS token.
 */
GLStaticApi const GLubyte *ApiEntry glGetString(GLenum name);

/**
 *	@brief return texture parameter values
 *	@param[in] target Specifies the symbolic name of the target texture. GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY, GL_TEXTURE_3D,
 *		GL_TEXTURE_RECTANGLE, and GL_TEXTURE_CUBE_MAP are accepted.
 *	@param[in] pname Specifies the symbolic name of a texture parameter. GL_TEXTURE_BASE_LEVEL,
 *		GL_TEXTURE_BORDER_COLOR, GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MAG_FILTER, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_MAX_LOD,
 *		GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MIN_LOD, GL_TEXTURE_SWIZZLE_R, GL_TEXTURE_SWIZZLE_G,
 *		GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_SWIZZLE_RGBA, GL_TEXTURE_WRAP_S,
 *		GL_TEXTURE_WRAP_T, and GL_TEXTURE_WRAP_R are accepted.
 *	@param[out] params Returns the texture parameters.
 */
GLStaticApi void ApiEntry glGetTexParameterfv(GLenum target, GLenum pname, float *params);

/**
 *	@brief return texture parameter values
 *	@param[in] target Specifies the symbolic name of the target texture. GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY, GL_TEXTURE_3D,
 *		GL_TEXTURE_RECTANGLE, and GL_TEXTURE_CUBE_MAP are accepted.
 *	@param[in] pname Specifies the symbolic name of a texture parameter. GL_TEXTURE_BASE_LEVEL,
 *		GL_TEXTURE_BORDER_COLOR, GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MAG_FILTER, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_MAX_LOD,
 *		GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MIN_LOD, GL_TEXTURE_SWIZZLE_R, GL_TEXTURE_SWIZZLE_G,
 *		GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_SWIZZLE_RGBA, GL_TEXTURE_WRAP_S,
 *		GL_TEXTURE_WRAP_T, and GL_TEXTURE_WRAP_R are accepted.
 *	@param[out] params Returns the texture parameters.
 */
GLStaticApi void ApiEntry glGetTexParameteriv(GLenum target, GLenum pname, int *params);

/**
 *	@brief specify implementation-specific hints
 *	@param[in] target Specifies a symbolic constant indicating the behavior to be controlled.
 *		GL_LINE_SMOOTH_HINT, GL_POLYGON_SMOOTH_HINT, GL_TEXTURE_COMPRESSION_HINT, and
 *		GL_FRAGMENT_SHADER_DERIVATIVE_HINT are accepted.
 *	@param[in] mode Specifies a symbolic constant indicating the desired behavior. GL_FASTEST,
 *		GL_NICEST, and GL_DONT_CARE are accepted.
 */
GLStaticApi void ApiEntry glHint(GLenum target, GLenum mode);

/**
 *	@brief test whether a capability is enabled
 *	@param[in] cap Specifies a symbolic constant indicating a GL capability.
 */
GLStaticApi GLboolean ApiEntry glIsEnabled(GLenum cap);

/**
 *	@brief determine if a name corresponds to a texture
 *	@param[in] texture Specifies a value that may be the name of a texture.
 */
GLStaticApi GLboolean ApiEntry glIsTexture(GLuint texture);

/**
 *	@brief specify the width of rasterized lines
 *	@param[in] width Specifies the width of rasterized lines. The initial value is 1.
 */
GLStaticApi void ApiEntry glLineWidth(float width);

/**
 *	@brief set pixel storage modes
 *	@param[in] pname Specifies the symbolic name of the parameter to be set. Six values affect the
 *		packing of pixel data into memory: GL_PACK_SWAP_BYTES, GL_PACK_LSB_FIRST,
 *		GL_PACK_ROW_LENGTH, GL_PACK_IMAGE_HEIGHT, GL_PACK_SKIP_PIXELS, GL_PACK_SKIP_ROWS,
 *		GL_PACK_SKIP_IMAGES, and GL_PACK_ALIGNMENT. Six more affect the unpacking of pixel data from
 *		memory: GL_UNPACK_SWAP_BYTES, GL_UNPACK_LSB_FIRST, GL_UNPACK_ROW_LENGTH,
 *		GL_UNPACK_IMAGE_HEIGHT, GL_UNPACK_SKIP_PIXELS, GL_UNPACK_SKIP_ROWS, GL_UNPACK_SKIP_IMAGES,
 *		and GL_UNPACK_ALIGNMENT.
 *	@param[in] param Specifies the value that pname is set to.
 */
GLStaticApi void ApiEntry glPixelStorei(GLenum pname, int param);

/**
 *	@brief set the scale and units used to calculate depth values
 *	@param[in] factor Specifies a scale factor that is used to create a variable depth offset for
 *		each polygon. The initial value is 0.
 *	@param[in] units Is multiplied by an implementation-specific value to create a constant depth
 *		offset. The initial value is 0.
 */
GLStaticApi void ApiEntry glPolygonOffset(float factor, float units);

/**
 *	@brief read a block of pixels from the frame buffer
 *	@param[in] x Specify the window coordinates of the first pixel that is read from the frame
 *		buffer. This location is the lower left corner of a rectangular block of pixels.
 *	@param[in] y Specify the window coordinates of the first pixel that is read from the frame
 *		buffer. This location is the lower left corner of a rectangular block of pixels.
 *	@param[in] width Specify the dimensions of the pixel rectangle. width and height of one
 *		correspond to a single pixel.
 *	@param[in] height Specify the dimensions of the pixel rectangle. width and height of one
 *		correspond to a single pixel.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_STENCIL_INDEX, GL_DEPTH_COMPONENT, GL_DEPTH_STENCIL, GL_RED, GL_GREEN, GL_BLUE,
 *		GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. Must be one of GL_UNSIGNED_BYTE,
 *		GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT, GL_HALF_FLOAT, GL_FLOAT,
 *		GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, GL_UNSIGNED_INT_2_10_10_10_REV,
 *		GL_UNSIGNED_INT_24_8, GL_UNSIGNED_INT_10F_11F_11F_REV, GL_UNSIGNED_INT_5_9_9_9_REV, or GL_FLOAT_32_UNSIGNED_INT_24_8_REV.
 *	@param[out] pixels (documantation not found)
 */
GLStaticApi void ApiEntry glReadPixels(int x, int y, GLsizei width, GLsizei height, GLenum format, GLenum type, void *pixels);

/**
 *	@brief define the scissor box
 *	@param[in] x Specify the lower left corner of the scissor box. Initially (0, 0).
 *	@param[in] y Specify the lower left corner of the scissor box. Initially (0, 0).
 *	@param[in] width Specify the width and height of the scissor box. When a GL context is first
 *		attached to a window, width and height are set to the dimensions of that window.
 *	@param[in] height Specify the width and height of the scissor box. When a GL context is first
 *		attached to a window, width and height are set to the dimensions of that window.
 */
GLStaticApi void ApiEntry glScissor(int x, int y, GLsizei width, GLsizei height);

/**
 *	@brief set front and back function and reference value for stencil testing
 *	@param[in] func Specifies the test function. Eight symbolic constants are valid: GL_NEVER,
 *		GL_LESS, GL_LEQUAL, GL_GREATER, GL_GEQUAL, GL_EQUAL, GL_NOTEQUAL, and GL_ALWAYS. The initial
 *		value is GL_ALWAYS.
 *	@param[in] ref Specifies the reference value for the stencil test. ref is clamped to the range 0
 *		2 n - 1, where n is the number of bitplanes in the stencil buffer. The initial value is 0.
 *	@param[in] mask Specifies a mask that is ANDed with both the reference value and the stored
 *		stencil value when the test is done. The initial value is all 1's.
 */
GLStaticApi void ApiEntry glStencilFunc(GLenum func, int ref, GLuint mask);

/**
 *	@brief control the front and back writing of individual bits in the stencil planes
 *	@param[in] mask Specifies a bit mask to enable and disable writing of individual bits in the
 *		stencil planes. Initially, the mask is all 1's.
 */
GLStaticApi void ApiEntry glStencilMask(GLuint mask);

/**
 *	@brief set front and back stencil test actions
 *	@param[in] fail (documantation not found)
 *	@param[in] zfail (documantation not found)
 *	@param[in] zpass (documantation not found)
 */
GLStaticApi void ApiEntry glStencilOp(GLenum fail, GLenum zfail, GLenum zpass);

/**
 *	@brief specify a two-dimensional texture image
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D, GL_PROXY_TEXTURE_2D,
 *		GL_TEXTURE_1D_ARRAY, GL_PROXY_TEXTURE_1D_ARRAY, GL_TEXTURE_RECTANGLE,
 *		GL_PROXY_TEXTURE_RECTANGLE, GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, GL_TEXTURE_CUBE_MAP_NEGATIVE_Z, or GL_PROXY_TEXTURE_CUBE_MAP.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image. If target is GL_TEXTURE_RECTANGLE or
 *		GL_PROXY_TEXTURE_RECTANGLE, level must be 0.
 *	@param[in] internalformat (documantation not found)
 *	@param[in] width Specifies the width of the texture image. All implementations support texture
 *		images that are at least 1024 texels wide.
 *	@param[in] height Specifies the height of the texture image, or the number of layers in a
 *		texture array, in the case of the GL_TEXTURE_1D_ARRAY and GL_PROXY_TEXTURE_1D_ARRAY targets.
 *		All implementations support 2D texture images that are at least 1024 texels high, and
 *		texture arrays that are at least 256 layers deep.
 *	@param[in] border This value must be 0.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_RED, GL_RG, GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. The following symbolic values are
 *		accepted: GL_UNSIGNED_BYTE, GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT,
 *		GL_FLOAT, GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, and GL_UNSIGNED_INT_2_10_10_10_REV.
 *	@param[in] pixels (documantation not found)
 */
GLStaticApi void ApiEntry glTexImage2D(GLenum target, int level, int internalformat, GLsizei width, GLsizei height, int border, GLenum format, GLenum type, const void *pixels);

/**
 *	@brief set texture parameters
 *	@param[in] target Specifies the target texture, which must be either GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, or GL_TEXTURE_CUBE_MAP.
 *	@param[in] pname Specifies the symbolic name of a single-valued texture parameter. pname can be
 *		one of the following: GL_TEXTURE_BASE_LEVEL, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
 *		GL_TEXTURE_MIN_LOD, GL_TEXTURE_MAX_LOD, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_SWIZZLE_R,
 *		GL_TEXTURE_SWIZZLE_G, GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_WRAP_S,
 *		GL_TEXTURE_WRAP_T, or GL_TEXTURE_WRAP_R.
 *	@param[in] param Specifies the value of pname.
 */
GLStaticApi void ApiEntry glTexParameterf(GLenum target, GLenum pname, float param);

/**
 *	@brief set texture parameters
 *	@param[in] target Specifies the target texture, which must be either GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, or GL_TEXTURE_CUBE_MAP.
 *	@param[in] pname Specifies the symbolic name of a texture parameter. pname can be one of the
 *		following: GL_TEXTURE_BASE_LEVEL, GL_TEXTURE_BORDER_COLOR, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
 *		GL_TEXTURE_MIN_LOD, GL_TEXTURE_MAX_LOD, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_SWIZZLE_R,
 *		GL_TEXTURE_SWIZZLE_G, GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_SWIZZLE_RGBA,
 *		GL_TEXTURE_WRAP_S, GL_TEXTURE_WRAP_T, or GL_TEXTURE_WRAP_R.
 *	@param[in] params Specifies a pointer to an array where the value or values of pname are stored.
 */
GLStaticApi void ApiEntry glTexParameterfv(GLenum target, GLenum pname, const float *params);

/**
 *	@brief set texture parameters
 *	@param[in] target Specifies the target texture, which must be either GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, or GL_TEXTURE_CUBE_MAP.
 *	@param[in] pname Specifies the symbolic name of a single-valued texture parameter. pname can be
 *		one of the following: GL_TEXTURE_BASE_LEVEL, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
 *		GL_TEXTURE_MIN_LOD, GL_TEXTURE_MAX_LOD, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_SWIZZLE_R,
 *		GL_TEXTURE_SWIZZLE_G, GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_WRAP_S,
 *		GL_TEXTURE_WRAP_T, or GL_TEXTURE_WRAP_R.
 *	@param[in] param Specifies the value of pname.
 */
GLStaticApi void ApiEntry glTexParameteri(GLenum target, GLenum pname, int param);

/**
 *	@brief set texture parameters
 *	@param[in] target Specifies the target texture, which must be either GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, or GL_TEXTURE_CUBE_MAP.
 *	@param[in] pname Specifies the symbolic name of a texture parameter. pname can be one of the
 *		following: GL_TEXTURE_BASE_LEVEL, GL_TEXTURE_BORDER_COLOR, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
 *		GL_TEXTURE_MIN_LOD, GL_TEXTURE_MAX_LOD, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_SWIZZLE_R,
 *		GL_TEXTURE_SWIZZLE_G, GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_SWIZZLE_RGBA,
 *		GL_TEXTURE_WRAP_S, GL_TEXTURE_WRAP_T, or GL_TEXTURE_WRAP_R.
 *	@param[in] params Specifies a pointer to an array where the value or values of pname are stored.
 */
GLStaticApi void ApiEntry glTexParameteriv(GLenum target, GLenum pname, const int *params);

/**
 *	@brief specify a two-dimensional texture subimage
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, or GL_TEXTURE_CUBE_MAP_NEGATIVE_Z.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_RED, GL_RG, GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. The following symbolic values are
 *		accepted: GL_UNSIGNED_BYTE, GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT,
 *		GL_FLOAT, GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, and GL_UNSIGNED_INT_2_10_10_10_REV.
 *	@param[in] pixels (documantation not found)
 */
GLStaticApi void ApiEntry glTexSubImage2D(GLenum target, int level, int xoffset, int yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *pixels);

/**
 *	@brief set the viewport
 *	@param[in] x Specify the lower left corner of the viewport rectangle, in pixels. The initial
 *		value is (0,0).
 *	@param[in] y Specify the lower left corner of the viewport rectangle, in pixels. The initial
 *		value is (0,0).
 *	@param[in] width Specify the width and height of the viewport. When a GL context is first
 *		attached to a window, width and height are set to the dimensions of that window.
 *	@param[in] height Specify the width and height of the viewport. When a GL context is first
 *		attached to a window, width and height are set to the dimensions of that window.
 */
GLStaticApi void ApiEntry glViewport(int x, int y, GLsizei width, GLsizei height);

#ifdef __cplusplus
}
#endif // __cplusplus

/**
 *	@brief select active texture unit
 *	@param[in] texture Specifies which texture unit to make active. The number of texture units is
 *		implementation dependent, but must be at least 80. texture must be one of GL_TEXTURE i,
 *		where i ranges from 0 ( GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS - 1). The initial value is GL_TEXTURE0.
 */
extern void (GLApi *GLFuncPtrName(glActiveTexture))(GLenum texture);

/**
 *	@brief Attaches a shader object to a program object
 *	@param[in] program Specifies the program object to which a shader object will be attached.
 *	@param[in] shader Specifies the shader object that is to be attached.
 */
extern void (GLApi *GLFuncPtrName(glAttachShader))(GLuint program, GLuint shader);

/**
 *	@brief Associates a generic vertex attribute index with a named attribute variable
 *	@param[in] program Specifies the handle of the program object in which the association is to be made.
 *	@param[in] index Specifies the index of the generic vertex attribute to be bound.
 *	@param[in] name Specifies a null terminated string containing the name of the vertex shader
 *		attribute variable to which index is to be bound.
 */
extern void (GLApi *GLFuncPtrName(glBindAttribLocation))(GLuint program, GLuint index, const GLchar *name);

/**
 *	@brief bind a named buffer object
 *	@param[in] target Specifies the target to which the buffer object is bound. The symbolic
 *		constant must be GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER,
 *		GL_DRAW_INDIRECT_BUFFER, GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER,
 *		GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] buffer Specifies the name of a buffer object.
 */
extern void (GLApi *GLFuncPtrName(glBindBuffer))(GLenum target, GLuint buffer);

/**
 *	@brief bind a framebuffer to a framebuffer target
 *	@param[in] target Specifies the framebuffer target of the binding operation.
 *	@param[in] framebuffer Specifies the name of the framebuffer object to bind.
 */
extern void (GLApi *GLFuncPtrName(glBindFramebuffer))(GLenum target, GLuint framebuffer);

/**
 *	@brief bind a renderbuffer to a renderbuffer target
 *	@param[in] target Specifies the renderbuffer target of the binding operation. target must be GL_RENDERBUFFER.
 *	@param[in] renderbuffer Specifies the name of the renderbuffer object to bind.
 */
extern void (GLApi *GLFuncPtrName(glBindRenderbuffer))(GLenum target, GLuint renderbuffer);

/**
 *	@brief set the blend color
 *	@param[in] red specify the components of GL_BLEND_COLOR
 *	@param[in] green specify the components of GL_BLEND_COLOR
 *	@param[in] blue specify the components of GL_BLEND_COLOR
 *	@param[in] alpha specify the components of GL_BLEND_COLOR
 */
extern void (GLApi *GLFuncPtrName(glBlendColor))(GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha);

/**
 *	@brief specify the equation used for both the RGB blend equation and the Alpha blend equation
 *	@param[in] mode specifies how source and destination colors are combined. It must be
 *		GL_FUNC_ADD, GL_FUNC_SUBTRACT, GL_FUNC_REVERSE_SUBTRACT, GL_MIN, GL_MAX.
 */
extern void (GLApi *GLFuncPtrName(glBlendEquation))(GLenum mode);

/**
 *	@brief set the RGB blend equation and the alpha blend equation separately
 *	@param[in] modeRGB specifies the RGB blend equation, how the red, green, and blue components of
 *		the source and destination colors are combined. It must be GL_FUNC_ADD, GL_FUNC_SUBTRACT,
 *		GL_FUNC_REVERSE_SUBTRACT, GL_MIN, GL_MAX.
 *	@param[in] modeAlpha specifies the alpha blend equation, how the alpha component of the source
 *		and destination colors are combined. It must be GL_FUNC_ADD, GL_FUNC_SUBTRACT,
 *		GL_FUNC_REVERSE_SUBTRACT, GL_MIN, GL_MAX.
 */
extern void (GLApi *GLFuncPtrName(glBlendEquationSeparate))(GLenum modeRGB, GLenum modeAlpha);

/**
 *	@brief specify pixel arithmetic for RGB and alpha components separately
 *	@param[in] srcRGB Specifies how the red, green, and blue blending factors are computed. The
 *		initial value is GL_ONE.
 *	@param[in] dstRGB Specifies how the red, green, and blue destination blending factors are
 *		computed. The initial value is GL_ZERO.
 *	@param[in] srcAlpha Specified how the alpha source blending factor is computed. The initial
 *		value is GL_ONE.
 *	@param[in] dstAlpha Specified how the alpha destination blending factor is computed. The initial
 *		value is GL_ZERO.
 */
extern void (GLApi *GLFuncPtrName(glBlendFuncSeparate))(GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha);

/**
 *	@brief creates and initializes a buffer object's data store
 *	@param[in] target Specifies the target buffer object. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] size Specifies the size in bytes of the buffer object's new data store.
 *	@param[in] data Specifies a pointer to data that will be copied into the data store for
 *		initialization, or NULL if no data is to be copied.
 *	@param[in] usage Specifies the expected usage pattern of the data store. The symbolic constant
 *		must be GL_STREAM_DRAW, GL_STREAM_READ, GL_STREAM_COPY, GL_STATIC_DRAW, GL_STATIC_READ,
 *		GL_STATIC_COPY, GL_DYNAMIC_DRAW, GL_DYNAMIC_READ, or GL_DYNAMIC_COPY.
 */
extern void (GLApi *GLFuncPtrName(glBufferData))(GLenum target, GLsizeiptr size, const void *data, GLenum usage);

/**
 *	@brief updates a subset of a buffer object's data store
 *	@param[in] target Specifies the target buffer object. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] offset Specifies the offset into the buffer object's data store where data
 *		replacement will begin, measured in bytes.
 *	@param[in] size Specifies the size in bytes of the data store region being replaced.
 *	@param[in] data Specifies a pointer to the new data that will be copied into the data store.
 */
extern void (GLApi *GLFuncPtrName(glBufferSubData))(GLenum target, GLintptr offset, GLsizeiptr size, const void *data);

/**
 *	@brief check the completeness status of a framebuffer
 *	@param[in] target Specify the target of the framebuffer completeness check.
 */
extern GLenum (GLApi *GLFuncPtrName(glCheckFramebufferStatus))(GLenum target);

/**
 *	@brief specify the clear value for the depth buffer
 *	@param[in] depth Specifies the depth value used when the depth buffer is cleared. The initial
 *		value is 1.
 */
extern void (GLApi *GLFuncPtrName(glClearDepthf))(GLclampf depth);

/**
 *	@brief Compiles a shader object
 *	@param[in] shader Specifies the shader object to be compiled.
 */
extern void (GLApi *GLFuncPtrName(glCompileShader))(GLuint shader);

/**
 *	@brief specify a two-dimensional texture image in a compressed format
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D, GL_PROXY_TEXTURE_2D,
 *		GL_TEXTURE_1D_ARRAY, GL_PROXY_TEXTURE_1D_ARRAY, GL_TEXTURE_CUBE_MAP_POSITIVE_X,
 *		GL_TEXTURE_CUBE_MAP_NEGATIVE_X, GL_TEXTURE_CUBE_MAP_POSITIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_NEGATIVE_Y, GL_TEXTURE_CUBE_MAP_POSITIVE_Z,
 *		GL_TEXTURE_CUBE_MAP_NEGATIVE_Z, or GL_PROXY_TEXTURE_CUBE_MAP.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] internalformat Specifies the format of the compressed image data stored at address data.
 *	@param[in] width Specifies the width of the texture image. All implementations support 2D
 *		texture images that are at least 64 texels wide and cube-mapped texture images that are at
 *		least 16 texels wide.
 *	@param[in] height Specifies the height of the texture image. All implementations support 2D
 *		texture images that are at least 64 texels high and cube-mapped texture images that are at
 *		least 16 texels high.
 *	@param[in] border This value must be 0.
 *	@param[in] imageSize Specifies the number of unsigned bytes of image data starting at the
 *		address specified by data.
 *	@param[in] data Specifies a pointer to the compressed image data in memory.
 */
extern void (GLApi *GLFuncPtrName(glCompressedTexImage2D))(GLenum target, int level, GLenum internalformat, GLsizei width, GLsizei height, int border, GLsizei imageSize, const void *data);

/**
 *	@brief specify a two-dimensional texture subimage in a compressed format
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, or GL_TEXTURE_CUBE_MAP_NEGATIVE_Z.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 *	@param[in] format Specifies the format of the compressed image data stored at address data.
 *	@param[in] imageSize Specifies the number of unsigned bytes of image data starting at the
 *		address specified by data.
 *	@param[in] data Specifies a pointer to the compressed image data in memory.
 */
extern void (GLApi *GLFuncPtrName(glCompressedTexSubImage2D))(GLenum target, int level, int xoffset, int yoffset, GLsizei width, GLsizei height, GLenum format, GLsizei imageSize, const void *data);

/**
 *	@brief Creates a program object
 */
extern GLuint (GLApi *GLFuncPtrName(glCreateProgram))();

/**
 *	@brief Creates a shader object
 *	@param[in] type (documantation not found)
 */
extern GLuint (GLApi *GLFuncPtrName(glCreateShader))(GLenum type);

/**
 *	@brief delete named buffer objects
 *	@param[in] n Specifies the number of buffer objects to be deleted.
 *	@param[in] buffers Specifies an array of buffer objects to be deleted.
 */
extern void (GLApi *GLFuncPtrName(glDeleteBuffers))(GLsizei n, const GLuint *buffers);

/**
 *	@brief delete framebuffer objects
 *	@param[in] n Specifies the number of framebuffer objects to be deleted.
 *	@param[in] framebuffers A pointer to an array containing n framebuffer objects to be deleted.
 */
extern void (GLApi *GLFuncPtrName(glDeleteFramebuffers))(GLsizei n, const GLuint *framebuffers);

/**
 *	@brief Deletes a program object
 *	@param[in] program Specifies the program object to be deleted.
 */
extern void (GLApi *GLFuncPtrName(glDeleteProgram))(GLuint program);

/**
 *	@brief delete renderbuffer objects
 *	@param[in] n Specifies the number of renderbuffer objects to be deleted.
 *	@param[in] renderbuffers A pointer to an array containing n renderbuffer objects to be deleted.
 */
extern void (GLApi *GLFuncPtrName(glDeleteRenderbuffers))(GLsizei n, const GLuint *renderbuffers);

/**
 *	@brief Deletes a shader object
 *	@param[in] shader Specifies the shader object to be deleted.
 */
extern void (GLApi *GLFuncPtrName(glDeleteShader))(GLuint shader);

/**
 *	@brief specify mapping of depth values from normalized device coordinates to window coordinates
 *	@param[in] nearVal Specifies the mapping of the near clipping plane to window coordinates. The
 *		initial value is 0.
 *	@param[in] farVal Specifies the mapping of the far clipping plane to window coordinates. The
 *		initial value is 1.
 */
extern void (GLApi *GLFuncPtrName(glDepthRangef))(GLclampf nearVal, GLclampf farVal);

/**
 *	@brief Detaches a shader object from a program object to which it is attached
 *	@param[in] program Specifies the program object from which to detach the shader object.
 *	@param[in] shader Specifies the shader object to be detached.
 */
extern void (GLApi *GLFuncPtrName(glDetachShader))(GLuint program, GLuint shader);

/**
 *	@brief Enable or disable a generic vertex attribute array
 *	@param[in] index Specifies the index of the generic vertex attribute to be enabled or disabled.
 */
extern void (GLApi *GLFuncPtrName(glDisableVertexAttribArray))(GLuint index);

/**
 *	@brief Enable or disable a generic vertex attribute array
 *	@param[in] index Specifies the index of the generic vertex attribute to be enabled or disabled.
 */
extern void (GLApi *GLFuncPtrName(glEnableVertexAttribArray))(GLuint index);

/**
 *	@brief attach a renderbuffer as a logical buffer to the currently bound framebuffer object
 *	@param[in] target Specifies the framebuffer target. target must be GL_DRAW_FRAMEBUFFER,
 *		GL_READ_FRAMEBUFFER, or GL_FRAMEBUFFER. GL_FRAMEBUFFER is equivalent to GL_DRAW_FRAMEBUFFER.
 *	@param[in] attachment Specifies the attachment point of the framebuffer.
 *	@param[in] renderbuffertarget Specifies the renderbuffer target and must be GL_RENDERBUFFER.
 *	@param[in] renderbuffer Specifies the name of an existing renderbuffer object of type
 *		renderbuffertarget to attach.
 */
extern void (GLApi *GLFuncPtrName(glFramebufferRenderbuffer))(GLenum target, GLenum attachment, GLenum renderbuffertarget, GLuint renderbuffer);

/**
 *	@brief attach a level of a texture object as a logical buffer to the currently bound framebuffer object
 *	@param[in] target Specifies the framebuffer target. target must be GL_DRAW_FRAMEBUFFER,
 *		GL_READ_FRAMEBUFFER, or GL_FRAMEBUFFER. GL_FRAMEBUFFER is equivalent to GL_DRAW_FRAMEBUFFER.
 *	@param[in] attachment Specifies the attachment point of the framebuffer. attachment must be
 *		GL_COLOR_ATTACHMENT i, GL_DEPTH_ATTACHMENT, GL_STENCIL_ATTACHMENT or GL_DEPTH_STENCIL_ATTACHMMENT.
 *	@param[in] textarget For glFramebufferTexture1D, glFramebufferTexture2D and
 *		glFramebufferTexture3D, specifies what type of texture is expected in the texture parameter,
 *		or for cube map textures, which face is to be attached.
 *	@param[in] texture Specifies the texture object to attach to the framebuffer attachment point
 *		named by attachment.
 *	@param[in] level Specifies the mipmap level of texture to attach.
 */
extern void (GLApi *GLFuncPtrName(glFramebufferTexture2D))(GLenum target, GLenum attachment, GLenum textarget, GLuint texture, int level);

/**
 *	@brief generate buffer object names
 *	@param[in] n Specifies the number of buffer object names to be generated.
 *	@param[out] buffers Specifies an array in which the generated buffer object names are stored.
 */
extern void (GLApi *GLFuncPtrName(glGenBuffers))(GLsizei n, GLuint *buffers);

/**
 *	@brief generate mipmaps for a specified texture target
 *	@param[in] target Specifies the target to which the texture whose mimaps to generate is bound.
 *		target must be GL_TEXTURE_1D, GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY,
 *		GL_TEXTURE_2D_ARRAY or GL_TEXTURE_CUBE_MAP.
 */
extern void (GLApi *GLFuncPtrName(glGenerateMipmap))(GLenum target);

/**
 *	@brief generate framebuffer object names
 *	@param[in] n Specifies the number of framebuffer object names to generate.
 *	@param[out] ids Specifies an array in which the generated framebuffer object names are stored.
 */
extern void (GLApi *GLFuncPtrName(glGenFramebuffers))(GLsizei n, GLuint *ids);

/**
 *	@brief generate renderbuffer object names
 *	@param[in] n Specifies the number of renderbuffer object names to generate.
 *	@param[out] renderbuffers Specifies an array in which the generated renderbuffer object names
 *		are stored.
 */
extern void (GLApi *GLFuncPtrName(glGenRenderbuffers))(GLsizei n, GLuint *renderbuffers);

/**
 *	@brief Returns information about an active attribute variable for the specified program object
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] index Specifies the index of the attribute variable to be queried.
 *	@param[in] bufSize Specifies the maximum number of characters OpenGL is allowed to write in the
 *		character buffer indicated by name.
 *	@param[out] length Returns the number of characters actually written by OpenGL in the string
 *		indicated by name (excluding the null terminator) if a value other than NULL is passed.
 *	@param[out] size Returns the size of the attribute variable.
 *	@param[out] type Returns the data type of the attribute variable.
 *	@param[out] name Returns a null terminated string containing the name of the attribute variable.
 */
extern void (GLApi *GLFuncPtrName(glGetActiveAttrib))(GLuint program, GLuint index, GLsizei bufSize, GLsizei *length, GLsizei *size, GLenum *type, GLchar *name);

/**
 *	@brief Returns information about an active uniform variable for the specified program object
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] index Specifies the index of the uniform variable to be queried.
 *	@param[in] bufSize Specifies the maximum number of characters OpenGL is allowed to write in the
 *		character buffer indicated by name.
 *	@param[out] length Returns the number of characters actually written by OpenGL in the string
 *		indicated by name (excluding the null terminator) if a value other than NULL is passed.
 *	@param[out] size Returns the size of the uniform variable.
 *	@param[out] type Returns the data type of the uniform variable.
 *	@param[out] name Returns a null terminated string containing the name of the uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glGetActiveUniform))(GLuint program, GLuint index, GLsizei bufSize, GLsizei *length, GLsizei *size, GLenum *type, GLchar *name);

/**
 *	@brief Returns the handles of the shader objects attached to a program object
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] maxCount Specifies the size of the array for storing the returned object names.
 *	@param[out] count Returns the number of names actually returned in objects.
 *	@param[out] shaders Specifies an array that is used to return the names of attached shader objects.
 */
extern void (GLApi *GLFuncPtrName(glGetAttachedShaders))(GLuint program, GLsizei maxCount, GLsizei *count, GLuint *shaders);

/**
 *	@brief Returns the location of an attribute variable
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] name Points to a null terminated string containing the name of the attribute variable
 *		whose location is to be queried.
 */
extern int (GLApi *GLFuncPtrName(glGetAttribLocation))(GLuint program, const GLchar *name);

/**
 *	@brief return parameters of a buffer object
 *	@param[in] target Specifies the target buffer object. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] value Specifies the symbolic name of a buffer object parameter. Accepted values are
 *		GL_BUFFER_ACCESS, GL_BUFFER_MAPPED, GL_BUFFER_SIZE, or GL_BUFFER_USAGE.
 *	@param[out] data Returns the requested parameter.
 */
extern void (GLApi *GLFuncPtrName(glGetBufferParameteriv))(GLenum target, GLenum value, int *data);

/**
 *	@brief return attachment parameters of a framebuffer object
 *	@param[in] target Specifies the target framebuffer object. The symbolic constant must be GL_FRAMEBUFFER.
 *	@param[in] attachment Specifies the symbolic name of a framebuffer object attachment point.
 *		Accepted values are GL_COLOR_ATTACHMENT0, GL_DEPTH_ATTACHMENT, and GL_STENCIL_ATTACHMENT.
 *	@param[in] pname Specifies the symbolic name of a framebuffer object attachment parameter.
 *		Accepted values are GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE,
 *		GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL, and GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE.
 *	@param[out] params Returns the requested parameter.
 */
extern void (GLApi *GLFuncPtrName(glGetFramebufferAttachmentParameteriv))(GLenum target, GLenum attachment, GLenum pname, int *params);

/**
 *	@brief Returns a parameter from a program object
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] pname Specifies the object parameter. Accepted symbolic names are GL_DELETE_STATUS,
 *		GL_LINK_STATUS, GL_VALIDATE_STATUS, GL_INFO_LOG_LENGTH, GL_ATTACHED_SHADERS,
 *		GL_ACTIVE_ATTRIBUTES, GL_ACTIVE_ATTRIBUTE_MAX_LENGTH, GL_ACTIVE_UNIFORMS,
 *		GL_ACTIVE_UNIFORM_BLOCKS, GL_ACTIVE_UNIFORM_BLOCK_MAX_NAME_LENGTH,
 *		GL_ACTIVE_UNIFORM_MAX_LENGTH, GL_PROGRAM_BINARY_LENGTH, GL_TRANSFORM_FEEDBACK_BUFFER_MODE,
 *		GL_TRANSFORM_FEEDBACK_VARYINGS, GL_TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH,
 *		GL_GEOMETRY_VERTICES_OUT, GL_GEOMETRY_INPUT_TYPE, and GL_GEOMETRY_OUTPUT_TYPE.
 *	@param[out] params Returns the requested object parameter.
 */
extern void (GLApi *GLFuncPtrName(glGetProgramiv))(GLuint program, GLenum pname, int *params);

/**
 *	@brief Returns the information log for a program object
 *	@param[in] program Specifies the program object whose information log is to be queried.
 *	@param[in] bufSize (documantation not found)
 *	@param[out] length Returns the length of the string returned in infoLog (excluding the null terminator).
 *	@param[out] infoLog Specifies an array of characters that is used to return the information log.
 */
extern void (GLApi *GLFuncPtrName(glGetProgramInfoLog))(GLuint program, GLsizei bufSize, GLsizei *length, GLchar *infoLog);

/**
 *	@brief retrieve information about a bound renderbuffer object
 *	@param[in] target Specifies the target of the query operation. target must be GL_RENDERBUFFER.
 *	@param[in] pname Specifies the parameter whose value to retrieve from the renderbuffer bound to target.
 *	@param[out] params Specifies the address of an array to receive the value of the queried parameter.
 */
extern void (GLApi *GLFuncPtrName(glGetRenderbufferParameteriv))(GLenum target, GLenum pname, int *params);

/**
 *	@brief Returns a parameter from a shader object
 *	@param[in] program (documantation not found)
 *	@param[in] pname Specifies the object parameter. Accepted symbolic names are GL_SHADER_TYPE,
 *		GL_DELETE_STATUS, GL_COMPILE_STATUS, GL_INFO_LOG_LENGTH, GL_SHADER_SOURCE_LENGTH.
 *	@param[out] params Returns the requested object parameter.
 */
extern void (GLApi *GLFuncPtrName(glGetShaderiv))(GLuint program, GLenum pname, int *params);

/**
 *	@brief Returns the information log for a shader object
 *	@param[in] shader Specifies the shader object whose information log is to be queried.
 *	@param[in] bufSize (documantation not found)
 *	@param[out] length Returns the length of the string returned in infoLog (excluding the null terminator).
 *	@param[out] infoLog Specifies an array of characters that is used to return the information log.
 */
extern void (GLApi *GLFuncPtrName(glGetShaderInfoLog))(GLuint shader, GLsizei bufSize, GLsizei *length, GLchar *infoLog);

/**
 *	@brief retrieve the range and precision for numeric formats supported by the shader compiler
 *	@param[in] shaderType Specifies the type of shader whose precision to query. shaderType must be
 *		GL_VERTEX_SHADER or GL_FRAGMENT_SHADER.
 *	@param[in] precisionType Specifies the numeric format whose precision and range to query.
 *	@param[out] range Specifies the address of array of two integers into which encodings of the
 *		implementation's numeric range are returned.
 *	@param[out] precision Specifies the address of an integer into which the numeric precision of
 *		the implementation is written.
 */
extern void (GLApi *GLFuncPtrName(glGetShaderPrecisionFormat))(GLenum shaderType, GLenum precisionType, int *range, int *precision);

/**
 *	@brief Returns the source code string from a shader object
 *	@param[in] shader Specifies the shader object to be queried.
 *	@param[in] bufSize Specifies the size of the character buffer for storing the returned source
 *		code string.
 *	@param[out] length Returns the length of the string returned in source (excluding the null terminator).
 *	@param[out] source Specifies an array of characters that is used to return the source code string.
 */
extern void (GLApi *GLFuncPtrName(glGetShaderSource))(GLuint shader, GLsizei bufSize, GLsizei *length, GLchar *source);

/**
 *	@brief Returns the value of a uniform variable
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] location Specifies the location of the uniform variable to be queried.
 *	@param[out] params Returns the value of the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glGetUniformfv))(GLuint program, int location, float *params);

/**
 *	@brief Returns the value of a uniform variable
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] location Specifies the location of the uniform variable to be queried.
 *	@param[out] params Returns the value of the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glGetUniformiv))(GLuint program, int location, int *params);

/**
 *	@brief Returns the location of a uniform variable
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] name Points to a null terminated string containing the name of the uniform variable
 *		whose location is to be queried.
 */
extern int (GLApi *GLFuncPtrName(glGetUniformLocation))(GLuint program, const GLchar *name);

/**
 *	@brief Return a generic vertex attribute parameter
 *	@param[in] index Specifies the generic vertex attribute parameter to be queried.
 *	@param[in] pname Specifies the symbolic name of the vertex attribute parameter to be queried.
 *		Accepted values are GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING, GL_VERTEX_ATTRIB_ARRAY_ENABLED,
 *		GL_VERTEX_ATTRIB_ARRAY_SIZE, GL_VERTEX_ATTRIB_ARRAY_STRIDE, GL_VERTEX_ATTRIB_ARRAY_TYPE,
 *		GL_VERTEX_ATTRIB_ARRAY_NORMALIZED, GL_VERTEX_ATTRIB_ARRAY_INTEGER,
 *		GL_VERTEX_ATTRIB_ARRAY_DIVISOR, or GL_CURRENT_VERTEX_ATTRIB.
 *	@param[out] params Returns the requested data.
 */
extern void (GLApi *GLFuncPtrName(glGetVertexAttribfv))(GLuint index, GLenum pname, float *params);

/**
 *	@brief Return a generic vertex attribute parameter
 *	@param[in] index Specifies the generic vertex attribute parameter to be queried.
 *	@param[in] pname Specifies the symbolic name of the vertex attribute parameter to be queried.
 *		Accepted values are GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING, GL_VERTEX_ATTRIB_ARRAY_ENABLED,
 *		GL_VERTEX_ATTRIB_ARRAY_SIZE, GL_VERTEX_ATTRIB_ARRAY_STRIDE, GL_VERTEX_ATTRIB_ARRAY_TYPE,
 *		GL_VERTEX_ATTRIB_ARRAY_NORMALIZED, GL_VERTEX_ATTRIB_ARRAY_INTEGER,
 *		GL_VERTEX_ATTRIB_ARRAY_DIVISOR, or GL_CURRENT_VERTEX_ATTRIB.
 *	@param[out] params Returns the requested data.
 */
extern void (GLApi *GLFuncPtrName(glGetVertexAttribiv))(GLuint index, GLenum pname, int *params);

/**
 *	@brief return the address of the specified generic vertex attribute pointer
 *	@param[in] index Specifies the generic vertex attribute parameter to be returned.
 *	@param[in] pname Specifies the symbolic name of the generic vertex attribute parameter to be
 *		returned. Must be GL_VERTEX_ATTRIB_ARRAY_POINTER.
 *	@param[out] pointer Returns the pointer value.
 */
extern void (GLApi *GLFuncPtrName(glGetVertexAttribPointerv))(GLuint index, GLenum pname, void **pointer);

/**
 *	@brief determine if a name corresponds to a buffer object
 *	@param[in] buffer Specifies a value that may be the name of a buffer object.
 */
extern GLboolean (GLApi *GLFuncPtrName(glIsBuffer))(GLuint buffer);

/**
 *	@brief determine if a name corresponds to a framebuffer object
 *	@param[in] framebuffer Specifies a value that may be the name of a framebuffer object.
 */
extern GLboolean (GLApi *GLFuncPtrName(glIsFramebuffer))(GLuint framebuffer);

/**
 *	@brief Determines if a name corresponds to a program object
 *	@param[in] program Specifies a potential program object.
 */
extern GLboolean (GLApi *GLFuncPtrName(glIsProgram))(GLuint program);

/**
 *	@brief determine if a name corresponds to a renderbuffer object
 *	@param[in] renderbuffer Specifies a value that may be the name of a renderbuffer object.
 */
extern GLboolean (GLApi *GLFuncPtrName(glIsRenderbuffer))(GLuint renderbuffer);

/**
 *	@brief Determines if a name corresponds to a shader object
 *	@param[in] shader Specifies a potential shader object.
 */
extern GLboolean (GLApi *GLFuncPtrName(glIsShader))(GLuint shader);

/**
 *	@brief Links a program object
 *	@param[in] program Specifies the handle of the program object to be linked.
 */
extern void (GLApi *GLFuncPtrName(glLinkProgram))(GLuint program);

/**
 *	@brief release resources consumed by the implementation's shader compiler
 */
extern void (GLApi *GLFuncPtrName(glReleaseShaderCompiler))();

/**
 *	@brief establish data storage, format and dimensions of a renderbuffer object's image
 *	@param[in] target Specifies a binding to which the target of the allocation and must be GL_RENDERBUFFER.
 *	@param[in] internalformat Specifies the internal format to use for the renderbuffer object's image.
 *	@param[in] width Specifies the width of the renderbuffer, in pixels.
 *	@param[in] height Specifies the height of the renderbuffer, in pixels.
 */
extern void (GLApi *GLFuncPtrName(glRenderbufferStorage))(GLenum target, GLenum internalformat, GLsizei width, GLsizei height);

/**
 *	@brief specify multisample coverage parameters
 *	@param[in] value Specify a single floating-point sample coverage value. The value is clamped to
 *		the range 0 1. The initial value is 1.0.
 *	@param[in] invert Specify a single boolean value representing if the coverage masks should be
 *		inverted. GL_TRUE and GL_FALSE are accepted. The initial value is GL_FALSE.
 */
extern void (GLApi *GLFuncPtrName(glSampleCoverage))(GLclampf value, GLboolean invert);

/**
 *	@brief load pre-compiled shader binaries
 *	@param[in] count Specifies the number of shader object handles contained in shaders.
 *	@param[in] shaders Specifies the address of an array of shader handles into which to load
 *		pre-compiled shader binaries.
 *	@param[in] binaryFormat Specifies the format of the shader binaries contained in binary.
 *	@param[in] binary Specifies the address of an array of bytes containing pre-compiled binary
 *		shader code.
 *	@param[in] length Specifies the length of the array whose address is given in binary.
 */
extern void (GLApi *GLFuncPtrName(glShaderBinary))(GLsizei count, const GLuint *shaders, GLenum binaryFormat, const void *binary, GLsizei length);

/**
 *	@brief Replaces the source code in a shader object
 *	@param[in] shader Specifies the handle of the shader object whose source code is to be replaced.
 *	@param[in] count Specifies the number of elements in the string and length arrays.
 *	@param[in] string Specifies an array of pointers to strings containing the source code to be
 *		loaded into the shader.
 *	@param[in] length Specifies an array of string lengths.
 */
extern void (GLApi *GLFuncPtrName(glShaderSource))(GLuint shader, GLsizei count, const GLchar **string, const int *length);

/**
 *	@brief set front and/or back function and reference value for stencil testing
 *	@param[in] face Specifies whether front and/or back stencil state is updated. Three symbolic
 *		constants are valid: GL_FRONT, GL_BACK, and GL_FRONT_AND_BACK.
 *	@param[in] func Specifies the test function. Eight symbolic constants are valid: GL_NEVER,
 *		GL_LESS, GL_LEQUAL, GL_GREATER, GL_GEQUAL, GL_EQUAL, GL_NOTEQUAL, and GL_ALWAYS. The initial
 *		value is GL_ALWAYS.
 *	@param[in] ref Specifies the reference value for the stencil test. ref is clamped to the range 0
 *		2 n - 1, where n is the number of bitplanes in the stencil buffer. The initial value is 0.
 *	@param[in] mask Specifies a mask that is ANDed with both the reference value and the stored
 *		stencil value when the test is done. The initial value is all 1's.
 */
extern void (GLApi *GLFuncPtrName(glStencilFuncSeparate))(GLenum face, GLenum func, int ref, GLuint mask);

/**
 *	@brief control the front and/or back writing of individual bits in the stencil planes
 *	@param[in] face Specifies whether the front and/or back stencil writemask is updated. Three
 *		symbolic constants are valid: GL_FRONT, GL_BACK, and GL_FRONT_AND_BACK.
 *	@param[in] mask Specifies a bit mask to enable and disable writing of individual bits in the
 *		stencil planes. Initially, the mask is all 1's.
 */
extern void (GLApi *GLFuncPtrName(glStencilMaskSeparate))(GLenum face, GLuint mask);

/**
 *	@brief set front and/or back stencil test actions
 *	@param[in] face Specifies whether front and/or back stencil state is updated. Three symbolic
 *		constants are valid: GL_FRONT, GL_BACK, and GL_FRONT_AND_BACK.
 *	@param[in] fail (documantation not found)
 *	@param[in] zfail (documantation not found)
 *	@param[in] zpass (documantation not found)
 */
extern void (GLApi *GLFuncPtrName(glStencilOpSeparate))(GLenum face, GLenum fail, GLenum zfail, GLenum zpass);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform1f))(int location, float v0);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform2f))(int location, float v0, float v1);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v2 Specifies the new values to be used for the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform3f))(int location, float v0, float v1, float v2);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v2 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v3 Specifies the new values to be used for the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform4f))(int location, float v0, float v1, float v2, float v3);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform1i))(int location, int v0);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform2i))(int location, int v0, int v1);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v2 Specifies the new values to be used for the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform3i))(int location, int v0, int v1, int v2);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v2 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v3 Specifies the new values to be used for the specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform4i))(int location, int v0, int v1, int v2, int v3);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform1fv))(int location, GLsizei count, const float *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform2fv))(int location, GLsizei count, const float *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform3fv))(int location, GLsizei count, const float *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform4fv))(int location, GLsizei count, const float *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform1iv))(int location, GLsizei count, const int *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform2iv))(int location, GLsizei count, const int *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform3iv))(int location, GLsizei count, const int *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniform4iv))(int location, GLsizei count, const int *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of matrices that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array of matrices, and 1 or more if it is an array
 *		of matrices.
 *	@param[in] transpose Specifies whether to transpose the matrix as the values are loaded into the
 *		uniform variable.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniformMatrix2fv))(int location, GLsizei count, GLboolean transpose, const float *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of matrices that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array of matrices, and 1 or more if it is an array
 *		of matrices.
 *	@param[in] transpose Specifies whether to transpose the matrix as the values are loaded into the
 *		uniform variable.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniformMatrix3fv))(int location, GLsizei count, GLboolean transpose, const float *value);

/**
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of matrices that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array of matrices, and 1 or more if it is an array
 *		of matrices.
 *	@param[in] transpose Specifies whether to transpose the matrix as the values are loaded into the
 *		uniform variable.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
extern void (GLApi *GLFuncPtrName(glUniformMatrix4fv))(int location, GLsizei count, GLboolean transpose, const float *value);

/**
 *	@brief Installs a program object as part of current rendering state
 *	@param[in] program Specifies the handle of the program object whose executables are to be used
 *		as part of current rendering state.
 */
extern void (GLApi *GLFuncPtrName(glUseProgram))(GLuint program);

/**
 *	@brief Validates a program object
 *	@param[in] program Specifies the handle of the program object to be validated.
 */
extern void (GLApi *GLFuncPtrName(glValidateProgram))(GLuint program);

/**
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] x (documantation not found)
 */
extern void (GLApi *GLFuncPtrName(glVertexAttrib1f))(GLuint index, float x);

/**
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] v Specifies a pointer to an array of values to be used for the generic vertex attribute.
 */
extern void (GLApi *GLFuncPtrName(glVertexAttrib1fv))(GLuint index, const float *v);

/**
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] x (documantation not found)
 *	@param[in] y (documantation not found)
 */
extern void (GLApi *GLFuncPtrName(glVertexAttrib2f))(GLuint index, float x, float y);

/**
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] v Specifies a pointer to an array of values to be used for the generic vertex attribute.
 */
extern void (GLApi *GLFuncPtrName(glVertexAttrib2fv))(GLuint index, const float *v);

/**
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] x (documantation not found)
 *	@param[in] y (documantation not found)
 *	@param[in] z (documantation not found)
 */
extern void (GLApi *GLFuncPtrName(glVertexAttrib3f))(GLuint index, float x, float y, float z);

/**
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] v Specifies a pointer to an array of values to be used for the generic vertex attribute.
 */
extern void (GLApi *GLFuncPtrName(glVertexAttrib3fv))(GLuint index, const float *v);

/**
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] x (documantation not found)
 *	@param[in] y (documantation not found)
 *	@param[in] z (documantation not found)
 *	@param[in] w (documantation not found)
 */
extern void (GLApi *GLFuncPtrName(glVertexAttrib4f))(GLuint index, float x, float y, float z, float w);

/**
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] v Specifies a pointer to an array of values to be used for the generic vertex attribute.
 */
extern void (GLApi *GLFuncPtrName(glVertexAttrib4fv))(GLuint index, const float *v);

/**
 *	@brief define an array of generic vertex attribute data
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] size Specifies the number of components per generic vertex attribute. Must be 1, 2,
 *		3, 4. Additionally, the symbolic constant GL_BGRA is accepted by glVertexAttribPointer. The
 *		initial value is 4.
 *	@param[in] type Specifies the data type of each component in the array. The symbolic constants
 *		GL_BYTE, GL_UNSIGNED_BYTE, GL_SHORT, GL_UNSIGNED_SHORT, GL_INT, and GL_UNSIGNED_INT are
 *		accepted by both functions. Additionally GL_HALF_FLOAT, GL_FLOAT, GL_DOUBLE, GL_FIXED,
 *		GL_INT_2_10_10_10_REV, and GL_UNSIGNED_INT_2_10_10_10_REV are accepted by
 *		glVertexAttribPointer. The initial value is GL_FLOAT.
 *	@param[in] normalized For glVertexAttribPointer , specifies whether fixed-point data values
 *		should be normalized ( GL_TRUE ) or converted directly as fixed-point values ( GL_FALSE)
 *		when they are accessed.
 *	@param[in] stride Specifies the byte offset between consecutive generic vertex attributes. If
 *		stride is 0, the generic vertex attributes are understood to be tightly packed in the array.
 *		The initial value is 0.
 *	@param[in] pointer Specifies a offset of the first component of the first generic vertex
 *		attribute in the array in the data store of the buffer currently bound to the
 *		GL_ARRAY_BUFFER target. The initial value is 0.
 */
extern void (GLApi *GLFuncPtrName(glVertexAttribPointer))(GLuint index, int size, GLenum type, GLboolean normalized, GLsizei stride, const void *pointer);

inline GLenum glCheckFramebufferStatus_glachwrap(GLenum target, const char *p_s_filename, int n_line) { GLenum result = GLFuncPtrName(glCheckFramebufferStatus)(target); GL_AUTOCHECK_BLOCK("glCheckFramebufferStatus", p_s_filename, n_line); return result; }
inline GLuint glCreateProgram_glachwrap(const char *p_s_filename, int n_line) { GLuint result = GLFuncPtrName(glCreateProgram)(); GL_AUTOCHECK_BLOCK("glCreateProgram", p_s_filename, n_line); return result; }
inline GLuint glCreateShader_glachwrap(GLenum type, const char *p_s_filename, int n_line) { GLuint result = GLFuncPtrName(glCreateShader)(type); GL_AUTOCHECK_BLOCK("glCreateShader", p_s_filename, n_line); return result; }
inline int glGetAttribLocation_glachwrap(GLuint program, const GLchar *name, const char *p_s_filename, int n_line) { int result = GLFuncPtrName(glGetAttribLocation)(program,name); GL_AUTOCHECK_BLOCK("glGetAttribLocation", p_s_filename, n_line); return result; }
inline int glGetUniformLocation_glachwrap(GLuint program, const GLchar *name, const char *p_s_filename, int n_line) { int result = GLFuncPtrName(glGetUniformLocation)(program,name); GL_AUTOCHECK_BLOCK("glGetUniformLocation", p_s_filename, n_line); return result; }
inline GLboolean glIsBuffer_glachwrap(GLuint buffer, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glIsBuffer)(buffer); GL_AUTOCHECK_BLOCK("glIsBuffer", p_s_filename, n_line); return result; }
inline GLboolean glIsFramebuffer_glachwrap(GLuint framebuffer, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glIsFramebuffer)(framebuffer); GL_AUTOCHECK_BLOCK("glIsFramebuffer", p_s_filename, n_line); return result; }
inline GLboolean glIsProgram_glachwrap(GLuint program, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glIsProgram)(program); GL_AUTOCHECK_BLOCK("glIsProgram", p_s_filename, n_line); return result; }
inline GLboolean glIsRenderbuffer_glachwrap(GLuint renderbuffer, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glIsRenderbuffer)(renderbuffer); GL_AUTOCHECK_BLOCK("glIsRenderbuffer", p_s_filename, n_line); return result; }
inline GLboolean glIsShader_glachwrap(GLuint shader, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glIsShader)(shader); GL_AUTOCHECK_BLOCK("glIsShader", p_s_filename, n_line); return result; }

#ifdef __HAVE_GL_AUTOCHECK

inline void glBindTexture_glachwrap(GLenum target, GLuint texture) { glBindTexture(target,texture); }
inline void glBlendFunc_glachwrap(GLenum sfactor, GLenum dfactor) { glBlendFunc(sfactor,dfactor); }
inline void glClear_glachwrap(GLbitfield mask) { glClear(mask); }
inline void glClearColor_glachwrap(GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha) { glClearColor(red,green,blue,alpha); }
inline void glClearStencil_glachwrap(int s) { glClearStencil(s); }
inline void glColorMask_glachwrap(GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha) { glColorMask(red,green,blue,alpha); }
inline void glCopyTexImage2D_glachwrap(GLenum target, int level, GLenum internalformat, int x, int y, GLsizei width, GLsizei height, int border) { glCopyTexImage2D(target,level,internalformat,x,y,width,height,border); }
inline void glCopyTexSubImage2D_glachwrap(GLenum target, int level, int xoffset, int yoffset, int x, int y, GLsizei width, GLsizei height) { glCopyTexSubImage2D(target,level,xoffset,yoffset,x,y,width,height); }
inline void glCullFace_glachwrap(GLenum mode) { glCullFace(mode); }
inline void glDeleteTextures_glachwrap(GLsizei n, const GLuint *textures) { glDeleteTextures(n,textures); }
inline void glDepthFunc_glachwrap(GLenum func) { glDepthFunc(func); }
inline void glDepthMask_glachwrap(GLboolean flag) { glDepthMask(flag); }
inline void glDisable_glachwrap(GLenum cap) { glDisable(cap); }
inline void glDrawArrays_glachwrap(GLenum mode, int first, GLsizei count) { glDrawArrays(mode,first,count); }
inline void glDrawElements_glachwrap(GLenum mode, GLsizei count, GLenum type, const void *indices) { glDrawElements(mode,count,type,indices); }
inline void glEnable_glachwrap(GLenum cap) { glEnable(cap); }
inline void glFinish_glachwrap() { glFinish(); }
inline void glFlush_glachwrap() { glFlush(); }
inline void glFrontFace_glachwrap(GLenum mode) { glFrontFace(mode); }
inline void glGenTextures_glachwrap(GLsizei n, GLuint *textures) { glGenTextures(n,textures); }
inline void glGetBooleanv_glachwrap(GLenum pname, GLboolean *params) { glGetBooleanv(pname,params); }
//inline GLenum glGetError_glachwrap(const char *p_s_filename, int n_line) { GLenum result = glGetError(); GL_AUTOCHECK_BLOCK("glGetError", p_s_filename, n_line); return result; }
inline void glGetFloatv_glachwrap(GLenum pname, float *params) { glGetFloatv(pname,params); }
inline void glGetIntegerv_glachwrap(GLenum pname, int *params) { glGetIntegerv(pname,params); }
inline const GLubyte *glGetString_glachwrap(GLenum name, const char *p_s_filename, int n_line) { const GLubyte *result = glGetString(name); GL_AUTOCHECK_BLOCK("glGetString", p_s_filename, n_line); return result; }
inline void glGetTexParameterfv_glachwrap(GLenum target, GLenum pname, float *params) { glGetTexParameterfv(target,pname,params); }
inline void glGetTexParameteriv_glachwrap(GLenum target, GLenum pname, int *params) { glGetTexParameteriv(target,pname,params); }
inline void glHint_glachwrap(GLenum target, GLenum mode) { glHint(target,mode); }
inline GLboolean glIsEnabled_glachwrap(GLenum cap, const char *p_s_filename, int n_line) { GLboolean result = glIsEnabled(cap); GL_AUTOCHECK_BLOCK("glIsEnabled", p_s_filename, n_line); return result; }
inline GLboolean glIsTexture_glachwrap(GLuint texture, const char *p_s_filename, int n_line) { GLboolean result = glIsTexture(texture); GL_AUTOCHECK_BLOCK("glIsTexture", p_s_filename, n_line); return result; }
inline void glLineWidth_glachwrap(float width) { glLineWidth(width); }
inline void glPixelStorei_glachwrap(GLenum pname, int param) { glPixelStorei(pname,param); }
inline void glPolygonOffset_glachwrap(float factor, float units) { glPolygonOffset(factor,units); }
inline void glReadPixels_glachwrap(int x, int y, GLsizei width, GLsizei height, GLenum format, GLenum type, void *pixels) { glReadPixels(x,y,width,height,format,type,pixels); }
inline void glScissor_glachwrap(int x, int y, GLsizei width, GLsizei height) { glScissor(x,y,width,height); }
inline void glStencilFunc_glachwrap(GLenum func, int ref, GLuint mask) { glStencilFunc(func,ref,mask); }
inline void glStencilMask_glachwrap(GLuint mask) { glStencilMask(mask); }
inline void glStencilOp_glachwrap(GLenum fail, GLenum zfail, GLenum zpass) { glStencilOp(fail,zfail,zpass); }
inline void glTexImage2D_glachwrap(GLenum target, int level, int internalformat, GLsizei width, GLsizei height, int border, GLenum format, GLenum type, const void *pixels) { glTexImage2D(target,level,internalformat,width,height,border,format,type,pixels); }
inline void glTexParameterf_glachwrap(GLenum target, GLenum pname, float param) { glTexParameterf(target,pname,param); }
inline void glTexParameterfv_glachwrap(GLenum target, GLenum pname, const float *params) { glTexParameterfv(target,pname,params); }
inline void glTexParameteri_glachwrap(GLenum target, GLenum pname, int param) { glTexParameteri(target,pname,param); }
inline void glTexParameteriv_glachwrap(GLenum target, GLenum pname, const int *params) { glTexParameteriv(target,pname,params); }
inline void glTexSubImage2D_glachwrap(GLenum target, int level, int xoffset, int yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *pixels) { glTexSubImage2D(target,level,xoffset,yoffset,width,height,format,type,pixels); }
inline void glViewport_glachwrap(int x, int y, GLsizei width, GLsizei height) { glViewport(x,y,width,height); }

/**
 *	@def glBindTexture
 *	@brief bind a named texture to a texturing target
 *	@param[in] target Specifies the target to which the texture is bound. Must be either
 *		GL_TEXTURE_1D, GL_TEXTURE_2D, GL_TEXTURE_3D, or GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, GL_TEXTURE_CUBE_MAP, GL_TEXTURE_2D_MULTISAMPLE or GL_TEXTURE_2D_MULTISAMPLE_ARRAY.
 *	@param[in] texture Specifies the name of a texture.
 */
#define glBindTexture(target,texture) do { glBindTexture_glachwrap(target, texture); GL_AUTOCHECK_BLOCK("glBindTexture", __FILE__, __LINE__); } while(0)

/**
 *	@def glBlendFunc
 *	@brief specify pixel arithmetic
 *	@param[in] sfactor Specifies how the red, green, blue, and alpha source blending factors are
 *		computed. The initial value is GL_ONE.
 *	@param[in] dfactor Specifies how the red, green, blue, and alpha destination blending factors
 *		are computed. The following symbolic constants are accepted: GL_ZERO, GL_ONE, GL_SRC_COLOR,
 *		GL_ONE_MINUS_SRC_COLOR, GL_DST_COLOR, GL_ONE_MINUS_DST_COLOR, GL_SRC_ALPHA,
 *		GL_ONE_MINUS_SRC_ALPHA, GL_DST_ALPHA, GL_ONE_MINUS_DST_ALPHA. GL_CONSTANT_COLOR,
 *		GL_ONE_MINUS_CONSTANT_COLOR, GL_CONSTANT_ALPHA, and GL_ONE_MINUS_CONSTANT_ALPHA. The initial
 *		value is GL_ZERO.
 */
#define glBlendFunc(sfactor,dfactor) do { glBlendFunc_glachwrap(sfactor, dfactor); GL_AUTOCHECK_BLOCK("glBlendFunc", __FILE__, __LINE__); } while(0)

/**
 *	@def glClear
 *	@brief clear buffers to preset values
 *	@param[in] mask Bitwise OR of masks that indicate the buffers to be cleared. The three masks are
 *		GL_COLOR_BUFFER_BIT, GL_DEPTH_BUFFER_BIT, and GL_STENCIL_BUFFER_BIT.
 */
#define glClear(mask) do { glClear_glachwrap(mask); GL_AUTOCHECK_BLOCK("glClear", __FILE__, __LINE__); } while(0)

/**
 *	@def glClearColor
 *	@brief specify clear values for the color buffers
 *	@param[in] red Specify the red, green, blue, and alpha values used when the color buffers are
 *		cleared. The initial values are all 0.
 *	@param[in] green Specify the red, green, blue, and alpha values used when the color buffers are
 *		cleared. The initial values are all 0.
 *	@param[in] blue Specify the red, green, blue, and alpha values used when the color buffers are
 *		cleared. The initial values are all 0.
 *	@param[in] alpha Specify the red, green, blue, and alpha values used when the color buffers are
 *		cleared. The initial values are all 0.
 */
#define glClearColor(red,green,blue,alpha) do { glClearColor_glachwrap(red, green, blue, alpha); GL_AUTOCHECK_BLOCK("glClearColor", __FILE__, __LINE__); } while(0)

/**
 *	@def glClearStencil
 *	@brief specify the clear value for the stencil buffer
 *	@param[in] s Specifies the index used when the stencil buffer is cleared. The initial value is 0.
 */
#define glClearStencil(s) do { glClearStencil_glachwrap(s); GL_AUTOCHECK_BLOCK("glClearStencil", __FILE__, __LINE__); } while(0)

/**
 *	@def glColorMask
 *	@brief enable and disable writing of frame buffer color components
 *	@param[in] red Specify whether red, green, blue, and alpha are to be written into the frame
 *		buffer. The initial values are all GL_TRUE, indicating that the color components are written.
 *	@param[in] green Specify whether red, green, blue, and alpha are to be written into the frame
 *		buffer. The initial values are all GL_TRUE, indicating that the color components are written.
 *	@param[in] blue Specify whether red, green, blue, and alpha are to be written into the frame
 *		buffer. The initial values are all GL_TRUE, indicating that the color components are written.
 *	@param[in] alpha Specify whether red, green, blue, and alpha are to be written into the frame
 *		buffer. The initial values are all GL_TRUE, indicating that the color components are written.
 */
#define glColorMask(red,green,blue,alpha) do { glColorMask_glachwrap(red, green, blue, alpha); GL_AUTOCHECK_BLOCK("glColorMask", __FILE__, __LINE__); } while(0)

/**
 *	@def glCopyTexImage2D
 *	@brief copy pixels into a 2D texture image
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, or GL_TEXTURE_CUBE_MAP_NEGATIVE_Z.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] internalformat Specifies the internal format of the texture. Must be one of the
 *		following symbolic constants: GL_COMPRESSED_RED, GL_COMPRESSED_RG, GL_COMPRESSED_RGB,
 *		GL_COMPRESSED_RGBA. GL_COMPRESSED_SRGB, GL_COMPRESSED_SRGB_ALPHA. GL_DEPTH_COMPONENT,
 *		GL_DEPTH_COMPONENT16, GL_DEPTH_COMPONENT24, GL_DEPTH_COMPONENT32, GL_RED, GL_RG, GL_RGB,
 *		GL_R3_G3_B2, GL_RGB4, GL_RGB5, GL_RGB8, GL_RGB10, GL_RGB12, GL_RGB16, GL_RGBA, GL_RGBA2,
 *		GL_RGBA4, GL_RGB5_A1, GL_RGBA8, GL_RGB10_A2, GL_RGBA12, GL_RGBA16, GL_SRGB, GL_SRGB8,
 *		GL_SRGB_ALPHA, or GL_SRGB8_ALPHA8.
 *	@param[in] x Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] y Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] width Specifies the width of the texture image. Must be 0 or 2 n + 2 border for some
 *		integer n.
 *	@param[in] height Specifies the height of the texture image. Must be 0 or 2 m + 2 border for
 *		some integer m.
 *	@param[in] border Specifies the width of the border. Must be either 0 or 1.
 */
#define glCopyTexImage2D(target,level,internalformat,x,y,width,height,border) do { glCopyTexImage2D_glachwrap(target, level, internalformat, x, y, width, height, border); GL_AUTOCHECK_BLOCK("glCopyTexImage2D", __FILE__, __LINE__); } while(0)

/**
 *	@def glCopyTexSubImage2D
 *	@brief copy a two-dimensional texture subimage
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, or GL_TEXTURE_CUBE_MAP_NEGATIVE_Z.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] x Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] y Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 */
#define glCopyTexSubImage2D(target,level,xoffset,yoffset,x,y,width,height) do { glCopyTexSubImage2D_glachwrap(target, level, xoffset, yoffset, x, y, width, height); GL_AUTOCHECK_BLOCK("glCopyTexSubImage2D", __FILE__, __LINE__); } while(0)

/**
 *	@def glCullFace
 *	@brief specify whether front- or back-facing facets can be culled
 *	@param[in] mode Specifies whether front- or back-facing facets are candidates for culling.
 *		Symbolic constants GL_FRONT, GL_BACK, and GL_FRONT_AND_BACK are accepted. The initial value
 *		is GL_BACK.
 */
#define glCullFace(mode) do { glCullFace_glachwrap(mode); GL_AUTOCHECK_BLOCK("glCullFace", __FILE__, __LINE__); } while(0)

/**
 *	@def glDeleteTextures
 *	@brief delete named textures
 *	@param[in] n Specifies the number of textures to be deleted.
 *	@param[in] textures Specifies an array of textures to be deleted.
 */
#define glDeleteTextures(n,textures) do { glDeleteTextures_glachwrap(n, textures); GL_AUTOCHECK_BLOCK("glDeleteTextures", __FILE__, __LINE__); } while(0)

/**
 *	@def glDepthFunc
 *	@brief specify the value used for depth buffer comparisons
 *	@param[in] func Specifies the depth comparison function. Symbolic constants GL_NEVER, GL_LESS,
 *		GL_EQUAL, GL_LEQUAL, GL_GREATER, GL_NOTEQUAL, GL_GEQUAL, and GL_ALWAYS are accepted. The
 *		initial value is GL_LESS.
 */
#define glDepthFunc(func) do { glDepthFunc_glachwrap(func); GL_AUTOCHECK_BLOCK("glDepthFunc", __FILE__, __LINE__); } while(0)

/**
 *	@def glDepthMask
 *	@brief enable or disable writing into the depth buffer
 *	@param[in] flag Specifies whether the depth buffer is enabled for writing. If flag is GL_FALSE,
 *		depth buffer writing is disabled. Otherwise, it is enabled. Initially, depth buffer writing
 *		is enabled.
 */
#define glDepthMask(flag) do { glDepthMask_glachwrap(flag); GL_AUTOCHECK_BLOCK("glDepthMask", __FILE__, __LINE__); } while(0)

/**
 *	@def glDisable
 *	@brief enable or disable server-side GL capabilities
 *	@param[in] cap Specifies a symbolic constant indicating a GL capability.
 */
#define glDisable(cap) do { glDisable_glachwrap(cap); GL_AUTOCHECK_BLOCK("glDisable", __FILE__, __LINE__); } while(0)

/**
 *	@def glDrawArrays
 *	@brief render primitives from array data
 *	@param[in] mode Specifies what kind of primitives to render. Symbolic constants GL_POINTS,
 *		GL_LINE_STRIP, GL_LINE_LOOP, GL_LINES, GL_LINE_STRIP_ADJACENCY, GL_LINES_ADJACENCY,
 *		GL_TRIANGLE_STRIP, GL_TRIANGLE_FAN, GL_TRIANGLES, GL_TRIANGLE_STRIP_ADJACENCY,
 *		GL_TRIANGLES_ADJACENCY and GL_PATCHES are accepted.
 *	@param[in] first Specifies the starting index in the enabled arrays.
 *	@param[in] count Specifies the number of indices to be rendered.
 */
#define glDrawArrays(mode,first,count) do { glDrawArrays_glachwrap(mode, first, count); GL_AUTOCHECK_BLOCK("glDrawArrays", __FILE__, __LINE__); } while(0)

/**
 *	@def glDrawElements
 *	@brief render primitives from array data
 *	@param[in] mode Specifies what kind of primitives to render. Symbolic constants GL_POINTS,
 *		GL_LINE_STRIP, GL_LINE_LOOP, GL_LINES, GL_LINE_STRIP_ADJACENCY, GL_LINES_ADJACENCY,
 *		GL_TRIANGLE_STRIP, GL_TRIANGLE_FAN, GL_TRIANGLES, GL_TRIANGLE_STRIP_ADJACENCY,
 *		GL_TRIANGLES_ADJACENCY and GL_PATCHES are accepted.
 *	@param[in] count Specifies the number of elements to be rendered.
 *	@param[in] type Specifies the type of the values in indices. Must be one of GL_UNSIGNED_BYTE,
 *		GL_UNSIGNED_SHORT, or GL_UNSIGNED_INT.
 *	@param[in] indices Specifies a pointer to the location where the indices are stored.
 */
#define glDrawElements(mode,count,type,indices) do { glDrawElements_glachwrap(mode, count, type, indices); GL_AUTOCHECK_BLOCK("glDrawElements", __FILE__, __LINE__); } while(0)

/**
 *	@def glEnable
 *	@brief enable or disable server-side GL capabilities
 *	@param[in] cap Specifies a symbolic constant indicating a GL capability.
 */
#define glEnable(cap) do { glEnable_glachwrap(cap); GL_AUTOCHECK_BLOCK("glEnable", __FILE__, __LINE__); } while(0)

/**
 *	@def glFinish
 *	@brief block until all GL execution is complete
 */
#define glFinish() do { glFinish_glachwrap(); GL_AUTOCHECK_BLOCK("glFinish", __FILE__, __LINE__); } while(0)

/**
 *	@def glFlush
 *	@brief force execution of GL commands in finite time
 */
#define glFlush() do { glFlush_glachwrap(); GL_AUTOCHECK_BLOCK("glFlush", __FILE__, __LINE__); } while(0)

/**
 *	@def glFrontFace
 *	@brief define front- and back-facing polygons
 *	@param[in] mode Specifies the orientation of front-facing polygons. GL_CW and GL_CCW are
 *		accepted. The initial value is GL_CCW.
 */
#define glFrontFace(mode) do { glFrontFace_glachwrap(mode); GL_AUTOCHECK_BLOCK("glFrontFace", __FILE__, __LINE__); } while(0)

/**
 *	@def glGenTextures
 *	@brief generate texture names
 *	@param[in] n Specifies the number of texture names to be generated.
 *	@param[out] textures Specifies an array in which the generated texture names are stored.
 */
#define glGenTextures(n,textures) do { glGenTextures_glachwrap(n, textures); GL_AUTOCHECK_BLOCK("glGenTextures", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetBooleanv
 *	@brief return the value or values of a selected parameter
 *	@param[in] pname Specifies the parameter value to be returned. The symbolic constants in the
 *		list below are accepted.
 *	@param[out] params Returns the value or values of the specified parameter.
 */
#define glGetBooleanv(pname,params) do { glGetBooleanv_glachwrap(pname, params); GL_AUTOCHECK_BLOCK("glGetBooleanv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetError
 *	@brief return error information
 */
//#define glGetError() glGetError_glachwrap(__FILE__, __LINE__)

/**
 *	@def glGetFloatv
 *	@brief return the value or values of a selected parameter
 *	@param[in] pname Specifies the parameter value to be returned. The symbolic constants in the
 *		list below are accepted.
 *	@param[out] params Returns the value or values of the specified parameter.
 */
#define glGetFloatv(pname,params) do { glGetFloatv_glachwrap(pname, params); GL_AUTOCHECK_BLOCK("glGetFloatv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetIntegerv
 *	@brief return the value or values of a selected parameter
 *	@param[in] pname Specifies the parameter value to be returned. The symbolic constants in the
 *		list below are accepted.
 *	@param[out] params Returns the value or values of the specified parameter.
 */
#define glGetIntegerv(pname,params) do { glGetIntegerv_glachwrap(pname, params); GL_AUTOCHECK_BLOCK("glGetIntegerv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetString
 *	@brief return a string describing the current GL connection
 *	@param[in] name Specifies a symbolic constant, one of GL_VENDOR, GL_RENDERER, GL_VERSION, or
 *		GL_SHADING_LANGUAGE_VERSION. Additionally, glGetStringi accepts the GL_EXTENSIONS token.
 */
#define glGetString(name) glGetString_glachwrap(name, __FILE__, __LINE__)

/**
 *	@def glGetTexParameterfv
 *	@brief return texture parameter values
 *	@param[in] target Specifies the symbolic name of the target texture. GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY, GL_TEXTURE_3D,
 *		GL_TEXTURE_RECTANGLE, and GL_TEXTURE_CUBE_MAP are accepted.
 *	@param[in] pname Specifies the symbolic name of a texture parameter. GL_TEXTURE_BASE_LEVEL,
 *		GL_TEXTURE_BORDER_COLOR, GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MAG_FILTER, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_MAX_LOD,
 *		GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MIN_LOD, GL_TEXTURE_SWIZZLE_R, GL_TEXTURE_SWIZZLE_G,
 *		GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_SWIZZLE_RGBA, GL_TEXTURE_WRAP_S,
 *		GL_TEXTURE_WRAP_T, and GL_TEXTURE_WRAP_R are accepted.
 *	@param[out] params Returns the texture parameters.
 */
#define glGetTexParameterfv(target,pname,params) do { glGetTexParameterfv_glachwrap(target, pname, params); GL_AUTOCHECK_BLOCK("glGetTexParameterfv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetTexParameteriv
 *	@brief return texture parameter values
 *	@param[in] target Specifies the symbolic name of the target texture. GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY, GL_TEXTURE_3D,
 *		GL_TEXTURE_RECTANGLE, and GL_TEXTURE_CUBE_MAP are accepted.
 *	@param[in] pname Specifies the symbolic name of a texture parameter. GL_TEXTURE_BASE_LEVEL,
 *		GL_TEXTURE_BORDER_COLOR, GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MAG_FILTER, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_MAX_LOD,
 *		GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MIN_LOD, GL_TEXTURE_SWIZZLE_R, GL_TEXTURE_SWIZZLE_G,
 *		GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_SWIZZLE_RGBA, GL_TEXTURE_WRAP_S,
 *		GL_TEXTURE_WRAP_T, and GL_TEXTURE_WRAP_R are accepted.
 *	@param[out] params Returns the texture parameters.
 */
#define glGetTexParameteriv(target,pname,params) do { glGetTexParameteriv_glachwrap(target, pname, params); GL_AUTOCHECK_BLOCK("glGetTexParameteriv", __FILE__, __LINE__); } while(0)

/**
 *	@def glHint
 *	@brief specify implementation-specific hints
 *	@param[in] target Specifies a symbolic constant indicating the behavior to be controlled.
 *		GL_LINE_SMOOTH_HINT, GL_POLYGON_SMOOTH_HINT, GL_TEXTURE_COMPRESSION_HINT, and
 *		GL_FRAGMENT_SHADER_DERIVATIVE_HINT are accepted.
 *	@param[in] mode Specifies a symbolic constant indicating the desired behavior. GL_FASTEST,
 *		GL_NICEST, and GL_DONT_CARE are accepted.
 */
#define glHint(target,mode) do { glHint_glachwrap(target, mode); GL_AUTOCHECK_BLOCK("glHint", __FILE__, __LINE__); } while(0)

/**
 *	@def glIsEnabled
 *	@brief test whether a capability is enabled
 *	@param[in] cap Specifies a symbolic constant indicating a GL capability.
 */
#define glIsEnabled(cap) glIsEnabled_glachwrap(cap, __FILE__, __LINE__)

/**
 *	@def glIsTexture
 *	@brief determine if a name corresponds to a texture
 *	@param[in] texture Specifies a value that may be the name of a texture.
 */
#define glIsTexture(texture) glIsTexture_glachwrap(texture, __FILE__, __LINE__)

/**
 *	@def glLineWidth
 *	@brief specify the width of rasterized lines
 *	@param[in] width Specifies the width of rasterized lines. The initial value is 1.
 */
#define glLineWidth(width) do { glLineWidth_glachwrap(width); GL_AUTOCHECK_BLOCK("glLineWidth", __FILE__, __LINE__); } while(0)

/**
 *	@def glPixelStorei
 *	@brief set pixel storage modes
 *	@param[in] pname Specifies the symbolic name of the parameter to be set. Six values affect the
 *		packing of pixel data into memory: GL_PACK_SWAP_BYTES, GL_PACK_LSB_FIRST,
 *		GL_PACK_ROW_LENGTH, GL_PACK_IMAGE_HEIGHT, GL_PACK_SKIP_PIXELS, GL_PACK_SKIP_ROWS,
 *		GL_PACK_SKIP_IMAGES, and GL_PACK_ALIGNMENT. Six more affect the unpacking of pixel data from
 *		memory: GL_UNPACK_SWAP_BYTES, GL_UNPACK_LSB_FIRST, GL_UNPACK_ROW_LENGTH,
 *		GL_UNPACK_IMAGE_HEIGHT, GL_UNPACK_SKIP_PIXELS, GL_UNPACK_SKIP_ROWS, GL_UNPACK_SKIP_IMAGES,
 *		and GL_UNPACK_ALIGNMENT.
 *	@param[in] param Specifies the value that pname is set to.
 */
#define glPixelStorei(pname,param) do { glPixelStorei_glachwrap(pname, param); GL_AUTOCHECK_BLOCK("glPixelStorei", __FILE__, __LINE__); } while(0)

/**
 *	@def glPolygonOffset
 *	@brief set the scale and units used to calculate depth values
 *	@param[in] factor Specifies a scale factor that is used to create a variable depth offset for
 *		each polygon. The initial value is 0.
 *	@param[in] units Is multiplied by an implementation-specific value to create a constant depth
 *		offset. The initial value is 0.
 */
#define glPolygonOffset(factor,units) do { glPolygonOffset_glachwrap(factor, units); GL_AUTOCHECK_BLOCK("glPolygonOffset", __FILE__, __LINE__); } while(0)

/**
 *	@def glReadPixels
 *	@brief read a block of pixels from the frame buffer
 *	@param[in] x Specify the window coordinates of the first pixel that is read from the frame
 *		buffer. This location is the lower left corner of a rectangular block of pixels.
 *	@param[in] y Specify the window coordinates of the first pixel that is read from the frame
 *		buffer. This location is the lower left corner of a rectangular block of pixels.
 *	@param[in] width Specify the dimensions of the pixel rectangle. width and height of one
 *		correspond to a single pixel.
 *	@param[in] height Specify the dimensions of the pixel rectangle. width and height of one
 *		correspond to a single pixel.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_STENCIL_INDEX, GL_DEPTH_COMPONENT, GL_DEPTH_STENCIL, GL_RED, GL_GREEN, GL_BLUE,
 *		GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. Must be one of GL_UNSIGNED_BYTE,
 *		GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT, GL_HALF_FLOAT, GL_FLOAT,
 *		GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, GL_UNSIGNED_INT_2_10_10_10_REV,
 *		GL_UNSIGNED_INT_24_8, GL_UNSIGNED_INT_10F_11F_11F_REV, GL_UNSIGNED_INT_5_9_9_9_REV, or GL_FLOAT_32_UNSIGNED_INT_24_8_REV.
 *	@param[out] pixels (documantation not found)
 */
#define glReadPixels(x,y,width,height,format,type,pixels) do { glReadPixels_glachwrap(x, y, width, height, format, type, pixels); GL_AUTOCHECK_BLOCK("glReadPixels", __FILE__, __LINE__); } while(0)

/**
 *	@def glScissor
 *	@brief define the scissor box
 *	@param[in] x Specify the lower left corner of the scissor box. Initially (0, 0).
 *	@param[in] y Specify the lower left corner of the scissor box. Initially (0, 0).
 *	@param[in] width Specify the width and height of the scissor box. When a GL context is first
 *		attached to a window, width and height are set to the dimensions of that window.
 *	@param[in] height Specify the width and height of the scissor box. When a GL context is first
 *		attached to a window, width and height are set to the dimensions of that window.
 */
#define glScissor(x,y,width,height) do { glScissor_glachwrap(x, y, width, height); GL_AUTOCHECK_BLOCK("glScissor", __FILE__, __LINE__); } while(0)

/**
 *	@def glStencilFunc
 *	@brief set front and back function and reference value for stencil testing
 *	@param[in] func Specifies the test function. Eight symbolic constants are valid: GL_NEVER,
 *		GL_LESS, GL_LEQUAL, GL_GREATER, GL_GEQUAL, GL_EQUAL, GL_NOTEQUAL, and GL_ALWAYS. The initial
 *		value is GL_ALWAYS.
 *	@param[in] ref Specifies the reference value for the stencil test. ref is clamped to the range 0
 *		2 n - 1, where n is the number of bitplanes in the stencil buffer. The initial value is 0.
 *	@param[in] mask Specifies a mask that is ANDed with both the reference value and the stored
 *		stencil value when the test is done. The initial value is all 1's.
 */
#define glStencilFunc(func,ref,mask) do { glStencilFunc_glachwrap(func, ref, mask); GL_AUTOCHECK_BLOCK("glStencilFunc", __FILE__, __LINE__); } while(0)

/**
 *	@def glStencilMask
 *	@brief control the front and back writing of individual bits in the stencil planes
 *	@param[in] mask Specifies a bit mask to enable and disable writing of individual bits in the
 *		stencil planes. Initially, the mask is all 1's.
 */
#define glStencilMask(mask) do { glStencilMask_glachwrap(mask); GL_AUTOCHECK_BLOCK("glStencilMask", __FILE__, __LINE__); } while(0)

/**
 *	@def glStencilOp
 *	@brief set front and back stencil test actions
 *	@param[in] fail (documantation not found)
 *	@param[in] zfail (documantation not found)
 *	@param[in] zpass (documantation not found)
 */
#define glStencilOp(fail,zfail,zpass) do { glStencilOp_glachwrap(fail, zfail, zpass); GL_AUTOCHECK_BLOCK("glStencilOp", __FILE__, __LINE__); } while(0)

/**
 *	@def glTexImage2D
 *	@brief specify a two-dimensional texture image
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D, GL_PROXY_TEXTURE_2D,
 *		GL_TEXTURE_1D_ARRAY, GL_PROXY_TEXTURE_1D_ARRAY, GL_TEXTURE_RECTANGLE,
 *		GL_PROXY_TEXTURE_RECTANGLE, GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, GL_TEXTURE_CUBE_MAP_NEGATIVE_Z, or GL_PROXY_TEXTURE_CUBE_MAP.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image. If target is GL_TEXTURE_RECTANGLE or
 *		GL_PROXY_TEXTURE_RECTANGLE, level must be 0.
 *	@param[in] internalformat (documantation not found)
 *	@param[in] width Specifies the width of the texture image. All implementations support texture
 *		images that are at least 1024 texels wide.
 *	@param[in] height Specifies the height of the texture image, or the number of layers in a
 *		texture array, in the case of the GL_TEXTURE_1D_ARRAY and GL_PROXY_TEXTURE_1D_ARRAY targets.
 *		All implementations support 2D texture images that are at least 1024 texels high, and
 *		texture arrays that are at least 256 layers deep.
 *	@param[in] border This value must be 0.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_RED, GL_RG, GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. The following symbolic values are
 *		accepted: GL_UNSIGNED_BYTE, GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT,
 *		GL_FLOAT, GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, and GL_UNSIGNED_INT_2_10_10_10_REV.
 *	@param[in] pixels (documantation not found)
 */
#define glTexImage2D(target,level,internalformat,width,height,border,format,type,pixels) do { glTexImage2D_glachwrap(target, level, internalformat, width, height, border, format, type, pixels); GL_AUTOCHECK_BLOCK("glTexImage2D", __FILE__, __LINE__); } while(0)

/**
 *	@def glTexParameterf
 *	@brief set texture parameters
 *	@param[in] target Specifies the target texture, which must be either GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, or GL_TEXTURE_CUBE_MAP.
 *	@param[in] pname Specifies the symbolic name of a single-valued texture parameter. pname can be
 *		one of the following: GL_TEXTURE_BASE_LEVEL, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
 *		GL_TEXTURE_MIN_LOD, GL_TEXTURE_MAX_LOD, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_SWIZZLE_R,
 *		GL_TEXTURE_SWIZZLE_G, GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_WRAP_S,
 *		GL_TEXTURE_WRAP_T, or GL_TEXTURE_WRAP_R.
 *	@param[in] param Specifies the value of pname.
 */
#define glTexParameterf(target,pname,param) do { glTexParameterf_glachwrap(target, pname, param); GL_AUTOCHECK_BLOCK("glTexParameterf", __FILE__, __LINE__); } while(0)

/**
 *	@def glTexParameterfv
 *	@brief set texture parameters
 *	@param[in] target Specifies the target texture, which must be either GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, or GL_TEXTURE_CUBE_MAP.
 *	@param[in] pname Specifies the symbolic name of a texture parameter. pname can be one of the
 *		following: GL_TEXTURE_BASE_LEVEL, GL_TEXTURE_BORDER_COLOR, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
 *		GL_TEXTURE_MIN_LOD, GL_TEXTURE_MAX_LOD, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_SWIZZLE_R,
 *		GL_TEXTURE_SWIZZLE_G, GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_SWIZZLE_RGBA,
 *		GL_TEXTURE_WRAP_S, GL_TEXTURE_WRAP_T, or GL_TEXTURE_WRAP_R.
 *	@param[in] params Specifies a pointer to an array where the value or values of pname are stored.
 */
#define glTexParameterfv(target,pname,params) do { glTexParameterfv_glachwrap(target, pname, params); GL_AUTOCHECK_BLOCK("glTexParameterfv", __FILE__, __LINE__); } while(0)

/**
 *	@def glTexParameteri
 *	@brief set texture parameters
 *	@param[in] target Specifies the target texture, which must be either GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, or GL_TEXTURE_CUBE_MAP.
 *	@param[in] pname Specifies the symbolic name of a single-valued texture parameter. pname can be
 *		one of the following: GL_TEXTURE_BASE_LEVEL, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
 *		GL_TEXTURE_MIN_LOD, GL_TEXTURE_MAX_LOD, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_SWIZZLE_R,
 *		GL_TEXTURE_SWIZZLE_G, GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_WRAP_S,
 *		GL_TEXTURE_WRAP_T, or GL_TEXTURE_WRAP_R.
 *	@param[in] param Specifies the value of pname.
 */
#define glTexParameteri(target,pname,param) do { glTexParameteri_glachwrap(target, pname, param); GL_AUTOCHECK_BLOCK("glTexParameteri", __FILE__, __LINE__); } while(0)

/**
 *	@def glTexParameteriv
 *	@brief set texture parameters
 *	@param[in] target Specifies the target texture, which must be either GL_TEXTURE_1D,
 *		GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY, GL_TEXTURE_2D_ARRAY,
 *		GL_TEXTURE_RECTANGLE, or GL_TEXTURE_CUBE_MAP.
 *	@param[in] pname Specifies the symbolic name of a texture parameter. pname can be one of the
 *		following: GL_TEXTURE_BASE_LEVEL, GL_TEXTURE_BORDER_COLOR, GL_TEXTURE_COMPARE_FUNC,
 *		GL_TEXTURE_COMPARE_MODE, GL_TEXTURE_LOD_BIAS, GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
 *		GL_TEXTURE_MIN_LOD, GL_TEXTURE_MAX_LOD, GL_TEXTURE_MAX_LEVEL, GL_TEXTURE_SWIZZLE_R,
 *		GL_TEXTURE_SWIZZLE_G, GL_TEXTURE_SWIZZLE_B, GL_TEXTURE_SWIZZLE_A, GL_TEXTURE_SWIZZLE_RGBA,
 *		GL_TEXTURE_WRAP_S, GL_TEXTURE_WRAP_T, or GL_TEXTURE_WRAP_R.
 *	@param[in] params Specifies a pointer to an array where the value or values of pname are stored.
 */
#define glTexParameteriv(target,pname,params) do { glTexParameteriv_glachwrap(target, pname, params); GL_AUTOCHECK_BLOCK("glTexParameteriv", __FILE__, __LINE__); } while(0)

/**
 *	@def glTexSubImage2D
 *	@brief specify a two-dimensional texture subimage
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, or GL_TEXTURE_CUBE_MAP_NEGATIVE_Z.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_RED, GL_RG, GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. The following symbolic values are
 *		accepted: GL_UNSIGNED_BYTE, GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT,
 *		GL_FLOAT, GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, and GL_UNSIGNED_INT_2_10_10_10_REV.
 *	@param[in] pixels (documantation not found)
 */
#define glTexSubImage2D(target,level,xoffset,yoffset,width,height,format,type,pixels) do { glTexSubImage2D_glachwrap(target, level, xoffset, yoffset, width, height, format, type, pixels); GL_AUTOCHECK_BLOCK("glTexSubImage2D", __FILE__, __LINE__); } while(0)

/**
 *	@def glViewport
 *	@brief set the viewport
 *	@param[in] x Specify the lower left corner of the viewport rectangle, in pixels. The initial
 *		value is (0,0).
 *	@param[in] y Specify the lower left corner of the viewport rectangle, in pixels. The initial
 *		value is (0,0).
 *	@param[in] width Specify the width and height of the viewport. When a GL context is first
 *		attached to a window, width and height are set to the dimensions of that window.
 *	@param[in] height Specify the width and height of the viewport. When a GL context is first
 *		attached to a window, width and height are set to the dimensions of that window.
 */
#define glViewport(x,y,width,height) do { glViewport_glachwrap(x, y, width, height); GL_AUTOCHECK_BLOCK("glViewport", __FILE__, __LINE__); } while(0)

#endif // __HAVE_GL_AUTOCHECK

/**
 *	@def glActiveTexture
 *	@brief select active texture unit
 *	@param[in] texture Specifies which texture unit to make active. The number of texture units is
 *		implementation dependent, but must be at least 80. texture must be one of GL_TEXTURE i,
 *		where i ranges from 0 ( GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS - 1). The initial value is GL_TEXTURE0.
 */
#define glActiveTexture(texture) do { GLFuncPtrName(glActiveTexture)(texture); GL_AUTOCHECK_BLOCK("glActiveTexture", __FILE__, __LINE__); } while(0)

/**
 *	@def glAttachShader
 *	@brief Attaches a shader object to a program object
 *	@param[in] program Specifies the program object to which a shader object will be attached.
 *	@param[in] shader Specifies the shader object that is to be attached.
 */
#define glAttachShader(program,shader) do { GLFuncPtrName(glAttachShader)(program, shader); GL_AUTOCHECK_BLOCK("glAttachShader", __FILE__, __LINE__); } while(0)

/**
 *	@def glBindAttribLocation
 *	@brief Associates a generic vertex attribute index with a named attribute variable
 *	@param[in] program Specifies the handle of the program object in which the association is to be made.
 *	@param[in] index Specifies the index of the generic vertex attribute to be bound.
 *	@param[in] name Specifies a null terminated string containing the name of the vertex shader
 *		attribute variable to which index is to be bound.
 */
#define glBindAttribLocation(program,index,name) do { GLFuncPtrName(glBindAttribLocation)(program, index, name); GL_AUTOCHECK_BLOCK("glBindAttribLocation", __FILE__, __LINE__); } while(0)

/**
 *	@def glBindBuffer
 *	@brief bind a named buffer object
 *	@param[in] target Specifies the target to which the buffer object is bound. The symbolic
 *		constant must be GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER,
 *		GL_DRAW_INDIRECT_BUFFER, GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER,
 *		GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] buffer Specifies the name of a buffer object.
 */
#define glBindBuffer(target,buffer) do { GLFuncPtrName(glBindBuffer)(target, buffer); GL_AUTOCHECK_BLOCK("glBindBuffer", __FILE__, __LINE__); } while(0)

/**
 *	@def glBindFramebuffer
 *	@brief bind a framebuffer to a framebuffer target
 *	@param[in] target Specifies the framebuffer target of the binding operation.
 *	@param[in] framebuffer Specifies the name of the framebuffer object to bind.
 */
#define glBindFramebuffer(target,framebuffer) do { GLFuncPtrName(glBindFramebuffer)(target, framebuffer); GL_AUTOCHECK_BLOCK("glBindFramebuffer", __FILE__, __LINE__); } while(0)

/**
 *	@def glBindRenderbuffer
 *	@brief bind a renderbuffer to a renderbuffer target
 *	@param[in] target Specifies the renderbuffer target of the binding operation. target must be GL_RENDERBUFFER.
 *	@param[in] renderbuffer Specifies the name of the renderbuffer object to bind.
 */
#define glBindRenderbuffer(target,renderbuffer) do { GLFuncPtrName(glBindRenderbuffer)(target, renderbuffer); GL_AUTOCHECK_BLOCK("glBindRenderbuffer", __FILE__, __LINE__); } while(0)

/**
 *	@def glBlendColor
 *	@brief set the blend color
 *	@param[in] red specify the components of GL_BLEND_COLOR
 *	@param[in] green specify the components of GL_BLEND_COLOR
 *	@param[in] blue specify the components of GL_BLEND_COLOR
 *	@param[in] alpha specify the components of GL_BLEND_COLOR
 */
#define glBlendColor(red,green,blue,alpha) do { GLFuncPtrName(glBlendColor)(red, green, blue, alpha); GL_AUTOCHECK_BLOCK("glBlendColor", __FILE__, __LINE__); } while(0)

/**
 *	@def glBlendEquation
 *	@brief specify the equation used for both the RGB blend equation and the Alpha blend equation
 *	@param[in] mode specifies how source and destination colors are combined. It must be
 *		GL_FUNC_ADD, GL_FUNC_SUBTRACT, GL_FUNC_REVERSE_SUBTRACT, GL_MIN, GL_MAX.
 */
#define glBlendEquation(mode) do { GLFuncPtrName(glBlendEquation)(mode); GL_AUTOCHECK_BLOCK("glBlendEquation", __FILE__, __LINE__); } while(0)

/**
 *	@def glBlendEquationSeparate
 *	@brief set the RGB blend equation and the alpha blend equation separately
 *	@param[in] modeRGB specifies the RGB blend equation, how the red, green, and blue components of
 *		the source and destination colors are combined. It must be GL_FUNC_ADD, GL_FUNC_SUBTRACT,
 *		GL_FUNC_REVERSE_SUBTRACT, GL_MIN, GL_MAX.
 *	@param[in] modeAlpha specifies the alpha blend equation, how the alpha component of the source
 *		and destination colors are combined. It must be GL_FUNC_ADD, GL_FUNC_SUBTRACT,
 *		GL_FUNC_REVERSE_SUBTRACT, GL_MIN, GL_MAX.
 */
#define glBlendEquationSeparate(modeRGB,modeAlpha) do { GLFuncPtrName(glBlendEquationSeparate)(modeRGB, modeAlpha); GL_AUTOCHECK_BLOCK("glBlendEquationSeparate", __FILE__, __LINE__); } while(0)

/**
 *	@def glBlendFuncSeparate
 *	@brief specify pixel arithmetic for RGB and alpha components separately
 *	@param[in] srcRGB Specifies how the red, green, and blue blending factors are computed. The
 *		initial value is GL_ONE.
 *	@param[in] dstRGB Specifies how the red, green, and blue destination blending factors are
 *		computed. The initial value is GL_ZERO.
 *	@param[in] srcAlpha Specified how the alpha source blending factor is computed. The initial
 *		value is GL_ONE.
 *	@param[in] dstAlpha Specified how the alpha destination blending factor is computed. The initial
 *		value is GL_ZERO.
 */
#define glBlendFuncSeparate(srcRGB,dstRGB,srcAlpha,dstAlpha) do { GLFuncPtrName(glBlendFuncSeparate)(srcRGB, dstRGB, srcAlpha, dstAlpha); GL_AUTOCHECK_BLOCK("glBlendFuncSeparate", __FILE__, __LINE__); } while(0)

/**
 *	@def glBufferData
 *	@brief creates and initializes a buffer object's data store
 *	@param[in] target Specifies the target buffer object. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] size Specifies the size in bytes of the buffer object's new data store.
 *	@param[in] data Specifies a pointer to data that will be copied into the data store for
 *		initialization, or NULL if no data is to be copied.
 *	@param[in] usage Specifies the expected usage pattern of the data store. The symbolic constant
 *		must be GL_STREAM_DRAW, GL_STREAM_READ, GL_STREAM_COPY, GL_STATIC_DRAW, GL_STATIC_READ,
 *		GL_STATIC_COPY, GL_DYNAMIC_DRAW, GL_DYNAMIC_READ, or GL_DYNAMIC_COPY.
 */
#define glBufferData(target,size,data,usage) do { GLFuncPtrName(glBufferData)(target, size, data, usage); GL_AUTOCHECK_BLOCK("glBufferData", __FILE__, __LINE__); } while(0)

/**
 *	@def glBufferSubData
 *	@brief updates a subset of a buffer object's data store
 *	@param[in] target Specifies the target buffer object. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] offset Specifies the offset into the buffer object's data store where data
 *		replacement will begin, measured in bytes.
 *	@param[in] size Specifies the size in bytes of the data store region being replaced.
 *	@param[in] data Specifies a pointer to the new data that will be copied into the data store.
 */
#define glBufferSubData(target,offset,size,data) do { GLFuncPtrName(glBufferSubData)(target, offset, size, data); GL_AUTOCHECK_BLOCK("glBufferSubData", __FILE__, __LINE__); } while(0)

/**
 *	@def glCheckFramebufferStatus
 *	@brief check the completeness status of a framebuffer
 *	@param[in] target Specify the target of the framebuffer completeness check.
 */
#define glCheckFramebufferStatus(target) GLAutoCheckFuncPtrName(glCheckFramebufferStatus)(target GLAutoCheckExtraParams)

/**
 *	@def glClearDepthf
 *	@brief specify the clear value for the depth buffer
 *	@param[in] depth Specifies the depth value used when the depth buffer is cleared. The initial
 *		value is 1.
 */
#define glClearDepthf(depth) do { GLFuncPtrName(glClearDepthf)(depth); GL_AUTOCHECK_BLOCK("glClearDepthf", __FILE__, __LINE__); } while(0)

/**
 *	@def glCompileShader
 *	@brief Compiles a shader object
 *	@param[in] shader Specifies the shader object to be compiled.
 */
#define glCompileShader(shader) do { GLFuncPtrName(glCompileShader)(shader); GL_AUTOCHECK_BLOCK("glCompileShader", __FILE__, __LINE__); } while(0)

/**
 *	@def glCompressedTexImage2D
 *	@brief specify a two-dimensional texture image in a compressed format
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D, GL_PROXY_TEXTURE_2D,
 *		GL_TEXTURE_1D_ARRAY, GL_PROXY_TEXTURE_1D_ARRAY, GL_TEXTURE_CUBE_MAP_POSITIVE_X,
 *		GL_TEXTURE_CUBE_MAP_NEGATIVE_X, GL_TEXTURE_CUBE_MAP_POSITIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_NEGATIVE_Y, GL_TEXTURE_CUBE_MAP_POSITIVE_Z,
 *		GL_TEXTURE_CUBE_MAP_NEGATIVE_Z, or GL_PROXY_TEXTURE_CUBE_MAP.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] internalformat Specifies the format of the compressed image data stored at address data.
 *	@param[in] width Specifies the width of the texture image. All implementations support 2D
 *		texture images that are at least 64 texels wide and cube-mapped texture images that are at
 *		least 16 texels wide.
 *	@param[in] height Specifies the height of the texture image. All implementations support 2D
 *		texture images that are at least 64 texels high and cube-mapped texture images that are at
 *		least 16 texels high.
 *	@param[in] border This value must be 0.
 *	@param[in] imageSize Specifies the number of unsigned bytes of image data starting at the
 *		address specified by data.
 *	@param[in] data Specifies a pointer to the compressed image data in memory.
 */
#define glCompressedTexImage2D(target,level,internalformat,width,height,border,imageSize,data) do { GLFuncPtrName(glCompressedTexImage2D)(target, level, internalformat, width, height, border, imageSize, data); GL_AUTOCHECK_BLOCK("glCompressedTexImage2D", __FILE__, __LINE__); } while(0)

/**
 *	@def glCompressedTexSubImage2D
 *	@brief specify a two-dimensional texture subimage in a compressed format
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_2D,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
 *		GL_TEXTURE_CUBE_MAP_POSITIVE_Z, or GL_TEXTURE_CUBE_MAP_NEGATIVE_Z.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 *	@param[in] format Specifies the format of the compressed image data stored at address data.
 *	@param[in] imageSize Specifies the number of unsigned bytes of image data starting at the
 *		address specified by data.
 *	@param[in] data Specifies a pointer to the compressed image data in memory.
 */
#define glCompressedTexSubImage2D(target,level,xoffset,yoffset,width,height,format,imageSize,data) do { GLFuncPtrName(glCompressedTexSubImage2D)(target, level, xoffset, yoffset, width, height, format, imageSize, data); GL_AUTOCHECK_BLOCK("glCompressedTexSubImage2D", __FILE__, __LINE__); } while(0)

/**
 *	@def glCreateProgram
 *	@brief Creates a program object
 */
#define glCreateProgram() GLAutoCheckFuncPtrName(glCreateProgram)(GLAutoCheckExtraParamsSA)

/**
 *	@def glCreateShader
 *	@brief Creates a shader object
 *	@param[in] type (documantation not found)
 */
#define glCreateShader(type) GLAutoCheckFuncPtrName(glCreateShader)(type GLAutoCheckExtraParams)

/**
 *	@def glDeleteBuffers
 *	@brief delete named buffer objects
 *	@param[in] n Specifies the number of buffer objects to be deleted.
 *	@param[in] buffers Specifies an array of buffer objects to be deleted.
 */
#define glDeleteBuffers(n,buffers) do { GLFuncPtrName(glDeleteBuffers)(n, buffers); GL_AUTOCHECK_BLOCK("glDeleteBuffers", __FILE__, __LINE__); } while(0)

/**
 *	@def glDeleteFramebuffers
 *	@brief delete framebuffer objects
 *	@param[in] n Specifies the number of framebuffer objects to be deleted.
 *	@param[in] framebuffers A pointer to an array containing n framebuffer objects to be deleted.
 */
#define glDeleteFramebuffers(n,framebuffers) do { GLFuncPtrName(glDeleteFramebuffers)(n, framebuffers); GL_AUTOCHECK_BLOCK("glDeleteFramebuffers", __FILE__, __LINE__); } while(0)

/**
 *	@def glDeleteProgram
 *	@brief Deletes a program object
 *	@param[in] program Specifies the program object to be deleted.
 */
#define glDeleteProgram(program) do { GLFuncPtrName(glDeleteProgram)(program); GL_AUTOCHECK_BLOCK("glDeleteProgram", __FILE__, __LINE__); } while(0)

/**
 *	@def glDeleteRenderbuffers
 *	@brief delete renderbuffer objects
 *	@param[in] n Specifies the number of renderbuffer objects to be deleted.
 *	@param[in] renderbuffers A pointer to an array containing n renderbuffer objects to be deleted.
 */
#define glDeleteRenderbuffers(n,renderbuffers) do { GLFuncPtrName(glDeleteRenderbuffers)(n, renderbuffers); GL_AUTOCHECK_BLOCK("glDeleteRenderbuffers", __FILE__, __LINE__); } while(0)

/**
 *	@def glDeleteShader
 *	@brief Deletes a shader object
 *	@param[in] shader Specifies the shader object to be deleted.
 */
#define glDeleteShader(shader) do { GLFuncPtrName(glDeleteShader)(shader); GL_AUTOCHECK_BLOCK("glDeleteShader", __FILE__, __LINE__); } while(0)

/**
 *	@def glDepthRangef
 *	@brief specify mapping of depth values from normalized device coordinates to window coordinates
 *	@param[in] nearVal Specifies the mapping of the near clipping plane to window coordinates. The
 *		initial value is 0.
 *	@param[in] farVal Specifies the mapping of the far clipping plane to window coordinates. The
 *		initial value is 1.
 */
#define glDepthRangef(nearVal,farVal) do { GLFuncPtrName(glDepthRangef)(nearVal, farVal); GL_AUTOCHECK_BLOCK("glDepthRangef", __FILE__, __LINE__); } while(0)

/**
 *	@def glDetachShader
 *	@brief Detaches a shader object from a program object to which it is attached
 *	@param[in] program Specifies the program object from which to detach the shader object.
 *	@param[in] shader Specifies the shader object to be detached.
 */
#define glDetachShader(program,shader) do { GLFuncPtrName(glDetachShader)(program, shader); GL_AUTOCHECK_BLOCK("glDetachShader", __FILE__, __LINE__); } while(0)

/**
 *	@def glDisableVertexAttribArray
 *	@brief Enable or disable a generic vertex attribute array
 *	@param[in] index Specifies the index of the generic vertex attribute to be enabled or disabled.
 */
#define glDisableVertexAttribArray(index) do { GLFuncPtrName(glDisableVertexAttribArray)(index); GL_AUTOCHECK_BLOCK("glDisableVertexAttribArray", __FILE__, __LINE__); } while(0)

/**
 *	@def glEnableVertexAttribArray
 *	@brief Enable or disable a generic vertex attribute array
 *	@param[in] index Specifies the index of the generic vertex attribute to be enabled or disabled.
 */
#define glEnableVertexAttribArray(index) do { GLFuncPtrName(glEnableVertexAttribArray)(index); GL_AUTOCHECK_BLOCK("glEnableVertexAttribArray", __FILE__, __LINE__); } while(0)

/**
 *	@def glFramebufferRenderbuffer
 *	@brief attach a renderbuffer as a logical buffer to the currently bound framebuffer object
 *	@param[in] target Specifies the framebuffer target. target must be GL_DRAW_FRAMEBUFFER,
 *		GL_READ_FRAMEBUFFER, or GL_FRAMEBUFFER. GL_FRAMEBUFFER is equivalent to GL_DRAW_FRAMEBUFFER.
 *	@param[in] attachment Specifies the attachment point of the framebuffer.
 *	@param[in] renderbuffertarget Specifies the renderbuffer target and must be GL_RENDERBUFFER.
 *	@param[in] renderbuffer Specifies the name of an existing renderbuffer object of type
 *		renderbuffertarget to attach.
 */
#define glFramebufferRenderbuffer(target,attachment,renderbuffertarget,renderbuffer) do { GLFuncPtrName(glFramebufferRenderbuffer)(target, attachment, renderbuffertarget, renderbuffer); GL_AUTOCHECK_BLOCK("glFramebufferRenderbuffer", __FILE__, __LINE__); } while(0)

/**
 *	@def glFramebufferTexture2D
 *	@brief attach a level of a texture object as a logical buffer to the currently bound framebuffer object
 *	@param[in] target Specifies the framebuffer target. target must be GL_DRAW_FRAMEBUFFER,
 *		GL_READ_FRAMEBUFFER, or GL_FRAMEBUFFER. GL_FRAMEBUFFER is equivalent to GL_DRAW_FRAMEBUFFER.
 *	@param[in] attachment Specifies the attachment point of the framebuffer. attachment must be
 *		GL_COLOR_ATTACHMENT i, GL_DEPTH_ATTACHMENT, GL_STENCIL_ATTACHMENT or GL_DEPTH_STENCIL_ATTACHMMENT.
 *	@param[in] textarget For glFramebufferTexture1D, glFramebufferTexture2D and
 *		glFramebufferTexture3D, specifies what type of texture is expected in the texture parameter,
 *		or for cube map textures, which face is to be attached.
 *	@param[in] texture Specifies the texture object to attach to the framebuffer attachment point
 *		named by attachment.
 *	@param[in] level Specifies the mipmap level of texture to attach.
 */
#define glFramebufferTexture2D(target,attachment,textarget,texture,level) do { GLFuncPtrName(glFramebufferTexture2D)(target, attachment, textarget, texture, level); GL_AUTOCHECK_BLOCK("glFramebufferTexture2D", __FILE__, __LINE__); } while(0)

/**
 *	@def glGenBuffers
 *	@brief generate buffer object names
 *	@param[in] n Specifies the number of buffer object names to be generated.
 *	@param[out] buffers Specifies an array in which the generated buffer object names are stored.
 */
#define glGenBuffers(n,buffers) do { GLFuncPtrName(glGenBuffers)(n, buffers); GL_AUTOCHECK_BLOCK("glGenBuffers", __FILE__, __LINE__); } while(0)

/**
 *	@def glGenerateMipmap
 *	@brief generate mipmaps for a specified texture target
 *	@param[in] target Specifies the target to which the texture whose mimaps to generate is bound.
 *		target must be GL_TEXTURE_1D, GL_TEXTURE_2D, GL_TEXTURE_3D, GL_TEXTURE_1D_ARRAY,
 *		GL_TEXTURE_2D_ARRAY or GL_TEXTURE_CUBE_MAP.
 */
#define glGenerateMipmap(target) do { GLFuncPtrName(glGenerateMipmap)(target); GL_AUTOCHECK_BLOCK("glGenerateMipmap", __FILE__, __LINE__); } while(0)

/**
 *	@def glGenFramebuffers
 *	@brief generate framebuffer object names
 *	@param[in] n Specifies the number of framebuffer object names to generate.
 *	@param[out] ids Specifies an array in which the generated framebuffer object names are stored.
 */
#define glGenFramebuffers(n,ids) do { GLFuncPtrName(glGenFramebuffers)(n, ids); GL_AUTOCHECK_BLOCK("glGenFramebuffers", __FILE__, __LINE__); } while(0)

/**
 *	@def glGenRenderbuffers
 *	@brief generate renderbuffer object names
 *	@param[in] n Specifies the number of renderbuffer object names to generate.
 *	@param[out] renderbuffers Specifies an array in which the generated renderbuffer object names
 *		are stored.
 */
#define glGenRenderbuffers(n,renderbuffers) do { GLFuncPtrName(glGenRenderbuffers)(n, renderbuffers); GL_AUTOCHECK_BLOCK("glGenRenderbuffers", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetActiveAttrib
 *	@brief Returns information about an active attribute variable for the specified program object
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] index Specifies the index of the attribute variable to be queried.
 *	@param[in] bufSize Specifies the maximum number of characters OpenGL is allowed to write in the
 *		character buffer indicated by name.
 *	@param[out] length Returns the number of characters actually written by OpenGL in the string
 *		indicated by name (excluding the null terminator) if a value other than NULL is passed.
 *	@param[out] size Returns the size of the attribute variable.
 *	@param[out] type Returns the data type of the attribute variable.
 *	@param[out] name Returns a null terminated string containing the name of the attribute variable.
 */
#define glGetActiveAttrib(program,index,bufSize,length,size,type,name) do { GLFuncPtrName(glGetActiveAttrib)(program, index, bufSize, length, size, type, name); GL_AUTOCHECK_BLOCK("glGetActiveAttrib", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetActiveUniform
 *	@brief Returns information about an active uniform variable for the specified program object
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] index Specifies the index of the uniform variable to be queried.
 *	@param[in] bufSize Specifies the maximum number of characters OpenGL is allowed to write in the
 *		character buffer indicated by name.
 *	@param[out] length Returns the number of characters actually written by OpenGL in the string
 *		indicated by name (excluding the null terminator) if a value other than NULL is passed.
 *	@param[out] size Returns the size of the uniform variable.
 *	@param[out] type Returns the data type of the uniform variable.
 *	@param[out] name Returns a null terminated string containing the name of the uniform variable.
 */
#define glGetActiveUniform(program,index,bufSize,length,size,type,name) do { GLFuncPtrName(glGetActiveUniform)(program, index, bufSize, length, size, type, name); GL_AUTOCHECK_BLOCK("glGetActiveUniform", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetAttachedShaders
 *	@brief Returns the handles of the shader objects attached to a program object
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] maxCount Specifies the size of the array for storing the returned object names.
 *	@param[out] count Returns the number of names actually returned in objects.
 *	@param[out] shaders Specifies an array that is used to return the names of attached shader objects.
 */
#define glGetAttachedShaders(program,maxCount,count,shaders) do { GLFuncPtrName(glGetAttachedShaders)(program, maxCount, count, shaders); GL_AUTOCHECK_BLOCK("glGetAttachedShaders", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetAttribLocation
 *	@brief Returns the location of an attribute variable
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] name Points to a null terminated string containing the name of the attribute variable
 *		whose location is to be queried.
 */
#define glGetAttribLocation(program,name) GLAutoCheckFuncPtrName(glGetAttribLocation)(program, name GLAutoCheckExtraParams)

/**
 *	@def glGetBufferParameteriv
 *	@brief return parameters of a buffer object
 *	@param[in] target Specifies the target buffer object. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] value Specifies the symbolic name of a buffer object parameter. Accepted values are
 *		GL_BUFFER_ACCESS, GL_BUFFER_MAPPED, GL_BUFFER_SIZE, or GL_BUFFER_USAGE.
 *	@param[out] data Returns the requested parameter.
 */
#define glGetBufferParameteriv(target,value,data) do { GLFuncPtrName(glGetBufferParameteriv)(target, value, data); GL_AUTOCHECK_BLOCK("glGetBufferParameteriv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetFramebufferAttachmentParameteriv
 *	@brief return attachment parameters of a framebuffer object
 *	@param[in] target Specifies the target framebuffer object. The symbolic constant must be GL_FRAMEBUFFER.
 *	@param[in] attachment Specifies the symbolic name of a framebuffer object attachment point.
 *		Accepted values are GL_COLOR_ATTACHMENT0, GL_DEPTH_ATTACHMENT, and GL_STENCIL_ATTACHMENT.
 *	@param[in] pname Specifies the symbolic name of a framebuffer object attachment parameter.
 *		Accepted values are GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE,
 *		GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL, and GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE.
 *	@param[out] params Returns the requested parameter.
 */
#define glGetFramebufferAttachmentParameteriv(target,attachment,pname,params) do { GLFuncPtrName(glGetFramebufferAttachmentParameteriv)(target, attachment, pname, params); GL_AUTOCHECK_BLOCK("glGetFramebufferAttachmentParameteriv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetProgramiv
 *	@brief Returns a parameter from a program object
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] pname Specifies the object parameter. Accepted symbolic names are GL_DELETE_STATUS,
 *		GL_LINK_STATUS, GL_VALIDATE_STATUS, GL_INFO_LOG_LENGTH, GL_ATTACHED_SHADERS,
 *		GL_ACTIVE_ATTRIBUTES, GL_ACTIVE_ATTRIBUTE_MAX_LENGTH, GL_ACTIVE_UNIFORMS,
 *		GL_ACTIVE_UNIFORM_BLOCKS, GL_ACTIVE_UNIFORM_BLOCK_MAX_NAME_LENGTH,
 *		GL_ACTIVE_UNIFORM_MAX_LENGTH, GL_PROGRAM_BINARY_LENGTH, GL_TRANSFORM_FEEDBACK_BUFFER_MODE,
 *		GL_TRANSFORM_FEEDBACK_VARYINGS, GL_TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH,
 *		GL_GEOMETRY_VERTICES_OUT, GL_GEOMETRY_INPUT_TYPE, and GL_GEOMETRY_OUTPUT_TYPE.
 *	@param[out] params Returns the requested object parameter.
 */
#define glGetProgramiv(program,pname,params) do { GLFuncPtrName(glGetProgramiv)(program, pname, params); GL_AUTOCHECK_BLOCK("glGetProgramiv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetProgramInfoLog
 *	@brief Returns the information log for a program object
 *	@param[in] program Specifies the program object whose information log is to be queried.
 *	@param[in] bufSize (documantation not found)
 *	@param[out] length Returns the length of the string returned in infoLog (excluding the null terminator).
 *	@param[out] infoLog Specifies an array of characters that is used to return the information log.
 */
#define glGetProgramInfoLog(program,bufSize,length,infoLog) do { GLFuncPtrName(glGetProgramInfoLog)(program, bufSize, length, infoLog); GL_AUTOCHECK_BLOCK("glGetProgramInfoLog", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetRenderbufferParameteriv
 *	@brief retrieve information about a bound renderbuffer object
 *	@param[in] target Specifies the target of the query operation. target must be GL_RENDERBUFFER.
 *	@param[in] pname Specifies the parameter whose value to retrieve from the renderbuffer bound to target.
 *	@param[out] params Specifies the address of an array to receive the value of the queried parameter.
 */
#define glGetRenderbufferParameteriv(target,pname,params) do { GLFuncPtrName(glGetRenderbufferParameteriv)(target, pname, params); GL_AUTOCHECK_BLOCK("glGetRenderbufferParameteriv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetShaderiv
 *	@brief Returns a parameter from a shader object
 *	@param[in] program (documantation not found)
 *	@param[in] pname Specifies the object parameter. Accepted symbolic names are GL_SHADER_TYPE,
 *		GL_DELETE_STATUS, GL_COMPILE_STATUS, GL_INFO_LOG_LENGTH, GL_SHADER_SOURCE_LENGTH.
 *	@param[out] params Returns the requested object parameter.
 */
#define glGetShaderiv(program,pname,params) do { GLFuncPtrName(glGetShaderiv)(program, pname, params); GL_AUTOCHECK_BLOCK("glGetShaderiv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetShaderInfoLog
 *	@brief Returns the information log for a shader object
 *	@param[in] shader Specifies the shader object whose information log is to be queried.
 *	@param[in] bufSize (documantation not found)
 *	@param[out] length Returns the length of the string returned in infoLog (excluding the null terminator).
 *	@param[out] infoLog Specifies an array of characters that is used to return the information log.
 */
#define glGetShaderInfoLog(shader,bufSize,length,infoLog) do { GLFuncPtrName(glGetShaderInfoLog)(shader, bufSize, length, infoLog); GL_AUTOCHECK_BLOCK("glGetShaderInfoLog", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetShaderPrecisionFormat
 *	@brief retrieve the range and precision for numeric formats supported by the shader compiler
 *	@param[in] shaderType Specifies the type of shader whose precision to query. shaderType must be
 *		GL_VERTEX_SHADER or GL_FRAGMENT_SHADER.
 *	@param[in] precisionType Specifies the numeric format whose precision and range to query.
 *	@param[out] range Specifies the address of array of two integers into which encodings of the
 *		implementation's numeric range are returned.
 *	@param[out] precision Specifies the address of an integer into which the numeric precision of
 *		the implementation is written.
 */
#define glGetShaderPrecisionFormat(shaderType,precisionType,range,precision) do { GLFuncPtrName(glGetShaderPrecisionFormat)(shaderType, precisionType, range, precision); GL_AUTOCHECK_BLOCK("glGetShaderPrecisionFormat", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetShaderSource
 *	@brief Returns the source code string from a shader object
 *	@param[in] shader Specifies the shader object to be queried.
 *	@param[in] bufSize Specifies the size of the character buffer for storing the returned source
 *		code string.
 *	@param[out] length Returns the length of the string returned in source (excluding the null terminator).
 *	@param[out] source Specifies an array of characters that is used to return the source code string.
 */
#define glGetShaderSource(shader,bufSize,length,source) do { GLFuncPtrName(glGetShaderSource)(shader, bufSize, length, source); GL_AUTOCHECK_BLOCK("glGetShaderSource", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetUniformfv
 *	@brief Returns the value of a uniform variable
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] location Specifies the location of the uniform variable to be queried.
 *	@param[out] params Returns the value of the specified uniform variable.
 */
#define glGetUniformfv(program,location,params) do { GLFuncPtrName(glGetUniformfv)(program, location, params); GL_AUTOCHECK_BLOCK("glGetUniformfv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetUniformiv
 *	@brief Returns the value of a uniform variable
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] location Specifies the location of the uniform variable to be queried.
 *	@param[out] params Returns the value of the specified uniform variable.
 */
#define glGetUniformiv(program,location,params) do { GLFuncPtrName(glGetUniformiv)(program, location, params); GL_AUTOCHECK_BLOCK("glGetUniformiv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetUniformLocation
 *	@brief Returns the location of a uniform variable
 *	@param[in] program Specifies the program object to be queried.
 *	@param[in] name Points to a null terminated string containing the name of the uniform variable
 *		whose location is to be queried.
 */
#define glGetUniformLocation(program,name) GLAutoCheckFuncPtrName(glGetUniformLocation)(program, name GLAutoCheckExtraParams)

/**
 *	@def glGetVertexAttribfv
 *	@brief Return a generic vertex attribute parameter
 *	@param[in] index Specifies the generic vertex attribute parameter to be queried.
 *	@param[in] pname Specifies the symbolic name of the vertex attribute parameter to be queried.
 *		Accepted values are GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING, GL_VERTEX_ATTRIB_ARRAY_ENABLED,
 *		GL_VERTEX_ATTRIB_ARRAY_SIZE, GL_VERTEX_ATTRIB_ARRAY_STRIDE, GL_VERTEX_ATTRIB_ARRAY_TYPE,
 *		GL_VERTEX_ATTRIB_ARRAY_NORMALIZED, GL_VERTEX_ATTRIB_ARRAY_INTEGER,
 *		GL_VERTEX_ATTRIB_ARRAY_DIVISOR, or GL_CURRENT_VERTEX_ATTRIB.
 *	@param[out] params Returns the requested data.
 */
#define glGetVertexAttribfv(index,pname,params) do { GLFuncPtrName(glGetVertexAttribfv)(index, pname, params); GL_AUTOCHECK_BLOCK("glGetVertexAttribfv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetVertexAttribiv
 *	@brief Return a generic vertex attribute parameter
 *	@param[in] index Specifies the generic vertex attribute parameter to be queried.
 *	@param[in] pname Specifies the symbolic name of the vertex attribute parameter to be queried.
 *		Accepted values are GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING, GL_VERTEX_ATTRIB_ARRAY_ENABLED,
 *		GL_VERTEX_ATTRIB_ARRAY_SIZE, GL_VERTEX_ATTRIB_ARRAY_STRIDE, GL_VERTEX_ATTRIB_ARRAY_TYPE,
 *		GL_VERTEX_ATTRIB_ARRAY_NORMALIZED, GL_VERTEX_ATTRIB_ARRAY_INTEGER,
 *		GL_VERTEX_ATTRIB_ARRAY_DIVISOR, or GL_CURRENT_VERTEX_ATTRIB.
 *	@param[out] params Returns the requested data.
 */
#define glGetVertexAttribiv(index,pname,params) do { GLFuncPtrName(glGetVertexAttribiv)(index, pname, params); GL_AUTOCHECK_BLOCK("glGetVertexAttribiv", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetVertexAttribPointerv
 *	@brief return the address of the specified generic vertex attribute pointer
 *	@param[in] index Specifies the generic vertex attribute parameter to be returned.
 *	@param[in] pname Specifies the symbolic name of the generic vertex attribute parameter to be
 *		returned. Must be GL_VERTEX_ATTRIB_ARRAY_POINTER.
 *	@param[out] pointer Returns the pointer value.
 */
#define glGetVertexAttribPointerv(index,pname,pointer) do { GLFuncPtrName(glGetVertexAttribPointerv)(index, pname, pointer); GL_AUTOCHECK_BLOCK("glGetVertexAttribPointerv", __FILE__, __LINE__); } while(0)

/**
 *	@def glIsBuffer
 *	@brief determine if a name corresponds to a buffer object
 *	@param[in] buffer Specifies a value that may be the name of a buffer object.
 */
#define glIsBuffer(buffer) GLAutoCheckFuncPtrName(glIsBuffer)(buffer GLAutoCheckExtraParams)

/**
 *	@def glIsFramebuffer
 *	@brief determine if a name corresponds to a framebuffer object
 *	@param[in] framebuffer Specifies a value that may be the name of a framebuffer object.
 */
#define glIsFramebuffer(framebuffer) GLAutoCheckFuncPtrName(glIsFramebuffer)(framebuffer GLAutoCheckExtraParams)

/**
 *	@def glIsProgram
 *	@brief Determines if a name corresponds to a program object
 *	@param[in] program Specifies a potential program object.
 */
#define glIsProgram(program) GLAutoCheckFuncPtrName(glIsProgram)(program GLAutoCheckExtraParams)

/**
 *	@def glIsRenderbuffer
 *	@brief determine if a name corresponds to a renderbuffer object
 *	@param[in] renderbuffer Specifies a value that may be the name of a renderbuffer object.
 */
#define glIsRenderbuffer(renderbuffer) GLAutoCheckFuncPtrName(glIsRenderbuffer)(renderbuffer GLAutoCheckExtraParams)

/**
 *	@def glIsShader
 *	@brief Determines if a name corresponds to a shader object
 *	@param[in] shader Specifies a potential shader object.
 */
#define glIsShader(shader) GLAutoCheckFuncPtrName(glIsShader)(shader GLAutoCheckExtraParams)

/**
 *	@def glLinkProgram
 *	@brief Links a program object
 *	@param[in] program Specifies the handle of the program object to be linked.
 */
#define glLinkProgram(program) do { GLFuncPtrName(glLinkProgram)(program); GL_AUTOCHECK_BLOCK("glLinkProgram", __FILE__, __LINE__); } while(0)

/**
 *	@def glReleaseShaderCompiler
 *	@brief release resources consumed by the implementation's shader compiler
 */
#define glReleaseShaderCompiler() do { GLFuncPtrName(glReleaseShaderCompiler)(); GL_AUTOCHECK_BLOCK("glReleaseShaderCompiler", __FILE__, __LINE__); } while(0)

/**
 *	@def glRenderbufferStorage
 *	@brief establish data storage, format and dimensions of a renderbuffer object's image
 *	@param[in] target Specifies a binding to which the target of the allocation and must be GL_RENDERBUFFER.
 *	@param[in] internalformat Specifies the internal format to use for the renderbuffer object's image.
 *	@param[in] width Specifies the width of the renderbuffer, in pixels.
 *	@param[in] height Specifies the height of the renderbuffer, in pixels.
 */
#define glRenderbufferStorage(target,internalformat,width,height) do { GLFuncPtrName(glRenderbufferStorage)(target, internalformat, width, height); GL_AUTOCHECK_BLOCK("glRenderbufferStorage", __FILE__, __LINE__); } while(0)

/**
 *	@def glSampleCoverage
 *	@brief specify multisample coverage parameters
 *	@param[in] value Specify a single floating-point sample coverage value. The value is clamped to
 *		the range 0 1. The initial value is 1.0.
 *	@param[in] invert Specify a single boolean value representing if the coverage masks should be
 *		inverted. GL_TRUE and GL_FALSE are accepted. The initial value is GL_FALSE.
 */
#define glSampleCoverage(value,invert) do { GLFuncPtrName(glSampleCoverage)(value, invert); GL_AUTOCHECK_BLOCK("glSampleCoverage", __FILE__, __LINE__); } while(0)

/**
 *	@def glShaderBinary
 *	@brief load pre-compiled shader binaries
 *	@param[in] count Specifies the number of shader object handles contained in shaders.
 *	@param[in] shaders Specifies the address of an array of shader handles into which to load
 *		pre-compiled shader binaries.
 *	@param[in] binaryFormat Specifies the format of the shader binaries contained in binary.
 *	@param[in] binary Specifies the address of an array of bytes containing pre-compiled binary
 *		shader code.
 *	@param[in] length Specifies the length of the array whose address is given in binary.
 */
#define glShaderBinary(count,shaders,binaryFormat,binary,length) do { GLFuncPtrName(glShaderBinary)(count, shaders, binaryFormat, binary, length); GL_AUTOCHECK_BLOCK("glShaderBinary", __FILE__, __LINE__); } while(0)

/**
 *	@def glShaderSource
 *	@brief Replaces the source code in a shader object
 *	@param[in] shader Specifies the handle of the shader object whose source code is to be replaced.
 *	@param[in] count Specifies the number of elements in the string and length arrays.
 *	@param[in] string Specifies an array of pointers to strings containing the source code to be
 *		loaded into the shader.
 *	@param[in] length Specifies an array of string lengths.
 */
#define glShaderSource(shader,count,string,length) do { GLFuncPtrName(glShaderSource)(shader, count, string, length); GL_AUTOCHECK_BLOCK("glShaderSource", __FILE__, __LINE__); } while(0)

/**
 *	@def glStencilFuncSeparate
 *	@brief set front and/or back function and reference value for stencil testing
 *	@param[in] face Specifies whether front and/or back stencil state is updated. Three symbolic
 *		constants are valid: GL_FRONT, GL_BACK, and GL_FRONT_AND_BACK.
 *	@param[in] func Specifies the test function. Eight symbolic constants are valid: GL_NEVER,
 *		GL_LESS, GL_LEQUAL, GL_GREATER, GL_GEQUAL, GL_EQUAL, GL_NOTEQUAL, and GL_ALWAYS. The initial
 *		value is GL_ALWAYS.
 *	@param[in] ref Specifies the reference value for the stencil test. ref is clamped to the range 0
 *		2 n - 1, where n is the number of bitplanes in the stencil buffer. The initial value is 0.
 *	@param[in] mask Specifies a mask that is ANDed with both the reference value and the stored
 *		stencil value when the test is done. The initial value is all 1's.
 */
#define glStencilFuncSeparate(face,func,ref,mask) do { GLFuncPtrName(glStencilFuncSeparate)(face, func, ref, mask); GL_AUTOCHECK_BLOCK("glStencilFuncSeparate", __FILE__, __LINE__); } while(0)

/**
 *	@def glStencilMaskSeparate
 *	@brief control the front and/or back writing of individual bits in the stencil planes
 *	@param[in] face Specifies whether the front and/or back stencil writemask is updated. Three
 *		symbolic constants are valid: GL_FRONT, GL_BACK, and GL_FRONT_AND_BACK.
 *	@param[in] mask Specifies a bit mask to enable and disable writing of individual bits in the
 *		stencil planes. Initially, the mask is all 1's.
 */
#define glStencilMaskSeparate(face,mask) do { GLFuncPtrName(glStencilMaskSeparate)(face, mask); GL_AUTOCHECK_BLOCK("glStencilMaskSeparate", __FILE__, __LINE__); } while(0)

/**
 *	@def glStencilOpSeparate
 *	@brief set front and/or back stencil test actions
 *	@param[in] face Specifies whether front and/or back stencil state is updated. Three symbolic
 *		constants are valid: GL_FRONT, GL_BACK, and GL_FRONT_AND_BACK.
 *	@param[in] fail (documantation not found)
 *	@param[in] zfail (documantation not found)
 *	@param[in] zpass (documantation not found)
 */
#define glStencilOpSeparate(face,fail,zfail,zpass) do { GLFuncPtrName(glStencilOpSeparate)(face, fail, zfail, zpass); GL_AUTOCHECK_BLOCK("glStencilOpSeparate", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform1f
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 */
#define glUniform1f(location,v0) do { GLFuncPtrName(glUniform1f)(location, v0); GL_AUTOCHECK_BLOCK("glUniform1f", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform2f
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 */
#define glUniform2f(location,v0,v1) do { GLFuncPtrName(glUniform2f)(location, v0, v1); GL_AUTOCHECK_BLOCK("glUniform2f", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform3f
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v2 Specifies the new values to be used for the specified uniform variable.
 */
#define glUniform3f(location,v0,v1,v2) do { GLFuncPtrName(glUniform3f)(location, v0, v1, v2); GL_AUTOCHECK_BLOCK("glUniform3f", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform4f
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v2 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v3 Specifies the new values to be used for the specified uniform variable.
 */
#define glUniform4f(location,v0,v1,v2,v3) do { GLFuncPtrName(glUniform4f)(location, v0, v1, v2, v3); GL_AUTOCHECK_BLOCK("glUniform4f", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform1i
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 */
#define glUniform1i(location,v0) do { GLFuncPtrName(glUniform1i)(location, v0); GL_AUTOCHECK_BLOCK("glUniform1i", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform2i
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 */
#define glUniform2i(location,v0,v1) do { GLFuncPtrName(glUniform2i)(location, v0, v1); GL_AUTOCHECK_BLOCK("glUniform2i", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform3i
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v2 Specifies the new values to be used for the specified uniform variable.
 */
#define glUniform3i(location,v0,v1,v2) do { GLFuncPtrName(glUniform3i)(location, v0, v1, v2); GL_AUTOCHECK_BLOCK("glUniform3i", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform4i
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform variable to be modified.
 *	@param[in] v0 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v1 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v2 Specifies the new values to be used for the specified uniform variable.
 *	@param[in] v3 Specifies the new values to be used for the specified uniform variable.
 */
#define glUniform4i(location,v0,v1,v2,v3) do { GLFuncPtrName(glUniform4i)(location, v0, v1, v2, v3); GL_AUTOCHECK_BLOCK("glUniform4i", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform1fv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniform1fv(location,count,value) do { GLFuncPtrName(glUniform1fv)(location, count, value); GL_AUTOCHECK_BLOCK("glUniform1fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform2fv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniform2fv(location,count,value) do { GLFuncPtrName(glUniform2fv)(location, count, value); GL_AUTOCHECK_BLOCK("glUniform2fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform3fv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniform3fv(location,count,value) do { GLFuncPtrName(glUniform3fv)(location, count, value); GL_AUTOCHECK_BLOCK("glUniform3fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform4fv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniform4fv(location,count,value) do { GLFuncPtrName(glUniform4fv)(location, count, value); GL_AUTOCHECK_BLOCK("glUniform4fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform1iv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniform1iv(location,count,value) do { GLFuncPtrName(glUniform1iv)(location, count, value); GL_AUTOCHECK_BLOCK("glUniform1iv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform2iv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniform2iv(location,count,value) do { GLFuncPtrName(glUniform2iv)(location, count, value); GL_AUTOCHECK_BLOCK("glUniform2iv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform3iv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniform3iv(location,count,value) do { GLFuncPtrName(glUniform3iv)(location, count, value); GL_AUTOCHECK_BLOCK("glUniform3iv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniform4iv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of elements that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array, and 1 or more if it is an array.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniform4iv(location,count,value) do { GLFuncPtrName(glUniform4iv)(location, count, value); GL_AUTOCHECK_BLOCK("glUniform4iv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniformMatrix2fv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of matrices that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array of matrices, and 1 or more if it is an array
 *		of matrices.
 *	@param[in] transpose Specifies whether to transpose the matrix as the values are loaded into the
 *		uniform variable.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniformMatrix2fv(location,count,transpose,value) do { GLFuncPtrName(glUniformMatrix2fv)(location, count, transpose, value); GL_AUTOCHECK_BLOCK("glUniformMatrix2fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniformMatrix3fv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of matrices that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array of matrices, and 1 or more if it is an array
 *		of matrices.
 *	@param[in] transpose Specifies whether to transpose the matrix as the values are loaded into the
 *		uniform variable.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniformMatrix3fv(location,count,transpose,value) do { GLFuncPtrName(glUniformMatrix3fv)(location, count, transpose, value); GL_AUTOCHECK_BLOCK("glUniformMatrix3fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUniformMatrix4fv
 *	@brief Specify the value of a uniform variable for the current program object
 *	@param[in] location Specifies the location of the uniform value to be modified.
 *	@param[in] count Specifies the number of matrices that are to be modified. This should be 1 if
 *		the targeted uniform variable is not an array of matrices, and 1 or more if it is an array
 *		of matrices.
 *	@param[in] transpose Specifies whether to transpose the matrix as the values are loaded into the
 *		uniform variable.
 *	@param[in] value Specifies a pointer to an array of count values that will be used to update the
 *		specified uniform variable.
 */
#define glUniformMatrix4fv(location,count,transpose,value) do { GLFuncPtrName(glUniformMatrix4fv)(location, count, transpose, value); GL_AUTOCHECK_BLOCK("glUniformMatrix4fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glUseProgram
 *	@brief Installs a program object as part of current rendering state
 *	@param[in] program Specifies the handle of the program object whose executables are to be used
 *		as part of current rendering state.
 */
#define glUseProgram(program) do { GLFuncPtrName(glUseProgram)(program); GL_AUTOCHECK_BLOCK("glUseProgram", __FILE__, __LINE__); } while(0)

/**
 *	@def glValidateProgram
 *	@brief Validates a program object
 *	@param[in] program Specifies the handle of the program object to be validated.
 */
#define glValidateProgram(program) do { GLFuncPtrName(glValidateProgram)(program); GL_AUTOCHECK_BLOCK("glValidateProgram", __FILE__, __LINE__); } while(0)

/**
 *	@def glVertexAttrib1f
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] x (documantation not found)
 */
#define glVertexAttrib1f(index,x) do { GLFuncPtrName(glVertexAttrib1f)(index, x); GL_AUTOCHECK_BLOCK("glVertexAttrib1f", __FILE__, __LINE__); } while(0)

/**
 *	@def glVertexAttrib1fv
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] v Specifies a pointer to an array of values to be used for the generic vertex attribute.
 */
#define glVertexAttrib1fv(index,v) do { GLFuncPtrName(glVertexAttrib1fv)(index, v); GL_AUTOCHECK_BLOCK("glVertexAttrib1fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glVertexAttrib2f
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] x (documantation not found)
 *	@param[in] y (documantation not found)
 */
#define glVertexAttrib2f(index,x,y) do { GLFuncPtrName(glVertexAttrib2f)(index, x, y); GL_AUTOCHECK_BLOCK("glVertexAttrib2f", __FILE__, __LINE__); } while(0)

/**
 *	@def glVertexAttrib2fv
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] v Specifies a pointer to an array of values to be used for the generic vertex attribute.
 */
#define glVertexAttrib2fv(index,v) do { GLFuncPtrName(glVertexAttrib2fv)(index, v); GL_AUTOCHECK_BLOCK("glVertexAttrib2fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glVertexAttrib3f
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] x (documantation not found)
 *	@param[in] y (documantation not found)
 *	@param[in] z (documantation not found)
 */
#define glVertexAttrib3f(index,x,y,z) do { GLFuncPtrName(glVertexAttrib3f)(index, x, y, z); GL_AUTOCHECK_BLOCK("glVertexAttrib3f", __FILE__, __LINE__); } while(0)

/**
 *	@def glVertexAttrib3fv
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] v Specifies a pointer to an array of values to be used for the generic vertex attribute.
 */
#define glVertexAttrib3fv(index,v) do { GLFuncPtrName(glVertexAttrib3fv)(index, v); GL_AUTOCHECK_BLOCK("glVertexAttrib3fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glVertexAttrib4f
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] x (documantation not found)
 *	@param[in] y (documantation not found)
 *	@param[in] z (documantation not found)
 *	@param[in] w (documantation not found)
 */
#define glVertexAttrib4f(index,x,y,z,w) do { GLFuncPtrName(glVertexAttrib4f)(index, x, y, z, w); GL_AUTOCHECK_BLOCK("glVertexAttrib4f", __FILE__, __LINE__); } while(0)

/**
 *	@def glVertexAttrib4fv
 *	@brief Specifies the value of a generic vertex attribute
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] v Specifies a pointer to an array of values to be used for the generic vertex attribute.
 */
#define glVertexAttrib4fv(index,v) do { GLFuncPtrName(glVertexAttrib4fv)(index, v); GL_AUTOCHECK_BLOCK("glVertexAttrib4fv", __FILE__, __LINE__); } while(0)

/**
 *	@def glVertexAttribPointer
 *	@brief define an array of generic vertex attribute data
 *	@param[in] index Specifies the index of the generic vertex attribute to be modified.
 *	@param[in] size Specifies the number of components per generic vertex attribute. Must be 1, 2,
 *		3, 4. Additionally, the symbolic constant GL_BGRA is accepted by glVertexAttribPointer. The
 *		initial value is 4.
 *	@param[in] type Specifies the data type of each component in the array. The symbolic constants
 *		GL_BYTE, GL_UNSIGNED_BYTE, GL_SHORT, GL_UNSIGNED_SHORT, GL_INT, and GL_UNSIGNED_INT are
 *		accepted by both functions. Additionally GL_HALF_FLOAT, GL_FLOAT, GL_DOUBLE, GL_FIXED,
 *		GL_INT_2_10_10_10_REV, and GL_UNSIGNED_INT_2_10_10_10_REV are accepted by
 *		glVertexAttribPointer. The initial value is GL_FLOAT.
 *	@param[in] normalized For glVertexAttribPointer , specifies whether fixed-point data values
 *		should be normalized ( GL_TRUE ) or converted directly as fixed-point values ( GL_FALSE)
 *		when they are accessed.
 *	@param[in] stride Specifies the byte offset between consecutive generic vertex attributes. If
 *		stride is 0, the generic vertex attributes are understood to be tightly packed in the array.
 *		The initial value is 0.
 *	@param[in] pointer Specifies a offset of the first component of the first generic vertex
 *		attribute in the array in the data store of the buffer currently bound to the
 *		GL_ARRAY_BUFFER target. The initial value is 0.
 */
#define glVertexAttribPointer(index,size,type,normalized,stride,pointer) do { GLFuncPtrName(glVertexAttribPointer)(index, size, type, normalized, stride, pointer); GL_AUTOCHECK_BLOCK("glVertexAttribPointer", __FILE__, __LINE__); } while(0)

#if !defined(GLES20EMU_CPP)
// do not patch this for GLES20Emu.cpp

void __gles20emupatch_glShaderSource(GLuint shader, GLsizei count, const GLchar** string, const GLint* length);
#ifdef __HAVE_GL_AUTOCHECK
const GLubyte *__gles20emupatch_glGetString(GLenum name, const char *p_s_file, int n_line);
#else // __HAVE_GL_AUTOCHECK
const GLubyte *__gles20emupatch_glGetString(GLenum name);
#endif // __HAVE_GL_AUTOCHECK
// patched function prototypes

#ifdef __HAVE_GL_AUTOCHECK
#undef glGetString
#endif // __HAVE_GL_AUTOCHECK
// statically linked functions are only redefined for autocheck

#undef glShaderSource
// dynamically linked functions are always redefined

#ifdef __HAVE_GL_AUTOCHECK
#define glShaderSource(shader,count,string,length) do { __gles20emupatch_glShaderSource((shader), (count), (string), (length)); GL_AUTOCHECK_BLOCK("glShaderSource", __FILE__, __LINE__); } while(0)
#define glGetString(name) __gles20emupatch_glGetString((name), __FILE__, __LINE__)
#else // __HAVE_GL_AUTOCHECK
#define glShaderSource(shader,count,string,length) __gles20emupatch_glShaderSource((shader), (count), (string), (length))
#define glGetString(name) __gles20emupatch_glGetString((name))
#endif // __HAVE_GL_AUTOCHECK
// use patched functions

#endif // !GLES20EMU_CPP

#endif // !GL_ES_VERSION_2_0

//								--- ~OpenGL ES 2.0 ---

//								--- GL_ARB_vertex_array_object ---

/**
 *	@def GL_ARB_vertex_array_object
 *	@brief if defined, GL_ARB_vertex_array_object entry points are available
 */
#ifndef GL_ARB_vertex_array_object
#define GL_ARB_vertex_array_object 1

/**
 *	@def __GENERATE_GL_ARB_vertex_array_object
 *	@brief if defined, GL_ARB_vertex_array_object entry points are available
 */
#define __GENERATE_GL_ARB_vertex_array_object

extern bool GLEH_ARB_vertex_array_object; /**< @brief set if GL_ARB_vertex_array_object is supported */

#define GL_VERTEX_ARRAY_BINDING										0x85b5 /**< @brief GL_ARB_vertex_array_object enum **/

/**
 *	@brief bind a vertex array object
 *	@param[in] array Specifies the name of the vertex array to bind.
 */
extern void (GLApi *GLFuncPtrName(glBindVertexArray))(GLuint array);

/**
 *	@brief delete vertex array objects
 *	@param[in] n Specifies the number of vertex array objects to be deleted.
 *	@param[in] arrays Specifies the address of an array containing the n names of the objects to be deleted.
 */
extern void (GLApi *GLFuncPtrName(glDeleteVertexArrays))(GLsizei n, const GLuint *arrays);

/**
 *	@brief generate vertex array object names
 *	@param[in] n Specifies the number of vertex array object names to generate.
 *	@param[out] arrays Specifies an array in which the generated vertex array object names are stored.
 */
extern void (GLApi *GLFuncPtrName(glGenVertexArrays))(GLsizei n, GLuint *arrays);

/**
 *	@brief determine if a name corresponds to a vertex array object
 *	@param[in] array Specifies a value that may be the name of a vertex array object.
 */
extern GLboolean (GLApi *GLFuncPtrName(glIsVertexArray))(GLuint array);

inline GLboolean glIsVertexArray_glachwrap(GLuint array, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glIsVertexArray)(array); GL_AUTOCHECK_BLOCK("glIsVertexArray", p_s_filename, n_line); return result; }

/**
 *	@def glBindVertexArray
 *	@brief bind a vertex array object
 *	@param[in] array Specifies the name of the vertex array to bind.
 */
#define glBindVertexArray(array) do { GLFuncPtrName(glBindVertexArray)(array); GL_AUTOCHECK_BLOCK("glBindVertexArray", __FILE__, __LINE__); } while(0)

/**
 *	@def glDeleteVertexArrays
 *	@brief delete vertex array objects
 *	@param[in] n Specifies the number of vertex array objects to be deleted.
 *	@param[in] arrays Specifies the address of an array containing the n names of the objects to be deleted.
 */
#define glDeleteVertexArrays(n,arrays) do { GLFuncPtrName(glDeleteVertexArrays)(n, arrays); GL_AUTOCHECK_BLOCK("glDeleteVertexArrays", __FILE__, __LINE__); } while(0)

/**
 *	@def glGenVertexArrays
 *	@brief generate vertex array object names
 *	@param[in] n Specifies the number of vertex array object names to generate.
 *	@param[out] arrays Specifies an array in which the generated vertex array object names are stored.
 */
#define glGenVertexArrays(n,arrays) do { GLFuncPtrName(glGenVertexArrays)(n, arrays); GL_AUTOCHECK_BLOCK("glGenVertexArrays", __FILE__, __LINE__); } while(0)

/**
 *	@def glIsVertexArray
 *	@brief determine if a name corresponds to a vertex array object
 *	@param[in] array Specifies a value that may be the name of a vertex array object.
 */
#define glIsVertexArray(array) GLAutoCheckFuncPtrName(glIsVertexArray)(array GLAutoCheckExtraParams)

#if !defined(GLES20EMU_CPP)
// do not patch this for GLES20Emu.cpp

extern GLuint __n_gles20emupatch_default_vertex_array;
// default vertex array in place of array 0 (array 0 is not allowed to be used for actual rendering in new OpenGL specifications)

inline void __gles20emupatch_glGetIntegerv(GLenum pname, GLint* params) { GLAutoCheckStaticFuncName(glGetIntegerv)(pname, params); if(pname == GL_VERTEX_ARRAY_BINDING && params != 0 && *params == __n_gles20emupatch_default_vertex_array) *params = 0; /* obfuscate the default vertex array and return 0 */ }
inline void __gles20emupatch_glBindVertexArray(GLuint array) { if(array) GLFuncPtrName(glBindVertexArray)(array); else GLFuncPtrName(glBindVertexArray)(__n_gles20emupatch_default_vertex_array); }
// patched function prototypes; make sure to call the original and not the autocheck wrapped one

#ifdef __HAVE_GL_AUTOCHECK
#undef glGetIntegerv
#endif // __HAVE_GL_AUTOCHECK
// statically linked functions are only redefined for autocheck

#undef glBindVertexArray
// dynamically linked functions are always redefined

#define glGetIntegerv(pname,params) do { __gles20emupatch_glGetIntegerv((pname), (params)); GL_AUTOCHECK_BLOCK("glGetIntegerv", __FILE__, __LINE__); } while(0)
#define glBindVertexArray(array) do { __gles20emupatch_glBindVertexArray((array)); GL_AUTOCHECK_BLOCK("glBindVertexArray", __FILE__, __LINE__); } while(0)
// use patched functions, make sure that autocheck will report the correct call origin

#endif // !GLES20EMU_CPP

#endif // !GL_ARB_vertex_array_object

//								--- ~GL_ARB_vertex_array_object ---

//								--- GL_OES_texture_3D ---

/**
 *	@def GL_OES_texture_3D
 *	@brief if defined, GL_OES_texture_3D entry points are available
 */
#ifndef GL_OES_texture_3D
#define GL_OES_texture_3D 1

/**
 *	@def __GENERATE_GL_OES_texture_3D
 *	@brief if defined, GL_OES_texture_3D entry points are available
 */
#define __GENERATE_GL_OES_texture_3D

extern bool GLEH_OES_texture_3D; /**< @brief set if GL_OES_texture_3D is supported */

#define GL_TEXTURE_3D_OES											0x806f /**< @brief GL_OES_texture_3D enum **/
#define GL_TEXTURE_WRAP_R_OES										0x8072 /**< @brief GL_OES_texture_3D enum **/
#define GL_MAX_3D_TEXTURE_SIZE_OES									0x8073 /**< @brief GL_OES_texture_3D enum **/
#define GL_TEXTURE_BINDING_3D_OES									0x806a /**< @brief GL_OES_texture_3D enum **/

/**
 *	@brief specify a three-dimensional texture image
 *	@param[in] target Specifies the target texture. Must be one of GL_TEXTURE_3D,
 *		GL_PROXY_TEXTURE_3D, GL_TEXTURE_2D_ARRAY or GL_PROXY_TEXTURE_2D_ARRAY.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the n th mipmap reduction image.
 *	@param[in] internalFormat Specifies the number of color components in the texture. Must be one
 *		of base internal formats given in Table 1, one of the sized internal formats given in Table
 *		2, or one of the compressed internal formats given in Table 3, below.
 *	@param[in] width Specifies the width of the texture image. All implementations support 3D
 *		texture images that are at least 16 texels wide.
 *	@param[in] height Specifies the height of the texture image. All implementations support 3D
 *		texture images that are at least 256 texels high.
 *	@param[in] depth Specifies the depth of the texture image, or the number of layers in a texture
 *		array. All implementations support 3D texture images that are at least 256 texels deep, and
 *		texture arrays that are at least 256 layers deep.
 *	@param[in] border This value must be 0.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_RED, GL_RG, GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. The following symbolic values are
 *		accepted: GL_UNSIGNED_BYTE, GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT,
 *		GL_FLOAT, GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, and GL_UNSIGNED_INT_2_10_10_10_REV.
 *	@param[in] data Specifies a pointer to the image data in memory.
 */
extern void (GLApi *GLFuncPtrName(glTexImage3DOES))(GLenum target, int level, GLenum internalFormat, GLsizei width, GLsizei height, GLsizei depth, int border, GLenum format, GLenum type, const void *data);

/**
 *	@brief specify a three-dimensional texture subimage
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_3D.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] zoffset Specifies a texel offset in the z direction within the texture array.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 *	@param[in] depth Specifies the depth of the texture subimage.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_RED, GL_RG, GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. The following symbolic values are
 *		accepted: GL_UNSIGNED_BYTE, GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT,
 *		GL_FLOAT, GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, and GL_UNSIGNED_INT_2_10_10_10_REV.
 *	@param[in] data Specifies a pointer to the image data in memory.
 */
extern void (GLApi *GLFuncPtrName(glTexSubImage3DOES))(GLenum target, int level, int xoffset, int yoffset, int zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *data);

/**
 *	@brief copy a three-dimensional texture subimage
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_3D
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] zoffset Specifies a texel offset in the z direction within the texture array.
 *	@param[in] x Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] y Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 */
extern void (GLApi *GLFuncPtrName(glCopyTexSubImage3DOES))(GLenum target, int level, int xoffset, int yoffset, int zoffset, int x, int y, GLsizei width, GLsizei height);

/**
 *	@brief specify a three-dimensional texture image in a compressed format
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_3D, GL_PROXY_TEXTURE_3D,
 *		GL_TEXTURE_2D_ARRAY or GL_PROXY_TEXTURE_2D_ARRAY.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] internalformat Specifies the format of the compressed image data stored at address data.
 *	@param[in] width Specifies the width of the texture image. All implementations support 3D
 *		texture images that are at least 16 texels wide.
 *	@param[in] height Specifies the height of the texture image. All implementations support 3D
 *		texture images that are at least 16 texels high.
 *	@param[in] depth Specifies the depth of the texture image. All implementations support 3D
 *		texture images that are at least 16 texels deep.
 *	@param[in] border This value must be 0.
 *	@param[in] imageSize Specifies the number of unsigned bytes of image data starting at the
 *		address specified by data.
 *	@param[in] data Specifies a pointer to the compressed image data in memory.
 */
extern void (GLApi *GLFuncPtrName(glCompressedTexImage3DOES))(GLenum target, int level, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, int border, GLsizei imageSize, const void *data);

/**
 *	@brief specify a three-dimensional texture subimage in a compressed format
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_3D.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] zoffset Specifies a texel offset in the z direction within the texture array.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 *	@param[in] depth Specifies the depth of the texture subimage.
 *	@param[in] format Specifies the format of the compressed image data stored at address data.
 *	@param[in] imageSize Specifies the number of unsigned bytes of image data starting at the
 *		address specified by data.
 *	@param[in] data Specifies a pointer to the compressed image data in memory.
 */
extern void (GLApi *GLFuncPtrName(glCompressedTexSubImage3DOES))(GLenum target, int level, int xoffset, int yoffset, int zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLsizei imageSize, const void *data);

/**
 *	@brief attach a level of a texture object as a logical buffer to the currently bound framebuffer object
 *	@param[in] target Specifies the framebuffer target. target must be GL_DRAW_FRAMEBUFFER,
 *		GL_READ_FRAMEBUFFER, or GL_FRAMEBUFFER. GL_FRAMEBUFFER is equivalent to GL_DRAW_FRAMEBUFFER.
 *	@param[in] attachment Specifies the attachment point of the framebuffer. attachment must be
 *		GL_COLOR_ATTACHMENT i, GL_DEPTH_ATTACHMENT, GL_STENCIL_ATTACHMENT or GL_DEPTH_STENCIL_ATTACHMMENT.
 *	@param[in] textarget For glFramebufferTexture1D, glFramebufferTexture2D and
 *		glFramebufferTexture3D, specifies what type of texture is expected in the texture parameter,
 *		or for cube map textures, which face is to be attached.
 *	@param[in] texture Specifies the texture object to attach to the framebuffer attachment point
 *		named by attachment.
 *	@param[in] level Specifies the mipmap level of texture to attach.
 *	@param[in] layer Specifies the layer of texture to attach.
 */
extern void (GLApi *GLFuncPtrName(glFramebufferTexture3DOES))(GLenum target, GLenum attachment, GLenum textarget, GLuint texture, int level, int layer);

/**
 *	@def glTexImage3DOES
 *	@brief specify a three-dimensional texture image
 *	@param[in] target Specifies the target texture. Must be one of GL_TEXTURE_3D,
 *		GL_PROXY_TEXTURE_3D, GL_TEXTURE_2D_ARRAY or GL_PROXY_TEXTURE_2D_ARRAY.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the n th mipmap reduction image.
 *	@param[in] internalFormat Specifies the number of color components in the texture. Must be one
 *		of base internal formats given in Table 1, one of the sized internal formats given in Table
 *		2, or one of the compressed internal formats given in Table 3, below.
 *	@param[in] width Specifies the width of the texture image. All implementations support 3D
 *		texture images that are at least 16 texels wide.
 *	@param[in] height Specifies the height of the texture image. All implementations support 3D
 *		texture images that are at least 256 texels high.
 *	@param[in] depth Specifies the depth of the texture image, or the number of layers in a texture
 *		array. All implementations support 3D texture images that are at least 256 texels deep, and
 *		texture arrays that are at least 256 layers deep.
 *	@param[in] border This value must be 0.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_RED, GL_RG, GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. The following symbolic values are
 *		accepted: GL_UNSIGNED_BYTE, GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT,
 *		GL_FLOAT, GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, and GL_UNSIGNED_INT_2_10_10_10_REV.
 *	@param[in] data Specifies a pointer to the image data in memory.
 */
#define glTexImage3DOES(target,level,internalFormat,width,height,depth,border,format,type,data) do { GLFuncPtrName(glTexImage3DOES)(target, level, internalFormat, width, height, depth, border, format, type, data); GL_AUTOCHECK_BLOCK("glTexImage3DOES", __FILE__, __LINE__); } while(0)

/**
 *	@def glTexSubImage3DOES
 *	@brief specify a three-dimensional texture subimage
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_3D.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] zoffset Specifies a texel offset in the z direction within the texture array.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 *	@param[in] depth Specifies the depth of the texture subimage.
 *	@param[in] format Specifies the format of the pixel data. The following symbolic values are
 *		accepted: GL_RED, GL_RG, GL_RGB, GL_BGR, GL_RGBA, and GL_BGRA.
 *	@param[in] type Specifies the data type of the pixel data. The following symbolic values are
 *		accepted: GL_UNSIGNED_BYTE, GL_BYTE, GL_UNSIGNED_SHORT, GL_SHORT, GL_UNSIGNED_INT, GL_INT,
 *		GL_FLOAT, GL_UNSIGNED_BYTE_3_3_2, GL_UNSIGNED_BYTE_2_3_3_REV, GL_UNSIGNED_SHORT_5_6_5,
 *		GL_UNSIGNED_SHORT_5_6_5_REV, GL_UNSIGNED_SHORT_4_4_4_4, GL_UNSIGNED_SHORT_4_4_4_4_REV,
 *		GL_UNSIGNED_SHORT_5_5_5_1, GL_UNSIGNED_SHORT_1_5_5_5_REV, GL_UNSIGNED_INT_8_8_8_8,
 *		GL_UNSIGNED_INT_8_8_8_8_REV, GL_UNSIGNED_INT_10_10_10_2, and GL_UNSIGNED_INT_2_10_10_10_REV.
 *	@param[in] data Specifies a pointer to the image data in memory.
 */
#define glTexSubImage3DOES(target,level,xoffset,yoffset,zoffset,width,height,depth,format,type,data) do { GLFuncPtrName(glTexSubImage3DOES)(target, level, xoffset, yoffset, zoffset, width, height, depth, format, type, data); GL_AUTOCHECK_BLOCK("glTexSubImage3DOES", __FILE__, __LINE__); } while(0)

/**
 *	@def glCopyTexSubImage3DOES
 *	@brief copy a three-dimensional texture subimage
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_3D
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] zoffset Specifies a texel offset in the z direction within the texture array.
 *	@param[in] x Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] y Specify the window coordinates of the lower left corner of the rectangular region
 *		of pixels to be copied.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 */
#define glCopyTexSubImage3DOES(target,level,xoffset,yoffset,zoffset,x,y,width,height) do { GLFuncPtrName(glCopyTexSubImage3DOES)(target, level, xoffset, yoffset, zoffset, x, y, width, height); GL_AUTOCHECK_BLOCK("glCopyTexSubImage3DOES", __FILE__, __LINE__); } while(0)

/**
 *	@def glCompressedTexImage3DOES
 *	@brief specify a three-dimensional texture image in a compressed format
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_3D, GL_PROXY_TEXTURE_3D,
 *		GL_TEXTURE_2D_ARRAY or GL_PROXY_TEXTURE_2D_ARRAY.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] internalformat Specifies the format of the compressed image data stored at address data.
 *	@param[in] width Specifies the width of the texture image. All implementations support 3D
 *		texture images that are at least 16 texels wide.
 *	@param[in] height Specifies the height of the texture image. All implementations support 3D
 *		texture images that are at least 16 texels high.
 *	@param[in] depth Specifies the depth of the texture image. All implementations support 3D
 *		texture images that are at least 16 texels deep.
 *	@param[in] border This value must be 0.
 *	@param[in] imageSize Specifies the number of unsigned bytes of image data starting at the
 *		address specified by data.
 *	@param[in] data Specifies a pointer to the compressed image data in memory.
 */
#define glCompressedTexImage3DOES(target,level,internalformat,width,height,depth,border,imageSize,data) do { GLFuncPtrName(glCompressedTexImage3DOES)(target, level, internalformat, width, height, depth, border, imageSize, data); GL_AUTOCHECK_BLOCK("glCompressedTexImage3DOES", __FILE__, __LINE__); } while(0)

/**
 *	@def glCompressedTexSubImage3DOES
 *	@brief specify a three-dimensional texture subimage in a compressed format
 *	@param[in] target Specifies the target texture. Must be GL_TEXTURE_3D.
 *	@param[in] level Specifies the level-of-detail number. Level 0 is the base image level. Level n
 *		is the nth mipmap reduction image.
 *	@param[in] xoffset Specifies a texel offset in the x direction within the texture array.
 *	@param[in] yoffset Specifies a texel offset in the y direction within the texture array.
 *	@param[in] zoffset Specifies a texel offset in the z direction within the texture array.
 *	@param[in] width Specifies the width of the texture subimage.
 *	@param[in] height Specifies the height of the texture subimage.
 *	@param[in] depth Specifies the depth of the texture subimage.
 *	@param[in] format Specifies the format of the compressed image data stored at address data.
 *	@param[in] imageSize Specifies the number of unsigned bytes of image data starting at the
 *		address specified by data.
 *	@param[in] data Specifies a pointer to the compressed image data in memory.
 */
#define glCompressedTexSubImage3DOES(target,level,xoffset,yoffset,zoffset,width,height,depth,format,imageSize,data) do { GLFuncPtrName(glCompressedTexSubImage3DOES)(target, level, xoffset, yoffset, zoffset, width, height, depth, format, imageSize, data); GL_AUTOCHECK_BLOCK("glCompressedTexSubImage3DOES", __FILE__, __LINE__); } while(0)

/**
 *	@def glFramebufferTexture3DOES
 *	@brief attach a level of a texture object as a logical buffer to the currently bound framebuffer object
 *	@param[in] target Specifies the framebuffer target. target must be GL_DRAW_FRAMEBUFFER,
 *		GL_READ_FRAMEBUFFER, or GL_FRAMEBUFFER. GL_FRAMEBUFFER is equivalent to GL_DRAW_FRAMEBUFFER.
 *	@param[in] attachment Specifies the attachment point of the framebuffer. attachment must be
 *		GL_COLOR_ATTACHMENT i, GL_DEPTH_ATTACHMENT, GL_STENCIL_ATTACHMENT or GL_DEPTH_STENCIL_ATTACHMMENT.
 *	@param[in] textarget For glFramebufferTexture1D, glFramebufferTexture2D and
 *		glFramebufferTexture3D, specifies what type of texture is expected in the texture parameter,
 *		or for cube map textures, which face is to be attached.
 *	@param[in] texture Specifies the texture object to attach to the framebuffer attachment point
 *		named by attachment.
 *	@param[in] level Specifies the mipmap level of texture to attach.
 *	@param[in] layer Specifies the layer of texture to attach.
 */
#define glFramebufferTexture3DOES(target,attachment,textarget,texture,level,layer) do { GLFuncPtrName(glFramebufferTexture3DOES)(target, attachment, textarget, texture, level, layer); GL_AUTOCHECK_BLOCK("glFramebufferTexture3DOES", __FILE__, __LINE__); } while(0)

#endif // !GL_OES_texture_3D

//								--- ~GL_OES_texture_3D ---

//								--- GL_OES_depth24 ---

/**
 *	@def GL_OES_depth24
 *	@brief if defined, GL_OES_depth24 entry points are available
 */
#ifndef GL_OES_depth24
#define GL_OES_depth24 1

extern bool GLEH_OES_depth24; /**< @brief set if GL_OES_depth24 is supported */

#define GL_DEPTH_COMPONENT24_OES									0x81a6 /**< @brief GL_OES_depth24 enum **/

// no new procedures or functions

#endif // !GL_OES_depth24

//								--- ~GL_OES_depth24 ---

//								--- GL_OES_depth32 ---

/**
 *	@def GL_OES_depth32
 *	@brief if defined, GL_OES_depth32 entry points are available
 */
#ifndef GL_OES_depth32
#define GL_OES_depth32 1

extern bool GLEH_OES_depth32; /**< @brief set if GL_OES_depth32 is supported */

#define GL_DEPTH_COMPONENT32_OES									0x81a7 /**< @brief GL_OES_depth32 enum **/

// no new procedures or functions

#endif // !GL_OES_depth32

//								--- ~GL_OES_depth32 ---

//								--- GL_OES_element_index_uint ---

/**
 *	@def GL_OES_element_index_uint
 *	@brief if defined, GL_OES_element_index_uint entry points are available
 */
#ifndef GL_OES_element_index_uint
#define GL_OES_element_index_uint 1

extern bool GLEH_OES_element_index_uint; /**< @brief set if GL_OES_element_index_uint is supported */

#define GL_UNSIGNED_INT												0x1405 /**< @brief GL_OES_element_index_uint enum **/

// no new procedures or functions

#endif // !GL_OES_element_index_uint

//								--- ~GL_OES_element_index_uint ---

//								--- GL_OES_mapbuffer ---

/**
 *	@def GL_OES_mapbuffer
 *	@brief if defined, GL_OES_mapbuffer entry points are available
 */
#ifndef GL_OES_mapbuffer
#define GL_OES_mapbuffer 1

/**
 *	@def __GENERATE_GL_OES_mapbuffer
 *	@brief if defined, GL_OES_mapbuffer entry points are available
 */
#define __GENERATE_GL_OES_mapbuffer

extern bool GLEH_OES_mapbuffer; /**< @brief set if GL_OES_mapbuffer is supported */

#define GL_WRITE_ONLY_OES											0x88b9 /**< @brief GL_OES_mapbuffer enum **/
#define GL_BUFFER_ACCESS_OES										0x88bb /**< @brief GL_OES_mapbuffer enum **/
#define GL_BUFFER_MAPPED_OES										0x88bc /**< @brief GL_OES_mapbuffer enum **/
#define GL_BUFFER_MAP_POINTER_OES									0x88bd /**< @brief GL_OES_mapbuffer enum **/

/**
 *	@brief return the pointer to a mapped buffer object's data store
 *	@param[in] target Specifies the target buffer object. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] pname Specifies the pointer to be returned. The symbolic constant must be GL_BUFFER_MAP_POINTER.
 *	@param[out] params Returns the pointer value specified by pname.
 */
extern void (GLApi *GLFuncPtrName(glGetBufferPointervOES))(GLenum target, GLenum pname, void **params);

/**
 *	@brief map a buffer object's data store
 *	@param[in] target Specifies the target buffer object being mapped. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_ATOMIC_COUNTER_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER,
 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER or GL_UNIFORM_BUFFER.
 *	@param[in] access Specifies the access policy, indicating whether it will be possible to read
 *		from, write to, or both read from and write to the buffer object's mapped data store. The
 *		symbolic constant must be GL_READ_ONLY, GL_WRITE_ONLY, or GL_READ_WRITE.
 */
extern void* (GLApi *GLFuncPtrName(glMapBufferOES))(GLenum target, GLenum access);

/**
 *	@brief map a buffer object's data store
 *	@param[in] target Specifies the target buffer object being unmapped. The symbolic constant must
 *		be GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER or GL_UNIFORM_BUFFER.
 */
extern GLboolean (GLApi *GLFuncPtrName(glUnmapBufferOES))(GLenum target);

inline void *glMapBufferOES_glachwrap(GLenum target, GLenum access, const char *p_s_filename, int n_line) { void *result = GLFuncPtrName(glMapBufferOES)(target,access); GL_AUTOCHECK_BLOCK("glMapBufferOES", p_s_filename, n_line); return result; }
inline GLboolean glUnmapBufferOES_glachwrap(GLenum target, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glUnmapBufferOES)(target); GL_AUTOCHECK_BLOCK("glUnmapBufferOES", p_s_filename, n_line); return result; }

/**
 *	@def glGetBufferPointervOES
 *	@brief return the pointer to a mapped buffer object's data store
 *	@param[in] target Specifies the target buffer object. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER, or GL_UNIFORM_BUFFER.
 *	@param[in] pname Specifies the pointer to be returned. The symbolic constant must be GL_BUFFER_MAP_POINTER.
 *	@param[out] params Returns the pointer value specified by pname.
 */
#define glGetBufferPointervOES(target,pname,params) do { GLFuncPtrName(glGetBufferPointervOES)(target, pname, params); GL_AUTOCHECK_BLOCK("glGetBufferPointervOES", __FILE__, __LINE__); } while(0)

/**
 *	@def glMapBufferOES
 *	@brief map a buffer object's data store
 *	@param[in] target Specifies the target buffer object being mapped. The symbolic constant must be
 *		GL_ARRAY_BUFFER, GL_ATOMIC_COUNTER_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER,
 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER or GL_UNIFORM_BUFFER.
 *	@param[in] access Specifies the access policy, indicating whether it will be possible to read
 *		from, write to, or both read from and write to the buffer object's mapped data store. The
 *		symbolic constant must be GL_READ_ONLY, GL_WRITE_ONLY, or GL_READ_WRITE.
 */
#define glMapBufferOES(target,access) GLAutoCheckFuncPtrName(glMapBufferOES)(target, access GLAutoCheckExtraParams)

/**
 *	@def glUnmapBufferOES
 *	@brief map a buffer object's data store
 *	@param[in] target Specifies the target buffer object being unmapped. The symbolic constant must
 *		be GL_ARRAY_BUFFER, GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_ELEMENT_ARRAY_BUFFER,
 *		GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER, GL_TEXTURE_BUFFER,
 *		GL_TRANSFORM_FEEDBACK_BUFFER or GL_UNIFORM_BUFFER.
 */
#define glUnmapBufferOES(target) GLAutoCheckFuncPtrName(glUnmapBufferOES)(target GLAutoCheckExtraParams)

#endif // !GL_OES_mapbuffer

//								--- ~GL_OES_mapbuffer ---

//								--- GL_OES_rgb8_rgba8 ---

/**
 *	@def GL_OES_rgb8_rgba8
 *	@brief if defined, GL_OES_rgb8_rgba8 entry points are available
 */
#ifndef GL_OES_rgb8_rgba8
#define GL_OES_rgb8_rgba8 1

extern bool GLEH_OES_rgb8_rgba8; /**< @brief set if GL_OES_rgb8_rgba8 is supported */

#define GL_RGB8_OES													0x8051 /**< @brief GL_OES_rgb8_rgba8 enum **/
#define GL_RGBA8_OES												0x8058 /**< @brief GL_OES_rgb8_rgba8 enum **/

// no new procedures or functions

#endif // !GL_OES_rgb8_rgba8

//								--- ~GL_OES_rgb8_rgba8 ---

//								--- GL_OES_stencil1 ---

/**
 *	@def GL_OES_stencil1
 *	@brief if defined, GL_OES_stencil1 entry points are available
 */
#ifndef GL_OES_stencil1
#define GL_OES_stencil1 1

extern bool GLEH_OES_stencil1; /**< @brief set if GL_OES_stencil1 is supported */

#define GL_STENCIL_INDEX1_OES										0x8d46 /**< @brief GL_OES_stencil1 enum **/

// no new procedures or functions

#endif // !GL_OES_stencil1

//								--- ~GL_OES_stencil1 ---

//								--- GL_OES_stencil4 ---

/**
 *	@def GL_OES_stencil4
 *	@brief if defined, GL_OES_stencil4 entry points are available
 */
#ifndef GL_OES_stencil4
#define GL_OES_stencil4 1

extern bool GLEH_OES_stencil4; /**< @brief set if GL_OES_stencil4 is supported */

#define GL_STENCIL_INDEX4_OES										0x8d47 /**< @brief GL_OES_stencil4 enum **/

// no new procedures or functions

#endif // !GL_OES_stencil4

//								--- ~GL_OES_stencil4 ---

//								--- GL_OES_stencil8 ---

/**
 *	@def GL_OES_stencil8
 *	@brief if defined, GL_OES_stencil8 entry points are available
 */
#ifndef GL_OES_stencil8
#define GL_OES_stencil8 1

extern bool GLEH_OES_stencil8; /**< @brief set if GL_OES_stencil8 is supported */

#define GL_STENCIL_INDEX8_OES										0x8d48 /**< @brief GL_OES_stencil8 enum **/

// no new procedures or functions

#endif // !GL_OES_stencil8

//								--- ~GL_OES_stencil8 ---

//								--- GL_OES_depth_texture ---

/**
 *	@def GL_OES_depth_texture
 *	@brief if defined, GL_OES_depth_texture entry points are available
 */
#ifndef GL_OES_depth_texture
#define GL_OES_depth_texture 1

extern bool GLEH_OES_depth_texture; /**< @brief set if GL_OES_depth_texture is supported */

//#define GL_DEPTH_COMPONENT										  0x1902 // (defined above)
//#define GL_UNSIGNED_SHORT											  0x1403 // (defined above)
//#define GL_UNSIGNED_INT											  0x1405 // (defined above)

// no new procedures or functions

#endif // !GL_OES_depth_texture

//								--- ~GL_OES_depth_texture ---

//								--- GL_OES_packed_depth_stencil ---

/**
 *	@def GL_OES_packed_depth_stencil
 *	@brief if defined, GL_OES_packed_depth_stencil entry points are available
 */
#ifndef GL_OES_packed_depth_stencil
#define GL_OES_packed_depth_stencil 1

extern bool GLEH_OES_packed_depth_stencil; /**< @brief set if GL_OES_packed_depth_stencil is supported */

#define GL_DEPTH_STENCIL_OES										0x84f9 /**< @brief GL_OES_packed_depth_stencil enum **/
#define GL_UNSIGNED_INT_24_8_OES									0x84fa /**< @brief GL_OES_packed_depth_stencil enum **/
#define GL_DEPTH24_STENCIL8_OES										0x88f0 /**< @brief GL_OES_packed_depth_stencil enum **/

// no new procedures or functions

#endif // !GL_OES_packed_depth_stencil

//								--- ~GL_OES_packed_depth_stencil ---

//								--- GL_OES_get_program_binary ---

/**
 *	@def GL_OES_get_program_binary
 *	@brief if defined, GL_OES_get_program_binary entry points are available
 */
#ifndef GL_OES_get_program_binary
#define GL_OES_get_program_binary 1

/**
 *	@def __GENERATE_GL_OES_get_program_binary
 *	@brief if defined, GL_OES_get_program_binary entry points are available
 */
#define __GENERATE_GL_OES_get_program_binary

extern bool GLEH_OES_get_program_binary; /**< @brief set if GL_OES_get_program_binary is supported */

#define GL_PROGRAM_BINARY_LENGTH_OES								0x8741 /**< @brief GL_OES_get_program_binary enum **/
#define GL_NUM_PROGRAM_BINARY_FORMATS_OES							0x87fe /**< @brief GL_OES_get_program_binary enum **/
#define GL_PROGRAM_BINARY_FORMATS_OES								0x87ff /**< @brief GL_OES_get_program_binary enum **/

/**
 *	@brief return a binary representation of a program object's compiled and linked executable source
 *	@param[in] program Specifies the name of a program object whose binary representation to retrieve.
 *	@param[in] bufSize Specifies the size of the buffer whose address is given by binary.
 *	@param[out] length Specifies the address of a variable to receive the number of bytes written
 *		into binary.
 *	@param[out] binaryFormat Specifies the address of a variable to receive a token indicating the
 *		format of the binary data returned by the GL.
 *	@param[out] binary Specifies the address an array into which the GL will return program's binary representation.
 */
extern void (GLApi *GLFuncPtrName(glGetProgramBinaryOES))(GLuint program, GLsizei bufSize, GLsizei *length, GLenum *binaryFormat, void *binary);

/**
 *	@brief load a program object with a program binary
 *	@param[in] program Specifies the name of a program object into which to load a program binary.
 *	@param[in] binaryFormat Specifies the format of the binary data in binary.
 *	@param[in] binary Specifies the address an array containing the binary to be loaded into program.
 *	@param[in] length Specifies the number of bytes contained in binary.
 */
extern void (GLApi *GLFuncPtrName(glProgramBinaryOES))(GLuint program, GLenum binaryFormat, const void *binary, int length);

/**
 *	@def glGetProgramBinaryOES
 *	@brief return a binary representation of a program object's compiled and linked executable source
 *	@param[in] program Specifies the name of a program object whose binary representation to retrieve.
 *	@param[in] bufSize Specifies the size of the buffer whose address is given by binary.
 *	@param[out] length Specifies the address of a variable to receive the number of bytes written
 *		into binary.
 *	@param[out] binaryFormat Specifies the address of a variable to receive a token indicating the
 *		format of the binary data returned by the GL.
 *	@param[out] binary Specifies the address an array into which the GL will return program's binary representation.
 */
#define glGetProgramBinaryOES(program,bufSize,length,binaryFormat,binary) do { GLFuncPtrName(glGetProgramBinaryOES)(program, bufSize, length, binaryFormat, binary); GL_AUTOCHECK_BLOCK("glGetProgramBinaryOES", __FILE__, __LINE__); } while(0)

/**
 *	@def glProgramBinaryOES
 *	@brief load a program object with a program binary
 *	@param[in] program Specifies the name of a program object into which to load a program binary.
 *	@param[in] binaryFormat Specifies the format of the binary data in binary.
 *	@param[in] binary Specifies the address an array containing the binary to be loaded into program.
 *	@param[in] length Specifies the number of bytes contained in binary.
 */
#define glProgramBinaryOES(program,binaryFormat,binary,length) do { GLFuncPtrName(glProgramBinaryOES)(program, binaryFormat, binary, length); GL_AUTOCHECK_BLOCK("glProgramBinaryOES", __FILE__, __LINE__); } while(0)

#endif // !GL_OES_get_program_binary

//								--- ~GL_OES_get_program_binary ---

//								--- GL_OES_fbo_render_mipmap ---

/**
 *	@def GL_OES_fbo_render_mipmap
 *	@brief if defined, GL_OES_fbo_render_mipmap entry points are available
 */
#ifndef GL_OES_fbo_render_mipmap
#define GL_OES_fbo_render_mipmap 1

extern bool GLEH_OES_fbo_render_mipmap; /**< @brief set if GL_OES_fbo_render_mipmap is supported */

// no new tokens

// no new procedures or functions

#endif // !GL_OES_fbo_render_mipmap

//								--- ~GL_OES_fbo_render_mipmap ---

//								--- GL_OES_vertex_half_float ---

/**
 *	@def GL_OES_vertex_half_float
 *	@brief if defined, GL_OES_vertex_half_float entry points are available
 */
#ifndef GL_OES_vertex_half_float
#define GL_OES_vertex_half_float 1

extern bool GLEH_OES_vertex_half_float; /**< @brief set if GL_OES_vertex_half_float is supported */

#define GL_HALF_FLOAT_OES											0x8d61 /**< @brief GL_OES_vertex_half_float enum **/

// no new procedures or functions

#endif // !GL_OES_vertex_half_float

//								--- ~GL_OES_vertex_half_float ---

//								--- GL_OES_texture_float ---

/**
 *	@def GL_OES_texture_float
 *	@brief if defined, GL_OES_texture_float entry points are available
 */
#ifndef GL_OES_texture_float
#define GL_OES_texture_float 1

extern bool GLEH_OES_texture_float; /**< @brief set if GL_OES_texture_float is supported */

//#define GL_HALF_FLOAT_OES											  0x8d61 // (defined above)
//#define GL_FLOAT													  0x1406 // (defined above)

// no new procedures or functions

#endif // !GL_OES_texture_float

//								--- ~GL_OES_texture_float ---

//								--- GL_OES_compressed_ETC1_RGB8_texture ---

/**
 *	@def GL_OES_compressed_ETC1_RGB8_texture
 *	@brief if defined, GL_OES_compressed_ETC1_RGB8_texture entry points are available
 */
#ifndef GL_OES_compressed_ETC1_RGB8_texture
#define GL_OES_compressed_ETC1_RGB8_texture 1

extern bool GLEH_OES_compressed_ETC1_RGB8_texture; /**< @brief set if GL_OES_compressed_ETC1_RGB8_texture is supported */

#define GL_ETC1_RGB8_OES											0x8d64 /**< @brief GL_OES_compressed_ETC1_RGB8_texture enum **/

// no new procedures or functions

#endif // !GL_OES_compressed_ETC1_RGB8_texture

//								--- ~GL_OES_compressed_ETC1_RGB8_texture ---

//								--- GL_OES_standard_derivatives ---

/**
 *	@def GL_OES_standard_derivatives
 *	@brief if defined, GL_OES_standard_derivatives entry points are available
 */
#ifndef GL_OES_standard_derivatives
#define GL_OES_standard_derivatives 1

extern bool GLEH_OES_standard_derivatives; /**< @brief set if GL_OES_standard_derivatives is supported */

#define GL_FRAGMENT_SHADER_DERIVATIVE_HINT_OES						0x8b8b /**< @brief GL_OES_standard_derivatives enum **/

// no new procedures or functions

#endif // !GL_OES_standard_derivatives

//								--- ~GL_OES_standard_derivatives ---

//								--- GL_EXT_texture_format_BGRA8888 ---

/**
 *	@def GL_EXT_texture_format_BGRA8888
 *	@brief if defined, GL_EXT_texture_format_BGRA8888 entry points are available
 */
#ifndef GL_EXT_texture_format_BGRA8888
#define GL_EXT_texture_format_BGRA8888 1

extern bool GLEH_EXT_texture_format_BGRA8888; /**< @brief set if GL_EXT_texture_format_BGRA8888 is supported */

#define GL_BGRA_EXT													0x80e1 /**< @brief GL_EXT_texture_format_BGRA8888 enum **/

// no new procedures or functions

#endif // !GL_EXT_texture_format_BGRA8888

//								--- ~GL_EXT_texture_format_BGRA8888 ---

//								--- GL_NV_fence ---

/**
 *	@def GL_NV_fence
 *	@brief if defined, GL_NV_fence entry points are available
 */
#ifndef GL_NV_fence
#define GL_NV_fence 1

/**
 *	@def __GENERATE_GL_NV_fence
 *	@brief if defined, GL_NV_fence entry points are available
 */
#define __GENERATE_GL_NV_fence

extern bool GLEH_NV_fence; /**< @brief set if GL_NV_fence is supported */

#define GL_ALL_COMPLETED_NV											0x84f2 /**< @brief GL_NV_fence enum **/
#define GL_FENCE_STATUS_NV											0x84f3 /**< @brief GL_NV_fence enum **/
#define GL_FENCE_CONDITION_NV										0x84f4 /**< @brief GL_NV_fence enum **/

extern void (GLApi *GLFuncPtrName(glGenFencesNV))(GLsizei n, GLuint *fences);
extern void (GLApi *GLFuncPtrName(glDeleteFencesNV))(GLsizei n, const GLuint *fences);
extern void (GLApi *GLFuncPtrName(glSetFenceNV))(GLuint fence, GLenum condition);
extern GLboolean (GLApi *GLFuncPtrName(glTestFenceNV))(GLuint fence);
extern void (GLApi *GLFuncPtrName(glFinishFenceNV))(GLuint fence);
extern GLboolean (GLApi *GLFuncPtrName(glIsFenceNV))(GLuint fence);
extern void (GLApi *GLFuncPtrName(glGetFenceivNV))(GLuint fence, GLenum pname, int *params);

inline GLboolean glTestFenceNV_glachwrap(GLuint fence, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glTestFenceNV)(fence); GL_AUTOCHECK_BLOCK("glTestFenceNV", p_s_filename, n_line); return result; }
inline GLboolean glIsFenceNV_glachwrap(GLuint fence, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glIsFenceNV)(fence); GL_AUTOCHECK_BLOCK("glIsFenceNV", p_s_filename, n_line); return result; }

#define glGenFencesNV(n,fences) do { GLFuncPtrName(glGenFencesNV)(n, fences); GL_AUTOCHECK_BLOCK("glGenFencesNV", __FILE__, __LINE__); } while(0)
#define glDeleteFencesNV(n,fences) do { GLFuncPtrName(glDeleteFencesNV)(n, fences); GL_AUTOCHECK_BLOCK("glDeleteFencesNV", __FILE__, __LINE__); } while(0)
#define glSetFenceNV(fence,condition) do { GLFuncPtrName(glSetFenceNV)(fence, condition); GL_AUTOCHECK_BLOCK("glSetFenceNV", __FILE__, __LINE__); } while(0)
#define glTestFenceNV(fence) GLAutoCheckFuncPtrName(glTestFenceNV)(fence GLAutoCheckExtraParams)
#define glFinishFenceNV(fence) do { GLFuncPtrName(glFinishFenceNV)(fence); GL_AUTOCHECK_BLOCK("glFinishFenceNV", __FILE__, __LINE__); } while(0)
#define glIsFenceNV(fence) GLAutoCheckFuncPtrName(glIsFenceNV)(fence GLAutoCheckExtraParams)
#define glGetFenceivNV(fence,pname,params) do { GLFuncPtrName(glGetFenceivNV)(fence, pname, params); GL_AUTOCHECK_BLOCK("glGetFenceivNV", __FILE__, __LINE__); } while(0)

#endif // !GL_NV_fence

//								--- ~GL_NV_fence ---

//								--- GL_EXT_texture_lod_bias ---

/**
 *	@def GL_EXT_texture_lod_bias
 *	@brief if defined, GL_EXT_texture_lod_bias entry points are available
 */
#ifndef GL_EXT_texture_lod_bias
#define GL_EXT_texture_lod_bias 1

extern bool GLEH_EXT_texture_lod_bias; /**< @brief set if GL_EXT_texture_lod_bias is supported */

#define GL_TEXTURE_FILTER_CONTROL_EXT								0x8500 /**< @brief GL_EXT_texture_lod_bias enum **/
#define GL_TEXTURE_LOD_BIAS_EXT										0x8501 /**< @brief GL_EXT_texture_lod_bias enum **/
#define GL_MAX_TEXTURE_LOD_BIAS_EXT									0x84fd /**< @brief GL_EXT_texture_lod_bias enum **/

// no new procedures or functions

#endif // !GL_EXT_texture_lod_bias

//								--- ~GL_EXT_texture_lod_bias ---

//								--- GL_NV_draw_buffers ---

/**
 *	@def GL_NV_draw_buffers
 *	@brief if defined, GL_NV_draw_buffers entry points are available
 */
#ifndef GL_NV_draw_buffers
#define GL_NV_draw_buffers 1

/**
 *	@def __GENERATE_GL_NV_draw_buffers
 *	@brief if defined, GL_NV_draw_buffers entry points are available
 */
#define __GENERATE_GL_NV_draw_buffers

extern bool GLEH_NV_draw_buffers; /**< @brief set if GL_NV_draw_buffers is supported */

#define GL_MAX_DRAW_BUFFERS_NV										0x8824 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER0_NV											0x8825 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER1_NV											0x8826 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER2_NV											0x8827 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER3_NV											0x8828 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER4_NV											0x8829 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER5_NV											0x882a /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER6_NV											0x882b /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER7_NV											0x882c /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER8_NV											0x882d /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER9_NV											0x882e /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER10_NV											0x882f /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER11_NV											0x8830 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER12_NV											0x8831 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER13_NV											0x8832 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER14_NV											0x8833 /**< @brief GL_NV_draw_buffers enum **/
#define GL_DRAW_BUFFER15_NV											0x8834 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT0_NV										0x8ce0 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT1_NV										0x8ce1 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT2_NV										0x8ce2 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT3_NV										0x8ce3 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT4_NV										0x8ce4 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT5_NV										0x8ce5 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT6_NV										0x8ce6 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT7_NV										0x8ce7 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT8_NV										0x8ce8 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT9_NV										0x8ce9 /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT10_NV									0x8cea /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT11_NV									0x8ceb /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT12_NV									0x8cec /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT13_NV									0x8ced /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT14_NV									0x8cee /**< @brief GL_NV_draw_buffers enum **/
#define GL_COLOR_ATTACHMENT15_NV									0x8cef /**< @brief GL_NV_draw_buffers enum **/

/**
 *	@brief Specifies a list of color buffers to be drawn into
 *	@param[in] n Specifies the number of buffers in bufs.
 *	@param[in] bufs Points to an array of symbolic constants specifying the buffers into which
 *		fragment colors or data values will be written.
 */
extern void (GLApi *GLFuncPtrName(glDrawBuffersNV))(GLsizei n, const GLenum *bufs);

/**
 *	@def glDrawBuffersNV
 *	@brief Specifies a list of color buffers to be drawn into
 *	@param[in] n Specifies the number of buffers in bufs.
 *	@param[in] bufs Points to an array of symbolic constants specifying the buffers into which
 *		fragment colors or data values will be written.
 */
#define glDrawBuffersNV(n,bufs) do { GLFuncPtrName(glDrawBuffersNV)(n, bufs); GL_AUTOCHECK_BLOCK("glDrawBuffersNV", __FILE__, __LINE__); } while(0)

#endif // !GL_NV_draw_buffers

//								--- ~GL_NV_draw_buffers ---

//								--- GL_NV_fbo_color_attachments ---

/**
 *	@def GL_NV_fbo_color_attachments
 *	@brief if defined, GL_NV_fbo_color_attachments entry points are available
 */
#ifndef GL_NV_fbo_color_attachments
#define GL_NV_fbo_color_attachments 1

extern bool GLEH_NV_fbo_color_attachments; /**< @brief set if GL_NV_fbo_color_attachments is supported */

#define GL_MAX_COLOR_ATTACHMENTS_NV									0x8cdf /**< @brief GL_NV_fbo_color_attachments enum **/
//#define GL_COLOR_ATTACHMENT0_NV									  0x8ce0 // (defined above)
//#define GL_COLOR_ATTACHMENT1_NV									  0x8ce1 // (defined above)
//#define GL_COLOR_ATTACHMENT2_NV									  0x8ce2 // (defined above)
//#define GL_COLOR_ATTACHMENT3_NV									  0x8ce3 // (defined above)
//#define GL_COLOR_ATTACHMENT4_NV									  0x8ce4 // (defined above)
//#define GL_COLOR_ATTACHMENT5_NV									  0x8ce5 // (defined above)
//#define GL_COLOR_ATTACHMENT6_NV									  0x8ce6 // (defined above)
//#define GL_COLOR_ATTACHMENT7_NV									  0x8ce7 // (defined above)
//#define GL_COLOR_ATTACHMENT8_NV									  0x8ce8 // (defined above)
//#define GL_COLOR_ATTACHMENT9_NV									  0x8ce9 // (defined above)
//#define GL_COLOR_ATTACHMENT10_NV									  0x8cea // (defined above)
//#define GL_COLOR_ATTACHMENT11_NV									  0x8ceb // (defined above)
//#define GL_COLOR_ATTACHMENT12_NV									  0x8cec // (defined above)
//#define GL_COLOR_ATTACHMENT13_NV									  0x8ced // (defined above)
//#define GL_COLOR_ATTACHMENT14_NV									  0x8cee // (defined above)
//#define GL_COLOR_ATTACHMENT15_NV									  0x8cef // (defined above)

// no new procedures or functions

#endif // !GL_NV_fbo_color_attachments

//								--- ~GL_NV_fbo_color_attachments ---

//								--- GL_NV_read_buffer ---

/**
 *	@def GL_NV_read_buffer
 *	@brief if defined, GL_NV_read_buffer entry points are available
 */
#ifndef GL_NV_read_buffer
#define GL_NV_read_buffer 1

/**
 *	@def __GENERATE_GL_NV_read_buffer
 *	@brief if defined, GL_NV_read_buffer entry points are available
 */
#define __GENERATE_GL_NV_read_buffer

extern bool GLEH_NV_read_buffer; /**< @brief set if GL_NV_read_buffer is supported */

#define GL_READ_BUFFER_NV											0x0c02 /**< @brief GL_NV_read_buffer enum **/

/**
 *	@brief select a color buffer source for pixels
 *	@param[in] mode Specifies a color buffer. Accepted values are GL_FRONT_LEFT, GL_FRONT_RIGHT,
 *		GL_BACK_LEFT, GL_BACK_RIGHT, GL_FRONT, GL_BACK, GL_LEFT, GL_RIGHT, and the constants
 *		GL_COLOR_ATTACHMENT i.
 */
extern void (GLApi *GLFuncPtrName(glReadBufferNV))(GLenum mode);

/**
 *	@def glReadBufferNV
 *	@brief select a color buffer source for pixels
 *	@param[in] mode Specifies a color buffer. Accepted values are GL_FRONT_LEFT, GL_FRONT_RIGHT,
 *		GL_BACK_LEFT, GL_BACK_RIGHT, GL_FRONT, GL_BACK, GL_LEFT, GL_RIGHT, and the constants
 *		GL_COLOR_ATTACHMENT i.
 */
#define glReadBufferNV(mode) do { GLFuncPtrName(glReadBufferNV)(mode); GL_AUTOCHECK_BLOCK("glReadBufferNV", __FILE__, __LINE__); } while(0)

#endif // !GL_NV_read_buffer

//								--- ~GL_NV_read_buffer ---

//								--- GL_EXT_texture_filter_anisotropic ---

/**
 *	@def GL_EXT_texture_filter_anisotropic
 *	@brief if defined, GL_EXT_texture_filter_anisotropic entry points are available
 */
#ifndef GL_EXT_texture_filter_anisotropic
#define GL_EXT_texture_filter_anisotropic 1

extern bool GLEH_EXT_texture_filter_anisotropic; /**< @brief set if GL_EXT_texture_filter_anisotropic is supported */

#define GL_TEXTURE_MAX_ANISOTROPY_EXT								0x84fe /**< @brief GL_EXT_texture_filter_anisotropic enum **/
#define GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT							0x84ff /**< @brief GL_EXT_texture_filter_anisotropic enum **/

// no new procedures or functions

#endif // !GL_EXT_texture_filter_anisotropic

//								--- ~GL_EXT_texture_filter_anisotropic ---

//								--- GL_EXT_unpack_subimage ---

/**
 *	@def GL_EXT_unpack_subimage
 *	@brief if defined, GL_EXT_unpack_subimage entry points are available
 */
#ifndef GL_EXT_unpack_subimage
#define GL_EXT_unpack_subimage 1

extern bool GLEH_EXT_unpack_subimage; /**< @brief set if GL_EXT_unpack_subimage is supported */

#define GL_UNPACK_ROW_LENGTH										0x0cf2 /**< @brief GL_EXT_unpack_subimage enum **/
#define GL_UNPACK_SKIP_ROWS											0x0cf3 /**< @brief GL_EXT_unpack_subimage enum **/
#define GL_UNPACK_SKIP_PIXELS										0x0cf4 /**< @brief GL_EXT_unpack_subimage enum **/

// no new procedures or functions

#endif // !GL_EXT_unpack_subimage

//								--- ~GL_EXT_unpack_subimage ---

//								--- GL_NV_texture_compression_s3tc_update ---

/**
 *	@def GL_NV_texture_compression_s3tc_update
 *	@brief if defined, GL_NV_texture_compression_s3tc_update entry points are available
 */
#ifndef GL_NV_texture_compression_s3tc_update
#define GL_NV_texture_compression_s3tc_update 1

extern bool GLEH_NV_texture_compression_s3tc_update; /**< @brief set if GL_NV_texture_compression_s3tc_update is supported */

// no new tokens

// no new procedures or functions

#endif // !GL_NV_texture_compression_s3tc_update

//								--- ~GL_NV_texture_compression_s3tc_update ---

//								--- GL_NV_read_depth_stencil ---

/**
 *	@def GL_NV_read_depth_stencil
 *	@brief if defined, GL_NV_read_depth_stencil entry points are available
 */
#ifndef GL_NV_read_depth_stencil
#define GL_NV_read_depth_stencil 1

extern bool GLEH_NV_read_depth_stencil; /**< @brief set if GL_NV_read_depth_stencil is supported */

// no new tokens

// no new procedures or functions

#endif // !GL_NV_read_depth_stencil

//								--- ~GL_NV_read_depth_stencil ---

//								--- GL_EXT_robustness ---

/**
 *	@def GL_EXT_robustness
 *	@brief if defined, GL_EXT_robustness entry points are available
 */
#ifndef GL_EXT_robustness
#define GL_EXT_robustness 1

/**
 *	@def __GENERATE_GL_EXT_robustness
 *	@brief if defined, GL_EXT_robustness entry points are available
 */
#define __GENERATE_GL_EXT_robustness

extern bool GLEH_EXT_robustness; /**< @brief set if GL_EXT_robustness is supported */

//#define GL_NO_ERROR												  0x0000 // (defined above)
#define GL_GUILTY_CONTEXT_RESET_EXT									0x8253 /**< @brief GL_EXT_robustness enum **/
#define GL_INNOCENT_CONTEXT_RESET_EXT								0x8254 /**< @brief GL_EXT_robustness enum **/
#define GL_UNKNOWN_CONTEXT_RESET_EXT								0x8255 /**< @brief GL_EXT_robustness enum **/
#define GL_CONTEXT_ROBUST_ACCESS_EXT								0x90f3 /**< @brief GL_EXT_robustness enum **/
#define GL_RESET_NOTIFICATION_STRATEGY_EXT							0x8256 /**< @brief GL_EXT_robustness enum **/
#define GL_LOSE_CONTEXT_ON_RESET_EXT								0x8252 /**< @brief GL_EXT_robustness enum **/
#define GL_NO_RESET_NOTIFICATION_EXT								0x8261 /**< @brief GL_EXT_robustness enum **/

extern GLenum (GLApi *GLFuncPtrName(glGetGraphicsResetStatusEXT))();
extern void (GLApi *GLFuncPtrName(glReadnPixelsEXT))(int x, int y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLsizei bufSize, void *data);
extern void (GLApi *GLFuncPtrName(glGetnUniformfvEXT))(GLuint program, int location, GLsizei bufSize, float *params);
extern void (GLApi *GLFuncPtrName(glGetnUniformivEXT))(GLuint program, int location, GLsizei bufSize, int *params);

inline GLenum glGetGraphicsResetStatusEXT_glachwrap(const char *p_s_filename, int n_line) { GLenum result = GLFuncPtrName(glGetGraphicsResetStatusEXT)(); GL_AUTOCHECK_BLOCK("glGetGraphicsResetStatusEXT", p_s_filename, n_line); return result; }

#define glGetGraphicsResetStatusEXT() GLAutoCheckFuncPtrName(glGetGraphicsResetStatusEXT)(GLAutoCheckExtraParamsSA)
#define glReadnPixelsEXT(x,y,width,height,format,type,bufSize,data) do { GLFuncPtrName(glReadnPixelsEXT)(x, y, width, height, format, type, bufSize, data); GL_AUTOCHECK_BLOCK("glReadnPixelsEXT", __FILE__, __LINE__); } while(0)
#define glGetnUniformfvEXT(program,location,bufSize,params) do { GLFuncPtrName(glGetnUniformfvEXT)(program, location, bufSize, params); GL_AUTOCHECK_BLOCK("glGetnUniformfvEXT", __FILE__, __LINE__); } while(0)
#define glGetnUniformivEXT(program,location,bufSize,params) do { GLFuncPtrName(glGetnUniformivEXT)(program, location, bufSize, params); GL_AUTOCHECK_BLOCK("glGetnUniformivEXT", __FILE__, __LINE__); } while(0)

#endif // !GL_EXT_robustness

//								--- ~GL_EXT_robustness ---

//								--- GL_EXT_occlusion_query_boolean ---

/**
 *	@def GL_EXT_occlusion_query_boolean
 *	@brief if defined, GL_EXT_occlusion_query_boolean entry points are available
 */
#ifndef GL_EXT_occlusion_query_boolean
#define GL_EXT_occlusion_query_boolean 1

/**
 *	@def __GENERATE_GL_EXT_occlusion_query_boolean
 *	@brief if defined, GL_EXT_occlusion_query_boolean entry points are available
 */
#define __GENERATE_GL_EXT_occlusion_query_boolean

extern bool GLEH_EXT_occlusion_query_boolean; /**< @brief set if GL_EXT_occlusion_query_boolean is supported */

#define GL_ANY_SAMPLES_PASSED_EXT									0x8c2f /**< @brief GL_EXT_occlusion_query_boolean enum **/
#define GL_ANY_SAMPLES_PASSED_CONSERVATIVE_EXT						0x8d6a /**< @brief GL_EXT_occlusion_query_boolean enum **/
#define GL_CURRENT_QUERY_EXT										0x8865 /**< @brief GL_EXT_occlusion_query_boolean enum **/
#define GL_QUERY_RESULT_EXT											0x8866 /**< @brief GL_EXT_occlusion_query_boolean enum **/
#define GL_QUERY_RESULT_AVAILABLE_EXT								0x8867 /**< @brief GL_EXT_occlusion_query_boolean enum **/

/**
 *	@brief generate query object names
 *	@param[in] n Specifies the number of query object names to be generated.
 *	@param[out] ids Specifies an array in which the generated query object names are stored.
 */
extern void (GLApi *GLFuncPtrName(glGenQueriesEXT))(GLsizei n, GLuint *ids);

/**
 *	@brief delete named query objects
 *	@param[in] n Specifies the number of query objects to be deleted.
 *	@param[in] ids Specifies an array of query objects to be deleted.
 */
extern void (GLApi *GLFuncPtrName(glDeleteQueriesEXT))(GLsizei n, const GLuint *ids);

/**
 *	@brief determine if a name corresponds to a query object
 *	@param[in] id Specifies a value that may be the name of a query object.
 */
extern GLboolean (GLApi *GLFuncPtrName(glIsQueryEXT))(GLuint id);

/**
 *	@brief delimit the boundaries of a query object
 *	@param[in] target Specifies the target type of query object established between glBeginQuery and
 *		the subsequent glEndQuery. The symbolic constant must be one of GL_SAMPLES_PASSED,
 *		GL_ANY_SAMPLES_PASSED, GL_PRIMITIVES_GENERATED, GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN, or GL_TIME_ELAPSED.
 *	@param[in] id Specifies the name of a query object.
 */
extern void (GLApi *GLFuncPtrName(glBeginQueryEXT))(GLenum target, GLuint id);

/**
 *	@brief delimit the boundaries of a query object
 *	@param[in] target Specifies the target type of query object to be concluded. The symbolic
 *		constant must be one of GL_SAMPLES_PASSED, GL_ANY_SAMPLES_PASSED, GL_PRIMITIVES_GENERATED,
 *		GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN, or GL_TIME_ELAPSED.
 */
extern void (GLApi *GLFuncPtrName(glEndQueryEXT))(GLenum target);

/**
 *	@brief return parameters of a query object target
 *	@param[in] target Specifies a query object target. Must be GL_SAMPLES_PASSED,
 *		GL_ANY_SAMPLES_PASSED, GL_PRIMITIVES_GENERATED, GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN,
 *		GL_TIME_ELAPSED, or GL_TIMESTAMP.
 *	@param[in] pname Specifies the symbolic name of a query object target parameter. Accepted values
 *		are GL_CURRENT_QUERY or GL_QUERY_COUNTER_BITS.
 *	@param[out] params Returns the requested data.
 */
extern void (GLApi *GLFuncPtrName(glGetQueryivEXT))(GLenum target, GLenum pname, int *params);

/**
 *	@brief return parameters of a query object
 *	@param[in] id Specifies the name of a query object.
 *	@param[in] pname Specifies the symbolic name of a query object parameter. Accepted values are
 *		GL_QUERY_RESULT or GL_QUERY_RESULT_AVAILABLE.
 *	@param[out] params Returns the requested data.
 */
extern void (GLApi *GLFuncPtrName(glGetQueryObjectuivEXT))(GLuint id, GLenum pname, GLuint *params);

inline GLboolean glIsQueryEXT_glachwrap(GLuint id, const char *p_s_filename, int n_line) { GLboolean result = GLFuncPtrName(glIsQueryEXT)(id); GL_AUTOCHECK_BLOCK("glIsQueryEXT", p_s_filename, n_line); return result; }

/**
 *	@def glGenQueriesEXT
 *	@brief generate query object names
 *	@param[in] n Specifies the number of query object names to be generated.
 *	@param[out] ids Specifies an array in which the generated query object names are stored.
 */
#define glGenQueriesEXT(n,ids) do { GLFuncPtrName(glGenQueriesEXT)(n, ids); GL_AUTOCHECK_BLOCK("glGenQueriesEXT", __FILE__, __LINE__); } while(0)

/**
 *	@def glDeleteQueriesEXT
 *	@brief delete named query objects
 *	@param[in] n Specifies the number of query objects to be deleted.
 *	@param[in] ids Specifies an array of query objects to be deleted.
 */
#define glDeleteQueriesEXT(n,ids) do { GLFuncPtrName(glDeleteQueriesEXT)(n, ids); GL_AUTOCHECK_BLOCK("glDeleteQueriesEXT", __FILE__, __LINE__); } while(0)

/**
 *	@def glIsQueryEXT
 *	@brief determine if a name corresponds to a query object
 *	@param[in] id Specifies a value that may be the name of a query object.
 */
#define glIsQueryEXT(id) GLAutoCheckFuncPtrName(glIsQueryEXT)(id GLAutoCheckExtraParams)

/**
 *	@def glBeginQueryEXT
 *	@brief delimit the boundaries of a query object
 *	@param[in] target Specifies the target type of query object established between glBeginQuery and
 *		the subsequent glEndQuery. The symbolic constant must be one of GL_SAMPLES_PASSED,
 *		GL_ANY_SAMPLES_PASSED, GL_PRIMITIVES_GENERATED, GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN, or GL_TIME_ELAPSED.
 *	@param[in] id Specifies the name of a query object.
 */
#define glBeginQueryEXT(target,id) do { GLFuncPtrName(glBeginQueryEXT)(target, id); GL_AUTOCHECK_BLOCK("glBeginQueryEXT", __FILE__, __LINE__); } while(0)

/**
 *	@def glEndQueryEXT
 *	@brief delimit the boundaries of a query object
 *	@param[in] target Specifies the target type of query object to be concluded. The symbolic
 *		constant must be one of GL_SAMPLES_PASSED, GL_ANY_SAMPLES_PASSED, GL_PRIMITIVES_GENERATED,
 *		GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN, or GL_TIME_ELAPSED.
 */
#define glEndQueryEXT(target) do { GLFuncPtrName(glEndQueryEXT)(target); GL_AUTOCHECK_BLOCK("glEndQueryEXT", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetQueryivEXT
 *	@brief return parameters of a query object target
 *	@param[in] target Specifies a query object target. Must be GL_SAMPLES_PASSED,
 *		GL_ANY_SAMPLES_PASSED, GL_PRIMITIVES_GENERATED, GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN,
 *		GL_TIME_ELAPSED, or GL_TIMESTAMP.
 *	@param[in] pname Specifies the symbolic name of a query object target parameter. Accepted values
 *		are GL_CURRENT_QUERY or GL_QUERY_COUNTER_BITS.
 *	@param[out] params Returns the requested data.
 */
#define glGetQueryivEXT(target,pname,params) do { GLFuncPtrName(glGetQueryivEXT)(target, pname, params); GL_AUTOCHECK_BLOCK("glGetQueryivEXT", __FILE__, __LINE__); } while(0)

/**
 *	@def glGetQueryObjectuivEXT
 *	@brief return parameters of a query object
 *	@param[in] id Specifies the name of a query object.
 *	@param[in] pname Specifies the symbolic name of a query object parameter. Accepted values are
 *		GL_QUERY_RESULT or GL_QUERY_RESULT_AVAILABLE.
 *	@param[out] params Returns the requested data.
 */
#define glGetQueryObjectuivEXT(id,pname,params) do { GLFuncPtrName(glGetQueryObjectuivEXT)(id, pname, params); GL_AUTOCHECK_BLOCK("glGetQueryObjectuivEXT", __FILE__, __LINE__); } while(0)

#endif // !GL_EXT_occlusion_query_boolean

//								--- ~GL_EXT_occlusion_query_boolean ---

//								--- GL_EXT_texture_array ---

/**
 *	@def GL_EXT_texture_array
 *	@brief if defined, GL_EXT_texture_array entry points are available
 */
#ifndef GL_EXT_texture_array
#define GL_EXT_texture_array 1

/**
 *	@def __GENERATE_GL_EXT_texture_array
 *	@brief if defined, GL_EXT_texture_array entry points are available
 */
#define __GENERATE_GL_EXT_texture_array

extern bool GLEH_EXT_texture_array; /**< @brief set if GL_EXT_texture_array is supported */

#define GL_TEXTURE_1D_ARRAY_EXT										0x8c18 /**< @brief GL_EXT_texture_array enum **/
#define GL_TEXTURE_2D_ARRAY_EXT										0x8c1a /**< @brief GL_EXT_texture_array enum **/
#define GL_PROXY_TEXTURE_2D_ARRAY_EXT								0x8c1b /**< @brief GL_EXT_texture_array enum **/
#define GL_PROXY_TEXTURE_1D_ARRAY_EXT								0x8c19 /**< @brief GL_EXT_texture_array enum **/
#define GL_TEXTURE_BINDING_1D_ARRAY_EXT								0x8c1c /**< @brief GL_EXT_texture_array enum **/
#define GL_TEXTURE_BINDING_2D_ARRAY_EXT								0x8c1d /**< @brief GL_EXT_texture_array enum **/
#define GL_MAX_ARRAY_TEXTURE_LAYERS_EXT								0x88ff /**< @brief GL_EXT_texture_array enum **/
#define GL_COMPARE_REF_DEPTH_TO_TEXTURE_EXT							0x884e /**< @brief GL_EXT_texture_array enum **/
#define GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER_EXT					0x8cd4 /**< @brief GL_EXT_texture_array enum **/
#define GL_SAMPLER_1D_ARRAY_EXT										0x8dc0 /**< @brief GL_EXT_texture_array enum **/
#define GL_SAMPLER_2D_ARRAY_EXT										0x8dc1 /**< @brief GL_EXT_texture_array enum **/
#define GL_SAMPLER_1D_ARRAY_SHADOW_EXT								0x8dc3 /**< @brief GL_EXT_texture_array enum **/
#define GL_SAMPLER_2D_ARRAY_SHADOW_EXT								0x8dc4 /**< @brief GL_EXT_texture_array enum **/

/**
 *	@brief attach a single layer of a texture to a framebuffer
 *	@param[in] target Specifies the framebuffer target. target must be GL_DRAW_FRAMEBUFFER,
 *		GL_READ_FRAMEBUFFER, or GL_FRAMEBUFFER. GL_FRAMEBUFFER is equivalent to GL_DRAW_FRAMEBUFFER.
 *	@param[in] attachment Specifies the attachment point of the framebuffer. attachment must be
 *		GL_COLOR_ATTACHMENT i, GL_DEPTH_ATTACHMENT, GL_STENCIL_ATTACHMENT or GL_DEPTH_STENCIL_ATTACHMMENT.
 *	@param[in] texture Specifies the texture object to attach to the framebuffer attachment point
 *		named by attachment.
 *	@param[in] level Specifies the mipmap level of texture to attach.
 *	@param[in] layer Specifies the layer of texture to attach.
 */
extern void (GLApi *GLFuncPtrName(glFramebufferTextureLayerEXT))(GLenum target, GLenum attachment, GLuint texture, int level, int layer);

/**
 *	@def glFramebufferTextureLayerEXT
 *	@brief attach a single layer of a texture to a framebuffer
 *	@param[in] target Specifies the framebuffer target. target must be GL_DRAW_FRAMEBUFFER,
 *		GL_READ_FRAMEBUFFER, or GL_FRAMEBUFFER. GL_FRAMEBUFFER is equivalent to GL_DRAW_FRAMEBUFFER.
 *	@param[in] attachment Specifies the attachment point of the framebuffer. attachment must be
 *		GL_COLOR_ATTACHMENT i, GL_DEPTH_ATTACHMENT, GL_STENCIL_ATTACHMENT or GL_DEPTH_STENCIL_ATTACHMMENT.
 *	@param[in] texture Specifies the texture object to attach to the framebuffer attachment point
 *		named by attachment.
 *	@param[in] level Specifies the mipmap level of texture to attach.
 *	@param[in] layer Specifies the layer of texture to attach.
 */
#define glFramebufferTextureLayerEXT(target,attachment,texture,level,layer) do { GLFuncPtrName(glFramebufferTextureLayerEXT)(target, attachment, texture, level, layer); GL_AUTOCHECK_BLOCK("glFramebufferTextureLayerEXT", __FILE__, __LINE__); } while(0)

#endif // !GL_EXT_texture_array

//								--- ~GL_EXT_texture_array ---

#endif // !__OPENGLES2_EMULATOR_INCLUDED
