/*
								+--------------------------------+
								|                                |
								|  ***  Mini OpenGL driver  ***  |
								|                                |
								|  Copyright  -tHE SWINe- 2009  |
								|                                |
								|            MiniGL.h            |
								|                                |
								+--------------------------------+
*/

#pragma once
#ifndef __MINI_OPENGL_DRIVER_INCLUDED
#define __MINI_OPENGL_DRIVER_INCLUDED

/**
 *	@file gl2/MiniGL.h
 *	@author -tHE SWINe-
 *	@date 2009
 *	@brief minimal OpenGL driver
 *
 *	@date 2009-10-08
 *
 *	added MINIGL_FORCE_GLUT, renamed GPU_LIB_WIN32_SHOW_WINDOW to MINIGL_SHOW_WINDOW
 *
 *	wrapped global functions as static members to CMiniGL class
 *
 *	@date 2012-06-19
 *
 *	Moved multiple inclusion guard before file documentation comment.
 *
 */

#include "OpenGL20.h"
#include "OpenGLState.h"

/**
 *	@def MINIGL_FORCE_GLUT
 *
 *	@brief if defined, GLUT is preferred interface even in windows
 *		(where WGL would be used otherwise)
 */
//#define MINIGL_FORCE_GLUT

/**
 *	@def MINIGL_SHOW_WINDOW
 *
 *	@brief if defined, window used for creating OpenGL context is visible
 *		(it contains image OpenGL 2.0 logo, inlined in the code)
 */
#define MINIGL_SHOW_WINDOW

/**
 *	@brief minimal OpenGL driver
 *
 *	This is used where rendering to window is not needed (GPGPU applications, etc.).
 *
 *	@note Supports both linux / windows.
 */
class CMiniGL {
protected:
	static CGLState *m_p_state;

public:
	/**
	 *	@brief initializes OpenGL. under linux uses GLUT, under windows uses
	 *		WGL system (unless MINIGL_FORCE_GLUT defined)
	 *
	 *	@return Returns true on success (or multiple initialization), false on failure.
	 *
	 *	@note there are no parameters with framebuffer requirements,
	 *		one should not make any assumptions about the framebuffer and
	 *		always use offscreen buffers (P-buffers / FBO's).
	 *	@note there's option to display window, containing "OpenGL 2.0"
	 *		logo (enabled using MINIGL_SHOW_WINDOW macro).
	 */
	static bool Init();

	/**
	 *	@brief determines if OpenGL is initialized
	 *
	 *	@return Returns true if OpenGL is initialized, otherwise returns false.
	 */
	static bool b_Initialized()
	{
		return m_p_state != 0;
	}

	/**
	 *	@brief gets pointer to OpenGL state guard
	 *
	 *	@return Returns pointer to OpenGL state guard, or 0 if OpenGL is not initialized.
	 *
	 *	@note Do not free the pointer!
	 */
	static CGLState *p_Get_GLState()
	{
		return m_p_state;
	}

	/**
	 *	@brief shuts OpenGL down, previously initialized by Init()
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note There's no function to shut down GLUT, this is expected
	 *		to be called just before exitting as it has no effect
	 *		(but repeating Init() - Shutdown() multiple times is still possible,
	 *		only OpenGL objects, such as textures, shaders, etc. are not deleted
	 *		and OpenGL state isn't reset to default state).
	 */
	static bool Shutdown();
};

#endif // !__MINI_OPENGL_DRIVER_INCLUDED
