/*****************************************************\
| File: packet_capture.c                              | 
| Author: Jan Drazil (xdrazi00)                       |
| Desc: Utility to capture data from uProbe exporter  |
|                                                     |
| Copyright (c) 2014 Brno University of Technology    |
\*****************************************************/

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <stdint.h>

#include <sys/socket.h>
#include <arpa/inet.h>
#include <linux/if_packet.h>
#include <net/ethernet.h> /* the L2 protocols */
#include <netdb.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <time.h>
#include <signal.h>

#include <pcap/pcap.h>

#include "packet_capture.h"

/**
 * Recv TCP connection from uProbe
 * port - listening port
 * Return socket ready for listening
 **/
int tcp_connection(char *port);

/**
 * Prepare pcap_handle for receiven from hardware uProbe export
 * interface - listening interface
 * Return pcap_handle of capturing interface
 **/
pcap_t *direct_connection(char *interface);

/**
 * Capture packets form export, parse INI3 and save in pcap
 * socket - socket where comes captured packets
 * output_file - ouptut pcap file open in write and binary mode
 * Return 0 if connection id normaly closed
 **/
int capture_TCP(int socket, FILE *output_file);

/**
 * Capture packets form hw export, parse INI3MAC and save in pcap
 * pcap - pcapt_handle with prepared promiscuid interface
 * output_file - ouptut pcap file open in write and binary mode
 * Return 0 if connection id normaly closed
 **/
int capture_DC(pcap_t *pcap, FILE *output_file);

int verbose = 0;
int stop = 0;

void sighandler(int signum)
{
	if(signum == SIGINT) {
		VERBOSE("Exiting...\n");
		stop = 1;
	}
}

int main(int argc, char **argv)
{

	struct sigaction act;
	memset(&act, 0, sizeof(struct sigaction));
	act.sa_handler = sighandler;
	sigaction(SIGINT, &act, NULL);


	int print_help = 0;
	int direct_export = -1;
	char source[128];
	char filename[256] = "dump.pcap";
	int opt;


	while ((opt = getopt(argc, argv, "vf:d:p:h")) != -1) {
		switch(opt) {
			case 'd':
				direct_export = 1;
				if(strlen(optarg) > 127) {
					fprintf(stderr, "Interface name can be long only 127 symbols.\n");
					exit(EXIT_FAILURE);
				}
				strcpy(source, optarg);
				break;
			case 'p':
				direct_export = 0;
				if(strlen(optarg) > 5) {
					fprintf(stderr, "Port can be 5 chars long only!\n");
					exit(EXIT_FAILURE);
				}
				strcpy(source, optarg);
				break;
			case 'v':
				verbose = 1;
				break;
			case'f':
				if(strlen(optarg) > 255) {
					fprintf(stderr, "Output file can be long only 255 symbols.\n");
					exit(EXIT_FAILURE);
				}
				strcpy(filename, optarg);
				break;
			case 'h':
			default:
				print_help = 1;
		}
	}

	if(direct_export < 0) {
		fprintf(stderr, "Expected parameter p or q!\n");
		print_help = 1;
	}

	if(print_help || argc == 1) {
		printf("Usage: %s [-h] [-f filename] {-d interface | -p port}\n", argv[0]);
		printf("\t-p port        - Save communication from uProbe when export is set to HDD export. Connection from uProbe is expected on given TCP port\n");
		printf("\t-d interface   - Save communication from uProbe when export is set to Direct export. Captured packets are expected on given interface.\n");
		//printf("\t-t timeout     - Set timeout to specified number of second. Defaults to 0 (infinity).\n");
		printf("\t-f filename    - Set output filename (default dump.pcap)\n");
		exit(EXIT_SUCCESS);
	}


	VERBOSE("Opening file %s\n", filename);
	FILE *fd = fopen(filename, "wb");

	if(direct_export == 0) {
		int socket = tcp_connection("");
		capture_TCP(socket, fd);
		close(socket);
		fclose(fd);
	} else {
		pcap_t *handle = direct_connection("enp0s25");
		capture_DC(handle, fd);
	}

	return 0;


}

/**
 * Recv TCP connection from uProbe
 * port - listening port
 * Return socket ready for listening
 **/
int tcp_connection(char *port)
{
	int sockfd, new_fd;  // listen on sock_fd, new connection on new_fd
	struct addrinfo hints, *servinfo, *p;
	struct sockaddr_storage their_addr; // connector's address information
	socklen_t sin_size;
	int reuse_addr=1;
	int rv;

	memset(&hints, 0, sizeof hints);
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_PASSIVE; // use my IP

	if ((rv = getaddrinfo(NULL, port, &hints, &servinfo)) != 0) {
		fprintf(stderr, "getaddrinfo: %s\n", gai_strerror(rv));
		return -1;
	}

	// loop through all the results and bind to the first we can
	for(p = servinfo; p != NULL; p = p->ai_next) {
		if ((sockfd = socket(p->ai_family, p->ai_socktype,
				p->ai_protocol)) == -1) {
			continue;
		}

		if (setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &reuse_addr,
				sizeof(int)) == -1) {
			return -1;
		}

		if (bind(sockfd, p->ai_addr, p->ai_addrlen) == -1) {
			close(sockfd);
			continue;
		}

		break;
	}

	if (p == NULL)  {
		fprintf(stderr, "ERROR: Failed to bind\n");
		return -1;
	}

	freeaddrinfo(servinfo); // all done with this structure

	if (listen(sockfd, 5) == -1) {
		fprintf(stderr, "ERROR: (listen) %s\n", strerror(errno));
		return -1;
	}

	VERBOSE("Waiting for connection...\n");

	sin_size = sizeof their_addr;
	new_fd = accept(sockfd, (struct sockaddr *)&their_addr, &sin_size);
	if (new_fd == -1) {
		fprintf(stderr, "ERROR (accept): %s\n", strerror(errno));
		return -1;
	}

	VERBOSE("New connection established.\n");
	close(sockfd);

	return new_fd;
}

/**
 * Capture packets form export, parse INI3 and save in pcap
 * socket - socket where comes captured packets
 * output_file - ouptut pcap file open in write and binary mode
 * Return 0 if connection id normaly closed
 **/
int capture_TCP(int socket, FILE *output_file)
{
	char buff[BUFF_SIZE];
	char pcap_item[BUFF_SIZE];

	int buffer_last_read;
	int buffer_read = 0;
	int pcap_item_seized = 0;
	struct ini3_header *ini3;
	struct pcap_packet_header pcap_h;

	int ini3_load_left = INI3_HEADER_SIZE;
	int data_load_left;

	int read_next = 1;

	fwrite(global_pcap_header_ini3, 1, GLOBAL_PCAP_HEADER_LEN, output_file);

	while(!stop) {

		if(read_next) {
			//Whole recv buffer was proccess, get new data
			buffer_last_read = recv(socket, buff, BUFF_SIZE, 0);
			if(buffer_last_read == -1) {
				if(errno != EINTR)
					fprintf(stderr, "Read error: %s\n", strerror(errno));
				break;

			} else if(buffer_last_read == 0) {
				VERBOSE("Connection closed.\n");
				return 0;
			}
			buffer_read = 0;
		}

		if(ini3_load_left) {
			//INI3 is load partially or not at all
			if(buffer_last_read-buffer_read >= ini3_load_left) {
				//All neccessary bytes are in actual recv buffer 
				
				//Copy all neccessary bytes of INI3 into buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, ini3_load_left);
				
				//Set pointer on position of unproccessed data in recv buffer
				buffer_read += ini3_load_left; 
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += ini3_load_left;
				
				//Mark INI3 as read
				ini3_load_left = 0;

				//Nasty "zero-copy" hack
				ini3 = (struct ini3_header*) pcap_item;
				
				//Prepare for loading captured packet
				data_load_left = ini3->blob_size;
			} else {
				//In recv buffer is not enought bytes for INI3
				
				//Copy all unproccessed bytes from recv buffer to buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, buffer_last_read-buffer_read);
				
				//Set count of bytes which are need for read rest of INI3
				ini3_load_left -= buffer_last_read-buffer_read;
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += buffer_last_read-buffer_read;
				
				//Mark recv buffer as fully proccessed
				read_next = 1;
				continue;
			}
		}

	   if(data_load_left) {
		   //Loading captured data
		   
			if(buffer_last_read-buffer_read == data_load_left) {
				//in unproccessed part of recv buff are only and exact captured packet bytes
				
				//Copy packet data into buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, data_load_left);
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += data_load_left;
				
				//Mark recv buffer as fully proccessed
				read_next = 1;				
			} else if(buffer_last_read-buffer_read > data_load_left) {
				//in unproccessed part of recv buff are exact captured packet bytes, but not only
				
				//Copy packet data into buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, data_load_left);
				
				//Set pointer on position of unproccessed data in recv buffer
				buffer_read += data_load_left;
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += data_load_left;
				
				//In next cycle DO NOT overwrite recv buffer
				read_next = 0;
			} else {
				//in unproccessed part of recv buff isn't whole captured packet
				
				//Copy packet data into buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, buffer_last_read-buffer_read);
				
				//Set count of bytes which are need for read rest of packet
				data_load_left -= buffer_last_read-buffer_read;
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += buffer_last_read-buffer_read;
				
				//Mark recv buffer as fully proccessed
				read_next = 1;
				continue;
			}
		}

		//Extraction meta-data from INI3 for save packet into pcap with valid params
		pcap_h.ts_sec = ini3->timestamp_s;
		pcap_h.ts_usec = ini3->timestamp_ns/1000;
		pcap_h.incl_len = ini3->blob_size;// + INI3_HEADER_SIZE;
		pcap_h.orig_len = pcap_h.incl_len;

		//Write packet into pcap
		fwrite(&pcap_h, 1, sizeof(struct pcap_packet_header), output_file);
		fwrite(pcap_item+INI3_HEADER_SIZE, 1, ini3->blob_size, output_file);
		//fflush(output_file);

		//Initialize for process of next packet
		ini3_load_left = INI3_HEADER_SIZE;
		pcap_item_seized = 0;

	}

	fflush(output_file);
	return 0;
}

/**
 * Prepare pcap_handle for receiven from hardware uProbe export
 * interface - listening interface
 * Return pcap_handle of capturing interface
 **/
pcap_t* direct_connection(char *interface)
{
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_t *handle = pcap_open_live(interface, BUFSIZ, 1, 0, errbuf);
	if(handle == NULL) {
		fprintf(stderr, "Open interface %s error: %s", interface, errbuf);
		return NULL;
	}

	if (pcap_datalink(handle) != DLT_EN10MB) {
		fprintf(stderr, "Device %s doesn't provide Ethernet headers - not supported\n", interface);
		return NULL;
	}

	return handle;
}

/**
 * Capture packets form hw export, parse INI3MAC and save in pcap
 * pcap - pcapt_handle with prepared promiscuid interface
 * output_file - ouptut pcap file open in write and binary mode
 * Return 0 if connection id normaly closed
 **/
int capture_DC(pcap_t *pcap, FILE *output_file) {
	struct pcap_pkthdr pkt_h;
	pcap_dumper_t *dump = pcap_dump_fopen(pcap, output_file);
	const u_char *packet;
	while(!stop) {
		packet = pcap_next(pcap, &pkt_h);

		//Timestamp extraction from INI3
		pkt_h.ts.tv_sec = *((uint32_t *)(packet+4));
		pkt_h.ts.tv_sec = *((uint32_t *)(packet+8));

		//Save paket into pcap
		pcap_dump((u_char *)dump, &pkt_h, packet);
	}

	pcap_dump_close(dump);
	return 0;

}
