#pragma once
#include "Benchmark.h"
#include "ProfilerD3D11.h"
#include "DWTSchemeD3D11.h"
#include "AppFrameD3D11.h"

#include <d3d11.h>
#include <wrl.h>
#include <memory>
#include <DirectXMath.h>

class AppFrameD3D11;

class BenchmarkD3D11Schemes :
    public Benchmark {
public:
    // Constructor
    BenchmarkD3D11Schemes( AppFrameD3D11* applicationFrame, Microsoft::WRL::ComPtr<ID3D11Device>& device, Microsoft::WRL::ComPtr<ID3D11DeviceContext>& context, Microsoft::WRL::ComPtr<IDXGISwapChain>& swapChain );

    // Management of benchmarked schemes
    virtual void PreviousScheme() override;
    virtual void PreviousResolution() override;
    virtual void NextScheme() override;
    virtual void NextResolution() override;

    // Run per-frame benchmark
    virtual void Run() override;

    // Start automatical benchmark
    virtual void Start() override;

protected:
    // Application frame
    AppFrameD3D11*  applicationFrame;

    // Initialize Direct3D for benchmarking
    void InitD3D();

    // Direct3D
    Microsoft::WRL::ComPtr<ID3D11Device>            device;
    Microsoft::WRL::ComPtr<ID3D11DeviceContext>     context;
    Microsoft::WRL::ComPtr<IDXGISwapChain>          swapChain;

    Microsoft::WRL::ComPtr<ID3D11RenderTargetView>  renderTargetView;

    Microsoft::WRL::ComPtr<ID3D11VertexShader>      vsMain;
    Microsoft::WRL::ComPtr<ID3D11PixelShader>       psCopyFloat4;
    Microsoft::WRL::ComPtr<ID3D11PixelShader>       psCopy4Tex;
    Microsoft::WRL::ComPtr<ID3D11PixelShader>       psRandomFloat4;
    Microsoft::WRL::ComPtr<ID3D11PixelShader>       psRandom4Tex;

    Microsoft::WRL::ComPtr<ID3D11Buffer>            constantBuffer;
    Microsoft::WRL::ComPtr<ID3D11SamplerState>      samplerState;

    // Profiler
    std::shared_ptr<ProfilerD3D11>                  profiler;

    // Shader file
    std::wstring                                    shaderFile = L"schemes/base.hlsl";

    // Constant buffer structure
    struct ConstantBuffer {
        DirectX::XMFLOAT4 textureRes;
    };

    // Schemes
    SchemesContainer<std::shared_ptr<DWTScheme>> immediateSchemes;
    //SchemesContainer<std::shared_ptr<DWTScheme>> deferredSchemes;

    // Time samplers per resolution
    std::vector<SchemesContainer<Sampler>> immediateTimeSamplers;
    //std::vector<SchemesContainer<Sampler>> deferredTimeSamplers;

    bool benchmarking = false;
    float maxVRAMUsage = 0.8f;
    unsigned int schemeIndex = 0;
    unsigned int schemesCount = 0;
    unsigned int resolutionIndex = 0;
    unsigned int resolutionsCountVRAMLimited = 0;
    unsigned int iterationIndex = 0;
    unsigned int runIndex = 0;
    unsigned int ignoreIterations = 15;
    unsigned int requiedTimeSamples = 80;
    unsigned int requiedRuns = 10;

    // Returns current scheme and corresponding time sampler
    void GetCurrentScheme( DWTSchemeD3D11 ** scheme, Sampler ** sampler );

    // Initialize current scheme
    void SetupScheme();

    // Print results to file
    void PrintResults(); 

    inline double GBPSMedian( Sampler::Results results, unsigned int pisels ) {
        return pisels * ( 1000.0 / results.median ) / 1000000000.0;
    }
};