#include "Application.h"
#include "AppFrameD3D11.h"
#include "DXHelper.h"
#include "Console.h"

#include <algorithm>
using Microsoft::WRL::ComPtr;
using namespace std;

AppFrameD3D11::AppFrameD3D11( wstring name, unsigned int width, unsigned int height ):
    AppFrame( name, width, height ) {
}

AppFrameD3D11::~AppFrameD3D11() {
}

void AppFrameD3D11::ConsoleInit() {
    // Initialize console 
    Console::Init();

    // Window
    int windowMode = -1;
    while ( windowMode != 1 && windowMode != 0 ) {
        wcout << "Show Preview in Window [0/1]:";
        cin >> windowMode;
    }
    useWindow = static_cast<bool>( windowMode );
}


void AppFrameD3D11::Init() {
    // Initialize Direct3D
    InitD3D();

    // Create benchmark
    benchmark.reset( new BenchmarkD3D11Schemes( this, device, context, swapChain ) );

    running = true;
}

void AppFrameD3D11::OnKeyPress( unsigned char key ) {
    switch ( key ) {
        case VK_LEFT:
            benchmark->PreviousResolution();
            break;
        case VK_RIGHT:
            benchmark->NextResolution();
            break;
        case VK_UP:
            benchmark->NextScheme();
            break;
        case VK_DOWN:
            benchmark->PreviousScheme();
            break;
        case VK_RETURN:
            benchmark->Start();
            break;
        case VK_ESCAPE:
            Application::Close();
            break;
    }
}

void AppFrameD3D11::Run() {
    benchmark->Run();
}

void AppFrameD3D11::Destroy() {
    // Release console 
    Console::Release();
}

void AppFrameD3D11::InitD3D() {
    HRESULT hr = S_OK;

    // Adapter
    ComPtr<IDXGIAdapter1> adapter;

    // IDXGIFactory interface for adapter enumeration
    {
        // Factory
        ComPtr<IDXGIFactory1> factoryDXGI = nullptr;
        hr = CreateDXGIFactory1( __uuidof( IDXGIFactory1 ), ( void** ) &factoryDXGI );
        ThrowIfFailed( hr, L"IDXGIFactory1::CreateDXGIFactory1 error" );

        // Adapter enumeration
        std::vector<ComPtr<IDXGIAdapter1>> adapters;
        std::vector<DXGI_ADAPTER_DESC1> adapterDescriptions;
        ComPtr<IDXGIAdapter1> adapterTMP = nullptr;
        for ( unsigned int i = 0; factoryDXGI->EnumAdapters1( i, &adapterTMP ) != DXGI_ERROR_NOT_FOUND; ++i ) {
            // Get adapter description
            DXGI_ADAPTER_DESC1 adapterDescription;
            adapterTMP->GetDesc1( &adapterDescription );
            ThrowIfFailed( hr, L"IDXGIAdapter1::GetDesc error" );
            if ( adapterDescription.DedicatedVideoMemory > 0 ) {
                // Store adapter and adapter description
                adapters.push_back( adapterTMP );
                adapterDescriptions.push_back( adapterDescription );
            }
        }

        unsigned int adapterIndex = 0;
        if ( adapters.size() > 1 ) {
            // Adapter selection
            wcout << "Available adapters:" << endl;
            for ( unsigned int i = 0; i < adapters.size(); ++i ) {
                wcout << "\t" << i + 1 << ": " << adapterDescriptions[ i ].Description << " (" << adapterDescriptions[ i ].DedicatedVideoMemory / 1024 / 1024 << " MB)" << endl;
            }
            while ( adapterIndex < 1 || adapterIndex > adapters.size() ) {
                wcout << "Select adapter [1.." << adapters.size() << "]: ";
                cin >> adapterIndex;
            }
            --adapterIndex;
        }
        hr = adapters[ adapterIndex ].As( &adapter );
        ThrowIfFailed( hr );
    }

    // Create Direct3D device and device context
    {
        // Requied feature level
        D3D_FEATURE_LEVEL featureLevels[] = {
            D3D_FEATURE_LEVEL_11_0,
        };
        UINT numFeatureLevels = _countof( featureLevels );

        UINT createDeviceFlags = 0;

        HWND window = Application::GetHwnd();
        if ( window ) {
            // Describe the swap chain
            DXGI_SWAP_CHAIN_DESC sd = {};
            sd.BufferCount = 1;
            sd.BufferDesc.Width = width;
            sd.BufferDesc.Height = height;
            sd.BufferDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
            sd.BufferDesc.RefreshRate.Numerator = 60;
            sd.BufferDesc.RefreshRate.Denominator = 1;
            sd.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
            sd.OutputWindow = Application::GetHwnd();
            sd.SampleDesc.Count = 1;
            sd.SampleDesc.Quality = 0;
            sd.Windowed = TRUE;

        #ifdef _DEBUG
            // Enable debug layer
            createDeviceFlags |= D3D11_CREATE_DEVICE_DEBUG;
        #endif

            // Create the device and swap chain
            hr = D3D11CreateDeviceAndSwapChain( adapter.Get(), D3D_DRIVER_TYPE_UNKNOWN, nullptr, createDeviceFlags, featureLevels, numFeatureLevels,
                                                D3D11_SDK_VERSION, &sd, swapChain.ReleaseAndGetAddressOf(), device.ReleaseAndGetAddressOf(), &featureLevel, context.ReleaseAndGetAddressOf() );
            ThrowIfFailed( hr, L"D3D11CreateDeviceAndSwapChain (DirectX 11 GPU required)" );
        }
        else {
        #ifdef _DEBUG
            // Enable debug layer
            createDeviceFlags |= D3D11_CREATE_DEVICE_DEBUG;
        #endif

            // Create the device and swap chain
            hr = D3D11CreateDevice( adapter.Get(), D3D_DRIVER_TYPE_UNKNOWN, nullptr, createDeviceFlags, featureLevels, numFeatureLevels,
                                                D3D11_SDK_VERSION, device.ReleaseAndGetAddressOf(), &featureLevel, context.ReleaseAndGetAddressOf() );
            ThrowIfFailed( hr, L"D3D11CreateDevice (DirectX 11 GPU required)" );

        }
    }

    // Get adapter description
    {
        ComPtr<IDXGIDevice> DXGIDevice;
        ComPtr<IDXGIAdapter> DXGIAdapter;

        hr = device.As( &DXGIDevice );
        ThrowIfFailed( hr );
        hr = DXGIDevice->GetAdapter( DXGIAdapter.GetAddressOf() );
        ThrowIfFailed( hr, L"IDXGIDevice::DXGIAdapter error" );
        hr = DXGIAdapter->GetDesc( &adapterDesc );
        ThrowIfFailed( hr, L"DXGIAdapter::GetDesc error" );
    }
}