#include <string>
#include <iostream>
#include "VideoMatcher.h"


class ClArgs {
public:
	typedef enum {
		OUT_UNKNOWN,
		OUT_SEGMENTS_TEXT,
		OUT_SEGMENTS_XML,
		OUT_DISSIMILARITIES_TEXT
	} OutputFormat;

protected:
	bool ok;
	std::string fileRef;
	std::string fileQuery;
	OutputFormat outputFormat;

	OutputFormat getOutputFormat(std::string outputFormatStr) {
		if(outputFormatStr == "seg") {
			return OUT_SEGMENTS_TEXT;
		}
		else if(outputFormatStr == "segxml") {
			return OUT_SEGMENTS_XML;
		}
		else if(outputFormatStr == "dis") {
			return OUT_DISSIMILARITIES_TEXT;
		}

		setError("Invalid output format type. Must be one of [seg|segxml|dis]");
		return OUT_UNKNOWN;
	}

public:
	ClArgs(int argc, char* argv[]) {
		emptyDefaults();
		ok = true;

		if(argc == 1) {
			printHelp();
			exit(0);
		}

		int i = 1;
		while(i < argc) {			
			// print help
			if(!strcmp(argv[i], "-h") || !strcmp(argv[i], "--help")) {
				printHelp();
				if(argc == 2) {
					exit(0);
				}
			}

			// input video filename
			else if(!strcmp(argv[i], "-r")) {
				if(i < argc-1) {
					if(fileRef == "") {
						fileRef = cv::string(argv[++i]);
					}
					else {
						setError("Duplicate -r parameter.");
					}
				}
				else {
					setError("Missing argument of -r parameter.");
				}
			}

			// output video filename
			else if(!strcmp(argv[i], "-q")) {
				if(i < argc-1) {
					if(fileQuery == "") {
						fileQuery = std::string(argv[++i]);
					}
					else {
						setError("Duplicate -q parameter.");
					}
				}
				else {
					setError("Missing argument of -q parameter.");
				}
			}

			// output format
			else if(!strcmp(argv[i], "-o")) {
				if(i < argc-1) {
					if(outputFormat == OUT_UNKNOWN) {
						outputFormat = getOutputFormat(argv[++i]);
					}
					else {
						setError("Duplicate -o parameter.");
					}
				}
				else {
					setError("Missing argument of -o parameter.");
				}
			}

			i++;
		}

		completeDefaults();

		// check args
		if(fileRef == "") {
			setError("Missing reference video file (-r).");
		}

		if(fileQuery == "") {
			setError("Missing test video file (-t).");
		}

		if(!ok) {
			std::cerr << "Bad command line arguments. Run this program with '-h' or '--help' for help." << std::endl;
		}
	}

	void emptyDefaults() {
		fileRef = "";
		fileQuery = "";
		outputFormat = OUT_UNKNOWN;
	}

	void completeDefaults() {
		if(outputFormat == OUT_UNKNOWN) {
			outputFormat = OUT_DISSIMILARITIES_TEXT;
		}
	}

	void printHelp() const {
		std::cout << "vMatch video comparison tool" << std::endl
				  << "----------------------------" << std::endl
				  << "Usage:" << std::endl
				  << "./vmatch-compare -r ref.avi -t qry.avi [-o seg|seg_xml|dis] [-h]" << std::endl
				  << std::endl
				  << "Arguments:" << std::endl
				  << "-r ref.avi        Reference video file name." << std::endl
				  << "-q qry.avi        Query video file name." << std::endl
				  << "-o seg|segxml|dis Select output format." << std::endl
				  << "-h                Print help." << std::endl;
	}

	void setError(std::string msg) {
		std::cerr << "Error: " << msg << std::endl;
		ok = false;
	}

	const std::string &getRefFileName() const { // asshole
		return fileRef;
	}

	const std::string &getQueryFileName() const {
		return fileQuery;
	}

	OutputFormat getOutputFormat() const {
		return outputFormat;
	}

	bool isOk() const {
		return ok;
	}
};


vmatch::PrinterPtr createSegmentsPrinter(const ClArgs args) {
	switch(args.getOutputFormat()) {
		case ClArgs::OUT_SEGMENTS_TEXT:
			return new vmatch::TextSegmentsPrinter();

		case ClArgs::OUT_SEGMENTS_XML:
			return new vmatch::XmlSegmentsPrinter();

		case ClArgs::OUT_DISSIMILARITIES_TEXT:
			return new vmatch::TextDissimilaritiesPrinter();

		default:
			CV_Assert(false);
			return NULL;
	}
}

const char *p_s_R_file, *p_s_Q_file;
bool b_use_autocorr;

class CGetFiles {
protected:
	std::vector<std::string> &m_r_files;
	const char *m_p_s_pattern;

public:
	CGetFiles(std::vector<std::string> &r_files, const char *p_s_pattern)
		:m_r_files(r_files), m_p_s_pattern(p_s_pattern)
	{}

	bool operator ()(const TFileInfo &f)
	{
		if(f.b_directory)
			return true;

		if(strstr(f.p_s_FileName(), m_p_s_pattern))
			m_r_files.push_back(f.p_s_FileName());

		return true;
	}
};

int main(int argc, char* argv[])
{
	b_use_autocorr = false; // todo - commandline?

	if(argc > 2 && !strcmp(argv[1], "--auto")) {
		b_use_autocorr = true;
		-- argc;
		++ argv; // hack - one rarely needs the name of the process in argv[0]
	}

	if(argc > 4 && !strcmp(argv[1], "--merge-to")) {
		const char *p_s_merge_dest = argv[2]; // contains "%d"
		std::string s_merge_dest = p_s_merge_dest;
		if(s_merge_dest.find('#') != std::string::npos) // todo - put these things from graphviewer to stdio utils, it is actually reusable
			s_merge_dest[s_merge_dest.find('#')] = '%';
		p_s_merge_dest = s_merge_dest.c_str();

		std::vector<std::vector<std::string> > sequences;
		size_t n_min_lenght = SIZE_MAX, n_max_length = 0;
		for(int a = 3; a < argc; ++ a) {
			const char *p_s_pattern = argv[a]; // pattern to merge

			std::vector<std::string> files;
			CDirTraversal::Traverse2("framedump/", CGetFiles(files, p_s_pattern), false);
			std::sort(files.begin(), files.end());
			sequences.push_back(files);
			n_min_lenght = std::min(n_min_lenght, files.size());
			n_max_length = std::max(n_max_length, files.size());
			printf("pattern \'%s\' matches %d files\n", p_s_pattern, int(files.size()));
		}
		// get file sequences

		std::vector<std::pair<int, int> > seq_sizes(sequences.size(), std::make_pair(0, 0));
		// sizes of images for when the frames run out

		TBmp *p_frame = 0;
		printf("have %d sequences, the shortest is %d frames, the longest is %d frames\n",
			int(sequences.size()), int(n_min_lenght), int(n_max_length));
		for(size_t i = 0; i < n_max_length; ++ i) { // merge all you can
			printf("merging frame %d\r", int(i));
			TBmp *p_bitmaps[64] = {0};
			for(size_t j = 0, m = sequences.size(); j < m; ++ j) {
				if(sequences[j].size() <= i)
					continue; // no more frames in this seq
				const char *p_s_filename = sequences[j][i].c_str();
				p_bitmaps[j] = CPngCodec::p_Load_PNG((std::string("framedump/") + p_s_filename).c_str());
				if(p_bitmaps[j]) {
					seq_sizes[j].first = p_bitmaps[j]->n_width;
					seq_sizes[j].second = p_bitmaps[j]->n_height;
				}
			}
			// load all the bitmaps

			int n_height = 0, n_width = 0;
			for(size_t j = 0, m = sequences.size(); j < m; ++ j) {
				n_width += seq_sizes[j].first;
				n_height = std::max(n_height, seq_sizes[j].second);
			}
			// calculate the size of the dest image (stack horz)

			if(p_frame && (p_frame->n_width != n_width || p_frame->n_height != n_height)) {
				p_frame->Delete();
				p_frame = 0;
			}
			if(!p_frame)
				p_frame = TBmp::p_Alloc(n_width, n_height);
			// get framebuffer

			p_frame->Clear(0xff000000);
			// black bk

			int n_x = 0;
			for(size_t j = 0, m = sequences.size(); j < m; ++ j) {
				if(p_bitmaps[j]) {
					const TBmp *p_src = p_bitmaps[j];
					for(int y = 0, h = p_src->n_height, w = p_src->n_width; y < h; ++ y) {
						for(int x = 0; x < w; ++ x)
							p_frame->p_buffer[n_x + x + y * n_width] = p_src->p_buffer[x + y * w];
					}
				}
				n_x += seq_sizes[j].first;
			}
			// put the frames together

			std::string s_filename;
			stl_ut::Format(s_filename, p_s_merge_dest, int(i));
			CPngCodec::Save_PNG(s_filename.c_str(), *p_frame, true);
		}
		if(p_frame)
			p_frame->Delete();
		printf("\ndone.\n");
		return 0;
	}

	try {
		ClArgs args(argc, argv);
		if(!args.isOk()) {
			return 1;
		}

		p_s_R_file = args.getRefFileName().c_str();
		p_s_Q_file = args.getQueryFileName().c_str();
		// hack

		std::string s_filename = p_s_R_file;
		if(s_filename.find_last_of("\\/") != std::string::npos)
			s_filename.erase(0, s_filename.find_last_of("\\/") + 1);
		s_filename.erase(s_filename.rfind('.'));
		// get file name only

		vmatch::VideoMatcher videoMatcher(vmatch::VideoSequenceDataSource::CONF_COLHIST);
		vmatch::Segments segments = videoMatcher.match(args.getRefFileName(), args.getQueryFileName());

		vmatch::PrinterPtr segmentsPrinter = createSegmentsPrinter(args);
		segmentsPrinter->print(segments);

		// run stuff

		std::vector<double> reltime_ref, reltime_query;
		int n_frames_ref, n_frames_query;
		FILE *p_fr;
		if((p_fr = fopen((s_filename + "sync_info.txt").c_str(), "r"))) {
			int n_step, n_stops_ref, n_stops_query, n_lerp_num;
			fscanf(p_fr, "%d\n""%d, %d\n""%d, %d\n""%d\n",
				&n_step, &n_stops_ref, &n_stops_query, &n_frames_ref, &n_frames_query, &n_lerp_num);
			reltime_ref.resize(n_lerp_num);
			reltime_query.resize(n_lerp_num);
			for(int i = 0; i < n_lerp_num; ++ i)
				fscanf(p_fr, "%lf, %lf\n", &reltime_ref[i], &reltime_query[i]);
			fclose(p_fr);
		} else {
			fprintf(stderr, "error: failed to read synchronization information\n");
			return -1;
		}
		printf("\nloaded %d syncinfo reltimes\n", int(reltime_ref.size()));
		// read synchronization info

		{
			std::string R = p_s_R_file, Q = p_s_Q_file;
			std::vector<std::string> Rf, Qf;
			stl_ut::Split(Rf, R, "*");
			stl_ut::Split(Qf, Q, "*");
			std::string s_run_designator = Rf[2] + "-" + Rf[3] + "_" + Qf[2] + "-" + Qf[3];

			remove((s_filename + "_sync_info_" + s_run_designator +
				((b_use_autocorr)? std::string("_auto") : std::string()) + "_.txt").c_str()); // otherwise rename fails
			remove((s_filename + "_sync_info_" + s_run_designator +
				((b_use_autocorr)? std::string("_auto") : std::string()) + "_.xml").c_str());
			rename((s_filename + "sync_info.txt").c_str(),
				(s_filename + "_sync_info_" + s_run_designator +
				((b_use_autocorr)? std::string("_auto") : std::string()) + "_.txt").c_str());
			rename((s_filename + "sync_info.xml").c_str(),
				(s_filename + "_sync_info_" + s_run_designator +
				((b_use_autocorr)? std::string("_auto") : std::string()) + "_.xml").c_str());
			const char *p_s_path_file = ((b_use_autocorr)? "_path_auto.png" : "_path.png");
			remove((s_filename + "_path_" + s_run_designator +
				((b_use_autocorr)? std::string("_auto") :
				std::string()) + "_.png").c_str()); // otherwise rename fails
			rename((s_filename + p_s_path_file).c_str(),
				(s_filename + "_path_" + s_run_designator +
				((b_use_autocorr)? std::string("_auto") :
				std::string()) + "_.png").c_str());
		}
		// rename the outputs so that they dont get overwritten by the next test

		std::vector<double> frametimes_query_to_ref(n_frames_ref); // contains floating point query frame index
		if(reltime_ref.front() >= 0.0 && reltime_ref.back() <= 1.0) { // should be
			if(reltime_ref.front() > 0) {
				reltime_ref.insert(reltime_ref.begin(), .0);
				reltime_query.insert(reltime_query.begin(), .0);
			}
			if(reltime_ref.back() < 1) {
				reltime_ref.push_back(1.0);
				reltime_query.push_back(1.0);
			}
			for(int i = 0; i < n_frames_ref; ++ i) {
				double t_ref = double(i) / n_frames_ref; // [0, 1)
				_ASSERTE(t_ref >= 0 && t_ref <= 1); // granted
				std::vector<double>::const_iterator p_it = std::lower_bound(reltime_ref.begin(), reltime_ref.end(), t_ref);
				_ASSERTE(p_it >= reltime_ref.begin() && p_it < reltime_ref.end()); // also granted
				_ASSERTE(*p_it >= t_ref);
				// find a position with a greater or equal time

				double t_query;
				if(*p_it == t_ref)
					t_query = reltime_query[p_it - reltime_ref.begin()]; // if they are equal, just sample from the other function
				else if(p_it + 1 == reltime_ref.end())
					t_query = reltime_query[p_it - reltime_ref.begin()]; // can't interpolate anyways
				else {
					_ASSERTE(p_it != reltime_ref.begin());
					double t0 = *(p_it - 1), t1 = *p_it;
					double t = (t_ref - t0) / (t1 - t0);
					_ASSERTE(t >= 0 && t <= 1); // should still be
					// get t in the current linear segment

					double q0 = reltime_query[(p_it - reltime_ref.begin()) - 1],
						q1 = reltime_query[p_it - reltime_ref.begin()];
					t_query = q0 + t * (q1 - q0);
					_ASSERTE(t >= 0 && t <= 1); // *inter*polation
					// interpolate t in the query thing

				}
				t_query *= (n_frames_query - 1);
				// convert to query frame index

				frametimes_query_to_ref[i] = t_query;
			}
		} else {
			fprintf(stderr, "error: reltime ref is not [0, 1], can't lerp\n");
			return -1;
		}
		// get frames

		CCueSheet_FFmpegVideo video(p_s_Q_file); // todo - support ordinary ffmpeg? not today.
		if(!video.b_Opened())
			CV_Error(CV_StsObjectNotFound, "Cannot open query video sequence.");
		video.NoDump();
		TBmp *p_prev_frame = video.t_Get_NextFrame().p_Clone(); // clone
		TBmp *p_blend_frame = p_prev_frame->p_Clone(true);
		TBmp &t_prev = *p_prev_frame;
		const TBmp &t_cur = video.t_Get_NextFrame(); // ref
		int n_prev_frame_time = 0, n_cur_frame_time = 1;
		for(int i = 0; i < n_frames_ref; ++ i) {
			double f_time = frametimes_query_to_ref[i];
			printf("writing synced frame %d (%.3f)\r", int(i), f_time);
			int n_frame = int(floor(f_time));
			float f_frac = float(f_time - n_frame);
			_ASSERTE(f_frac >= 0 && f_frac <= 1); // not equal

			char p_s_filename[256];
			sprintf(p_s_filename, (b_use_autocorr)?
				"framedump/sequence_%s_auto-synced_frame_%05" _PRIsize ".png" :
				"framedump/sequence_%s_synced_frame_%05"
				_PRIsize ".png", video.s_Seq_Name().c_str(), i);

			if(f_frac < 1.0f / 256) { // a single frame
				for(;;) {
					if(n_prev_frame_time == n_frame)
						CPngCodec::Save_PNG(p_s_filename, t_prev);
					else if(n_cur_frame_time == n_frame)
						CPngCodec::Save_PNG(p_s_filename, t_cur);
					else {
						// need to read more frames
						_ASSERTE(n_frame > n_cur_frame_time); // make sure that the wanted frame is ahead and not in the past
						++ n_cur_frame_time;
						++ n_prev_frame_time;
						memcpy(t_prev.p_buffer, t_cur.p_buffer,
							t_cur.n_width * t_cur.n_height * sizeof(uint32_t));
						const TBmp &t_next = video.t_Get_NextFrame();
						if(!t_next.p_buffer) {
							fprintf(stderr, "error: not enough frames for lerp\n");
							return -1;
						}
						_ASSERTE(&t_next == &t_cur); // points to the same object inside the codec, no need to do anything
						continue; // try again
					}
					break;
				}
			} else { // two frames blend
				_ASSERTE(f_frac >= 0); // checked already though
				const int n_alpha = int(f_frac * 255);
				for(;;) {
					if(n_prev_frame_time == n_frame && n_cur_frame_time == n_frame + 1) {
						for(size_t j = 0, m = size_t(t_cur.n_width) * t_cur.n_height; j < m; ++ j) {
							p_blend_frame->p_buffer[j] = TBmp::n_Modulate_RGB(t_prev.p_buffer[j], 255 - n_alpha) +
								TBmp::n_Modulate_RGB(t_cur.p_buffer[j], n_alpha);
						}
						// blend frame

						CPngCodec::Save_PNG(p_s_filename, *p_blend_frame);
					} else {
						// need to read more frames
						_ASSERTE(n_prev_frame_time < n_frame); // make sure that the wanted frame is ahead and not in the past
						++ n_cur_frame_time;
						++ n_prev_frame_time;
						memcpy(t_prev.p_buffer, t_cur.p_buffer,
							t_cur.n_width * t_cur.n_height * sizeof(uint32_t));
						const TBmp &t_next = video.t_Get_NextFrame();
						if(!t_next.p_buffer) {
							fprintf(stderr, "error: not enough frames for lerp\n");
							return -1;
						}
						_ASSERTE(&t_next == &t_cur); // points to the same object inside the codec, no need to do anything
						continue; // try again
					}
					break;
				}
			}
			// produce a single frame of the synchronized video
		}
		p_prev_frame->Delete();
		p_blend_frame->Delete();
		printf("\ndone.\n");
	}
	catch(cv::Exception ex) {
		std::cerr << ex.what() << std::endl;
		return 1;
	}
	catch(...) {
		std::cerr << "Unkown exception arrised!" << std::endl;
		return 1;
	}

	return 0;
}
