/*
								+----------------------------------+
								|                                  |
								| ***  Random generator model  *** |
								|                                  |
								|   Copyright  -tHE SWINe- 2007   |
								|                                  |
								|            RandGen.h             |
								|                                  |
								+----------------------------------+
*/

#pragma once
#ifndef __RANDOM_GENERATORS_INCLUDED
#define __RANDOM_GENERATORS_INCLUDED

/**
 *	@file RandGen.h
 *	@brief generic random generator models
 *	@date 2007
 *	@author -tHE SWINe-
 *
 *	@date 2007-10-17
 *
 *	added CMTGenerator::CMTGenerator(unsigned int n_seed = 0) constructor for greater convenience
 *
 */

#include "Integer.h"
#include "Mersene.h" // Mersene Twister
#include <stdlib.h> // srand(), rand()
#include <math.h> // floor()

/**
 *	@brief generic minimal random number generator class (intended
 *		for experiments with random number generation methods)
 *
 *	@tparam b_use_virtual_interface is common interface flag
 */
template <bool b_use_virtual_interface = true>
class CRandomGeneratorModel {
public:
	/**
	 *	@brief sets a new seed and restarts the generator with this seed
	 *	@param[in] n_seed is random generator seed
	 */
	virtual void Seed(unsigned int n_seed) = 0;

	/**
	 *	@brief generates a 32-bit unsigned random number
	 *	@return Returns the generated random number.
	 */
	virtual uint32_t n_Rand32() = 0;

	/**
	 *	@brief generates a random number on [0, 1] real interval
	 *	@return Returns floating-point random number in range 0 to 1 inclusive.
	 */
	virtual double f_Rand() = 0;

	/**
	 *	@brief generates a random number on [0, 1) real interval
	 *	@return Returns floating-point random number in range 0 to 1 non-inclusive on the right.
	 */
	virtual double f_Rand_LeftInclusive() = 0;

	/**
	 *	@brief generates a random number on (0, 1) real interval
	 *	@return Returns floating-point random number in range 0 to 1 non-inclusive.
	 */
	virtual double f_Rand_NonInclusive() = 0;
};

/**
 *	@brief generic minimal random number generator class (specialization for no common interface)
 */
template <>
class CRandomGeneratorModel<false> {};

/**
 *	@brief random number generator traits
 *	@tparam CGenerator is random number genetator type
 */
template <class CGenerator>
class CRandomGeneratorTraits {
public:
	/**
	 *	@brief random number generator traits, stored as enum
	 */
	enum {
		b_seedable = CGenerator::b_seedable, /**< @brief seed not ignored flag */
		b_64bit_rand = CGenerator::b_64bit_rand, /**< @brief 64-bit random number capability flag */
		b_crypto_safe = CGenerator::b_crypto_safe /**< @brief cryptographically safe flag */
	};
};

/**
 *	@brief Mersene Twister-based random number generator
 *	@tparam b_use_virtual_interface is common interface flag (default true)
 */
template <bool b_use_virtual_interface = true>
class CMTGenerator : public CRandomGeneratorModel<b_use_virtual_interface> {
public:
	/**
	 *	@brief random number generator traits, stored as enum
	 */
	enum {
		b_seedable = true, /**< @brief seed not ignored flag */
		b_64bit_rand = false, /**< @brief 64-bit random number capability flag */
		b_crypto_safe = false /**< @brief cryptographically safe flag */
	};

protected:
	CMerseneTwister m_twister; /**< @brief Mersene Twister instance */

public:
	/**
	 *	@brief default constructor; has no effect
	 */
	inline CMTGenerator()
	{}

	/**
	 *	@brief constructor; initializes the random generator
	 */
	inline CMTGenerator(unsigned int n_seed)
	{
		Seed(n_seed);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::Seed()
	 */
	void Seed(unsigned int n_seed)
	{
		m_twister.init_genrand(n_seed);
	}

	/**
	 *	@brief generates a 32-bit unsigned random number
	 *	@return Returns the generated random number.
	 */
	uint32_t n_Rand32()
	{
		return m_twister.genrand_int32();
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand()
	 */
	double f_Rand()
	{
		return m_twister.genrand_real1();
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_LeftInclusive()
	 */
	double f_Rand_LeftInclusive()
	{
		return m_twister.genrand_real2();
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_NonInclusive()
	 */
	double f_Rand_NonInclusive()
	{
		return m_twister.genrand_real3();
	}
};

/**
 *	@brief random number generator, based on standard library
 *	@tparam b_use_virtual_interface is common interface flag (default true)
 *	@note Only a single instance of this class may exist at a time. Note that this is not enforced by any checks.
 */
template <bool b_use_virtual_interface = true>
class CCLibGenerator : public CRandomGeneratorModel<b_use_virtual_interface> {
public:
	/**
	 *	@brief random number generator traits, stored as enum
	 */
	enum {
		b_seedable = true, /**< @brief seed not ignored flag */
		b_64bit_rand = false, /**< @brief 64-bit random number capability flag */
		b_crypto_safe = false /**< @brief cryptographically safe flag */
	};

public:
	/**
	 *	@brief default constructor; has no effect
	 */
	inline CCLibGenerator()
	{}

	/**
	 *	@brief constructor; initializes the random generator
	 */
	inline CCLibGenerator(unsigned int n_seed)
	{
		Seed(n_seed);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::Seed()
	 */
	void Seed(unsigned int n_seed)
	{
		srand(n_seed);
	}

	/**
	 *	@brief generates a 32-bit unsigned random number
	 *	@return Returns the generated random number.
	 */
	uint32_t n_Rand32()
	{
		if(RAND_MAX == UINT32_MAX || (RAND_MAX > UINT32_MAX &&
		   (!(RAND_MAX + 1) || b_Is_POT_Static(RAND_MAX + 1)))) // either exact match, or larger and a power of two
			return rand();
		else {
			const uint32_t n_bit_num = n_Log2_Static(RAND_MAX); // not exactly nice if it is not a power of two
			const uint32_t n_mask = n_Mask_Static(n_bit_num);
			uint32_t n_result = rand() & n_mask;
			for(int i = n_bit_num; i < 32; i += n_bit_num) {
				n_result <<= n_bit_num;
				n_result |= rand() & n_mask;
			}
			return n_result;
		}
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand()
	 */
	double f_Rand()
	{
		return rand() / double(RAND_MAX);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_LeftInclusive()
	 */
	double f_Rand_LeftInclusive()
	{
		_ASSERTE(RAND_MAX < INT_MAX);
		return rand() / double(uint64_t(RAND_MAX) + 1);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_NonInclusive()
	 */
	double f_Rand_NonInclusive()
	{
		_ASSERTE(RAND_MAX < INT_MAX - 1);
		return (rand() + 1.0) / double(uint64_t(RAND_MAX) + 2);
	}
};

/**
 *	@brief a simple and fast congruential random nunber generator
 *	@tparam b_use_virtual_interface is common interface flag (default true)
 */
template <bool b_use_virtual_interface = true>
class CCongruentialGenerator : public CRandomGeneratorModel<b_use_virtual_interface> {
public:
	/**
	 *	@brief random number generator traits, stored as enum
	 */
	enum {
		b_seedable = true, /**< @brief seed not ignored flag */
		b_64bit_rand = false, /**< @brief 64-bit random number capability flag */
		b_crypto_safe = false /**< @brief cryptographically safe flag */
	};

protected:
	uint32_t m_n_x; /**< @brief random generator state */

public:
	/**
	 *	@brief default constructor; has no effect
	 */
	inline CCongruentialGenerator()
	{}

	/**
	 *	@brief constructor; initializes the random generator
	 */
	inline CCongruentialGenerator(unsigned int n_seed)
	{
		Seed(n_seed);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::Seed()
	 */
	void Seed(unsigned int n_seed)
	{
		m_n_x = n_seed;
		n_Rand32(); // do not return seed
	}

	/**
	 *	@brief generates a 32-bit unsigned random number
	 *	@return Returns the generated random number.
	 */
	uint32_t n_Rand32()
	{
		m_n_x = m_n_x * 1664525U + 1013904223U;
		return m_n_x;
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand()
	 */
	double f_Rand()
	{
		return n_Rand32() / double(UINT32_MAX);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_LeftInclusive()
	 */
	double f_Rand_LeftInclusive()
	{
		return n_Rand32() / (double(UINT32_MAX) + 1);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_NonInclusive()
	 */
	double f_Rand_NonInclusive()
	{
		return (double(n_Rand32()) + 1) / (double(UINT32_MAX) + 2);
	}
};

/**
 *	@brief Numerical Recipes' long generator (period 3.138 * 10^57)
 *	@tparam b_use_virtual_interface is common interface flag (default true)
 */
template <bool b_use_virtual_interface = true>
class CNRLongGenerator : public CRandomGeneratorModel<b_use_virtual_interface> {
public:
	/**
	 *	@brief random number generator traits, stored as enum
	 */
	enum {
		b_seedable = true, /**< @brief seed not ignored flag */
		b_64bit_rand = true, /**< @brief 64-bit random number capability flag */
		b_crypto_safe = false /**< @brief cryptographically safe flag */
	};

protected:
	uint64_t m_n_u; /**< @brief random generator state */
	uint64_t m_n_v; /**< @brief random generator state */
	uint64_t m_n_w; /**< @brief random generator state */

public:
	/**
	 *	@brief constructor; initializes the random generator
	 */
	CNRLongGenerator(uint64_t n_seed = 0)
	{
		Seed64(n_seed);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::Seed()
	 */
	void Seed(unsigned int n_seed)
	{
		Seed64(n_seed);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::Seed()
	 */
	void Seed64(uint64_t n_seed)
	{
		m_n_v = (uint64_t(0x38ecac5f) << 32) |	0xb3251641U; // 4101842887655102017LL
		m_n_w = 1;
		//_ASSERTE(n_seed != m_n_v); // can happen, don't want this to kill my process in the unlikely event
		m_n_u = n_seed ^ m_n_v;
		n_Rand64();
		m_n_v = m_n_u;
		n_Rand64();
		m_n_w = m_n_v;
		n_Rand64();
	}

	/**
	 *	@brief generates a 64-bit unsigned random number
	 *	@return Returns the generated random number.
	 */
	uint64_t n_Rand64()
	{
		m_n_u = m_n_u * ((uint64_t(0x27bb2ee6U) << 32) | 0x87b0b0fdU)/*2862933555777941757LL*/ +
			((uint64_t(0x61c88646U) << 32) | 0x80b583bfU)/*7046029254386353087LL*/;
		m_n_v ^= m_n_v >> 17;
		m_n_v ^= m_n_v << 31;
		m_n_v ^= m_n_v >> 8;
		m_n_w = 4294957665U * (m_n_w & 0xffffffff) + (m_n_w >> 32);
		uint64_t x = m_n_u ^ (m_n_u << 21);
		x ^= x >> 35;
		x ^= x << 4;
		return (x + m_n_v) ^ m_n_w;
	}

	/**
	 *	@brief generates a 32-bit unsigned random number
	 *	@return Returns the generated random number.
	 */
	uint32_t n_Rand32()
	{
		return uint32_t(n_Rand64());
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand()
	 */
	double f_Rand()
	{
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER <= 1200
		return (int64_t)(n_Rand64() & INT64_MAX) / double(INT64_MAX); // msvc 6.0 does not implement conversion from uint64_t to double
#else // _MSC_VER) && !__MWERKS__ && _MSC_VER <= 1200
		return n_Rand64() / double(UINT64_MAX);
#endif // _MSC_VER) && !__MWERKS__ && _MSC_VER <= 1200
		//return double(n_Rand64() * 5.42101086242752217e-20);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_LeftInclusive()
	 */
	double f_Rand_LeftInclusive()
	{
		//return f_Rand() * (1 - 1 / double(UINT64_MAX)); // this is not good, 1 / UINT64_MAX will be below epsilon, will return the same numbers as f_Rand()
		//return double(n_Rand32() / (double(UINT32_MAX) + 1)); // this throws away too much precision
		const uint64_t n_max = (uint64_t(1) << 52) - 1; // use 52 bits
		_ASSERTE(1.0 + 1.0 / double(n_max + 1) != 1.0);
		return (n_Rand64() & n_max) / double(n_max + 1);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_NonInclusive()
	 */
	double f_Rand_NonInclusive()
	{
		//return f_Rand() * (1 - 2 / double(UINT64_MAX)) + 1 / double(UINT64_MAX); // this is not good, 1 / UINT64_MAX will be below epsilon, will return the same numbers as f_Rand()
		//return double((double(n_Rand32()) + 1) / (double(UINT32_MAX) + 2)); // this throws away too much precision
		const uint64_t n_max = (uint64_t(1) << 52) - 1; // use 52 bits
		_ASSERTE(1.0 + 1.0 / double(n_max + 1) != 1.0);
		return ((n_Rand64() & n_max) + 1) / double(n_max + 2);
	}
};

/**
 *	@brief Numerical Recipes' fast generator Ranq1 (period 1.8 * 10^19)
 *	@tparam b_use_virtual_interface is common interface flag (default true)
 */
template <bool b_use_virtual_interface = true>
class CNRFastGenerator : public CRandomGeneratorModel<b_use_virtual_interface> {
public:
	/**
	 *	@brief random number generator traits, stored as enum
	 */
	enum {
		b_seedable = true, /**< @brief seed not ignored flag */
		b_64bit_rand = true, /**< @brief 64-bit random number capability flag */
		b_crypto_safe = false /**< @brief cryptographically safe flag */
	};

protected:
	uint64_t m_n_v; /**< @brief random generator state */

public:
	/**
	 *	@brief constructor; initializes the random generator
	 */
	CNRFastGenerator(uint64_t n_seed = 0)
	{
		Seed64(n_seed);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::Seed()
	 */
	void Seed(unsigned int n_seed)
	{
		Seed64(n_seed);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::Seed()
	 */
	void Seed64(uint64_t n_seed)
	{
		m_n_v = (uint64_t(0x38ecac5f) << 32) |	0xb3251641U; // 4101842887655102017LL
		//_ASSERTE(n_seed != m_n_v); // can happen, don't want this to kill my process in the unlikely event
		m_n_v ^= n_seed;
		m_n_v = n_Rand64();
	}

	/**
	 *	@brief generates a 64-bit unsigned random number
	 *	@return Returns the generated random number.
	 */
	uint64_t n_Rand64()
	{
		m_n_v ^= m_n_v >> 21;
		m_n_v ^= m_n_v << 35;
		m_n_v ^= m_n_v >> 4;
		return m_n_v * ((uint64_t(0x2545f491) << 32) | 0x4f6cdd1dU); // 2685821657736338717LL;
	}

	/**
	 *	@brief generates a 32-bit unsigned random number
	 *	@return Returns the generated random number.
	 */
	uint32_t n_Rand32()
	{
		return uint32_t(n_Rand64());
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand()
	 */
	double f_Rand()
	{
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER <= 1200
		return (int64_t)(n_Rand64() & INT64_MAX) / double(INT64_MAX); // msvc 6.0 does not implement conversion from uint64_t to double
#else // _MSC_VER) && !__MWERKS__ && _MSC_VER <= 1200
		return n_Rand64() / double(UINT64_MAX);
#endif // _MSC_VER) && !__MWERKS__ && _MSC_VER <= 1200
		//return double(n_Rand64() * 5.42101086242752217e-20);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_LeftInclusive()
	 */
	double f_Rand_LeftInclusive()
	{
		//return f_Rand() * (1 - 1 / double(UINT64_MAX)); // this is not good, 1 / UINT64_MAX will be below epsilon, will return the same numbers as f_Rand()
		//return double(n_Rand32() / (double(UINT32_MAX) + 1)); // this throws away too much precision
		const uint64_t n_max = (uint64_t(1) << 52) - 1; // use 52 bits
		_ASSERTE(1.0 + 1.0 / double(n_max + 1) != 1.0);
		return (n_Rand64() & n_max) / double(n_max + 1);
	}

	/**
	 *	@copydoc CRandomGeneratorModel::f_Rand_NonInclusive()
	 */
	double f_Rand_NonInclusive()
	{
		//return f_Rand() * (1 - 2 / double(UINT64_MAX)) + 1 / double(UINT64_MAX); // this is not good, 1 / UINT64_MAX will be below epsilon, will return the same numbers as f_Rand()
		//return double((double(n_Rand32()) + 1) / (double(UINT32_MAX) + 2)); // this throws away too much precision
		const uint64_t n_max = (uint64_t(1) << 52) - 1; // use 52 bits
		_ASSERTE(1.0 + 1.0 / double(n_max + 1) != 1.0);
		return ((n_Rand64() & n_max) + 1) / double(n_max + 2);
	}
};

/**
 *	@brief utility to generate 64-bit random numbers using any generator
 *	@tparam CRandomNumberGenerator is random number generator model
 */
template <class CRandomNumberGenerator>
class CRand64Caller {
public:
	/**
	 *	@brief intermediates stored as enum
	 */
	enum {
		b_has_64 = CRandomGeneratorTraits<CRandomNumberGenerator>::b_64bit_rand /**< @brief 64-bit rand support */
	};

protected:
	template <const bool b_has64 MSVC_OMMIT_ARG(class GppDummy)>
	class CCall {
	public:
		static uint64_t n_Rand64(CRandomNumberGenerator &r_gen)
		{
			return r_gen.n_Rand64();
		}
	};

	template <MSVC_OMMIT(class GppDummy)>
	class CCall<false MSVC_OMMIT_ARG(GppDummy)> {
	public:
		static uint64_t n_Rand64(CRandomNumberGenerator &r_gen)
		{
			return (uint64_t(r_gen.n_Rand32()) << 32) | r_gen.n_Rand32();
		}
	};

public:
	/**
	 *	@brief generates a 64-bit unsigned random number
	 *	@param[in,out] r_gen is reference to the generator state
	 *	@return Returns the generated random number.
	 */
	static uint64_t n_Rand64(CRandomNumberGenerator &r_gen)
	{
		return CCall<b_has_64 MSVC_OMMIT_ARG(void)>::n_Rand64(r_gen);
	}
};

/**
 *	@brief uniform integer distribution
 *	@tparam CIntType is integer data type
 */
template <class CIntType>
class CUniformIntegerDistribution {
protected:
	CIntType m_n_low; /**< @brief low bound */
	CIntType m_n_high; /**< @brief high bound */

public:
	/**
	 *	@brief default constructor
	 *
	 *	@param[in] n_low is low bound (inclusive)
	 *	@param[in] n_high is high bound (inclusive)
	 */
	CUniformIntegerDistribution(CIntType n_low, CIntType n_high)
		:m_n_low(n_low), m_n_high(n_high)
	{
		_ASSERTE(m_n_low <= m_n_high); // would break overflow checks
	}

	/**
	 *	@brief gets a random number in the given distribution, using the specified generator
	 *	@tparam CRandomNumberGenerator is random number generator model
	 *	@param[in,out] r_gen is random number generator instance
	 *	@return Returns a random number in the given distribution.
	 */
	template <class CRandomNumberGenerator>
	CIntType operator ()(CRandomNumberGenerator &r_gen)
	{
#if 1
		_ASSERTE(m_n_high <= UINT64_MAX);
		uint64_t n_range = m_n_high;
		_ASSERTE((m_n_low < 0 && m_n_high <= UINT64_MAX + m_n_low) ||
			(m_n_low >= 0 && m_n_high - m_n_low <= UINT64_MAX));
		n_range -= m_n_low;
		_ASSERTE(n_range <= UINT64_MAX); // but is still exclusive
		// calculate the required (exclusive) range of values (note that we cant use
		// CIntType here, as the range might not fit in it if it is signed)
		// also note that inclusive range might not be representable

		if(n_range <= UINT32_MAX) {
			uint32_t n_range32 = uint32_t(n_range);
			uint32_t n_highest_modulus32 = (n_range32 < UINT32_MAX)?
				UINT32_MAX - ((UINT32_MAX % (n_range32 + 1)) + 1) % (n_range32 + 1) : UINT32_MAX; // so that it is inclusive
			for(;;) {
				uint32_t n_rand = r_gen.n_Rand32();
				if(n_rand <= n_highest_modulus32)
					return m_n_low + CIntType(n_rand % n_range32);
			}
		} else {
			uint64_t n_highest_modulus = (n_range < UINT64_MAX)?
				UINT64_MAX - ((UINT64_MAX % (n_range + 1)) + 1) % (n_range + 1) : UINT64_MAX; // so that it is inclusive
			for(;;) {
				uint64_t n_rand = CRand64Caller<CRandomNumberGenerator>::n_Rand64(r_gen);
				if(n_rand <= n_highest_modulus)
					return m_n_low + CIntType(n_rand % n_range);
			}
		}
		// generate a number and see if it falls under the largest modulus (so that
		// unbiased distribution is generated), otherwise generate one more random number
#else // 1
		// note that this is flawed; should use random integers, either uint32_t or uint64_t
		// and use modulo with repeated generation

		double f_random = r_gen.f_Rand_LeftInclusive();
		// get a random number in [0, 1)

		/*bool a = m_n_low < INT64_MIN || m_n_low > INT64_MAX,
			b = m_n_high < INT64_MIN || m_n_high > INT64_MAX,
			c = m_n_low < INT64_MIN + 1,
			d = (int64_t(m_n_low) - 1) < 0 && m_n_high > INT64_MAX + (int64_t(m_n_low) - 1),
			e = (int64_t(m_n_low) - 1) > 0 && m_n_high < INT64_MIN + (int64_t(m_n_low) - 1),
			f = floor(m_n_high - (double(m_n_low) - 1)) <= m_n_high - int64_t(m_n_low) + 1;*/ // debug
		_ASSERTE(m_n_low < INT64_MIN || m_n_low > INT64_MAX || // m_n_low doesn't fit signed 64-bit integer
			m_n_high < INT64_MIN || m_n_high > INT64_MAX || // m_n_high doesn't fit signed 64-bit integer
			m_n_low < INT64_MIN + 1 || // m_n_low - 1 underflows
			((int64_t(m_n_low) - 1) < 0 && m_n_high > INT64_MAX + (int64_t(m_n_low) - 1) || // m_n_high - (m_n_low - 1) overflows
			(int64_t(m_n_low) - 1) > 0 && m_n_high < INT64_MIN + (int64_t(m_n_low) - 1)) || // m_n_high - (m_n_low - 1) underflows
			floor(m_n_high - (double(m_n_low) - 1)) <= m_n_high - int64_t(m_n_low) + 1); // the scaler is greater than the integer version
		// make sure that the float value is smaller than the integer value, if the integer value can be calculated

		f_random *= floor(m_n_high - (double(m_n_low) - 1)); // watch out for overflow (both integer and float)
		// scale the random number to the number of integers in the specified interval

		CIntType n_result = CIntType(f_random) + m_n_low;
		_ASSERTE(n_result >= m_n_low && n_result <= m_n_high); // make sure it is in the bounds
		// round down, add low

		return n_result;
#endif // 1
	}
};

/**
 *	@brief a simple generator of random permutation of unique bounded values
 */
class CUniqueRandomPermutation {
public:
	/**
	 *	@brief custom comparison operator for std::lower_bound / std::upper_bound
	 *
	 *	Compares a reference value to the value in the vector minus its index. Use as follows:
	 *
	 *	@code
	 *	std::vector<int> sequence = ...; // a strictly increasing sequence
	 *	int n = ...; // needle
	 *
	 *	typedef CCompareWithOffset<std::vector<int> > Comparator;
	 *
	 *	std::vector<int>::iterator iter = std::upper_bound(sequence.begin(), sequence.end(),
	 *		Comparator::TNeedle(n), Comparator(sequence.begin()));
	 *	// finds the first i, such that sequence[i] - i > n
	 *	@endcode
	 *
	 *	@tparam CScalar is scalar type
	 */
	template <class CContainer>
	class CCompareWithOffset {
	public:
		typedef typename CContainer::value_type CScalar;
		typedef typename CContainer::const_iterator CConstIter;

		/**
		 *	@brief a dummy type that unambiguously marks the needle
		 */
		struct TNeedle {
			CScalar n_value; /**< @brief value of the needle */

			/**
			 *	@brief default constructor
			 *	@param[in] r_n is value of the needle
			 */
			inline TNeedle(const CScalar &r_n)
				:n_value(r_n)
			{}
		};

	protected:
		CConstIter m_p_begin_it; /**< @brief iterator pointing to the first element of the vector */

	public:
		/**
		 *	@brief default constructor
		 *	@param[in] p_begin_it is iterator pointing to the first element of the vector
		 */
		inline CCompareWithOffset(CConstIter p_begin_it)
			:m_p_begin_it(p_begin_it)
		{}

		/**
		 *	@brief less-than comparison operator; compares a value from the vector to a reference
		 *
		 *	@param[in] r_value is const reference to a values in the vector
		 *	@param[in] n_reference is reference value
		 *
		 *	@return Returns true if the left value minus its index is lower than
		 *		the reference value, otherwise returns false.
		 */
		inline bool operator ()(const CScalar &r_value, TNeedle n_reference) const
		{
			size_t n_index = &r_value - &*m_p_begin_it;
			// calculate index in the vector

			return r_value < CScalar(n_reference.n_value + n_index);
		}

		/**
		 *	@brief less-than comparison operator; compares a reference to a value from the vector
		 *
		 *	@param[in] n_reference is reference value
		 *	@param[in] r_value is const reference to a values in the vector
		 *
		 *	@return Returns true if the reference value is lower than
		 *		the right value minus its index, otherwise returns false.
		 */
		inline bool operator ()(TNeedle n_reference, const CScalar &r_value) const
		{
			size_t n_index = &r_value - &*m_p_begin_it;
			// calculate index in the vector

			return CScalar(n_reference.n_value + n_index) < r_value;
		}

		/**
		 *	@brief less-than comparison operator; compares two values from the vector
		 *
		 *	@param[in] r_value_l is const reference to a values in the vector
		 *	@param[in] r_value_r is const reference to a values in the vector
		 *
		 *	@return Returns true if the left value minus its index is lower than
		 *		the right value minus its index, otherwise returns false.
		 *
		 *	@note This is required by Visual Studio for debug checking that
		 *		the sequence in the vector is indeed sorted.
		 */
		inline bool operator ()(const CScalar &r_value_l, const CScalar &r_value_r) const
		{
			size_t n_index_l = &r_value_l - &*m_p_begin_it;
			size_t n_index_r = &r_value_r - &*m_p_begin_it;
			// calculate indices of both elements in the vector

			return CScalar(r_value_l + n_index_r) < CScalar(r_value_r + n_index_l);
			// or equivalently r_value_l - n_index_l < r_value_r - n_index_r
		}
	};

public:
	template <class CContainer, class CRandomNumberGenerator>
	static void Generate(CContainer &r_permutation, size_t n_permutation_size,
		size_t n_set_size, CRandomNumberGenerator &random)
	{
		typedef typename CContainer::value_type CScalar;
		typedef typename CContainer::iterator CIterator;

		r_permutation.clear(); // !!
		if(n_permutation_size < 75) {
			// b1: 3187.000 msec (the fastest)
			// b2: 3734.000 msec
			for(size_t i = 0; i < n_permutation_size; ++ i) {
				CUniformIntegerDistribution<CScalar> distribution(0, n_set_size - i - 1);
				CScalar n = distribution(random);
				// get a random number

				CIterator p_where_it = r_permutation.begin();
				for(size_t j = 0; j < i; ++ j, ++ p_where_it, ++ n) {
					if(n < *p_where_it)
						break;
				}
				// see where it should be inserted

				r_permutation.insert(p_where_it, 1, n);
				// insert it, maintain a sorted sequence
			}
		} else {
			// b1: 3578.000 msec
			// b2: 1703.000 msec (the fastest)
			for(size_t i = 0; i < n_permutation_size; ++ i) {
				CUniformIntegerDistribution<CScalar> distribution(0, n_set_size - i - 1);
				CScalar n = distribution(random);
				// get a random number

				typedef CCompareWithOffset<CContainer> TCompare;
				CIterator p_begin_it = r_permutation.begin();
				CIterator p_where_it = std::upper_bound(p_begin_it, r_permutation.end(),
					typename TCompare::TNeedle(n), TCompare(p_begin_it));
				// see where it should be inserted

				r_permutation.insert(p_where_it, 1, n + (p_where_it - p_begin_it)); // needs to be in parentheses for VS 2008, otherwise the safe iterators go crazy
				// insert it in the list, maintain a sorted sequence
			}

			// this does binary search with the comparison "n + j < rand_num[j]" with non-constant needle,
			// which can be rewritten to one with constant needle and a modified sequence "n < rand_num[j] - j".
			// it is easily shown that since the lowest distance between two elements of the original
			// sequence "rand_num[j]" is:
			//
			//		rand_num[j] - rand_num[j - 1] >= 1
			//
			// (the generated numbers are unique), and at the same time, the difference
			// between different elements of "-j" is:
			//
			//		-j - -(j - 1) = -1
			//
			// The difference between elements of their sum "rand_num[j] - j" is:
			//
			//		(rand_num[j] - j) - (rand_num[j - 1] - (j - 1)) >= 0
			//
			// therefore, the sequence is nondecreasing and this comparison can be used.
		}
	}
};

#endif // !__RANDOM_GENERATORS_INCLUDED
