#include "SegmentsExtractor.h"


#include "UberLame_src/iface/PngLoad.h"
#include "UberLame_src/StlUtils.h"
#include "UberLame_src/Timer.h"

extern const char *p_s_R_file, *p_s_Q_file;
extern bool b_use_autocorr;

class CScale {
protected:
	double m_f_scale;

public:
	CScale(double f_scale)
		:m_f_scale(f_scale)
	{}

	double operator ()(double f_x) const
	{
		return f_x * m_f_scale;
	}
};

namespace vmatch {


void SegmentsExtractor::convertSegments(ExtractedSegments extractedSegments, VideoSequenceDataPtr dataRef, VideoSequenceDataPtr dataQuery, Segments & segments) {
	CV_Assert(!dataRef->getKeyFrames().empty());
	CV_Assert(!dataQuery->getKeyFrames().empty());

	KeyFrames keyframesRef = dataRef->getKeyFrames();
	KeyFrames keyframesQuery = dataQuery->getKeyFrames();
	
	bool useKeySegments = keyframesRef[0]->getFrameDescriptor()->getType() == FrameDescriptor::_GROUP && keyframesQuery[0]->getFrameDescriptor()->getType() == FrameDescriptor::_GROUP;

	for(size_t i = 0; i < extractedSegments.size(); i++) {			
		int refStart = 0;
		int refEnd = 0;
		int queryStart = 0;
		int queryEnd = 0;
		
		int refStartIdx = extractedSegments[i][0];
		int refEndIdx = extractedSegments[i][1];
		int queryStartIdx = extractedSegments[i][2];
		int queryEndIdx = extractedSegments[i][3];

		KeyFrame *kfRefStart = keyframesRef[refStartIdx];
		KeyFrame *kfRefEnd = keyframesRef[refEndIdx];
		KeyFrame *kfQueryStart = keyframesQuery[queryStartIdx];
		KeyFrame *kfQueryEnd = keyframesQuery[queryEndIdx];
		
		if(useKeySegments) {
			refStart = ((KeySegment *)kfRefStart)->getStart();
			refEnd = ((KeySegment *)kfRefEnd)->getEnd();
			queryStart = ((KeySegment *)kfQueryStart)->getStart();
			queryEnd = ((KeySegment *)kfQueryEnd)->getEnd();
		}
		else {
			refStart = kfRefStart->getFrameNumber();
			refEnd = kfRefEnd->getFrameNumber();
			queryStart = kfQueryStart->getFrameNumber();
			queryEnd = kfQueryEnd->getFrameNumber();
		}

		if(refStart < refEnd && queryStart < queryEnd) {
			SegmentPtr segment = new Segment(refStart, refEnd, queryStart, queryEnd);
			segments.push_back(segment);
		}
	}
}

Segments SegmentsExtractor::extract(const SimilarityMatrix & similarity, VideoSequenceDataPtr dataRef, VideoSequenceDataPtr dataQuery) {
	// TODO jen na ladeni
	tmpDataRef = dataRef;
	tmpDataQuery = dataQuery;
	// TODO jen na ladeni
	
	CV_Assert(!similarity.empty());

	ExtractedSegments extractedSegments;
	extractFunc(similarity, dataRef, dataQuery, extractedSegments);

	Segments segments(dataRef->getMetaData(), dataQuery->getMetaData());
	convertSegments(extractedSegments, dataRef, dataQuery, segments);

	
	//// TODO jen na ladeni
	//cv::Mat img;
	//similarity.render(img);
	//cv::cvtColor(img, img, CV_GRAY2BGR);

	//cv::imshow("segments", img);
	//cv::waitKey(0);

	//for(size_t i = 0; i < extractedSegments.size(); i++) {
	//	cv::line(img, cv::Point(extractedSegments[i][2], extractedSegments[i][0]), cv::Point(extractedSegments[i][3], extractedSegments[i][1]), cv::Scalar(0, 0, 255), 2);
	//	cv::imshow("segments", img);
	//	cv::waitKey(0);
	//}

	//std::cout << "segm_count=" << extractedSegments.size() << std::endl;
	//// TODO jen na ladeni
	

	return segments;
}


HoughSegmentsExtractor::HoughSegmentsExtractor(double threshold) : SegmentsExtractor() {
	setThreshold(threshold);
}


void HoughSegmentsExtractor::setThreshold(double threshold) {
	if(threshold <= 0) {
		CV_Error(CV_StsBadArg, "Invalid frame distance threshold value.");
	}

	this->threshold = threshold;
}


void HoughSegmentsExtractor::preprocessImage(cv::Mat & img) const {
	cv::dilate(img, img, cv::Mat());
	cv::medianBlur(img, img, 3);
	cv::erode(img, img, cv::Mat());
}


void HoughSegmentsExtractor::detectLines(cv::Mat & img, ExtractedSegments & lines) const {
	lines.clear();
	preprocessImage(img);
	cv::HoughLinesP(img, lines, 2.0, CV_PI/1440, 30, 10, 5);
}


bool HoughSegmentsExtractor::isNear(int xA, int yA, int xB, int yB, int threshold) const {
	float dist = sqrt((float)(xA - xB)*(xA - xB) + (float)(yA - yB)*(yA - yB));
	return dist < threshold;
}


bool HoughSegmentsExtractor::isInside(const cv::Vec4i & A, const cv::Vec4i & B, int threshold) const {
	// test zda jedna usecka lezi uvnitr druhe (soradnicove)
	bool test1 = (A[0] <= B[0]) && (A[1] <= B[1]);
	bool test2 = (A[2] >= B[2]) && (A[3] >= B[3]);

	// spocita smernici a parametr "q" pro obe usecky
	float x1 = (float)A[0];
	float x2 = (float)A[2];
	float y1 = (float)A[1];
	float y2 = (float)A[3];

	float kA = (y2 - y1)/(x2 - x1);
	float qA = (x2*y1 - x1*y2)/(x2 - x1);

	x1 = (float)B[0];
	x2 = (float)B[2];
	y1 = (float)B[1];
	y2 = (float)B[3];

	float kB = (y2 - y1)/(x2 - x1);
	float qB = (x2*y1 - x1*y2)/(x2 - x1);

	//test na prusecik
	float xX = (qB - qA)/(kA- kB);
	float yX = (kA * xX) + qA;
	bool test4 = (xX >= cv::min(A[0], A[2])) && (xX <= cv::max(A[0], A[2])) && (xX >= cv::min(B[0], B[2])) && (xX <= cv::max(B[0], B[2]));
	bool test5 = (yX >= cv::min(A[1], A[3])) && (yX <= cv::max(A[1], A[3])) && (yX >= cv::min(B[1], B[3])) && (yX <= cv::max(B[1], B[3]));

	// test zda usecky lezi priblizne na sobe
	bool test3 = (abs(kA - kB) < 0.4) && (abs(qA - qB) < threshold);
	
	return (test1 && test2 && (test3 || (test4 && test5)));
}


bool HoughSegmentsExtractor::isExtension(const cv::Vec4i & A, const cv::Vec4i & B, int threshold) const {
	// test jestli usecky navazuji
	bool test1 = (A[0] <= B[0]) && (A[1] <= B[1]);
	bool test2 = (A[2] >= B[0]) && (A[3] >= B[1]);
	bool test3 = (A[2] <= B[2]) && (A[3] <= B[3]);

	// spocita smernici a parametr "q" pro obe usecky
	float x1 = (float)A[0];
	float x2 = (float)A[2];
	float y1 = (float)A[1];
	float y2 = (float)A[3];

	float kA = (y2 - y1)/(x2 - x1);
	float qA = (x2*y1 - x1*y2)/(x2 - x1);

	x1 = (float)B[0];
	x2 = (float)B[2];
	y1 = (float)B[1];
	y2 = (float)B[3];

	float kB = (y2 - y1)/(x2 - x1);
	float qB = (x2*y1 - x1*y2)/(x2 - x1);

	// test zda usecky lezi priblizne na sobe a maji stejnou smernici
	bool test4 = (abs(kA - kB) < 0.3) && (abs(qA - qB) < threshold);

	return (test1 && test2 && test3 && test4);
}


void HoughSegmentsExtractor::removeOrthogonalLines(ExtractedSegments & lines) const {
	ExtractedSegments result;

	for(size_t i = 0; i < lines.size(); i++) {
		float dx = (float)lines[i][0] - (float)lines[i][2];
		float dy = (float)lines[i][1] - (float)lines[i][3];
		
		if(abs(atan2(dy,dx)) < 0.2) continue;
		if(abs(atan2(dy,dx)) > 2.8 && abs(atan2(dy,dx)) < 3.5) continue;
		if(abs(atan2(dy,dx)) > 1.2 && abs(atan2(dy,dx)) < 1.9) continue;
		if(abs(atan2(dy,dx)) > 4.4 && abs(atan2(dy,dx)) < 5) continue;
	
		result.push_back(lines[i]);
	}

	lines = result;
}


void HoughSegmentsExtractor::removeSimilarLines(ExtractedSegments & lines) const {
	ExtractedSegments result;
	
	for(size_t l = 0; l < lines.size(); l++) {
		float xA = (float)lines[l][0];
		float yA = (float)lines[l][1];
		float xB = (float)lines[l][2];
		float yB = (float)lines[l][3];

		for(size_t a = 0; a < lines.size(); a++) {
			if(a == l) continue;

			if(isNear(lines[l][0], lines[l][1], lines[a][0], lines[a][1], 30)) {
				if(xA > lines[a][0]) xA = (float)lines[a][0];				
				if(yA > lines[a][1]) yA = (float)lines[a][1];
				
			}
			if(isNear(lines[l][2], lines[l][3], lines[a][2], lines[a][3], 30)) {
				if(xB < lines[a][2]) xB = (float)lines[a][2];				
				if(yB < lines[a][3]) yB = (float)lines[a][3];				
			}
			
		}
		
		result.push_back(cv::Vec4i((int)xA, (int)yA, (int)xB, (int)yB));
	}

	lines = result;
}


void HoughSegmentsExtractor::removeOverlappingLines(ExtractedSegments & lines) const {
	for(size_t l = 0; l < lines.size(); l++) {
		for(size_t a = 0; a < lines.size(); a++) {
			if(a == l) continue;

			if(isInside(lines[l], lines[a], 50)) {
				lines.erase(lines.begin()+a);
				a = a-1;
				if(l > 0) l = l-1;
			}
		}
	}
}


void HoughSegmentsExtractor::mergeLines(ExtractedSegments & lines) const {
	for(size_t l = 0; l < lines.size(); l++) {
		for(size_t a = 0; a < lines.size(); a++) {
			if(a == l) continue;

			if(isExtension(lines[l], lines[a], 30)){
				lines[l][2] = lines[a][2];
				lines[l][3] = lines[a][3];
				
				lines.erase(lines.begin()+a);
				a = a-1;
				if(l > 0) l = l-1;
			}
		}
	}
}


void HoughSegmentsExtractor::processLines(ExtractedSegments & lines) const {
	removeOrthogonalLines(lines);
	removeSimilarLines(lines);
	removeOverlappingLines(lines);
	mergeLines(lines);
}


void HoughSegmentsExtractor::extractFunc(const SimilarityMatrix & similarity, VideoSequenceDataPtr dataRef, VideoSequenceDataPtr dataQuery, ExtractedSegments & segments) {
	cv::Mat similarityRender;
	similarity.threshold(similarityRender, threshold, true);
	
	detectLines(similarityRender, segments);
	processLines(segments);
}


SubdivisionSegmentsExtractor::SubdivisionSegmentsExtractor(double threshold) : SegmentsExtractor() {
	setThreshold(threshold);
}


void SubdivisionSegmentsExtractor::setThreshold(double threshold) {
	if(threshold <= 0) {
		CV_Error(CV_StsBadArg, "Invalid frame distance threshold value.");
	}

	this->threshold = threshold;
}

void SubdivisionSegmentsExtractor::extractFunc(const SimilarityMatrix & similarity, VideoSequenceDataPtr dataRef, VideoSequenceDataPtr dataQuery, ExtractedSegments & segments) {
	SimilarityMatrix sm2 = similarity.normalize();

	cv::Mat similarityProcessed;
	sm2.threshold(similarityProcessed, threshold, true);
    //cv::imshow("sim-thresholded", similarityProcessed);
    //cv::waitKey(0);
	//preprocessImage(similarityProcessed);

	cv::Mat length, endX, endY;
	getSegmentLengthMatrix(similarityProcessed, length, endX, endY);
	
	recursion(segments, length, endX, endY, cv::Rect(0, 0, similarity.cols-1, similarity.rows-1), 10, 5);
}


void SubdivisionSegmentsExtractor::recursion(ExtractedSegments & segments, const cv::Mat & length, const cv::Mat & endX, const cv::Mat & endY, const cv::Rect & roi, float minLength, int depth) {
	if((roi.width*roi.width + roi.height*roi.height) < minLength*minLength) {
		return;
	}
	
	const cv::Mat lengthRoi(length, roi);
	const cv::Mat endXRoi(endX, roi);
	const cv::Mat endYRoi(endY, roi);

	cv::Point start, end;
	bool found = findDominantSegment(lengthRoi, endXRoi, endYRoi, start, end, minLength);

	if(found) {
		//correct roi shift
		start.x += roi.x;
		start.y += roi.y;

		//crop the line with roi
		start.x = MAX(start.x, roi.x);
		start.y = MAX(start.y, roi.y);
		end.x = MIN(end.x, roi.x+roi.width);
		end.y = MIN(end.y, roi.y+roi.height);

		segments.push_back(cv::Vec4i(start.y, end.y, start.x, end.x));

		cv::Point roi1A(roi.x, roi.y);
		cv::Point roi1B(start.x, start.y);

		cv::Point roi2A(end.x, end.y);
		cv::Point roi2B(roi.x+roi.width-1, roi.y+roi.height-1);

		cv::Rect roi1(roi1A, roi1B);
		cv::Rect roi2(roi2A, roi2B);

		if(roi1A.x <= roi1B.x && roi1A.y <= roi1B.y) {
			recursion(segments, length, endX, endY, roi1, minLength, depth+1);
		}

		if(roi2A.x <= roi2B.x && roi2A.y <= roi2B.y) {
			recursion(segments, length, endX, endY, roi2, minLength, depth+1);
		}
	}
}


bool SubdivisionSegmentsExtractor::findDominantSegment(const cv::Mat & length, const cv::Mat & endX, const cv::Mat & endY, cv::Point & start, cv::Point & end, float minLength) const {
	float maxLength = 0;

	for(int r = 0; r < length.rows; r++) {
		float* lengthPtr = (float *)length.ptr(r);
		int *endXPtr = (int *)endX.ptr(r);
		int *endYPtr = (int *)endY.ptr(r);

		for(int c = 0; c < length.cols; c++) {
			if(*lengthPtr > maxLength) {
				maxLength = *lengthPtr;
				start = cv::Point(c, r);
				end = cv::Point(*endXPtr, *endYPtr);
			}
			
			lengthPtr++;
			endXPtr++;
			endYPtr++;
		}
	}
	
	return maxLength > minLength;
}


void SubdivisionSegmentsExtractor::getSegmentLengthMatrix(const cv::Mat & similarityProcessed, cv::Mat & length, cv::Mat & endX, cv::Mat & endY) const {
	length = cv::Mat::zeros(similarityProcessed.rows, similarityProcessed.cols, CV_32F);
	endX = cv::Mat::zeros(similarityProcessed.rows, similarityProcessed.cols, CV_32S);
	endY = cv::Mat::zeros(similarityProcessed.rows, similarityProcessed.cols, CV_32S);

	for(int r = 0; r < similarityProcessed.rows; r++) {
		uchar* similarityProcessedPtr = (uchar*)similarityProcessed.ptr(r);
		float *lengthPtr = (float *)length.ptr(r);
		int *endXPtr = (int *)endX.ptr(r);
		int *endYPtr = (int *)endY.ptr(r);

		for(int c = 0; c < similarityProcessed.cols; c++) {
			 cv::Point start(c,r);
			 cv::Point end = extendSegment(similarityProcessed, start, 0);
			 float distance = sqrt((float)((start.x-end.x)*(start.x-end.x) + (start.y-end.y)*(start.y-end.y)));

			 *lengthPtr = checkAngle(start, end)? distance : 0;
			 *endXPtr = end.x;
			 *endYPtr = end.y;

			 similarityProcessedPtr++;
			 lengthPtr++;
			 endXPtr++;
			 endYPtr++;
		}
	}
}


void SubdivisionSegmentsExtractor::preprocessImage(cv::Mat & img) const {
	float kernelData[] = {0.2f, 0,    0,    0,    0,
						  0,    0.2f, 0,    0,    0,
						  0,    0,    0.2f, 0,    0,
						  0,    0,    0,    0.2f, 0,
						  0,    0,    0,    0,    0.2f
	};
 
	cv::Mat kernel(5, 5, CV_32FC1, kernelData);

	cv::filter2D(img, img, 8, kernel, cv::Point(-1, -1));
	cv::threshold(img, img, 150, 255, CV_8U);
}


bool SubdivisionSegmentsExtractor::checkAngle(const cv::Point & start, const cv::Point & end) const {
	return true; // TODO
}


cv::Point SubdivisionSegmentsExtractor::extendSegment(const cv::Mat & similarityProcessed, const cv::Point & start, int maxGap) const {
	int x = start.x;
	int y = start.y;

	for(;;) {
		if(x >= similarityProcessed.cols-1 || y >= similarityProcessed.rows-1) {
			break;
		}

		// try diagonal step
		if(similarityProcessed.ptr(y+1)[x+1] == 255) {
			x++;
			y++;
		}
		// try horizontal step
		else if(similarityProcessed.ptr(y)[x+1] == 255) {
			x++;
		}
		// try vertical step
		else if(similarityProcessed.ptr(y)[x+1] == 255) {
			y++;
		}
		else {
			// go back horizontally and try to extend line somewhere else
			int tmpX = x;
			while(tmpX >= 0 && similarityProcessed.ptr(y)[tmpX] == 255 && similarityProcessed.ptr(y+1)[tmpX] != 255) {
				tmpX--;
			}

			// ok, proper extension found
			if(similarityProcessed.ptr(y+1)[tmpX] == 255) {
				x = tmpX;
				y++;
			}
			else {
				// jump over some pixels and found best extension in particular directions
				if(maxGap > 0) {
					cv::Point ptX = extendSegment(similarityProcessed, cv::Point(x+1, y), maxGap-1);
					cv::Point ptY = extendSegment(similarityProcessed, cv::Point(x, y+1), maxGap-1);
					cv::Point ptXY = extendSegment(similarityProcessed, cv::Point(x+1, y+1), maxGap-1);

					double distX = sqrt((float)((start.x-ptX.x)*(start.x-ptX.x) + (start.y-ptX.y)*(start.y-ptX.y)));
					double distY = sqrt((float)((start.x-ptY.x)*(start.x-ptY.x) + (start.y-ptY.y)*(start.y-ptY.y)));
					double distXY = sqrt((float)((start.x-ptXY.x)*(start.x-ptXY.x) + (start.y-ptXY.y)*(start.y-ptXY.y)));

					if(distXY >= distX && distXY >= distY) {
						return ptXY;
					}
					else {
						if(distX > distY) {
							return ptX;
						}
						else {
							return ptY;
						}
					}
				}
				// end of the current segment
				else {
					break;
				}
			}
		}
	}

	return cv::Point(x, y);
}


void NeedlemanWunschSegmentsExtractor::findAlignment(const SimilarityMatrix & similarity,
													 double gapWeight, cv::Mat & score, cv::Mat & trace) {
	score = cv::Mat::zeros(similarity.rows/*+1*/, similarity.cols/*+1*/, CV_64F);
	trace = cv::Mat::zeros(similarity.rows/*+1*/, similarity.cols/*+1*/, CV_8U);

	// TODO optimalizovat pristup k maticim pres ukazatele
	for(int r = 0; r < score.rows; r++) {
		for(int c = 0; c < score.cols; c++) {
			/*if(c + 1 == score.cols && r + 1 == score.rows) {
				score.at<double>(r, c) = score.at<double>(r-1, c-1);
				trace.at<uchar>(r, c) = TR_DIAGONAL;
			} else if(c + 1 == score.cols) {
				score.at<double>(r, c) = score.at<double>(r, c-1);
				trace.at<uchar>(r, c) = TR_HORIZONTAL;
			} else if(r + 1 == score.rows) {
				score.at<double>(r, c) = score.at<double>(r-1, c);
				trace.at<uchar>(r, c) = TR_VERTICAL;
			} else*/ if(c > 0 && r > 0) {
				double weightDiagonal = -similarity.at<float>(r-1,c-1) + score.at<double>(r-1, c-1);//score.at<double>(r-1, c-1) - similarity.at<float>(r-1,c-1);
				double weightVertical = -similarity.at<float>(r-1,c) + score.at<double>(r-1, c);///*similarity.at<float>(r,c) -*/ score.at<double>(r-1, c) + gapWeight;
				double weightHorizontal = -similarity.at<float>(r,c-1) + score.at<double>(r, c-1);///*similarity.at<float>(r,c) -*/ score.at<double>(r, c-1) + gapWeight;

				if(weightDiagonal >= weightVertical && weightDiagonal >= weightHorizontal) {
					score.at<double>(r, c) = weightDiagonal;
					trace.at<uchar>(r, c) = TR_DIAGONAL;
				} else if(weightVertical > weightHorizontal) {
					score.at<double>(r, c) = weightVertical;
					trace.at<uchar>(r, c) = TR_VERTICAL;
				} else {
					score.at<double>(r, c) = weightHorizontal;
					trace.at<uchar>(r, c) = TR_HORIZONTAL;
				}
			} else if(c > 0) {
				double weightHorizontal = -similarity.at<float>(r,c-1) + score.at<double>(r, c-1);//score.at<double>(r, c-1) + gapWeight;
				score.at<double>(r, c) = weightHorizontal;
				trace.at<uchar>(r, c) = TR_HORIZONTAL;
			} else if(r > 0) {
				double weightVertical = -similarity.at<float>(r-1,c) + score.at<double>(r-1, c);//score.at<double>(r-1, c) + gapWeight;
				score.at<double>(r, c) = weightVertical;
				trace.at<uchar>(r, c) = TR_VERTICAL;
			}
		}
	}
	// this is Levenshtein now.
}


NeedlemanWunschSegmentsExtractor::Points NeedlemanWunschSegmentsExtractor::backTrace(const cv::Mat & trace) {
	std::vector<cv::Point> points;

	int r = trace.rows-1;
	int c = trace.cols-1;

	// TODO optimalizovat pristup k matici pres ukazatele
	while(r > 0 && c > 0) {
		points.insert(points.begin(), cv::Point(c, r));

		uchar traceDirection = trace.at<uchar>(r, c);
		switch(traceDirection) {
			case TR_DIAGONAL:
				r--;
				c--;
				break;

			case TR_VERTICAL:
				r--;
				break;

			case TR_HORIZONTAL:
				c--;
				break;
		}
	}

	points.insert(points.begin(), cv::Point(c, r));

	return points;
}


double NeedlemanWunschSegmentsExtractor::linePointDistance(cv::Point pt, cv::Point lineStart, cv::Point lineEnd) {
	cv::Point u(lineEnd.x-lineStart.x, lineEnd.y-lineStart.y);
	double a = u.y;
	double b = -u.x;
	double c = -a*lineStart.x - b*lineStart.y;

	return abs(a*pt.x + b*pt.y + c)/sqrt(a*a + b*b);
}


int NeedlemanWunschSegmentsExtractor::mostDivergingFromSegment(Points & points, double & maxDist) {
	if(points.size() < 3) {
		return -1;
	}

	cv::Point lineStart = points[0];
	cv::Point lineEnd = points[points.size()-1];

	maxDist = 0;
	int maxIdx = -1;

	for(int i = 1; i < (signed)points.size()-1; i++) {
		double dist = linePointDistance(points[i], lineStart, lineEnd);

		if(dist > maxDist) {
			maxDist = dist;
			maxIdx = i;
		}
	}

	return maxIdx;
}


void NeedlemanWunschSegmentsExtractor::geometricSubdivision(PointSegments & segments, double distThr) {
	bool changed = true;

	while(changed) {
		PointSegments newSegments;
		changed = false;

		for(std::vector<Points>::iterator it = segments.begin(); it != segments.end(); it++) {
			double dist = 0;
			int divergIdx = mostDivergingFromSegment(*it, dist);

			if(divergIdx != -1 && dist > distThr) {
				Points seg1;
				Points seg2;
				
				for(int i = 0; i < (signed)it->size(); i++) {
					if(i < divergIdx) {
						seg1.push_back((*it)[i]);
					}
					else {
						seg2.push_back((*it)[i]);
					}
				}

				newSegments.push_back(seg1);
				newSegments.push_back(seg2);

				changed = true;
			}
			else {
				newSegments.push_back(*it);
			}
		}

		segments = newSegments;
	}
}


void NeedlemanWunschSegmentsExtractor::similaritySubdivision(PointSegments & segments, SimilarityMatrix & similarityNormalized, double distThr) {
	PointSegments newSegments;
	
	for(PointSegments::iterator it = segments.begin(); it != segments.end(); it++) {		
		Points tmpSegment;
		Points currentSegment = *it;

		for(Points::iterator it2 = it->begin(); it2 != it->end(); it2++) {
			if(similarityNormalized.at<float>(it2->y, it2->x) < distThr) {
				tmpSegment.push_back(*it2);
			}
			else if(tmpSegment.size() > 0) {
				newSegments.push_back(tmpSegment);
				tmpSegment = Points();
			}
		}

		if(tmpSegment.size() > 0) {
			newSegments.push_back(tmpSegment);
		}
	}

	segments = newSegments;
}


void NeedlemanWunschSegmentsExtractor::filterSegments(PointSegments & segments, bool angle, double minLength) {
	PointSegments newSegments;

	for(PointSegments::iterator it = segments.begin(); it != segments.end(); it++) {
		if(it->size() > 2 && (it->size() >= minLength || minLength < 0)) {
			cv::Point ptStart = (*it)[0];
			cv::Point ptEnd = (*it)[it->size()-1];

			double tanV = ((double)(ptEnd.y - ptStart.y)) / (ptEnd.x - ptStart.x);
			double atanV = atan(abs(tanV));

			if(!angle || (atanV > 0.3926994 && atanV < 1.178097)) {
				newSegments.push_back(*it);
			}
		}
	}

	segments = newSegments;
}

extern double f_video_time, f_matrix_time, f_matrix_time2;
extern int length_ref, length_qry;

void NeedlemanWunschSegmentsExtractor::extractFunc(const SimilarityMatrix & similarity, VideoSequenceDataPtr dataRef, VideoSequenceDataPtr dataQuery, ExtractedSegments & segments) {
	SimilarityMatrix similarityNormalized = similarity.normalize();
	CTimer t;
	cv::Mat score, trace;
	findAlignment(similarityNormalized, -.4, score, trace);

	Points points = backTrace(trace);

	double f_matching_time = t.f_Time();

	int n_perturb_type = 3;
#if 1
	n_perturb_type = -1; // no GT
	std::vector<double> scaled_time;
#else
	const char *p_s_perturb_type = strstr(p_s_R_file, "_perturbed"); // R is peRturbed, Q is original
	if(!p_s_perturb_type) {
		fprintf(stderr, "error: failed to parse perturb type\n");
		exit(-1);
	}
	p_s_perturb_type += strlen("_perturbed");
	if(!isdigit(*p_s_perturb_type)) {
		fprintf(stderr, "error: failed to parse perturb type\n");
		exit(-1);
	}
	n_perturb_type = *p_s_perturb_type - '0';
	if(n_perturb_type < 0 || n_perturb_type > 3) {
		fprintf(stderr, "error: failed to parse perturb type\n");
		exit(-1);
	}
	// not sure why i dont use exceptions ...

	std::vector<double> relative_speed;
	{
		char p_s_perturb_data[256];
		stl_ut::Format(p_s_perturb_data, sizeof(p_s_perturb_data), "relspeed%d.png", n_perturb_type);
		TBmp *p_relspeed;
		if(!(p_relspeed = CPngCodec::p_Load_PNG(p_s_perturb_data))) {
			fprintf(stderr, "error: failed to open \'%s\'\n", p_s_perturb_data);
			exit(-1);
		}
		const uint32_t n_bk_color = p_relspeed->p_buffer[0];
		for(int i = 0, w = p_relspeed->n_width, h = p_relspeed->n_height; i < w; ++ i) {
			int n_position = -1;
			for(int j = 0; j < h; ++ j) {
				if(p_relspeed->p_buffer[i + w * j] != n_bk_color) {
					n_position = j;
					break;
				}
			}
			if(n_position == -1) {
				fprintf(stderr, "error: column %d does not have a marker\n", i);
				continue;
			}
			n_position = h - 1 - n_position; // upside down - max is supposed to be on the top
			relative_speed.push_back(max(.0, min(1.0, ((n_position - h * .5) / (h * .5) + 1) / w)));
		}
		p_relspeed->Delete();
	}
	// load a relative speed vector

	std::vector<double> scaled_time = relative_speed;
	scaled_time.push_back(0); // not the last one
	stl_ut::ExclusiveScan(scaled_time.begin(), scaled_time.end());
	const double f_scaled_time = scaled_time.back(); // get the total time of the scaler (should be about 1, depending on the contents)
	std::transform(scaled_time.begin(), scaled_time.end(),
		scaled_time.begin(), CScale(1 / f_scaled_time));
	// convert the relative speed to scaled time
#endif // 0

	const double f_frame_num = similarity.cols;
	// get the number of video frames

	const size_t n_new_frame_num = similarity.rows;
	// get the new video length

	size_t n_cur_frame = 0;
	/*TBmp t_next_frame = v.t_Get_NextFrame();
	TBmp *p_cur_frame = t_next_frame.p_Clone();
	const size_t n_image_size = t_next_frame.n_width * t_next_frame.n_height * sizeof(uint32_t);
	t_next_frame = v.t_Get_NextFrame();
	v.t_Get_NextFrame();
	TBmp *p_blend = p_cur_frame->p_Clone(true);
	TBmp &t_cur_frame = *p_cur_frame, &t_blend = *p_blend; // convenience*/

	// make sure there are no leftovers from a previous run

	std::vector<double> ground_truth; // one per each row

	if(!scaled_time.empty()) {
		const size_t n_scaler_size = scaled_time.size();
		for(size_t i = 0; i < n_new_frame_num; ++ i) { // similarity.rows
			double f_rel_time = double(i) / (n_new_frame_num - 1);
			// relative frame time

			double f_scl_index = f_rel_time * (n_scaler_size - 1);
			size_t n_scl_index = min(size_t(f_scl_index), n_scaler_size - 1);
			double f_scl_lerp = max(.0, f_scl_index - n_scl_index);
			_ASSERTE(f_scl_lerp >= 0 && f_scl_lerp <= 1);
			double f_scl_time = scaled_time[n_scl_index] * (1 - f_scl_lerp) +
				scaled_time[min(n_scl_index + 1, n_scaler_size - 1)] * f_scl_lerp;
			// calculate scaled time (in the original video)

			double f_src_frame = f_scl_time * (f_frame_num - 1);

			ground_truth.push_back(f_src_frame);
		}
	}

	cv::Mat frame, frame1, frame2;
	similarity.render(frame);
	cv::cvtColor(frame, frame1, CV_GRAY2BGR);
	const float scl = 4;
	cv::resize(frame1, frame2, cv::Size(0, 0), scl, scl, cv::INTER_NEAREST);

	float f_sum_of_squares = 0, f_max_square = 0;
	if(!ground_truth.empty()) {
		for(Points::const_iterator it = points.begin(); it != points.end(); it++) {
			float f_found_x = float((*it).x), f_found_y = float((*it).y);
			float ff = std::min(f_found_x / frame.cols, f_found_y / frame.rows);
			f_found_x += ff;
			f_found_y += ff;
			float f_best_x, f_best_y;
			float f_best = float(1e100);
			for(size_t y = 0; y < n_new_frame_num; ++ y) {
				float f_y = float(y) + float(y) / (n_new_frame_num - 1);
				float f_x = float(ground_truth[y]);
				float f_error2 = (f_found_x - f_x) * (f_found_x - f_x) +
					(f_found_y - f_y) * (f_found_y - f_y);
				if(f_error2 < f_best) {
					f_best_x = f_x;
					f_best_y = f_y;
					f_best = f_error2;
				}
			}
			/*{
				cv::line(frame2, cv::Point(f_found_x * scl, f_found_y * scl),
					cv::Point(f_best_x * scl, f_best_y * scl), cv::Scalar(128,128,128), 1, CV_AA);
				// debug
			}*/
			f_sum_of_squares += f_best;
			if(f_max_square < f_best)
				f_max_square = f_best;
		}
	}
	f_sum_of_squares /= std::min(n_new_frame_num, size_t(f_frame_num));
	// calculate sum of squares of errors of the solution

	std::string s_filename = p_s_R_file;
	if(s_filename.find_last_of("\\/") != std::string::npos)
		s_filename.erase(0, s_filename.find_last_of("\\/") + 1);
	// get file name only

	if(!ground_truth.empty()) {
		FILE *p_fw;
		if((p_fw = fopen("precision_logs.txt", "a"))) {
			fprintf(p_fw, "%s;%d;%.5f;%.5f;%.3f;%.3f;%.3f;%.3f\n", s_filename.c_str(), n_perturb_type,
				f_sum_of_squares, f_max_square, f_video_time * 1000, f_matrix_time * 1000, f_matrix_time2 * 1000, f_matching_time * 1000);
			fclose(p_fw);
		} else
			fprintf(stderr, "error: failed to write log\n");
	}

	if(!ground_truth.empty()) {
		//frame2.setTo(255.0); // draw a white-bk frame for the paper

		float f_prev_y, f_prev_x;
		for(size_t y = 0; y < n_new_frame_num; ++ y) {
			float f_y = y + float(y) / (n_new_frame_num - 1);
			float f_x = float(ground_truth[y]);

			if(y) {
				cv::line(frame2, cv::Point(int(f_x * scl), int(f_y * scl)),
					cv::Point(int(f_prev_x * scl), int(f_prev_y * scl)), cv::Scalar(0, 255, 0), 2, CV_AA);
			}

			f_prev_y = f_y, f_prev_x = f_x;
		}
	}
	// draw GT

	std::vector<cv::Point> points_filtered(points);
	for(size_t i = 1, n = points_filtered.size(); i < n; ++ i) {
		if(points_filtered[i - 1].x == points_filtered[i].x ||
		   points_filtered[i - 1].y == points_filtered[i].y) { // filter out vertical or horizontal line segments
			points_filtered.erase(points_filtered.begin() + i);
			-- i;
			-- n;
		}
	}
	// try to filter the points

	/*for(Points::const_iterator it = points.begin()+1; it != points.end(); it++) {		
		cv::Point ptPrev = *(it-1);
		cv::Point ptCurrent = *it;
		float fprev = std::min(float(ptPrev.x) / frame.cols, float(ptPrev.y) / frame.rows);
		float ff = std::min(float(ptCurrent.x) / frame.cols, float(ptCurrent.y) / frame.rows);
		cv::line(frame2, cv::Point((ptPrev.x + fprev) * scl, (ptPrev.y + fprev) * scl),
			cv::Point((ptCurrent.x + ff) * scl, (ptCurrent.y + ff) * scl), cv::Scalar(0,0,255), 2, CV_AA);

		/*cv::Scalar color = cv::Scalar(rand()%255,rand()%255,rand()%255);
		cv::circle(frame2, cv::Point((ptCurrent.x + .5f) * scl, (ptCurrent.y + .5f) * scl), 1 * scl, color, 2);* /
		// note that the first circle is not displayed
	}*/
	for(Points::const_iterator it = points_filtered.begin()+1; it != points_filtered.end(); it++) {		
		cv::Point ptPrev = *(it-1);
		cv::Point ptCurrent = *it;
		float fprev = std::min(float(ptPrev.x) / frame.cols, float(ptPrev.y) / frame.rows);
		float ff = std::min(float(ptCurrent.x) / frame.cols, float(ptCurrent.y) / frame.rows);
		cv::line(frame2, cv::Point(int((ptPrev.x + fprev) * scl), int((ptPrev.y + fprev) * scl)),
			cv::Point(int((ptCurrent.x + ff) * scl), int((ptCurrent.y + ff) * scl)), cv::Scalar(0, 255, 0), 2, CV_AA);

		/*cv::Scalar color = cv::Scalar(rand()%255,rand()%255,rand()%255);
		cv::circle(frame2, cv::Point((ptCurrent.x + .5f) * scl, (ptCurrent.y + .5f) * scl), 1 * scl, color, 2);*/
		// note that the first circle is not displayed
	}

	s_filename.erase(s_filename.rfind('.'));

	cv::imwrite(s_filename + ((b_use_autocorr)? "_path_auto.png" : "_path.png"), frame2);

	FILE *p_fw;
	if((p_fw = fopen((s_filename + "sync_info.txt").c_str(), "w"))) {
		fprintf(p_fw, "%d\n", 8); // step
		fprintf(p_fw, "%d, %d\n", similarity.rows, similarity.cols); // ref, query
		fprintf(p_fw, "%d, %d\n", length_ref, length_qry); // ref, query in frames
		fprintf(p_fw, "%d\n", points_filtered.size());
		for(Points::const_iterator it = points_filtered.begin(); it != points_filtered.end(); it++) {
			cv::Point ptCurrent = *it;
			fprintf(p_fw, "%.15f, %.15f\n", double(ptCurrent.y) / points_filtered.back().y,
				double(ptCurrent.x) / points_filtered.back().x);
		}
		fprintf(p_fw, "%d\n", points.size());
		for(Points::const_iterator it = points.begin(); it != points.end(); it++) {
			cv::Point ptCurrent = *it;
			fprintf(p_fw, "%d, %d\n", ptCurrent.y, ptCurrent.x);
		}
		fclose(p_fw);
	} else
		fprintf(stderr, "error: failed to write synchronization information\n");
	// write synchronization info as txt

	if((p_fw = fopen((s_filename + "sync_info.xml").c_str(), "w"))) {
		fprintf(p_fw, "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n", 8); // header
		fprintf(p_fw, "<sync-data granularity=\"%d\" knots=\"%d\">\n", 8, points_filtered.size()); // step
		fprintf(p_fw, "\t<query-sequence length=\"%d\"/>\n", length_qry); // in frames
		fprintf(p_fw, "\t<reference-sequence length=\"%d\"/>\n", length_ref); // in frames
		//fprintf(p_fw, "%d\n", points_filtered.size());
		for(Points::const_iterator it = points_filtered.begin(); it != points_filtered.end(); it++) {
			cv::Point ptCurrent = *it;
			fprintf(p_fw, "\t<sync q=\"%.15f\" r=\"%.15f\"/>\n", double(ptCurrent.y) / points_filtered.back().y,
				double(ptCurrent.x) / points_filtered.back().x);
		}
		fprintf(p_fw, "\t<keyframe-sync-data granularity=\"%d\" knots=\"%d\">\n", 8, points.size()); // step
		fprintf(p_fw, "\t\t<query-sequence length=\"%d\"/>\n", similarity.cols); // in frames
		fprintf(p_fw, "\t\t<reference-sequence length=\"%d\"/>\n", similarity.rows); // in frames
		for(Points::const_iterator it = points.begin(); it != points.end(); it++) {
			cv::Point ptCurrent = *it;
			fprintf(p_fw, "\t\t<kf-sync q=\"%d\" r=\"%d\"/>\n", ptCurrent.y, ptCurrent.x);
		}
		fprintf(p_fw, "</keyframe-sync-data>\n", 8);
		fprintf(p_fw, "</sync-data>\n", 8);
		fclose(p_fw);
	} else
		fprintf(stderr, "error: failed to write synchronization information XML\n");
	// write synchronization info as xml

	//cv::namedWindow("path", cv::WINDOW_NORMAL);
	//cv::resizeWindow("path", 512, 512);
    //cv::imshow("path", frame2);

	PointSegments pointSegments;
	pointSegments.push_back(points);

	geometricSubdivision(pointSegments, 5);
	
	cv::resize(frame1, frame2, cv::Size(0, 0), scl, scl, cv::INTER_NEAREST);
	//cv::cvtColor(frame1, frame2, CV_GRAY2BGR);

	for(std::vector<Points>::iterator it = pointSegments.begin(); it != pointSegments.end(); it++) {
		cv::Scalar color = cv::Scalar(rand()%255,rand()%255,rand()%255);

		for(int i = 0; i < (signed)it->size(); i++) {
			cv::Point ptCurrent = (*it)[i];
			cv::circle(frame2, cv::Point(int((ptCurrent.x + .5f) * scl),
				int((ptCurrent.y + .5f) * scl)), int(1 * scl), color, 2);
		}
	}

	//cv::namedWindow("segments-geometric", cv::WINDOW_NORMAL);
	//cv::resizeWindow("segments-geometric", 512, 512);
    //cv::imshow("segments-geometric", frame2);

	//cv::imwrite(s_filename + "_segments-geometric.png", frame2);



	filterSegments(pointSegments, true, 5);
	similaritySubdivision(pointSegments, similarityNormalized, 0.3);
	filterSegments(pointSegments, false, 5);

	cv::resize(frame1, frame2, cv::Size(0, 0), scl, scl, cv::INTER_NEAREST);
	//cv::cvtColor(frame1, frame2, CV_GRAY2BGR);

	for(std::vector<Points>::iterator it = pointSegments.begin(); it != pointSegments.end(); it++) {
		cv::Scalar color = cv::Scalar(rand()%255,rand()%255,rand()%255);

		for(int i = 0; i < (signed)it->size(); i++) {
			cv::Point ptCurrent = (*it)[i];
			cv::circle(frame2, cv::Point(int((ptCurrent.x + .5f) * scl),
				int((ptCurrent.y + .5f) * scl)), int(1 * scl), color, 2);
		}
	}

    //cv::imshow("segments-geometric+similarity", frame2);

	//cv::namedWindow("segments-geometric+similarity", cv::WINDOW_NORMAL);
	//cv::resizeWindow("segments-geometric+similarity", 512, 512);
    //cv::imshow("segments-geometric+similarity", frame2);

	//cv::imwrite(s_filename + "_segments-geometric+similarity.png", frame2);

	//cv::waitKey(0);


    //cv::waitKey(1);
	









	for(std::vector<Points>::iterator it = pointSegments.begin(); it != pointSegments.end(); it++) {
		if(it->size() < 1) {
			continue;
		}

		cv::Point start = (*it)[0];
		cv::Point end = (*it)[it->size()-1];
		
		segments.push_back(cv::Vec4i(start.y, end.y, start.x, end.x));
	}
}


//void NeedlemanWunschSegmentsExtractor::extractFunc(const SimilarityMatrix & similarity, VideoSequenceDataPtr dataRef, VideoSequenceDataPtr dataQuery, ExtractedSegments & segments) {
//	SimilarityMatrix similarityNormalized = similarity.normalize();
//	
//	cv::Mat score, trace;
//	findAlignment(similarityNormalized, -0.5, score, trace);
//
//	Points points = backTrace(trace);
//	NeedlemanWunschSegmentsExtractor::Segments segmentsDet;
//
//	if(points.empty()) {
//		return;
//	}
//
//	float threshold = 0.3;
//
//	cv::Point ptStart = *points.begin();
//	cv::Point ptCurrent = ptStart;
//	bool currentOk = similarityNormalized.at<float>(ptCurrent.y, ptCurrent.x) < threshold;
//
//	for(Points::const_iterator it = points.begin()+1; it != points.end()-1; it++) {
//		ptCurrent = *it;
//		
//		if(similarityNormalized.at<float>(ptCurrent.y, ptCurrent.x) > threshold) {
//			if(currentOk) {
//				NeedlemanWunschSegmentsExtractor::Segment currentSegment(ptStart, ptCurrent);
//				segmentsDet.push_back(currentSegment);
//				currentOk = false;
//			}
//		}
//		else if(!currentOk) {
//			ptStart = *it;
//			currentOk = true;
//		}
//	}
//
//	ptCurrent = *(points.end()-1);
//	NeedlemanWunschSegmentsExtractor::Segment currentSegment(ptStart, ptCurrent);
//	segmentsDet.push_back(currentSegment);
//
//	cv::Mat frame;
//	similarity.render(frame);
//	cv::cvtColor(frame, frame, CV_GRAY2BGR);
//
//	cv::imshow("sim", frame);
//
//	for(NeedlemanWunschSegmentsExtractor::Segments::iterator it = segmentsDet.begin(); it != segmentsDet.end(); it++) {
//		segments.push_back(cv::Vec4i(it->start.y, it->end.y, it->start.x, it->end.x));
//		cv::Scalar color = cv::Scalar(rand()%255, rand()%255, rand()%255);
//		cv::line(frame, it->start, it->end, color, 1);
//	}
//
//	cv::imshow("sim-alignment", frame);
//	//cv::waitKey(0);
//	//exit(0);
//}


//void NeedlemanWunschSegmentsExtractor::extractFunc(const SimilarityMatrix & similarity, VideoSequenceDataPtr dataRef, VideoSequenceDataPtr dataQuery, ExtractedSegments & segments) {
//	//==================== ZAROVNANI VIDEI ====================
//	SimilarityMatrix similarityNormalized = similarity.normalize();
//	
//	cv::Mat score, trace;
//	findAlignment(similarityNormalized, -0.5, score, trace);
//
//	Points points = backTrace(trace);
//
//	
//
//	//==================== EXTRAKCE KF SE SURFY ====================
//	PresetKeyFramesExtractor::FrameNumbers fnRef, fnQry;
//	KeyFrames kfRef = tmpDataRef->getKeyFrames();
//	KeyFrames kfQry = tmpDataQuery->getKeyFrames();
//
//	for(KeyFrames::iterator it = kfRef.begin(); it != kfRef.end(); it++) {
//		fnRef.push_back((*it)->getFrameNumber());
//	}
//
//	for(KeyFrames::iterator it = kfQry.begin(); it != kfQry.end(); it++) {
//		fnQry.push_back((*it)->getFrameNumber());
//	}	
//
//	CvVideoSequence vsRef("ref.avi", new SurfFrameDescriptorExtractor());
//	CvVideoSequence vsQry("qry.avi", new SurfFrameDescriptorExtractor());
//
//	PresetKeyFramesExtractor keyFramesExtractor(fnRef);
//	kfRef = keyFramesExtractor.extract(&vsRef);
//	std::cout << "kfref=" << kfRef.size() << "\n";
//
//	keyFramesExtractor.setFrameNumbers(fnQry);
//	kfQry = keyFramesExtractor.extract(&vsQry);
//
//
//
//	//==================== VYKRESLENI ====================
//	cv::Mat frame;
//	similarity.render(frame);
//	cv::cvtColor(frame, frame, CV_GRAY2BGR);
//
//	cv::imshow("sim", frame);
//
//	for(Points::const_iterator it = points.begin()+1; it != points.end(); it++) {
//		cv::line(frame, *it, *(it-1), cv::Scalar(0,0,255*(1-similarityNormalized.at<float>((*it).y, (*it).x))), 3);
//	}
//
//	cv::imshow("sim-alignment-hist", frame);
//
//
//	similarity.render(frame);
//	cv::cvtColor(frame, frame, CV_GRAY2BGR);
//	for(Points::const_iterator it = points.begin()+1; it != points.end(); it++) {
//		cv::line(frame, *it, *(it-1), cv::Scalar(0,0,255*(1-kfRef[(*it).y]->getFrameDescriptor()->compare(*kfQry[(*it).x]->getFrameDescriptor()))), 3);
//	}
//	cv::imshow("sim-alignment-surf", frame);
//
//	cv::waitKey(0);
//	exit(0);
//}


//void NeedlemanWunschSegmentsExtractor::extractFunc(const SimilarityMatrix & similarity, VideoSequenceDataPtr dataRef, VideoSequenceDataPtr dataQuery, ExtractedSegments & segments) {
//	SimilarityMatrix similarityNormalized = similarity.normalize();
//
//	cv::Mat score, trace;
//	findAlignment(similarityNormalized, -0.5, score, trace);
//
//	Points points = backTrace(trace);
//	NeedlemanWunschSegmentsExtractor::Segments segmentsDet;
//
//	if(points.empty()) {
//		return;
//	}
//
//	float threshold = 0.3f;
//
//	cv::Point ptStart = *points.begin();
//	cv::Point ptCurrent = ptStart;
//	bool inSegment = similarityNormalized.at<float>(ptCurrent.y, ptCurrent.x) < threshold;
//
//	for(Points::const_iterator it = points.begin()+1; it != points.end()-1; it++) {
//		ptCurrent = *it;
//
//			//---squaremin---//
//			float squaremin = FLT_MAX;
//			int squareSize = 1;
//			
//			for(int x = MAX(0, ptCurrent.x-squareSize); x <= MIN(similarityNormalized.cols-1, ptCurrent.x+squareSize); x++) {				
//				for(int y = MAX(0, ptCurrent.y-squareSize); y <= MIN(similarityNormalized.rows-1, ptCurrent.y+squareSize); y++) {
//					squaremin = MIN(squaremin, similarityNormalized.at<float>(y, x));
//				}
//			}
//			
//			//---squaremin---//
//		
//		if(squaremin > threshold) {
//			//---crosscheck---//
//			bool crosscheck = true;
//			int crossSize = 1;
//			
//			KeyFrames keyframesRef = dataRef->getKeyFrames();
//			KeyFrames keyframesQuery = dataQuery->getKeyFrames();
//			
//			int segmentId = -1;
//
//			for(int x = MAX(0, ptCurrent.x-crossSize); x <= MIN(similarityNormalized.cols-1, ptCurrent.x+crossSize); x++) {				
//				int sss = keyframesQuery[x]->getSegmentId();
//				
//				if(segmentId == -1) {
//					segmentId = sss;
//				}
//
//				crosscheck &= (sss == segmentId);
//			}
//
//			segmentId = -1;
//			for(int y = MAX(0, ptCurrent.y-crossSize); y <= MIN(similarityNormalized.rows-1, ptCurrent.y+crossSize); y++) {
//				int sss = keyframesRef[y]->getSegmentId();
//				
//				if(segmentId == -1) {
//					segmentId = sss;
//				}
//
//				crosscheck &= (sss == segmentId);	
//			}
//
//			//---crosscheck---//
//
//			if(inSegment && !crosscheck) {
//				NeedlemanWunschSegmentsExtractor::Segment currentSegment(ptStart, ptCurrent);
//				segmentsDet.push_back(currentSegment);
//				inSegment = false;
//			}
//		}
//		else {
//			 if(!inSegment) {
//				 ptStart = *it;
//				 inSegment = true;
//			 }
//		}
//	}
//
//	ptCurrent = *(points.end()-1);
//	NeedlemanWunschSegmentsExtractor::Segment currentSegment(ptStart, ptCurrent);
//	segmentsDet.push_back(currentSegment);
//
//	cv::Mat frame;
//	similarity.render(frame);
//	cv::cvtColor(frame, frame, CV_GRAY2BGR);
//
//	cv::imshow("sim", frame);
//
//	for(NeedlemanWunschSegmentsExtractor::Segments::iterator it = segmentsDet.begin(); it != segmentsDet.end(); it++) {
//		segments.push_back(cv::Vec4i(it->start.y, it->end.y, it->start.x, it->end.x));
//		cv::Scalar color = cv::Scalar(rand()%255, rand()%255, rand()%255);
//		cv::line(frame, it->start, it->end, color, 1);
//	}
//
//	cv::imshow("sim-alignment", frame);
//
//
//	similarity.render(frame);
//	cv::cvtColor(frame, frame, CV_GRAY2BGR);
//
//	for(Points::const_iterator it = points.begin()+1; it != points.end()-1; it++) {
//		cv::Point ptPrev = *(it-1);
//		cv::Point ptCurrent = *it;
//
//		cv::line(frame, ptPrev, ptCurrent, cv::Scalar(0,0,255));
//	}
//
//	cv::imshow("path", frame);
//	cv::waitKey(0);
//	exit(0);
//}


NeedlemanWunschSegmentsExtractor::NeedlemanWunschSegmentsExtractor() {
}


}
