#include "plot.h"
#include "vec2.h"
#include "transform.h"
#include "timer.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

static
double eval_streaming(int N, int size_x, int size_y, int J)
{
	const struct vec2_t cb_exp = { 6, 6 };
	const struct vec2_t tc0 = { 5, 128 };
	const struct vec2_t size = { size_x, size_y };
	const struct vec2_t tc1 = vec2_add(tc0, size);

	double min_time = +INFINITY;

	for(int n = 0; n < N; n++)
	{
		struct imageptr_t *imageptr = imageptr_create_sparse(size);

		if( !imageptr )
			return -1;

		struct transform_t *transform = transform_create(
			tc0,
			tc1,
			cb_exp,
			1,
			J,
			NULL,
			NULL
		);

		if( !transform )
			return -1;

		int64_t clocks = 0;

		while( !transform_finished(transform) )
		{
			const struct range2_t lines = transform_prepare_strip(transform);

			const struct vec2_t data_offset = { 0, lines.i0 };
			const struct vec2_t data_size = { size.x, lines.i1 - lines.i0 };

			struct imageptr_t *data_lines = imageptr_viewport(imageptr, data_offset);

			imageptr_fill_strip(
				data_lines,
				data_offset,
				data_size
			);

			free(data_lines);

			int64_t start = timer_get_clock();

			transform_process_strip(
				transform,
				imageptr,
				vec2_zero
			);

			int64_t stop = timer_get_clock();

			clocks += (stop - start);

			// discard already processed data
			imageptr_discard_sparse(
				imageptr,
				vec2_create( 0, lines.i1 )
			);
		}

		double time = (double)clocks / timer_get_frequency() * 1e9 / ( (long)size.x * size.y );

		if( time < min_time )
			min_time = time;

		transform_destroy(transform);

		imageptr_destroy_sparse(imageptr, size);
	}

	return min_time;
}

static
double eval_allocated(int N, int size_x, int size_y, int J)
{
	const struct vec2_t cb_exp = { 6, 6 };
	const struct vec2_t tc0 = { 5, 128 };
	const struct vec2_t size = { size_x, size_y };
	const struct vec2_t tc1 = vec2_add(tc0, size);

	struct imageptr_t *imageptr = imageptr_create_sparse(size);

	double min_time = +INFINITY;

	for(int n = 0; n < N; n++)
	{
		imageptr_fill(imageptr, size);

		int64_t clocks = -timer_get_clock();

		if( transform_process_tile(tc0, tc1, cb_exp, J, imageptr) )
			return -1;

		double time = timer_clock_to_nsec_pel(clocks+timer_get_clock(), size.x*size.y);

		if( time < min_time )
			min_time = time;
	}

	imageptr_destroy_sparse(imageptr, size);

	return min_time;
}

static
void eval_wrapper_streaming(FILE *file, int size_x, int size_y)
{
	const int J = 8;
	const int N = 100 + (1<<30) / size_x / size_y;
	double time;

	if( size_x*size_y*sizeof(int) > (2UL<<30) )
		return;

	fprintf(stdout, "TEST(streaming) \t (%5i,%5i) \t N=%5i \t %10li [pel] \t %4zu MiB \t measuring... \t ", size_x, size_y, N, (long)size_x*size_y, ((size_t)size_x*size_y*sizeof(int))>>20);
	fflush(stdout);

	time = eval_streaming(N, size_x, size_y, J);

	fprintf(stdout, "%f [ns/pel] \n", time);
	fflush(stdout);

	fprintf(file, "%li\t%f\n", (long)size_x*size_y, time);
	fflush(file);
}

static
void eval_wrapper_allocated(FILE *file, int size_x, int size_y)
{
	const int J = 8;
	const int N = 100 + (1<<30) / size_x / size_y;
	double time;

	if( size_x*size_y*sizeof(int) > (2UL<<30) )
		return;

	fprintf(stdout, "TEST(allocated) \t (%5i,%5i) \t N=%5i \t %10li [pel] \t %4zu MiB \t measuring... \t ", size_x, size_y, N, (long)size_x*size_y, ((size_t)size_x*size_y*sizeof(int))>>20);
	fflush(stdout);

	time = eval_allocated(N, size_x, size_y, J);

	fprintf(stdout, "%f [ns/pel] \n", time);
	fflush(stdout);

	fprintf(file, "%li\t%f\n", (long)size_x*size_y, time);
	fflush(file);
}

static
void annotate_wrapper_allocated(const char *message, int size_x, int size_y)
{
	const int J = 8;
	const int N = 100 + (1<<30) / size_x / size_y;
	double time;

	if( size_x*size_y*sizeof(int) > (2UL<<30) )
		return;

	fprintf(stdout, " (%5i,%5i) \t %10li pel \t %4zu MiB \t", size_x, size_y, (long)size_x*size_y, ((size_t)size_x*size_y*sizeof(int))>>20);
	fflush(stdout);

	time = eval_allocated(N, size_x, size_y, J);

	fprintf(stdout, " %9.6f ns/pel \t %s\n", time, message);
	fflush(stdout);
}

void plot_generate_streaming()
{
	FILE *file;
	int shift;

	if( !(file = fopen("plot/streaming.txt", "w+")) )
	{
		abort();
	}

	for(shift = 1; shift < 8; shift++)
	{
		long a = 1<<shift;

		eval_wrapper_streaming(file, 15*16*a, 15*9*a);
		eval_wrapper_streaming(file, 16*16*a, 15*9*a);
		eval_wrapper_streaming(file, (3*15*16*a)>>1, (3*15*9*a)>>1);
		eval_wrapper_streaming(file, (3*16*16*a)>>1, (3*15*9*a)>>1);
	}

	fclose(file);
}

void plot_generate_allocated()
{
	FILE *file;
	int shift;

	if( !(file = fopen("plot/allocated.txt", "w+")) )
	{
		abort();
	}

	for(shift = 1; shift < 8; shift++)
	{
		long a = 1<<shift;

		eval_wrapper_allocated(file, 15*16*a, 15*9*a);
		eval_wrapper_allocated(file, 16*16*a, 15*9*a);
		eval_wrapper_allocated(file, (3*15*16*a)>>1, (3*15*9*a)>>1);
		eval_wrapper_allocated(file, (3*16*16*a)>>1, (3*15*9*a)>>1);
	}

	fclose(file);
}

void plot_annotate_allocated()
{
	for(int shift = 3; shift < 7; shift++)
	{
		long a = 1<<shift;

		annotate_wrapper_allocated("digital television", 15*16*a, 15*9*a);
		annotate_wrapper_allocated("digital cinema",     16*16*a, 15*9*a);
	}
}
