#ifndef IMAGEPTR_H
#define IMAGEPTR_H

#include "system.h"
#include "vec2.h"
#include "stride2.h"
#include <assert.h>
#include <stdint.h>

/**
 * @brief A pointer to image data.
 */
struct imageptr_t {
	/** pointer to the beginning of data area */
	void *data;

	/** steps in X and Y dimensions */
	struct stride2_t stride;

	/** pointer to the beginning of mapped data / the end of discarded data */
	void *valid;

	// TODO: add 'size'
};

/**
 * @brief Create (allocate) the image.
 *
 * Every pixel at the x=0 coordinate is aligned at a suitable boundary.
 *
 * @param[in] size the size of image
 */
struct imageptr_t *imageptr_create(
	const struct vec2_t size
);

/**
 * @brief Create (allocate) the image.
 *
 * Every pixel at the x=0 coordinate is aligned at a suitable boundary.
 *
 * @param[in] size the size of image
 */
struct imageptr_t *imageptr_create_normal(
	const struct vec2_t size
);

/**
 * @brief Create (allocate) the image using @c mmap.
 *
 * Every pixel at the x=0 coordinate is aligned at a suitable boundary.
 *
 * @param[in] size the size of image
 */
struct imageptr_t *imageptr_create_sparse(
	const struct vec2_t size
);

/**
 * @brief Create (allocate) the image using @c mmap and huge pages.
 *
 * Every pixel at the x=0 coordinate is aligned at a suitable boundary.
 *
 * @param[in] size the size of image
 */
struct imageptr_t *imageptr_create_sparse_huge(
	struct vec2_t size
);

/**
 * @brief Create (allocate) the image mapped into memory window using @c mmap.
 *
 * Every pixel at the x=0 coordinate is aligned at a suitable boundary.
 * The lines are congruent modulo @e lines.
 *
 * @warning Expects the tmpfs to be mounted in /dev/shm.
 *
 * @param[in] size the size of image
 * @param[in] lines wrap the window every so many lines
 */
struct imageptr_t *imageptr_create_window(
	const struct vec2_t size,
	int lines
);

/**
 * @brief Create (allocate) the image mapped into memory window using @c mmap and huge pages.
 *
 * Every pixel at the x=0 coordinate is aligned at a suitable boundary.
 * The lines are congruent modulo @e lines.
 *
 * @param[in] size the size of image
 * @param[in] lines wrap the window every so many lines
 */
struct imageptr_t *imageptr_create_window_huge(
	const struct vec2_t size,
	int lines
);

/**
 * @brief Create the image for LL subband.
 */
struct imageptr_t *imageptr_create_llband(
	const struct vec2_t size,
	const struct vec2_t cb_size1
);

/**
 * @brief Destroy (free) the image.
 */
void imageptr_destroy(
	struct imageptr_t *imageptr,
	const struct vec2_t size
);

/**
 * @brief Destroy (free) the image.
 */
void imageptr_destroy_normal(
	struct imageptr_t *imageptr,
	const struct vec2_t size
);

/**
 * @brief Destroy (free) the image using @c munmap.
 *
 * @param[in] size the size of image
 */
void imageptr_destroy_sparse(
	struct imageptr_t *imageptr,
	const struct vec2_t size
);

/**
 * @brief Destroy (free) the image using @c munmap.
 *
 * @param[in] size the size of image
 */
void imageptr_destroy_sparse_huge(
	struct imageptr_t *imageptr,
	const struct vec2_t size
);

/**
 * @brief Destroy (free) the image allocated by @ref imageptr_create_wrapped.
 *
 * @param[in] size the size of image
 * @param[in] lines window size in lines
 */
void imageptr_destroy_window(
	struct imageptr_t *imageptr,
	const struct vec2_t size,
	int lines
);

/**
 * @brief Destroy (free) the image allocated by @ref imageptr_create_wrapped.
 *
 * @param[in] size the size of image
 * @param[in] lines window size in lines
 */
void imageptr_destroy_window_huge(
	struct imageptr_t *imageptr,
	const struct vec2_t size,
	int lines
);

/**
 * @brief Destroy the image for LL subband.
 */
void imageptr_destroy_llband(
	struct imageptr_t *imageptr,
	const struct vec2_t size,
	const struct vec2_t cb_size1
);

/**
 * @brief Partially discard (free) the image data using @c munmap.
 *
 * @param[in] size the coordinate up to that discard the data
 */
void imageptr_discard(
	struct imageptr_t *imageptr,
	const struct vec2_t size
);

/**
 * @brief Partially discard (free) the image data using @c munmap.
 *
 * @param[in] size the coordinate up to that discard the data
 */
void imageptr_discard_sparse(
	struct imageptr_t *imageptr,
	const struct vec2_t size
);

/**
 * @brief Partially discard (free) the image data using @c munmap.
 *
 * @param[in] size the coordinate up to that discard the data
 */
void imageptr_discard_sparse_huge(
	struct imageptr_t *imageptr,
	const struct vec2_t size
);

/**
 * @brief Pointer to an image pixel.
 *
 * @param[in] offset coordinates the pixel
 *
 * @return pointer to an image pixel
 */
UNUSED_FUNCTION
static
float *imageptr_pixel(
	const struct imageptr_t *imageptr, // NOTE: the structure itself is const, however, the data are not
	const struct vec2_t offset
)
{
	assert( imageptr );
	assert( imageptr->data );

	return (void *)( (intptr_t)imageptr->data + offset.x * imageptr->stride.x + offset.y * imageptr->stride.y );
}

UNUSED_FUNCTION
static
float *imageptr_pixel_offset_0_0(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 0 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
float *imageptr_pixel_offset_0_1(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 1 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
float *imageptr_pixel_offset_0_2(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 2 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
float *imageptr_pixel_offset_0_3(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 3 * imageptr->stride.y );
}

/**
 * @brief Pointer to a constant image pixel.
 *
 * @param[in] offset coordinates of the pixel
 *
 * @return pointer to a constant image pixel
 */
UNUSED_FUNCTION
static
const float *imageptr_pixel_const(
	const struct imageptr_t *imageptr,
	const struct vec2_t offset
)
{
	assert( imageptr );
	assert( imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + offset.x * imageptr->stride.x + offset.y * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_0_0_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 0 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_0_1_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 1 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_0_2_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 2 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_0_3_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 3 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_0_4_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 4 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_0_5_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 5 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_0_6_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 6 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_0_7_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 0 * imageptr->stride.x + 7 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_4_0_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 4 * imageptr->stride.x + 0 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_4_1_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 4 * imageptr->stride.x + 1 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_4_2_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 4 * imageptr->stride.x + 2 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_4_3_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 4 * imageptr->stride.x + 3 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_4_4_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 4 * imageptr->stride.x + 4 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_4_5_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 4 * imageptr->stride.x + 5 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_4_6_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 4 * imageptr->stride.x + 6 * imageptr->stride.y );
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_offset_4_7_const(
	const struct imageptr_t *imageptr
)
{
	assert( imageptr && imageptr->data );

	return (const void *)( (intptr_t)imageptr->data + 4 * imageptr->stride.x + 7 * imageptr->stride.y );
}

/**
 * @brief Access the image pixel through the symmetric periodic extension.
 *
 * @param[in] data_offset an user-specific data offset; se to zero if @c imageptr refers to the beginning of the tile
 * @param[in] coord the global coordinates (not an offset)
 * @param[in] tc0 the coordinates of the upper left hand sample (inclusive) of the tile-component
 * @param[in] tc1 the coordinates of the lower right hand sample (exclusive) of the tile-component
 *
 * @return pointer to a constant image pixel
 *
 * @note this function is not very efficient
 */
UNUSED_FUNCTION
static
const float *imageptr_pixel_ext(
	const struct imageptr_t *imageptr,
	const struct vec2_t data_offset,
	const struct vec2_t coord,
	const struct vec2_t tc0,
	const struct vec2_t tc1
)
{
	// symmetric periodic extension
	struct vec2_t offset = vec2_coord_extr(
		vec2_sub(coord, tc0),
		vec2_sub(tc1, tc0)
	);

	// add the data offset
	offset = vec2_sub(offset, data_offset);

	// direct access
	return imageptr_pixel_const(imageptr, offset);
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_ext_limited(
	const struct imageptr_t *imageptr,
	const struct vec2_t data_offset,
	const struct vec2_t coord,
	const struct vec2_t tc0,
	const struct vec2_t tc1
)
{
	// symmetric periodic extension
	struct vec2_t offset = vec2_coord_extr_limited(
		vec2_sub(coord, tc0),
		vec2_sub(tc1, tc0)
	);

	// add the data offset
	offset = vec2_sub(offset, data_offset);

	// direct access
	return imageptr_pixel_const(imageptr, offset);
}

UNUSED_FUNCTION
static
const float *imageptr_pixel_ext_limited2(
	const struct imageptr_t *imageptr,
	const struct vec2_t local,
	const struct vec2_t size
)
{
	// symmetric periodic extension
	const struct vec2_t offset = vec2_coord_extr_limited(
		local,
		size
	);

	// direct access
	return imageptr_pixel_const(imageptr, offset);
}

/**
 * @brief Fill the whole image with a test pattern.
 *
 * @param[in] size the size of image
 */
void imageptr_fill(
	struct imageptr_t *imageptr,
	const struct vec2_t size
);

/**
 * @brief Fill the image strip with a test pattern.
 *
 * @param[in] data_offset offset of the strip beginning
 * @param[in] data_size the size of the strip
 */
void imageptr_fill_strip(
	struct imageptr_t *imageptr,
	const struct vec2_t data_offset,
	const struct vec2_t data_size
);

/**
 * @brief Store the image into PGM-file. Useful for the original image.
 *
 * @param[in] size the size of the image
 * @param[in] path file path
 *
 * @note http://netpbm.sourceforge.net/doc/pgm.html
 */
void imageptr_dump(
	struct imageptr_t *imageptr,
	const struct vec2_t size,
	const char *path
);

/**
 * @brief Store a logarithm of the image into PGM-file. Useful for the wavelet transform.
 *
 * @param[in] size the size of the image
 * @param[in] path file path
 */
void imageptr_log_dump(
	struct imageptr_t *imageptr,
	const struct vec2_t size,
	const char *path
);

/**
 * @brief Store a logarithm of the image into PGM-file. Useful for the wavelet transform.
 *
 * @param[in] size the size of the image
 * @param[in] path file path
 */
void imageptr_log_dump_fmt(
	struct imageptr_t *imageptr,
	const struct vec2_t size,
	const char *path,
	...
);

/**
 * @brief Get pointer to an image viewport specified by an offset.
 *
 * @param[in] offset offset of the viewport
 */
struct imageptr_t *imageptr_viewport(
	struct imageptr_t *imageptr,
	const struct vec2_t offset
);

/**
 * @brief Store a logarithm of the image viewport into PGM-file. Useful for the wavelet transform.
 *
 * @param[in] offset offset of the viewport
 * @param[in] size the size of the image viewport
 * @param[in] path file path
 */
void imageptr_log_dump_viewport_fmt(
	struct imageptr_t *imageptr,
	const struct vec2_t offset,
	const struct vec2_t size,
	const char *path,
	...
);

#endif
