#include "timer.h"
#include "vec2.h"
#include "imageptr.h"
#include "range2.h"
#include "transform.h"
#include "system.h"
#include "plot.h"
#include <stdio.h>
#include <math.h>

int main()
{
	system_setup_debug();

	const int N = 8;

	const struct vec2_t cb_exp = { 6, 6 };
	const struct vec2_t tc0 = { 5, 128 };
// 	const struct vec2_t size = { 1920, 1080 };
// 	const struct vec2_t size = { 4096, 2160 };
	const struct vec2_t size = { 7680, 4320 };
// 	const struct vec2_t size = { 1<<16, 1<<16 };
// 	const struct vec2_t size = { 1000000, 1000000 };
// 	const struct vec2_t size = { 4920, 12840 };
	const struct vec2_t tc1 = vec2_add(tc0, size);

	const int MAX = 1000;

	double min_time = +INFINITY;

	for(int i = 0; i < MAX; i++)
	{
		struct imageptr_t *imageptr = imageptr_create_sparse(size);

		if( !imageptr )
			return -1;

		struct transform_t *transform = transform_create(
			tc0,
			tc1,
			cb_exp,
			1,
			N,
			NULL,
			NULL
		);

		if( !transform )
			return -1;

		int64_t clocks = 0;

		while( !transform_finished(transform) )
		{
			const struct range2_t lines = transform_prepare_strip(transform);

			// pointing to i0-th row of the tile
			const struct vec2_t data_offset = { 0, lines.i0 };
			const struct vec2_t data_size = { size.x, lines.i1 - lines.i0 };

			dprintf("REQUIRED: %i lines starting from %i (local coordinates in the input image), i.e. y [%i..%i]\n", data_size.y, data_offset.y, lines.i0, lines.i1 - 1);

			struct imageptr_t *data_lines = imageptr_viewport(imageptr, data_offset);

			imageptr_fill_strip(
				data_lines,
				data_offset,
				data_size
			);

			free(data_lines);

			clocks -= timer_get_clock();

			transform_process_strip(
				transform,
				imageptr,
				vec2_zero
			);

			clocks += timer_get_clock();

			// discard already processed data
			imageptr_discard_sparse(
				imageptr,
				vec2_create( 0, lines.i1 )
			);
		}

		double time = timer_clock_to_nsec_pel(clocks, (long)size.x * size.y);

		if( time < min_time )
			min_time = time;

		printf("(streaming) pass %4i/%i: elapsed %f nsecs/pel in total (minimum %f nsecs/pel)\n", i, MAX, time, min_time);

		transform_destroy(transform);

		imageptr_destroy_sparse(imageptr, size);
	}

	return 0;
}
