function detector = lbpModify( detector, varargin )
% Modify aggregate channel features object detector.
%
% Takes an object detector trained by acfTrain() and modifies it. Only
% certain modifications are allowed to the detector and the detector should
% never be modified directly (this may cause the detector to be invalid and
% cause segmentation faults). Any valid modification to a detector after it
% is trained should be performed using acfModify().
%
% The parameters 'nPerOct', 'nOctUp', 'nApprox', 'lambdas', 'pad', 'minDs'
% modify the channel feature pyramid created (see help of chnsPyramid.m for
% more details) and primarily control the scales used. The parameters
% 'pNms', 'stride', 'cascThr' and 'cascCal' modify the detector behavior
% (see help of acfTrain.m for more details). Finally, 'rescale' can be
% used to rescale the trained detector (this change is irreversible).
%
% USAGE
%  detector = acfModify( detector, pModify )
%
% INPUTS
%  detector   - detector trained via acfTrain
%  pModify    - parameters (struct or name/value pairs)
%   .nPerOct    - [] number of scales per octave
%   .nOctUp     - [] number of upsampled octaves to compute
%   .nApprox    - [] number of approx. scales to use
%   .lambdas    - [] coefficients for power law scaling (see BMVC10)
%   .pad        - [] amount to pad channels (along T/B and L/R)
%   .minDs      - [] minimum image size for channel computation
%   .pNms       - [] params for non-maximal suppression (see bbNms.m)
%   .stride     - [] spatial stride between detection windows
%   .cascThr    - [] constant cascade threshold (affects speed/accuracy)
%   .cascCal    - [] cascade calibration (affects speed/accuracy)
%   .stages     - [] Set number of staes
%
% OUTPUTS
%  detector   - modified object detector
%
% EXAMPLE
%
% See also chnsPyramid, bbNms, acfTrain, acfDetect
%
% Piotr's Computer Vision Matlab Toolbox      Version 3.20
% Copyright 2014 Piotr Dollar.  [pdollar-at-gmail.com]
% Licensed under the Simplified BSD License [see external/bsd.txt]
%
% Modified by Roman Juranek [ijuranek-at-fit.vutbr.cz]
% - Removed rescale
% - Added stages

% get parameters (and copy to detector and pPyramid structs)
opts=detector.opts; p=opts.pPyramid;
dfs={ 'nPerOct',p.nPerOct, 'nOctUp',p.nOctUp, 'nApprox',p.nApprox, ...
  'lambdas',p.lambdas, 'pad',p.pad, 'minDs',p.minDs, 'pNms',opts.pNms, ...
  'stride',opts.stride,'cascThr',opts.cascThr,'cascCal',0,'stages',inf };
[p.nPerOct,p.nOctUp,p.nApprox,p.lambdas,p.pad,p.minDs,opts.pNms,...
  opts.stride,opts.cascThr,cascCal,stages] = getPrmDflt(varargin,dfs,1);

% finalize pPyramid and opts
p.complete=0; p.pChns.complete=0; p=chnsPyramid([],p); p=p.pPyramid;
p.complete=1; p.pChns.complete=1; shrink=p.pChns.shrink;
opts.stride=max(1,round(opts.stride/shrink))*shrink;
opts.pPyramid=p; detector.opts=opts;

detector.clf.hs = detector.clf.hs+cascCal;

% Truncate detector
if stages < size(detector.clf.ftr,2)
    detector.clf.thr = detector.clf.thr(1:stages);
    detector.clf.hs = detector.clf.hs(:,1:stages);
    detector.clf.ftr = detector.clf.ftr(:,1:stages);
end

end

