/*
 * API for object detection in images. This is common to both MATLAB and C++
 * versions of detector.
 *
 * LBP Detector Toolbox
 * Roman Juranek <ijuranek@fit.vutbr.cz>
 * Faculty of Information Technology, BUT, Brno
 *
 */

#ifndef _DETECTOR_H_
#define _DETECTOR_H_

#include <vector>
#include <array>
#include "common.h"

namespace LBPDetector {

// Detector structure
// T - Number of features
// w,h - Dimensions
// ftr - LBP features
// alpha - Responses
// thr - Termination thresholds
struct Detector
{
    // Initialized from file
    std::string label {""};
    int T {1};
    int wdim0 {1};
    int wdim1 {1};
    unsigned char * ftr {0};
    float * alpha {0};
    float cascThr {-1};
    bool dynamic {true};
    typedef int (*FtrEvalFunc)(const Image&,int,int,int,int,int);
    // Initialized from initDetector()
    std::vector <FtrEvalFunc> FtrEvaluator;
    std::vector <int> CachePlane;
    int numCachePlanes {1};
};

// Parameters of feature channels
struct ChnsParams
{
    int shrink {1};

    // Color channel
    enum ColorSpace {CS_GRAY, CS_RGB, CS_LUV};
    ColorSpace colorSpace {CS_GRAY};
    int colorSmooth {0};

    // TODO: GradMag, GradHist

    // Pyramid
    int pyrSmooth {0};
    int nPerOct {6};
    int nOctUp {0};
    std::vector<float> lambdas; // Not used now
};

// Representation of bounding box
struct BB
{
    float x, y, width, height, scale;
    BB(float _x, float _y, float _w, float _h, float _s):
        x(_x), y(_y), width(_w), height(_h), scale(_s){};
};

// Structure with object detection result - list of detected boundig boxes,
// list of responses, and numbers of evaluated samples and features.
struct ScalarResult
{
    std::vector<BB> bbs;
    std::vector<float> hs;
    unsigned long long nf {0};
    unsigned long long ns {0};
    float cacheEffeciency {0.0f};
};

// Create new detector form ptrs to data.
Detector * createDetector(int sz0, int sz1, int T, unsigned char * ftrs, float * hs, float cascThr);

// Initialize detector structure
bool initDetector(Detector&);

// Free detector from memory.
void destroyDetector(Detector**);

// Scan an image area with detector
ScalarResult scalarScanImage(const Image&, const int, const int, const int, const int, const int, const Detector&, const float, bool);

} // namespace

#endif
