/*
 * Stand-alone tool for object detection in a set of images.
 * Each image generate one line JSON object. The structure is following:
 *
 *  {
 *      "fileName": string,
 *      "width": int,
 *      "height": int,
 *      "time": float,
 *      "nWins": int,
 *      "nFtrs": int,
 *      "cacheEffeciency": float,
 *      "bbs": [
 *          [x,y,w,h,score],
 *          ...
 *      ]
 *  }
 *
 * Usage:
 *  lbp-detector --help
 *
 * LBP Detector Toolbox
 * Roman Juranek <ijuranek@fit.vutbr.cz>
 * Faculty of Information Technology, BUT, Brno
 *
 * TODO:
 * * -nocache option. This will need support in lbpdetector module
 * * Add support for detector label
 *   * Detector structure - Detector, DetectorProto
 *   * JSON output
 */

#include <vector>
#include <list>
#include <cmath>
#include <chrono>
#include <cstdio>
#include <iostream>
#include <fstream>
#include <opencv2/opencv.hpp>
#include <argtable2.h>
#include <json/json.h>
#include "lbpdetector_cv.h"
#include "detector.h"

using namespace std;
using namespace cv;
using namespace LBPDetector;

struct Options{
    float thr {0.0f};
    int stride {1};
    int nPerOct {-1};
    int nOctUp {-1};
    unsigned int nThreads {1};
    int tile {1000};
    bool useCache {true};
    bool nonms {false};
    int min_group {1};
    float resize_x {1.0f};
    float resize_y {1.0f};
    float group_thr {0.5f};
    bool show {false};
    bool save {false};
    bool nopause {false};
    string detector {""};
    list<string> files {};
    int err {0};
} opts;

///////////////////////////////////////////////////////////////////////////////

Options processArguments(int argc, char ** argv)
{
    Options opts; // opts with default values

    // Process arguments
    const char * bin = "lbpdetector";
    arg_file * files = arg_filen(NULL, NULL, "FILES", 0, argc-1, "Input file(s)");
    arg_file * filelist = arg_filen("l", NULL, "<FILE>", 0, 1, "File list");
    arg_file * detector = arg_file1("d", NULL, "<FILE>", "Detector");
    arg_dbl * threshold = arg_dbln("t", "threshold", "f", 0, 1, "Detection threshold (0)");
    arg_lit * nonms = arg_lit0(NULL, "nonms", "Do not perform non-maxima suppression");
    arg_int * grp_count = arg_intn(NULL, "min-group", "n", 0, 1, "Minimum detections in one group (1)");
    arg_dbl * grp_thr = arg_dbln(NULL, "group-thr", "n", 0, 1, "Overlap of detections in a group (0.5)");
    arg_int * nPerOct = arg_intn(NULL, "nPerOct", "n", 0, 1, "Number of levels per scale octave (6)");
    arg_int * nOctUp = arg_intn(NULL, "nOctUp", "n", 0, 1, "Upscale factor (1)");
    arg_int * stride = arg_intn(NULL, "stride", "n", 0, 1, "Spatial step of detector (in shrinked image) (1)");
    arg_lit * nocache = arg_lit0(NULL, "nocache", "Do not use feature cache");
    arg_dbl * resize = arg_dbln("s", "scale", "ratio", 0, 1, "Detecion window scale factor");
    arg_dbl * resize_x = arg_dbln(NULL, "sx", "ratio", 0, 1, "Horizontal detection window scale factor");
    arg_dbl * resize_y = arg_dbln(NULL, "sy", "ratio", 0, 1, "Vertical detection window scale factor");
    arg_int * threads = arg_intn(NULL, "threads", "n", 0, 1, "Number of threads to use (1)");
    arg_int * tilesz = arg_intn(NULL, "tile", "n", 0, 1, "Tile size for threaded");
    arg_lit * show = arg_lit0(NULL, "show", "Show detection results on screen");
    arg_lit * save = arg_lit0(NULL, "save", "Save images with detected objects");
    arg_lit * nopause = arg_lit0(NULL, "nopause", "Dont pause when showing image output");
    arg_lit * help = arg_lit0("h", "help", "Display this help and exit");
    arg_str * otype = arg_strn("o", "output-type", "<TYPE>", 0, 1, "Output type - JSON or CSV (default)");
    struct arg_end * end = arg_end(20);
    void *argtable[] = {detector, threshold, nonms, grp_count, grp_thr, nPerOct, nOctUp, nocache, resize, resize_x, resize_y, stride, filelist, otype, threads, tilesz, show, save, nopause, help, files, end};
    opts.err = arg_parse(argc, argv, argtable);

    // Check for any errors in agrs
    if(help->count > 0)
    {
        fprintf(stderr, "Usage: %s", bin);
        arg_print_syntax(stderr, argtable, "\n\n");
        arg_print_glossary(stderr, argtable, "  %-30s %s\n");
        arg_freetable(argtable, sizeof(argtable)/sizeof(argtable[0]));
        opts.err = 1;
        return opts;
    }
    if (opts.err > 0)
    {
        arg_print_errors(stderr, end, bin);
        fprintf(stderr, "Try '%s --help' for more information.\n", bin);
        arg_freetable(argtable, sizeof(argtable)/sizeof(argtable[0]));
        return opts;
    }

    if (threshold->count>0) opts.thr = threshold->dval[0];
    if (stride->count > 0) opts.stride = min(max(stride->ival[0], 1), 8);
    if (nPerOct->count > 0) opts.nPerOct = min(max(nPerOct->ival[0], 1), 32);
    if (nOctUp->count > 0) opts.nOctUp = nOctUp->ival[0];
    if (tilesz->count > 0) opts.tile = tilesz->ival[0];
    if (threads->count > 0) opts.nThreads = threads->ival[0];
    if (nonms->count > 0) opts.nonms = true;
    if (grp_count->count > 0) opts.min_group = grp_count->ival[0];
    if (grp_thr->count > 0) opts.group_thr = grp_thr->dval[0];
    if (show->count > 0) opts.show = true;
    if (nopause->count > 0) opts.nopause = true;
    if (save->count > 0) opts.save = true;
    if (nocache->count > 0) opts.useCache = false;
    if (grp_count->count > 0) opts.min_group = grp_count->ival[0];
    if (resize->count > 0)
    {
        opts.resize_x = resize->dval[0];
        opts.resize_y = resize->dval[0];
    }
    else
    {
        if (resize_x->count > 0) opts.resize_x = resize_x->dval[0];
        if (resize_y->count > 0) opts.resize_y = resize_y->dval[0];
    }
    opts.detector = detector->filename[0];
    if (filelist->count > 0)
    {
        ifstream f(filelist->filename[0]);
        string fn;
        while (f >> fn) opts.files.emplace_back(fn);
    }
    for (int i = 0; i < files->count; i++)
    {
        opts.files.emplace_back(files->filename[i]);
    }

    opts.tile = ceil(opts.tile / opts.stride) * opts.stride;

    if (opts.files.empty())
    {
        fprintf(stderr, "%s: No input files\n", bin);
        arg_freetable(argtable, sizeof(argtable)/sizeof(argtable[0]));
        opts.err = 1;
        return opts;
    }

    arg_freetable(argtable, sizeof(argtable)/sizeof(argtable[0]));

    return opts;
}

int main(int argc, char ** argv)
{
    Options opts = processArguments(argc, argv);
    if (opts.err > 0) return 1;

    // Load detector
    Detector * D = loadDetectorFromFile(opts.detector);
    ChnsParams C = loadChnsParamsFromFile(opts.detector);
    if (!D)
    {
        cerr << "Cannot load detector " << opts.detector << endl;
        return 0;
    }

    // Setup params in C according to opts
    if (opts.nPerOct >= 0) C.nPerOct = opts.nPerOct;
    if (opts.nOctUp >= 0) C.nOctUp = opts.nOctUp;

    // Init JSON writer
    Json::FastWriter writer;

    // Go through files
    size_t i = 0;
    for (auto filename = opts.files.begin(); filename != opts.files.end(); ++filename, ++i)
    {
        // Load image
        Mat image_rgb = imread(*filename);

        // Check if loaded successfully
        if (!image_rgb.data)
        {
            cerr << "Warning: cannot load " << *filename << endl;
            continue;
        }

        // Time measurement start
        std::chrono::time_point<std::chrono::system_clock> tm_start, tm_end;
        tm_start = std::chrono::system_clock::now();

        ////////////////////////////////
        // Actual detection happens HERE
        Detections dets = lbpDetectImage(image_rgb, D, C, opts.thr, opts.stride,
            opts.nonms ? 0 : opts.min_group, opts.group_thr, opts.resize_x, opts.resize_y,
            opts.nThreads, opts.tile, opts.useCache);
        ////////////////////////////////

        // Point for measurement stop
        tm_end = std::chrono::system_clock::now();
        std::chrono::duration<double,std::micro> tm_elapsed = tm_end - tm_start;

        // Output results
        Json::Value lst;
        if (!D->label.empty()) lst["label"] = D->label;
        lst["fileName"] = *filename;
        lst["time"] = tm_elapsed.count();
        lst["width"] = image_rgb.cols;
        lst["height"] = image_rgb.rows;
        lst["nFtrs"] = dets.nf;
        lst["nWins"] = dets.ns;
        lst["cacheEffeciency"] = dets.cacheEffeciency;
        Json::Value objs(Json::arrayValue);
        for (size_t j = 0; j < dets.bbs.size(); ++j)
        {
            cv::Rect & bb = dets.bbs[j]; float h = dets.hs[j];
            Json::Value bb1(Json::arrayValue);
            bb1.append(bb.x);
            bb1.append(bb.y);
            bb1.append(bb.width);
            bb1.append(bb.height);
            bb1.append(h);
            objs.append(bb1);
        }
        lst["bbs"] = objs;
        cout << writer.write(lst);

        // Preview and/or save image
        if (opts.show || opts.save)
        {
            float resolution = 1200;
            cv::Size image_sz = image_rgb.size();
            float f = (max(image_sz.width, image_sz.height) <= resolution) ? 1.f : resolution/max(image_sz.width, image_sz.height);
            if (f < 1) resize(image_rgb, image_rgb, Size(f*image_sz.width, f*image_sz.height));
            for (auto bb:dets.bbs)
            {
                if (opts.nonms)
                {
                    rectangle(image_rgb, Point(f*bb.x,f*bb.y), Point(f*(bb.x+bb.width), f*(bb.y+bb.height)),Scalar(255,255,255),1);
                }
                else
                {
                    rectangle(image_rgb, Point(f*bb.x,f*bb.y), Point(f*(bb.x+bb.width), f*(bb.y+bb.height)),Scalar(0,255,0),3);
                    rectangle(image_rgb, Point(f*bb.x,f*bb.y), Point(f*(bb.x+bb.width), f*(bb.y+bb.height)),0,2);
                }
            }
            
            if (opts.save)
            {
                imwrite(basename(filename->c_str()), image_rgb);
            }

            if (opts.show)
            {
                imshow("im", image_rgb);
                if (opts.nopause) waitKey(5);
                else waitKey();
            }
        } // Preview and save
    } // File loop

    destroyDetector(&D);

    // Done
    return 0;
}
