#!/usr/bin/env python3
# Copyright (C) 2015 Barbora Frankova
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

########################################################################
# SDN-tap was developed to simplify data capturing in SDN. It installs #
# OpenFlow rules to duplicate, mark and forward flows based on IP      #
# addresses, 3-tuples (src IP address, dest IP address, protocol) and  #
# 5-tuples (src IP address, src port, dest IP address, dest port,      #
# protocol).                                                           #
# Usage:  ./SDN-tap                             start daemon           #
#         ./insertIntercept 10.0.0.1            add intercept          #
#         ./DeleteIntercept 10.0.0.1            delete intercept       #
# See README for further details.                                      #
########################################################################

import os
import time
import sys
import requests
import getopt
import json
import subprocess
import networkx as nx
import re
from threading import Thread
sys.path.append(os.path.join(os.path.dirname(__file__)))
import Node
import Flow
import socket
from Flows import *
from Topology import *
from datetime import datetime
from networkx.algorithms import isomorphism
import matplotlib.pyplot as plt
from networkx.readwrite import json_graph

#sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..'))
#from modules.sockets.li import ptptAsClient

LOGIN = "admin"
PASSWD = "admin"
URL = "http://localhost"
PORT = "8181"

ACTIVE_INTERCEPTS = []
DEBUG = False

def saveGraphFile(G, switches, hosts, probes):
    # save graph to DOT file
    f = open('topo', 'w')
    f.write("graph G{\n")
    f.write("node [shape=none, fontsize=10, fixedsize=true, splines=polyline, width=.6, height=.6];\n")
    for node in switches:
        f.write("\"" + node.switchId + "\" [color=none, image=\"switch.png\", label=<<br/><br/><br/>" + node.switchId + ">];\n")
    for host in hosts:
        f.write("\"" + host.hostId + "\" [color=none, image=\"host.png\", label=<<br/><br/><br/>" + host.ipAddress + ">];\n")
    for probe in probes:
        f.write("\"" + probe.probeId + "\" [color=none, image=\"sonda.png\", label=<<br/><br/><br/>" + probe.probeId + ">];\n")
    for line in nx.generate_edgelist(G):
        node1, node2, rest = line.split(" ", 2)
        f.write("\"" + node1 + "\" -- \"" + node2 + "\"")
        # weights
        rest = rest.strip("{\'weight\': ")
        rest = rest.strip("}")
        if rest == "5":
            f.write("[style = bold]")
        if rest == "1000":
            f.write("[style = dotted]")
        f.write(";\n")
    f.write("overlap=false\n}")
    f.close()
    FNULL = open(os.devnull, 'w')
    subprocess.call(["neato", "-Gsplines=polyline" , "-Tsvg", "topo", "-o", "topo.svg"], stdout=FNULL, stderr=subprocess.STDOUT)
    
    f = open('topo.json', 'w')
    f.write(json.dumps(json_graph.node_link_data(G)))
    f.close()
    

def getProbePosition(G, switches, hosts):
    ''' Get probe position from config file, update topology graph and list of interfaces'''
    PROBE_LIST = []
    global DEBUG
    if DEBUG:
        print("getting probes...")
    try:
        f = open('ProbeConfig', 'r')
    except:
        print("Failed to open ProbeConfig.")
        return()
    # open configuration file
    for line in f:
        switch2 = None
        interface2 = None
        #print line,
        # skip comments
        if '#' not in line:
            if 'probe' in line:
                line = line.strip('\n')
                if line.count(' ') == 2:
                    probeId, switch1, vlanTag = line.split(' ')
                else:
                    probeId, switch1, switch2, vlanTag = line.split(' ')
                    switch2a, switch2b, interface2 = switch2.split(':')
                    switch2 = switch2a + ":" + switch2b
                switch1a, switch1b, interface1 = switch1.split(':')
                switch1 = switch1a + ":" + switch1b
                
                # probe is connected to switch which is not in topology!!
                if G.has_node(switch1) is False:
                    return
                
                # add probe to list
                probe = Node.Probe(probeId, int(vlanTag), switch1, interface1, switch2, interface2)
                PROBE_LIST.append(probe)
                # add probe to graph
                G.add_node(probe.probeId)
                #if switch2 is None:
                #    G.add_edge(probe.probeId, switch1, weight=1)
                #else:
                #    G.remove_edge(switch1, switch2)
                #    G.add_edge(probe.probeId, switch1, weight=1)
                #    G.add_edge(probe.probeId, switch2, weight=1)
                # update interface dict
                for switch in switches:
                    if switch.switchId == switch1:
                        intId = 0
                        for neigh in switch.interfaces:
                            if switch.interfaces[neigh] == interface1:
                                intId = neigh
                                for host in hosts:
                                    if host.ipAddress == neigh:
                                        hosts.remove(host)
                                        nextNodes = G.neighbors(host.hostId)
                                        #print(nextNodes)
                                        for nextNode in nextNodes:
                                            G.add_edge(nextNode, probe.probeId)
                                            G[nextNode][probe.probeId]['weight'] = 1
                                        #if nextNode is not None:
                                        #    G.add_edge(probe.probeId, nextNode)
                                        G.remove_node(host.hostId)
                        if intId != 0:
                            switch.interfaces[probe.probeId] = interface1
                            switch.interfaces.pop(intId, None)
                        break
                if switch2 != None:
                    for switch in switches:
                        if switch.switchId == switch2:
                            switch.interfaces[probeId] = interface2
                            break
            elif 'weight' in line:
                line = line.strip('\n')
                switch1 = line.split(' ', 4)[1]
                switch2 = line.split(' ', 4)[2]
                weight = line.split(' ', 4)[3]
                G[switch1][switch2]['weight'] = int(weight)
    f.close()
    #for edge in nx.generate_edgelist(G):
    #    print(edge)
    return(PROBE_LIST)
    #for switch in switches:
    #    print(switch.interfaces)
    if DEBUG:
        print("done")

def getInterceptsPerProbe(switches, probes):
    global DEBUG
    if DEBUG:
        print("getting intercepts per probe")
    # reset counter
    for probe in probes:
        probe.numOfIntercepts = 0
    # get intercepts from flows
    for switch in switches:
        for flow in switch.flows[0]:
            if 'dest-ip' in switch.flows[0][flow].match or 'src-ip' in switch.flows[0][flow].match:
                #print(switch.flows[0][flow].actions)
                if 'push:vlan' in switch.flows[0][flow].actions:
                    vlanTag = switch.flows[0][flow].actions.split(',', 1)[0]
                    vlanTag = vlanTag.strip("push:vlan")
                    probe = findProbeInListByVlan(vlanTag, probes)
                    if probe is not None:
                        probe.numOfIntercepts = probe.numOfIntercepts + 1
                elif 'outport:' in switch.flows[0][flow].actions and 'go-to:table2' in switch.flows[0][flow].actions:
                    outport = switch.flows[0][flow].actions.split(',', 1)[0]
                    outport = outport.strip("outport:")
                    for interface in switch.interfaces:
                        if switch.interfaces[interface] == outport:
                            probe = findProbeInList(interface, probes)
                            if probe is not None:
                                probe.numOfIntercepts = probe.numOfIntercepts + 1
    #for probe in probes:
    #    print(probe)
    if DEBUG:
        print("done")
    
def getStats(switches, probes):
    for switch in switches:
        print(switch)
        for flow in switch.flows[0]:
            print(switch.flows[0][flow])
        print("---")
        for flow in switch.flows[1]:
            print(switch.flows[1][flow])
        print('-------------------')
    
    for probe in probes:
        print(probe)
    
def findHostInHosts(ipAddress, hosts):
    for host in hosts:
        if host.ipAddress == ipAddress:
            return(host)
    return(None)

def findSwitchInList(switchId, switches):
    for switch in switches:
        if switch.switchId == switchId:
            return(switch)
    return(None)

def findProbeInList(probeId, probes):
    for probe in probes:
        if probe.probeId == probeId:
            return(probe)
    return(None)

def findProbeInListByVlan(vlanTag, probes):
    for probe in probes:
        #print(probe)
        if int(probe.vlanTag) == int(vlanTag):
            return(probe)
    return(None)

def setInitFlow(url, switches):
    ''' Insert initial flows in table 0.
        Rules:  match: [*]          actions: [go-to:table2]
                match: [type:vlan]  actions: [go-to:table1]'''
    table = 0
    flowId = 1
    for switch in switches:
        createInitFlow(url, switch.switchId, table, flowId)
        flow1 = Flow.Flow(flowId, switch.switchId, table, "2", "*", "go-to:table2")
        createVlanGotoTableFlow(url, switch.switchId, table, flowId + 1)
        flow2 = Flow.Flow(flowId + 1, switch.switchId, table, "10", "type:vlan", "go-to:table1")
        switch.flows[table][flowId] = flow1
        switch.flows[table][flowId + 1] = flow2
    print("\tinitial flow rules added")

def setVlanForwardingFlow(url, G, switches, PROBE_LIST):
    global DEBUG
    ''' Insert flow forwarding rules in table 1.
        Rules:                          match:[vlan:1]   actions:[outport:X]
        On switch closest to probe:     match:[vlan:1]   actions: [pop:vlan1,outport:X]'''
    #nx.draw(G)
    #plt.show()
    #edgeList = nx.generate_edgelist(G)
    #for edge in edgeList:
    #    print(edge)
    for probe in PROBE_LIST:
        for switch in switches:
            #print(switch.switchId)
            try:
                path = nx.shortest_path(G, switch.switchId, probe.probeId)
                #print(path)
                #print(switch.interfaces[path[1]])
                # switch is directly connected to probe, add forwarding flow to table 1
                if len(path) == 2:
                    setVlanTagRemoveFlow(url, switch.switchId, 1, probe.vlanTag, probe.vlanTag, switch.interfaces[probe.probeId])
                    flow = Flow.Flow(probe.vlanTag, switch.switchId,  1, '10', "vlan:vlan" + str(probe.vlanTag), "pop:vlan" + str(probe.vlanTag) + ",outport:" + str(switch.interfaces[probe.probeId]))
                # switch is not directly connected to probe, add forwarding flow to table 1
                else:
                    createVlanForwardFlow(url, switch.switchId, 1, probe.vlanTag, probe.vlanTag, switch.interfaces[path[1]])
                    flow = Flow.Flow(probe.vlanTag, switch.switchId, 1, '10', "vlan:vlan" + str(probe.vlanTag), "outport:" + str(switch.interfaces[path[1]]))
                switch.flows[1][probe.vlanTag] = flow
            except:
                print("No path found between " + switch.switchId + " and " + probe.probeId)
    #for switch in switches:
    #    print(switch.flows[1])
    if DEBUG:
        print("\tVLAN forwarding flows updated (table 1)")

def renewVlanForwardingFlows(url, G, switches, PROBE_LIST):
    global DEBUG
    deleteFlows = []
    currentSwitches = []
    checkedProbes = []
    if DEBUG:
        print("updating VLAN forwarding flows...")
    for switch in switches:
        #print(switch.switchId)
        for flow in switch.flows[1]:
            if switch.flows[1][flow].match is not None:
                #print(switch.flows[1][flow].match)
                if "vlan:vlan" in switch.flows[1][flow].match:
                    probe = findProbeInListByVlan(switch.flows[1][flow].match.strip("vlan:vlan"), PROBE_LIST)
                    # probe not active
                    if probe is not None and probe not in checkedProbes:
                        checkedProbes.append(probe)
                    if probe is None:
                        deleteFlow(url, switch.switchId, 1, switch.flows[1][flow].Id)
                        deleteFlows.append(switch.flows[1][flow].Id)
                    else:
                        try:
                            path = nx.shortest_path(G, switch.switchId, probe.probeId)
                            # interface changed (switch down or something)
                            if switch.interfaces[path[1]] not in switch.flows[1][flow].actions:
                                deleteFlow(url, switch.switchId, 1, switch.flows[1][flow].Id)
                                setVlanForwardingFlow(url, G, [switch,], [probe,])
                        except:
                            pass
        for key in deleteFlows:
            switch.flows[1].pop(key, None)
    
    # install rules for new probes new probes    
    for probe in PROBE_LIST:
        if probe not in checkedProbes:
            setVlanForwardingFlow(url, G, switches, [probe,])
    #for switch in switches:
    #    print(switch.flows)
    if DEBUG:
        print("done")

def setInterceptFlow(url, probe, switch, intercept):
    ''' Insert match intercept + set vlan tag rule in table 0
        Rule:   match:[ip:X.X.X.X]  actions:[push:vlan1,outport:X,pop:vlan1,go-to:table2]
                match:[ip:X.X.X.X]  actions:[outport:X,go-to:table2]'''
    global ACTIVE_INTERCEPTS
    #print(intercept)
    #print(probe.probeId)
    table = 0
    pushVlan = True
    flowId = switch.getNextFlowId(table)
    outport = -1
    # get vlan tag of current probe
    vlan = probe.vlanTag
    #print(intercept[0])
    #print(switch.switchId)
    #print(probe.probeId)
    # find outport to selected probe
    for key in switch.flows[1]:
        #print(switch.flows[1][key])
        if ("vlan:vlan" + str(vlan)) in switch.flows[1][key].match:
            #print(switch.flows[1][key].actions)
            if switch.flows[1][key].actions.find("outport:") != -1:
                outport = switch.flows[1][key].actions[switch.flows[1][key].actions.find("outport:") + len("outport:"):]
                #print(outport)
            if switch.flows[1][key].actions.find("pop:vlan" + str(vlan)) != -1:
                pushVlan = False
    # failed to get output interface
    if outport is -1:
        print("Outport not set. Check flows in table 1.")
        return()
    # probe is directly connected to this switch -> no need to push vlan tag, only to forward traffic
    if pushVlan:
        actions = "push:vlan" + str(vlan) + ",outport:" + str(outport) + ",pop:vlan" + str(vlan) + ",go-to:table2"
    else:
        actions = "outport:" + str(outport) + ",go-to:table2"
    #print(actions)
    # add intercept
    if len(intercept) is 1 or len(intercept) is 3:
        # match destination IP address
        createInterceptFlow(url, switch.switchId, table, flowId, intercept, actions, vlan, outport, "destination")
        if(len(intercept) is 1):
            flow = Flow.Flow(flowId, switch.switchId, table, "5", "dest-ip:" + intercept[0], actions)
        else:
            flow = Flow.Flow(flowId, switch.switchId, table, "5", "dest-ip:" + intercept[0] + ",port:" + intercept[1] + ",proto:" + intercept[2], actions)
        switch.flows[table][flowId] = flow
        probe.numOfIntercepts = probe.numOfIntercepts + 1
        
        # match source IP address
        flowId = switch.getNextFlowId(table)
        createInterceptFlow(url, switch.switchId, table, flowId, intercept, actions, vlan, outport, "source")
        if(len(intercept) is 1):
            flow = Flow.Flow(flowId, switch.switchId, table, "5", "src-ip:" + intercept[0], actions)
        else:
            flow = Flow.Flow(flowId, switch.switchId, table, "5", "src-ip:" + intercept[0] + ",port:" + intercept[1] + ",proto:" + intercept[2], actions)
        switch.flows[table][flowId] = flow
        probe.numOfIntercepts = probe.numOfIntercepts + 1
    else:
        # match source and destination IP address
        flowId = switch.getNextFlowId(table)
        createInterceptFlow(url, switch.switchId, table, flowId, intercept, actions, vlan, outport, "")
        flow = Flow.Flow(flowId, switch.switchId, table, "5", "src-ip:" + intercept[0] + ",dest-ip:" + intercept[1] + ",src-port:" + intercept[2] + ",dest-port:" + intercept[3] +  ",proto:" + intercept[4], actions)
        switch.flows[table][flowId] = flow
        probe.numOfIntercepts = probe.numOfIntercepts + 1
    ACTIVE_INTERCEPTS.append(intercept)
    #print(ACTIVE_INTERCEPTS)
    #print("-------> SET PROBE:" + str(vlan) + ", " + str(intercept))

def deleteInterceptFlow(url, switches, intercept):
    global ACTIVE_INTERCEPTS
    global DEBUG
    deleteFlows = []
    if len(intercept) == 1:
        pattern = intercept[0]
    elif len(intercept) == 3:
        pattern = intercept[0] + ",port:" + intercept[1] + ",proto:" + intercept[2]
    else:
        # dest-ip:192.168.1.91,src-ip:192.168.1.92,src-port:80,dest-port:2568,proto:TCP
        pattern = "dest-ip:" + intercept[0] + ",src-ip:" + intercept[1] + ",src-port:" + intercept[2] + ",dest-port:" + intercept[3] + ",proto:" + intercept[4]

    #print(pattern)
    #print("------>find intercept id to delete")
    for switch in switches:
        #print(switch.switchId)
        for flow in switch.flows[0]:
            if switch.flows[0][flow].match is not None:
                #print(switch.flows[0][flow].match)
                if pattern in switch.flows[0][flow].match:
                    currentFlow = []
                    deleteFlow(url, switch.switchId, 0, switch.flows[0][flow].Id)
                    currentFlow.append(switch.flows[0][flow].Id)
                    #print("deleting")
                    #print(switch.flows[0][flow].actions)
                    # intercepting host at switch which is directly connected to probe
                    if switch.flows[0][flow].actions.count(',') is 1:
                        action1, action2 = switch.flows[0][flow].actions.split(',')
                        outport = action1.strip("outport:")
                        for interface in switch.interfaces:
                            if switch.interfaces[interface] is outport:
                                vlanTag = interface.strip("probe:")
                                #print(vlanTag)
                    else:
                        action1, action2, action3, action4 = switch.flows[0][flow].actions.split(',')
                        vlanTag = action1.strip("push:vlan")
                    currentFlow.append(vlanTag)
                    deleteFlows.append(currentFlow)
        for item in deleteFlows:
            switch.flows[0].pop(item[0], None)
            # TODO: delete intercept from probe
        deleteFlows = []
    #if intercept in ACTIVE_INTERCEPTS: ACTIVE_INTERCEPTS.remove(intercept)
    #print(ACTIVE_INTERCEPTS)
    if DEBUG:
        print("-------> UNSET PROBE:" + vlanTag + ", " + str(intercept))
    return(vlanTag)
    #print("-------> UNSET PROBE:" + vlanTag + ", " + str(intercept))

def checkIsomorphism(G, H):
    ''' Check if two graphs are different '''
    if(H == None or G == None):
        return(True)
    GM = isomorphism.GraphMatcher(G, H)
    if GM.is_isomorphic():
        return(False)
    else:
        return(True)

def findShortestPathForIP(G, hosts, PROBE_LIST, node):
    ''' Find shortest paths in network graph between probe and host (IP, port, protocol) '''
    found = False
    if hosts is None or node is None:
        return(None, None)
    # find matching node ID
    for currentHost in hosts:
        if currentHost.ipAddress == node:
            found = True
            break
    if found is False:
        print("IP address " + node + " not found.")
        return(None, None)
    # check if host is in topology
    if currentHost.hostId not in G.nodes():
        print("IP address " + node + " not in topology graph.")
        return(None, None)
    # find all shortest paths
    allProbePaths = []
    for probe in PROBE_LIST:
        if nx.has_path(G, currentHost.hostId, probe.probeId):
            paths = nx.all_shortest_paths(G, currentHost.hostId, probe.probeId, weight="weight")
            allProbePaths.append(paths)
    
    if len(allProbePaths) is 0:
        print("No path from " + node + " to probe was found.")
        return(None, None)
    
    minLen = 5000
    minIntercepts = 5000
    for probePath in allProbePaths:
        for path in probePath:
            #print(path)
            probe = findProbeInList(path[-1], PROBE_LIST)
            # find shortest path to probe
            pathLen = 0
            for i in range(0, len(path) - 1):
                pathLen = pathLen + int(G[path[i]][path[i+1]]['weight'])
            # hosts on same switch as probe will allwasy get this probe
            if(pathLen == 2):
                minLen = pathLen
                minIntercepts = probe.numOfIntercepts
                closeSwitchId = path[1]
                closeProbeId = path[-1]
                break
            #print(pathLen)
            # load-balancing based on number of intercepts
            #print("path len: " + str(pathLen))
            #print("min intercepts: " + str(minIntercepts))
            #print(probe.probeId + " " + str(probe.numOfIntercepts))
            #print("  -> " + str(minIntercepts / probe.numOfIntercepts))
            #print(str(minIntercepts - probe.numOfIntercepts) + " >= " + str(minIntercepts / 2))
            # when possible, use shortest weighted path
            if pathLen < minLen:
                if minIntercepts == 0:
                    minIntercepts = 1
                if (probe.numOfIntercepts / minIntercepts < 5) or (minLen > 1000 and minLen < 5000):
                    minLen = pathLen
                    minIntercepts = probe.numOfIntercepts
                    closeSwitchId = path[1]
                    closeProbeId = path[-1]
                else:
                    pass
            # in case of multiple shortest paths, select the one with less active intercepts
            if pathLen == minLen:
                # TODO: check
                if probe.numOfIntercepts < minIntercepts:
                    minIntercepts = probe.numOfIntercepts
                    closeSwitchId = path[1]
                    closeProbeId = path[-1]
            if pathLen > minLen:
                if probe.numOfIntercepts == 0:
                    currProbeIntercepts = 1
                else:
                    currProbeIntercepts = probe.numOfIntercepts
                #print("-> ", minIntercepts / currProbeIntercepts)
                if minIntercepts / currProbeIntercepts > 5 and pathLen < 1000:
                    minLen = pathLen
                    minIntercepts = probe.numOfIntercepts
                    closeSwitchId = path[1]
                    closeProbeId = path[-1]
                else:
                    pass
            #print('------')
    
    return(closeSwitchId, closeProbeId)

def setInitialFlowRules(url, switches):
    global DEBUG
    for switch in switches:
        if 1 in switch.flows[0] and "go-to:table2" in switch.flows[0][1].actions:
            if DEBUG:
                print("match: *, actions: go-to:table2 already present")
        if 2 in switch.flows[0] and "go-to:table1" in switch.flows[0][2].actions:
            if DEBUG:
                print("match: type:vlan, actions: go-to:table1 already present")
        else:
            if 1 in switch.flows[0]:
                deleteFlow(url, switch.switchId, 0, 1)
                switch.flows[0].pop(1, None)
            if 2 in switch.flows[0]:
                deleteFlow(url, switch.switchId, 0, 2)
                switch.flows[0].pop(2, None)
            setInitFlow(url, switches)

def newIntercept(G, url, switches, hosts, PROBE_LIST, intercept):
    #for switch in switches:
    #    print(switch)
    #for host in hosts:
    #    print(host)
    #nx.draw(G)
    #plt.show()
    #e = nx.generate_edgelist(G)
    #for item in e:
    #    print(item)
    if hosts is None:
        return(0)
    # find shortest path from source address
    closestSwitchId, closeProbeId = findShortestPathForIP(G, hosts, PROBE_LIST, intercept[0])
    switch = findSwitchInList(closestSwitchId, switches)
    probe = findProbeInList(closeProbeId, PROBE_LIST)
    # need vlan number to set for intercept rule
    if switch is not None and probe is not None:
        setInterceptFlow(url, probe, switch, intercept)
    else:
        return(0)
    return([probe.probeId, probe.vlanTag])

def getAllActiveIntercepts(switches):
    global DEBUG
    flows = []
    ongoingIntercepts = []
    usedProbes = []
    outports = []
    usedSwitches = []
    if DEBUG:
        print("gathering active intercepts")
    
    for switch in switches:
        for key in switch.flows[0]:
            intercept = []
            vlanTag = 0
            outport = 0
            if switch.flows[0][key].match is not None:
                # intercept targeted only on IP address
                if ',' not in switch.flows[0][key].match and ("src-ip" in switch.flows[0][key].match or "dest-ip" in switch.flows[0][key].match):
                    desc, match = switch.flows[0][key].match.split(':')
                    intercept.append(match)
                    if 'push:vlan' in switch.flows[0][key].actions:
                        vlanTag = switch.flows[0][key].actions.split(',', 1)[0]
                        vlanTag = vlanTag.strip("push:vlan")
                        outport = switch.flows[0][key].actions.split(',', 2)[1]
                        outport = outport.strip("outport:")
                    else:
                        outport = switch.flows[0][key].actions
                        outport = outport.strip("outport:").split(',', 1)[0]
                        outport = outport.strip("outport:")
                # intercept is IP address, port, protocol
                if switch.flows[0][key].match.count(',') == 2 and ("src-ip" in switch.flows[0][key].match or "dest-ip" in switch.flows[0][key].match):
                    ip, port, proto = switch.flows[0][key].match.split(',')
                    ip = ip.split(':')[1]
                    port = port.split(':')[1]
                    proto = proto.split(':')[1]
                    intercept.append(ip)
                    intercept.append(port)
                    intercept.append(proto)
                    if 'push:vlan' in switch.flows[0][key].actions:
                        vlanTag = switch.flows[0][key].actions.split(',', 1)[0]
                        vlanTag = vlanTag.strip("push:vlan")
                        outport = switch.flows[0][key].actions.split(',', 2)[1]
                        outport = outport.strip("outport:")
                    else:
                        outport = switch.flows[0][key].actions
                        outport = outport.strip("outport:").split(',', 1)[0]
                        outport = outport.strip("outport:")
                if "src-ip" in switch.flows[0][key].match and "dest-ip" in switch.flows[0][key].match:
                    ip1, ip2, port1, port2, proto = switch.flows[0][key].match.split(',')
                    ip1 = ip1.split(':')[1]
                    ip2 = ip2.split(':')[1]
                    port1 = port1.split(':')[1]
                    port2 = port2.split(':')[1]
                    proto = proto.split(':')[1]
                    intercept.append(ip1)
                    intercept.append(ip2)
                    intercept.append(port1)
                    intercept.append(port2)
                    intercept.append(proto)
                    if 'push:vlan' in switch.flows[0][key].actions:
                        vlanTag = switch.flows[0][key].actions.split(',', 1)[0]
                        vlanTag = vlanTag.strip("push:vlan")
                        outport = switch.flows[0][key].actions.split(',', 2)[1]
                        outport = outport.strip("outport:")
                    else:
                        outport = switch.flows[0][key].actions
                        outport = outport.strip("outport:").split(',', 1)[0]
                        outport = outport.strip("outport:")
                if len(intercept) > 0 and intercept not in ongoingIntercepts:
                    #print("----")
                    #print(intercept)
                    #print(vlanTag)
                    #print(outport)
                    ongoingIntercepts.append(intercept)
                    usedProbes.append(vlanTag)
                    outports.append(outport)
                    usedSwitches.append(switch)
                if len(intercept) > 0:
                    flows.append(switch.flows[0][key])
    if DEBUG:
        print("done")
    return(flows, ongoingIntercepts, usedProbes, outports, usedSwitches)

def createInterceptMsg(intercept):
    if len(intercept) == 1:
        return(intercept[0])
    elif len(intercept) == 3:
        if intercept[2] == "TCP":
            return("tcp3:(" + intercept[0] + "_" + intercept[1] + ")")
        else:
            return("udp3:(" + intercept[0] + "_" + intercept[1] + ")")
    else:
        if intercept[4] == "TCP":
            return("tcp:(" + intercept[0] + ", " + intercept[2] + ", " + intercept[1] + ", " + intercept[3] + ")")
        else:
            return("udp:(" + intercept[0] + ", " + intercept[2] + ", " + intercept[1] + ", " + intercept[3] + ")")

def setProbe(probeId1, probeId2, intercept):
    #s = ptptAsClient("odltrigger")
    #s.send(str(probeId1) + "-" + str(probeId2) + "-" + createInterceptMsg(intercept))
    return()

def renewInterceptFlows(G, url, switches, PROBE_LIST, hosts):
    global ACTIVE_INTERCEPTS
    global DEBUG
    if DEBUG:
        print("updating flow rules for VLAN tagging (table 0)...")
    updatedIntercepts = []
    flows, ongoingIntercepts, usedProbes, outports, usedSwitches = getAllActiveIntercepts(switches)
    #for i in xrange(0, len(ongoingIntercepts)):
    #    print(ongoingIntercepts[i]),
    #    print(usedProbes[i]),
    #    print(outports[i])
    #for flow in flows:
    #    print(flow)
    #for probe in PROBE_LIST:
    #    print(probe)
    #for switch in switches:
    #    print(switch)

    i = 0
    # check if host with active intercept is up again
    for intercept in ongoingIntercepts:
        needToRenew = False
        #print(intercept)
        #print(ACTIVE_INTERCEPTS)
        if intercept not in ACTIVE_INTERCEPTS:
            #print("not in active")
            if len(intercept) == 1 or len(intercept) == 3:
                try:
                    deleteFlow(URL + ":" + PORT + '/restconf/config/opendaylight-inventory:nodes/node', usedSwitches[i].switchId, 0, flows[i].Id)
                    deleteFlow(URL + ":" + PORT + '/restconf/config/opendaylight-inventory:nodes/node', usedSwitches[i].switchId, 0, flows[i + 1].Id)
                    usedSwitches[i].flows[0].pop(int(flows[i].Id), None)
                    usedSwitches[i].flows[0].pop(int(flows[i + 1].Id), None)
                except:
                    pass
            else:
                deleteFlow(URL + ":" + PORT + '/restconf/config/opendaylight-inventory:nodes/node', usedSwitches[i].switchId, 0, flows[i].Id)
                usedSwitches[i].flows[0].pop(int(flows[i].Id), None)
                i = i + 1
                continue

        outport = -1
        host = findHostInHosts(intercept[0], hosts)
        attachementSwitch = host.attachementSwitch
        attachementSwitch = findSwitchInList(attachementSwitch, switches)
        #print(attachementSwitch.flows[0])
        # host connected to switch with probe and interface exists -> no change
        if usedProbes[i] == 0 and outports[i] in attachementSwitch.interfaces.values():
            print("host connected directly to probe, no change: "),
            print(intercept)
            i = i + 1
            continue
        probe = findProbeInListByVlan(usedProbes[i], PROBE_LIST)
        if probe is None:
            needToRenew = True
        else:
            #print(probe)
            #print(attachementSwitch)
            #print(host)
            #print('---')
            #print(ongoingIntercepts[i]),
            #print(usedProbes[i]),
            #print(outports[i])
            if 'src-ip:' in flows[i].match:
                ipAddress = flows[i].match.split(',', 1)[0]
                ipAddress = ipAddress.strip("src-ip:")
            if 'dest-ip:' in flows[i].match:
                ipAddress = flows[i].match.split(',', 1)[0]
                ipAddress = ipAddress.strip("dest-ip:")
            closestSwitchId, closestProbeId = findShortestPathForIP(G, hosts, PROBE_LIST, ipAddress)
            print(closestSwitchId)
            print(closestProbeId)
            
            if nx.has_path(G, host.hostId, attachementSwitch.switchId):
                #print("has path")
                #print(attachementSwitch.flows[0])
                for key in attachementSwitch.flows[1]:
                    if ("vlan:vlan" + str(probe.vlanTag)) in attachementSwitch.flows[1][key].match:
                        #print(attachementSwitch.flows[1][key].actions)
                        if attachementSwitch.flows[1][key].actions.find("outport:") != -1:
                            outport = attachementSwitch.flows[1][key].actions[attachementSwitch.flows[1][key].actions.find("outport:") + len("outport:"):]
                            #print(outport)
                if outports[i] != outport:
                    needToRenew = True

            if nx.has_path(G, host.hostId, probe.probeId) == False:
                needToRenew = True
        # update flow if needed
        if needToRenew is False:
            if DEBUG:
                print("no change: "),
                print(intercept)
        else:
            if DEBUG:
                print("need to update rule - "),
                print(intercept)
            if len(intercept) == 1 or len(intercept) == 3:
                deleteFlow(URL + ":" + PORT + '/restconf/config/opendaylight-inventory:nodes/node', attachementSwitch.switchId, 0, flows[i].Id)
                deleteFlow(URL + ":" + PORT + '/restconf/config/opendaylight-inventory:nodes/node', attachementSwitch.switchId, 0, flows[i + 1].Id)
            else:
                deleteFlow(URL + ":" + PORT + '/restconf/config/opendaylight-inventory:nodes/node', attachementSwitch.switchId, 0, flows[i].Id)
            if usedProbes[i] == 0:
                if DEBUG:
                    print("-------> TRY TO UNSET PROBE:" + str(usedProbes[i]) + ", " + str(intercept))
            else:
                if DEBUG:
                    print("-------> UNSET PROBE:" + str(usedProbes[i]) + ", " + str(intercept))
            print("call MF&CCTF to unset probe " + str(usedProbes[i]) + ", intercept: " + str(intercept))
            #unsetProbe(usedProbes[i], intercept)
            attachementSwitch.flows[0].pop(int(flows[i].Id), None)
            if len(intercept)  != 5:
                attachementSwitch.flows[0].pop(int(flows[i + 1].Id), None)
            #print("---")
            #print(attachementSwitch.flows[0])
            #print("---")
            #print("current port: " + str(outports[i]) + ", new port: " + str(outport) + ", current vlan: " + str(usedProbes[i]) + ", new vlan: ?")
            #deleteInterceptFlow(url, switches, intercept)

            interceptStatus = newIntercept(G, url, switches, hosts, PROBE_LIST, intercept)
            #print(attachementSwitch.switchId + ", " + flows[i].Id)
            
            if interceptStatus is 0:
                if DEBUG:
                    print("intercept not updated!")
            else:
                if DEBUG:
                    print("SET PROBE " + str(interceptStatus[0]) + ", VLAN: " + str(interceptStatus[1]))
                print("call MF&CCTF to set probe " + str(interceptStatus[0]) + ", intercept: " + str(intercept))
                setProbe(usedProbes[i], interceptStatus[0], intercept)
        i = i + 1
        #print(attachementSwitch.flows[0])
    if DEBUG:
        print("done")

def odl_daemon_socket():
    global ACTIVE_INTERCEPTS
    if os.path.exists( "/tmp/odl_deamon" ):
        os.remove( "/tmp/odl_deamon" )
    server = socket.socket( socket.AF_UNIX, socket.SOCK_DGRAM )
    server.bind("/tmp/odl_deamon")
 
    while True:
        datagram = server.recv( 1024 )
        if not datagram:
            break
        else:
            intercept = []
            action, data = datagram.decode('ascii').split(':')
            #print(action)
            #print(data)
            if ',' not in data:
                intercept.append(data)
            elif data.count(',') is 2:
                ip, port, proto = data.split(',')
                intercept.append(ip)
                intercept.append(port)
                intercept.append(proto)
            else:
                ip1, port1, ip2, port2, proto = data.split(',')
                intercept.append(ip1)
                intercept.append(port1)
                intercept.append(ip2)
                intercept.append(port2)
                if proto == 'tcp':
                    proto = 'TCP'
                else:
                    proto = 'UDP'
                intercept.append(proto)
            if action == 'insert':
                ACTIVE_INTERCEPTS.append(intercept)
            else:
                if intercept in ACTIVE_INTERCEPTS:
                    ACTIVE_INTERCEPTS.remove(intercept)
            #print(ACTIVE_INTERCEPTS)

def odl_daemon():
    """ Main body of the program. """
    global URL, PORT, DEBUG
    restconf_topo_url = URL + ":" + PORT + '/restconf/operational/network-topology:network-topology/topology/flow:1'
    restconf_flow_url = URL + ":" + PORT + '/restconf/operational/opendaylight-inventory:nodes/node'
    restconf_modify_flow_url = URL + ":" + PORT + '/restconf/config/opendaylight-inventory:nodes/node'
    prevProbes = []
    while True:
        try:
            # initialization
            G, switches, hosts = getTopology(restconf_topo_url)
            # get spanning tree and edge weights
            getLoopFreeTopo(G, restconf_flow_url, switches)
            # get probe position from configuration file
            probes = getProbePosition(G, switches, hosts)
            #for switch in switches:
            #    print(switch.interfaces)
            if DEBUG:
                print(probes)
            #print(prevProbes)
            if probes is None or len(probes) == 0:
                print("No probe found.")
            else:
                # recover last graph from file
                try:
                    H = json_graph.node_link_graph(json.load(open("topo.json")))
                except:
                    H = None
                topologyChanged = checkIsomorphism(G, H)
                if topologyChanged == True or probes != prevProbes:
                    print("Topology changed, start updating rules")
                    getFlows(restconf_flow_url, switches)
                    getInterceptsPerProbe(switches, probes)
                    setInitialFlowRules(restconf_modify_flow_url, switches)
                    # first run
                    if H is None:
                        setVlanForwardingFlow(restconf_modify_flow_url, G, switches, probes)
                    else:
                        renewVlanForwardingFlows(restconf_modify_flow_url, G, switches, probes)
                    renewInterceptFlows(G, restconf_modify_flow_url, switches, probes, hosts)

            # save graph (svg, json)
            saveGraphFile(G, switches, hosts, probes)
            prevProbes = probes
            #print("OK")
            time.sleep(5)
        except:
            e = sys.exc_info()[0]
            print("FAIL, try again in 5 secs")
            print(e)
            time.sleep(5)

def run():
    thread1 = Thread(target=odl_daemon_socket, args=())
    thread2 = Thread(target=odl_daemon, args=())
    thread1.start()
    thread2.start()
    thread1.join()
    thread2.join()

def main():
    thread1 = Thread(target=odl_daemon_socket, args=())
    thread2 = Thread(target=odl_daemon, args=())
    thread1.start()
    thread2.start()
    thread1.join()
    thread2.join()

if __name__ == "__main__":
    main()

