# Class describing CIN table for IRI-CORE
#
# Copyright (C) 2013 Radek Hranicky
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

class CINTable(object):
    def __init__(self):
        self.cinTable = set()
        
    def update(self, nidcc, liid, cin):
        """ Adds a new record or updates an existing one """
        recExists = False
        for record in self.cinTable:
            if record.getNIDCC() == nidcc and record.getLIID() == liid:
                recExists = True
                record.setCIN(cin)
        if not recExists:
            newRecord = CINTableRecord(nidcc, liid, cin)
            self.cinTable.add(newRecord)
            
    def delete(self, nidcc, liid):
        """ Deletes a record with given NIDCC and LIID """
        for record in self.cinTable:
            if record.getNIDCC() == nidcc and record.getLIID() == liid:
                self.cinTable.discard(record)
                break
                
    def deleteWithLIID(self, liid):
        """ Deletes all records with given LIID """
        self.cinTable = {record for record in self.cinTable if record.getLIID() != liid}
                
    def getCIN(self, nidcc, liid):
        """ Returns CIN of a record with given NIDCC and LIID """
        for record in self.cinTable:
            if record.getNIDCC() == nidcc and record.getLIID() == liid:
                return record.getCIN()
        return None
                
    def recordExists(self, nidcc, liid):
        for record in self.cinTable:
            if record.getNIDCC() == nidcc and record.getLIID() == liid:
                return True
        return False
        
    def debug_printCinTable(self):
        """ Prints the content of the CIN table """
        print("CIN TABLE CONTENT")
        print("===========================================================================")
        print("NIDCC\t\t | LIID\t\t\t\t | CIN")
        print("---------------------------------------------------------------------------")
        for record in self.cinTable:
            print(record.getNIDCC(), "\t |", record.getLIID(), "\t\t |", record.getCIN())

class CINTableRecord(object):
    def __init__(self, nidcc, liid, cin):
        self.__nidcc = nidcc
        self.__liid = liid
        self.__cin = cin
    
    def getNIDCC(self):
        """ Returns NIDcc value of the record """
        return self.__nidcc
        
    def getLIID(self):
        """ Returns LIID value of the record """
        return self.__liid
        
    def getCIN(self):
        """ Returns CIN value of the record """
        return self.__cin
        
    def setCIN(self, newCin):
        """ Sets a new CIN value of the record """
        self.__cin = newCin
