# -*- coding: utf-8 -*-
# CC table for CCTF and CC-IIF
# Copyright (C) 2011 Matěj Grégr, Michal Kajan, Libor Polčák, Vladimír Veselý
# 
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from .cc_table_base import CCTableBase
from .liexceptions import LIException

from IPy import IP

class CCTable(CCTableBase):

    def __init__(self, content):
        """ Extends """
        CCTableBase.__init__(self, content)
        self.__sid_nid = {}
        self.__nid_sid = {}
        for sid, nid in content:
            self.add(sid, nid)

    def getContent(self):
        """ Overrides """
        content = []
        for sid, nid in self.__sid_nid.items():
            content.append((sid, nid))
        return content

    def add(self, sid, nid):
        """ Overrides """
        if sid in self.__sid_nid:
            if self.__sid_nid[sid] == nid:
                return False
            else:
                raise LIException("Trying to register NID %s but SID %s already registered with NID %s" % (nid, sid, self.__sid_nid[sid]))
        if nid in self.__nid_sid:
            raise LIException("Trying to register NID %s with SID %s but it was already registered with SID %s" % (nid, sid, self.__nid_sid[nid]))
        self.__sid_nid[sid] = nid
        self.__nid_sid[nid] = sid
        return True

    def findSID(self, nid):
        """ Overrides """
        
        if not nid: # Non-IP packet
            return None
        sid = None
        nidF = IP("::/0")             # Initiali se to biggest possible subnet
        captureEverything_sid = None  # Matching SID for interception to ::/0 if there is any
        for n, s in self.__nid_sid.items():
            if nid in n:
                # TODO - check if the behavior is correct!
                if (nid.getType() == "IPv6" and n.getType() == "IPv6") \
                or (nid.getType() == "IPv4" and n.getType() == "IPv4"):
                    v = n.getValue()
                    if nidF > v:
                        nidF = v
                        sid = s
                    if (n == nidF):
                        captureEverything_sid = s
                else:
                    sid = s
                    if (n.getType() == "NIDTCP"):
                        break
        if (sid == 0 and captureEverything_sid != None):
            sid = captureEverything_sid
        return sid

    def findNID(self, sid):
        """ Overrides """
        try:
            return self.__sid_nid[sid]
        except KeyError: # unknown SID
            return None

    def remove(self, sid):
        """ Removes sid from the table """
        try:
            nid = self.__sid_nid[sid]
            del self.__sid_nid[sid]
            del self.__nid_sid[nid]
        except KeyError:
            pass # Attempt to remove non-existing entry
            
    def debug_print(self):
        print("================== CC Table: SID-->NID ==================")
        print(self.__sid_nid)
        print("================== CC Table: NID-->SID ==================")
        print(self.__nid_sid)
        
    def empty(self):
        if len(self.__nid_sid) == 0:
            return True
        else:
            return False

    def __repr__(self):
        return "CCTable(%s)" % str(self.getContent())
