#include <stdio.h>
#include <algorithm>
#include <vector>
#include "../UberLame_src/Vector.h"
#include "../UberLame_src/gl3/Transform.h"
#include "../UberLame_src/gles2/GLES20Emu.h"
#include "../UberLame_src/gles2/Shader.h"
#include "../UberLame_src/gles2/BufferObjects.h"
#include "../UberLame_src/gles2/FrameBuffer.h"
#include "../UberLame_src/Timer.h"
#include "../UberLame_src/Dir.h"
#include "../UberLame_src/ExpEval.h"
#include "../UberLame_src/iface/PNGLoad.h"
#include "../UberLame_src/Thread.h"
typedef double GLdouble;
#ifdef _MAC
#include <GLUT/glut.h>
#else // _MAC
#include <GL/glut.h>
#endif // _MAC
#include "SkyDome.h"
#include "Scene.h"
#include "Grid.h"

#define MAINDEMO

static int n_width = 720, n_height = 480;
static Vector3f v_camera_pos(3, 2, -5);
static float f_angle_x = -.2f, f_angle_y = .5f, f_angle_z = 0;
//#define __GL_WINDOW_TRACKBALL_STYLE_CAMERA
// camera

#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER >= 1400
#define sscanf sscanf_s
#define fscanf fscanf_s
#endif //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
// get rid of annoying warnings

#include "Shaders.h"

#include "../UberLame_src/lml/PlatPrim.h"
#include "../UberLame_src/lml/GeoSphere.h"
#include "../UberLame_src/lml/TexCoords.h"
#include "../UberLame_src/lml/BooleanOps.h"
#include "../UberLame_src/lml/KochanekBartels.h"

#include "Deco.h"

/**
 *	@brief called by GLUT when the window changes size, and also when it is created
 *
 *	@param[in] n_new_width is the new width of the window, in pixels
 *	@param[in] n_new_height is the new height of the window, in pixels
 */
void onResize(int n_new_width, int n_new_height)
{
	n_width = n_new_width;
	n_height = n_new_height;
	glViewport(0, 0, n_new_width, n_new_height);
}

static const double f_range_sigma00 = 10;
static const double p_dt_sigma[21] = {
	1, 0, 0,   0,   0,   0,
	   1, 0,   0,   0,   0,
	      1,   0,   0,   0,
		     100,   0,   0,
		          100,   0,
			           100
};

#ifndef GL_LINE_SMOOTH // GL ES does not have it
#define GL_LINE_SMOOTH												0x0B20
#endif

static bool b_inhibit_texture_referesh = false;

class CDemo {
protected:
	CSimpleShader m_simple_shader;
	CBuoyShader m_buoy_shader;
	CColorShader m_color_shader;
	CPointShader m_point_shader;
	CGLSkyDome m_sky_dome;
	CGLIslandScene m_island_scene;
	CInfiniteGridGeometry m_inf_grid;
	CWaterShader m_reflection_shader;
	CRefractionShader m_refraction_shader;
	CBuoyObject m_buoy;
	CSubObject m_diver;
	bool m_b_first_frame;
	CTimer m_timer;
	CGUIWidget m_gui;

	std::vector<Vector3f> m_buoy_list;

	CGLTexture_2D *m_p_refraction_tex;
	CGLTexture_2D *m_p_reflection_tex;

	CRecordedTrajectory trajectory;

	CKochanekBartelsQuatSpline<Vector3f> m_camera_spline, m_spline; // m_spline is a copy of m_camera_spline, except that it has the first and last control point duplicated
	bool m_b_animate;
	double m_f_anim_poses_offset;

public:
	CDemo()
		:m_b_first_frame(true), m_p_refraction_tex(0), m_p_reflection_tex(0), m_gui(Vector2f(-.9f, -1), .5f),
		trajectory("data/spiral_long"), m_b_animate(false)
	{
		m_simple_shader.Compile();
		m_color_shader.Compile();
		m_point_shader.Compile();
		m_buoy_shader.Compile();
		m_reflection_shader.Compile();
		m_refraction_shader.Compile();

		m_buoy_list.push_back(Vector3f(-1, 0, -1));
		m_buoy_list.push_back(Vector3f( 1, 0, -1));
		m_buoy_list.push_back(Vector3f( 1, 0,  1));
		m_buoy_list.push_back(Vector3f(-1, 0,  1));
		m_buoy_list.push_back(Vector3f(0, -1,  0));
		// list of buoys

		m_gui.Set_Title("test\nlat lon 123 456");

		double f_time = 0;
	}

	void onMouseClick(int n_button, int n_state, Vector2f v_position)
	{}

	void onMouseMotion(int n_button, Vector2f v_position)
	{}

	void onPushSplinePose()
	{
		double f_time = m_timer.f_Time();

		if(m_camera_spline.b_Empty())
			m_f_anim_poses_offset = f_time;

		Matrix4f t_camera_matrix;
		t_camera_matrix.Identity();
#ifdef __GL_WINDOW_TRACKBALL_STYLE_CAMERA
		t_camera_matrix.Translate(v_camera_pos);
#endif // __GL_WINDOW_TRACKBALL_STYLE_CAMERA
		t_camera_matrix.RotateZ(f_angle_z);
		t_camera_matrix.RotateX(f_angle_x);
		t_camera_matrix.RotateY(f_angle_y);
#ifndef __GL_WINDOW_TRACKBALL_STYLE_CAMERA
		t_camera_matrix.Translate(v_camera_pos);
#endif // !__GL_WINDOW_TRACKBALL_STYLE_CAMERA
		// calculate final camera matrix (with pitch and roll)

		CKochanekBartelsQuatSpline<Vector3f>::_TyKeyPoint kp(-t_camera_matrix.t_Inverse().v_Transform_Pos(Vector3f(0, 0, 0)),
			Quatf(t_camera_matrix.t_RotationPart()).t_Positive(), f_time - m_f_anim_poses_offset);
		// make a keypoint

		m_camera_spline.PushBack(kp);
		// put it in the spline
	}

	void onResetAnimation()
	{
		m_b_animate = false;
		m_camera_spline.Erase();
	}

	void onLoadAnimation()
	{
		FILE *p_fr = fopen("animation.txt", "r");
		if(!p_fr)
			return;
		size_t n_point_num;
		fscanf(p_fr, PRIsize "\n", &n_point_num);
		std::vector<CKochanekBartelsQuatSpline<Vector3f>::_TyKeyPoint> kps(n_point_num);
		for(size_t i = 0; i < n_point_num; ++ i) {
			CKochanekBartelsQuatSpline<Vector3f>::_TyKeyPoint &kp = kps[i];
			fscanf(p_fr, "%f %f %f %f %f %f %f %f %f %f %f\n",
				&kp.f_time, &kp.f_tension, &kp.f_continuity, &kp.f_bias,
				&kp.v_position.x, &kp.v_position.y, &kp.v_position.z,
				&kp.t_rotation.x, &kp.t_rotation.y, &kp.t_rotation.z, &kp.t_rotation.w);
		}
		fclose(p_fr);
		printf("loaded " PRIsize " keypoints\n", n_point_num);
		m_b_animate = false;
		m_camera_spline.Erase();
		m_camera_spline.Insert(0, kps.begin(), kps.end());
	}

	void onSaveAnimation()
	{
		FILE *p_fw = fopen("animation.txt", "w");
		if(!p_fw)
			return;
		fprintf(p_fw, PRIsize "\n", m_camera_spline.n_Point_Num());
		for(size_t i = 0, n = m_camera_spline.n_Point_Num(); i < n; ++ i) {
			CKochanekBartelsQuatSpline<Vector3f>::_TyKeyPoint kp = m_camera_spline.r_Point(i);
			fprintf(p_fw, "%.15f %.15f %.15f %.15f %.15f %.15f %.15f %.15f %.15f %.15f %.15f\n",
				kp.f_time, kp.f_tension, kp.f_continuity, kp.f_bias,
				kp.v_position.x, kp.v_position.y, kp.v_position.z,
				kp.t_rotation.x, kp.t_rotation.y, kp.t_rotation.z, kp.t_rotation.w);
		}
		fclose(p_fw);
		printf("saved " PRIsize " keypoints\n", m_camera_spline.n_Point_Num());
	}

	float f_AnimationLength() const
	{
		return (m_camera_spline.b_Empty())? 0 :
			m_camera_spline.r_Point(m_camera_spline.n_Point_Num() - 1).f_time;
	}

	void onPlayAnimation()
	{
		m_b_animate = true;
		m_spline = m_camera_spline;
		if(!m_spline.b_Empty()) {
			m_spline.Insert(0, m_spline.r_Point(0));
			m_spline.PushBack(m_spline.r_Point(m_spline.n_Point_Num() - 1));
		}
		//float f_len = m_spline.f_Length(); // this takes some time, if done with great precision
		// duplicate the first and the last point
		m_timer.Reset();
	}

	void onDisplay()
	{
		if(m_b_first_frame) {
			m_b_first_frame = false;
			m_timer.Reset();
		}
		double f_time = m_timer.f_Time();
		// ...

		onDisplay(f_time);
	}

	void onDisplay(double f_time)
	{
		int p_wp[4];
		glGetIntegerv(GL_VIEWPORT, p_wp);
		if(!b_inhibit_texture_referesh && (!m_p_refraction_tex ||
		   m_p_refraction_tex->n_Width() != p_wp[2] ||
		   m_p_refraction_tex->n_Height() != p_wp[3])) {
			if(m_p_refraction_tex)
				delete m_p_refraction_tex;
			if(m_p_reflection_tex)
				delete m_p_reflection_tex;
			m_p_reflection_tex = new CGLTexture_2D(p_wp[2], p_wp[3], GL_RGBA, false);
			m_p_reflection_tex->Bind();
			m_p_reflection_tex->Set_Wrap_S(GL_CLAMP_TO_EDGE);
			m_p_reflection_tex->Set_Wrap_T(GL_CLAMP_TO_EDGE);
			m_p_refraction_tex = new CGLTexture_2D(p_wp[2], p_wp[3], GL_RGBA, false);
			m_p_refraction_tex->Bind();
			m_p_refraction_tex->Set_Wrap_S(GL_CLAMP_TO_EDGE);
			m_p_refraction_tex->Set_Wrap_T(GL_CLAMP_TO_EDGE);
		}
		// alloc the reflection and refraction texture

		Matrix4f t_modelview, t_projection, t_mvp;
		{
			CGLTransform::Perspective(t_projection, 60, float(n_width) / n_height, .1f, 10000);
			// spocitame matici perspektivni projekce

			Matrix4f t_camera_matrix;
			t_camera_matrix.Identity();
			if(m_b_animate) {
				if(m_spline.r_Point(m_spline.n_Point_Num() - 1).f_time < f_time)
					m_b_animate = false; // end animation after it is finished
				CSplineSampler_ExplicitTime_UniformStep<CKochanekBartelsQuatSpline<Vector3f> > sampler(m_spline);
				CKochanekBartelsQuatSpline<Vector3f>::_TyPoint p = sampler.v_Position(float(f_time));
				t_camera_matrix.Set_RotationPart(p.t_rotation.t_ToMatrix());
				t_camera_matrix.Translate(p.v_position);
				// sample and convert to a matrix
			} else {
#ifdef __GL_WINDOW_TRACKBALL_STYLE_CAMERA
				t_camera_matrix.Translate(v_camera_pos);
#endif // __GL_WINDOW_TRACKBALL_STYLE_CAMERA
				t_camera_matrix.RotateZ(f_angle_z);
				t_camera_matrix.RotateX(f_angle_x);
				t_camera_matrix.RotateY(f_angle_y);
#ifndef __GL_WINDOW_TRACKBALL_STYLE_CAMERA
				t_camera_matrix.Translate(v_camera_pos);
#endif // !__GL_WINDOW_TRACKBALL_STYLE_CAMERA
			}
			// calculate final camera matrix (with pitch and roll)

			Matrix4f t_model_matrix;
			t_model_matrix.Identity();
			// calculate object matrix

			t_modelview = t_camera_matrix * t_model_matrix;
			// fuse object and camera matrix to modelview

			t_mvp = t_projection * t_modelview;
		}
		// calculate the modelview - projection matrix, required to transform the geometry

		Matrix4f t_mirror;
		CGLTransform::Mirror(t_mirror, Plane3f(Vector3f(0, 1, 0), 0));
		// mirror plane

		glEnable(GL_DEPTH_TEST);
		glEnable(GL_CULL_FACE);

		bool b_camera_underwater = (t_modelview.t_Inverse().v_Transform_Pos(Vector3f(0, 0, 0)).y < 0);
		if(!b_inhibit_texture_referesh) {
			glDisable(GL_CULL_FACE);
			// to avoid sharp cusps at the edges of the reflected objects

			if(!b_camera_underwater) {
				Matrix4f t_oblique_proj;
				CGLTransform::ObliqueClipping(t_oblique_proj, Plane3f(Vector3f(0, 1, 0), .1f), // let it draw slightly deeper to cover seams
					t_modelview * t_mirror, t_projection);
				Matrix4f t_mvp2 = t_oblique_proj * (t_modelview * t_mirror);
				// clipping and mirroring matrices

				glFrontFace(GL_CW); // due to mirror
				glClearColor(.3f, .9f, 1, 1.0); // sky blue (low horizon tint)
				glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

				m_sky_dome.Draw(t_mvp2, float(f_time));
				m_island_scene.Draw(t_mvp2);
				for(size_t i = 0, n = trajectory.n_Buoy_Num(); i < n; ++ i)
					m_buoy.Draw(t_mvp2, t_modelview, trajectory.v_Buoy(i));
				m_diver.Draw(t_mvp2, t_modelview, trajectory.v_Diver(f_time),
					trajectory.v_Diver_Direction(f_time));
				// draw non-transparent scene objects

				m_p_reflection_tex->Bind();
				glCopyTexImage2D(m_p_reflection_tex->n_Target(), 0, GL_RGBA, 0, 0,
					m_p_reflection_tex->n_Width(), m_p_reflection_tex->n_Height(), 0);
				glFrontFace(GL_CCW); // due to mirror
				// render reflection texture (could use a FBO here)
				// note that the depth with oblique frustum is not comparable to depth without!
			} else {
				Matrix4f t_oblique_proj;
				CGLTransform::ObliqueClipping(t_oblique_proj, Plane3f(Vector3f(0, 1, 0), .1f), // let it draw slightly deeper to cover seams
					t_modelview, t_projection);
				Matrix4f t_mvp2 = t_oblique_proj * t_modelview;
				// clipping and mirroring matrices

				glClearColor(0, 0, 0, 1);
				glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

				m_sky_dome.Draw_UW(t_mvp2, float(f_time));
				m_island_scene.Draw_UW(t_mvp2, t_modelview); // would be better to have a postprocess pass, this is kind of hacky
				for(size_t i = 0, n = trajectory.n_Buoy_Num(); i < n; ++ i)
					m_buoy.Draw(t_mvp2, t_modelview, trajectory.v_Buoy(i));
				m_diver.Draw(t_mvp2, t_modelview, trajectory.v_Diver(f_time),
					trajectory.v_Diver_Direction(f_time));
				// draw non-transparent scene objects

				m_p_refraction_tex->Bind();
				glCopyTexImage2D(m_p_refraction_tex->n_Target(), 0, GL_RGBA, 0, 0,
					m_p_refraction_tex->n_Width(), m_p_refraction_tex->n_Height(), 0);
				// render refraction texture (could use FBO if had framebuffer blit extension, otherwise will copy images anyway)
				// note that this is a "fake" refraction as it can also contain objects closer to the camera (would have to draw it twice)
			}
			glEnable(GL_CULL_FACE); // reenable
		}

		glClearColor(0, 0, 0, 1);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		if(b_camera_underwater) {
			m_sky_dome.Draw_UW(t_mvp, float(f_time));
			m_island_scene.Draw_UW(t_mvp, t_modelview); // would be better to have a postprocess pass, this is kind of hacky
		} else {
			m_sky_dome.Draw(t_mvp, float(f_time));
			m_island_scene.Draw(t_mvp);
		}
		for(size_t i = 0, n = trajectory.n_Buoy_Num(); i < n; ++ i)
			m_buoy.Draw(t_mvp, t_modelview, trajectory.v_Buoy(i));
		Vector3f v_diver_pos = trajectory.v_Diver(f_time);
		Vector3f v_diver_heading = trajectory.v_Diver_Direction(f_time);
		m_diver.Draw(t_mvp, t_modelview, v_diver_pos, v_diver_heading);

		if(!b_camera_underwater) {
			glLineWidth((n_width > 2048)? 13 : (n_width > 1024)? 9 : 3);
			glDisable(GL_LINE_SMOOTH); // makes the line thicker
			m_color_shader.Bind(t_mvp, Vector4f(1, 0, 1, 1));
			trajectory.Draw_Trajectory(true, f_time);
			m_point_shader.Bind(t_mvp, Vector4f(1, 0, 1, 1), (n_width > 2048)? 11 : (n_width > 1024)? 9 : 3);
			trajectory.Draw_Trajectory(true, f_time, GL_POINTS);
			m_color_shader.Bind(t_mvp, Vector4f(0, 1, 0, 1));
			trajectory.Draw_Trajectory(false, f_time);
			m_point_shader.Bind(t_mvp, Vector4f(0, 1, 0, 1), (n_width > 2048)? 11 : (n_width > 1024)? 9 : 3);
			trajectory.Draw_Trajectory(false, f_time, GL_POINTS);
			// draw trajectory (have to draw without blending because of depth test issues)
		}

		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glEnable(GL_BLEND);
		m_p_reflection_tex->Bind();
		m_reflection_shader.Bind(t_mvp, t_modelview, float(f_time));
		m_inf_grid.Draw();
		glDisable(GL_BLEND);
		// draw water surface

		glDepthMask(0); // !!
		glFrontFace(GL_CW); // due to mirror
		m_p_refraction_tex->Bind();
		m_refraction_shader.Bind(t_mvp, t_modelview, float(f_time));
		m_inf_grid.Draw();
		glFrontFace(GL_CCW); // due to mirror
		glDepthMask(1);
		// draw view from under water

		if(b_camera_underwater) {
			glLineWidth((n_width > 2048)? 13 : (n_width > 1024)? 9 : 3);
			glDepthMask(0);
			glEnable(GL_LINE_SMOOTH);
			glEnable(GL_BLEND);
			m_color_shader.Bind(t_mvp, Vector4f(1, 0, 1, 1));
			trajectory.Draw_Trajectory(true, f_time);
			m_point_shader.Bind(t_mvp, Vector4f(1, 0, 1, 1), (n_width > 2048)? 11 : (n_width > 1024)? 9 : 3);
			trajectory.Draw_Trajectory(true, f_time, GL_POINTS);
			m_color_shader.Bind(t_mvp, Vector4f(0, 1, 0, 1));
			trajectory.Draw_Trajectory(false, f_time);
			m_point_shader.Bind(t_mvp, Vector4f(0, 1, 0, 1), (n_width > 2048)? 11 : (n_width > 1024)? 9 : 3);
			trajectory.Draw_Trajectory(false, f_time, GL_POINTS);
			glDisable(GL_BLEND);
			glDepthMask(1);
			// draw trajectory

			glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			glEnable(GL_BLEND);
			glDepthMask(0);
			for(size_t i = 0, n = trajectory.n_Buoy_Num(); i < n; ++ i)
				m_buoy.Draw_Halo(t_mvp, trajectory.v_Buoy(i), float(f_time));
			glDisable(GL_BLEND);
			glDepthMask(1);
			// draw the buoy halos
		}
		// draw transparent objects

		Vector2d v_GPS(20.838278, -79.013672); // lat, lon, degrees
		v_GPS.x += v_diver_pos.z / 111111.0; // 111,111 meters (111.111 km) in the y direction is 1 degree (of latitude)
		v_GPS.y += v_diver_pos.x / (111111.0 * cos(v_GPS.x)); // 111,111 * cos(latitude) meters in the x direction is 1 degree (of longitude).
		// calculate GPS coordinates

		glDisable(GL_DEPTH_TEST);
		std::basic_string<wchar_t> s_title;
		stl_ut::FormatW(s_title, L"lat: %02d%02d'%06.3f\"%c\n"
			L"lon: %02d%02d'%06.3f\"%c\ndepth: %.2f m\nspeed: %.2f knots",
			int(floor(fabs(v_GPS.x))), int(floor(fmod(fabs(v_GPS.x * 60), 60))),
			fmod(fabs(v_GPS.x * 3600), 60), (v_GPS.x > 0)? 'N' : 'S',
			int(floor(fabs(v_GPS.y))), int(floor(fmod(fabs(v_GPS.y * 60), 60))),
			fmod(fabs(v_GPS.y * 3600), 60), (v_GPS.y > 0)? 'E' : 'W', -v_diver_pos.y,
			v_diver_heading.f_Length() * 0.514444f);
		std::string s_title8;
		CUniConv::n_UTF16_to_UTF8(s_title.data(), s_title.length() * sizeof(wchar_t), s_title8, false, false);
		m_gui.Set_Title(s_title8.c_str(), .18f);
		//m_gui.Set_Title("test\nlat lon 123 456");
		m_gui.Draw();

		/*{
			std::vector<Vector4f> buoy_list_ds(m_buoy_list.size());
			for(size_t i = 0, n = m_buoy_list.size() - 1; i < n; ++ i)
				buoy_list_ds[i] = Vector4f(m_buoy_list[i], 0);
			buoy_list_ds[m_buoy_list.size() - 1] = Vector4f(m_buoy_list[m_buoy_list.size() - 1], 1); // mark the diver
			struct TDepthSorter {
				Matrix4f t_mvp;

				TDepthSorter(Matrix4f _t_mvp)
					:t_mvp(_t_mvp)
				{}

				bool operator ()(Vector4f a, Vector4f b) const
				{
					Vector4f ta = t_mvp * Vector4f(a.v_xyz(), 1);
					Vector4f tb = t_mvp * Vector4f(b.v_xyz(), 1);
					return ta.z / ta.w > tb.z / tb.w;
				}
			};
			std::sort(buoy_list_ds.begin(), buoy_list_ds.end(), TDepthSorter(t_mvp));
			// a simple depth sorting of the buoys

			glEnable(GL_BLEND);
			glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			glDepthMask(0);

			m_buoy_shader.Bind(t_mvp, t_modelview, f_time);
			CGLArraySetup buoys(&buoy_list_ds.front().x, sizeof(Vector4f) *
				buoy_list_ds.size(), 0, GL_FLOAT, 4, 0, 0, 0, 0, GL_POINTS);
			buoys.Draw();
			// draw the buoys

			glDisable(GL_BLEND);
			glDepthMask(1);
		}*/
		// draw buoys

		/*static double f_last_time = 0;
		static Vector3f v_last_pose = Vector3f(sin(.0 * .5f), -1 - .5 * (.5 + .5 * sin(.0 / 10)), cos(.0 * .5f));

		double f_dt = f_time - f_last_time;*/

		m_buoy_list.back() = Vector3f(sin(f_time * .5f), -1 - .5 * (.5 + .5 * sin(f_time / 10)), cos(f_time * .5f));
		/*Vector3f v_last_velo = (m_buoy_list.back() - v_last_pose) / f_dt;

		Vector3f v_velo(.5*cos(.5*f_time), -1.0/40*cos(.1*f_time), -.5*sin(.5 * f_time)); // velocity, from matlab
		// diver motion

		static int n_frame_id = 0;
		if(n_frame_id == 1) {
			//m_buoy_list.back() = Vector3f(sin(f_time * .5f), -1 - .5 * (.5 + .5 * sin(f_time / 10)), cos(f_time * .5f));
			//Vector3f v_velo(.5*cos(.5*f_time), -1.0/40*cos(.1*f_time), -.5*sin(.5 * f_time)); // velocity, from matlab
			printf("ROCV:RECEIVER 0 %.15g %.15g %.15g %.15g %.15g %.15g\n",
				v_last_pose.x, v_last_pose.y, v_last_pose.z, v_last_velo.x, v_last_velo.y, v_last_velo.z);
			printf("ROCV:TRANSMITTER 1 %g %g %g\n", m_buoy_list[0].x, m_buoy_list[0].y, m_buoy_list[0].z);
			printf("ROCV:TRANSMITTER 2 %g %g %g\n", m_buoy_list[1].x, m_buoy_list[1].y, m_buoy_list[1].z);
			printf("ROCV:TRANSMITTER 3 %g %g %g\n", m_buoy_list[2].x, m_buoy_list[2].y, m_buoy_list[2].z);
			printf("ROCV:TRANSMITTER 4 %g %g %g\n", m_buoy_list[3].x, m_buoy_list[3].y, m_buoy_list[3].z);
			printf("ROCV:RANGE %d 1 %g %g\n", 0, (v_last_pose - m_buoy_list[0]).f_Length(), f_range_sigma00);
			printf("ROCV:RANGE %d 2 %g %g\n", 0, (v_last_pose - m_buoy_list[1]).f_Length(), f_range_sigma00);
			printf("ROCV:RANGE %d 3 %g %g\n", 0, (v_last_pose - m_buoy_list[2]).f_Length(), f_range_sigma00);
			printf("ROCV:RANGE %d 4 %g %g\n", 0, (v_last_pose - m_buoy_list[3]).f_Length(), f_range_sigma00);
			// print the beginning of the graph file

			fprintf(stderr, "%g ", ((v_last_pose + v_last_velo * f_dt) - m_buoy_list.back()).f_Length());

			n_frame_id = 4; // so it is 5 next time
		} else if(n_frame_id > 4) {
			printf("ROCV:RECEIVER_GT %d %.15g %.15g %.15g %.15g %.15g %.15g\n", n_frame_id,
				v_last_pose.x, v_last_pose.y, v_last_pose.z, v_last_velo.x, v_last_velo.y, v_last_velo.z);
			printf("ROCV:DELTA_TIME %d %d %.15g"
				" %g %g %g %g %g %g"
				   " %g %g %g %g %g"
					  " %g %g %g %g"
						 " %g %g %g"
							" %g %g"
							   " %g"
				"\n", (n_frame_id == 5)? 0 : n_frame_id - 1, n_frame_id, f_time - f_last_time,
				p_dt_sigma[0], p_dt_sigma[1], p_dt_sigma[2], p_dt_sigma[3], p_dt_sigma[4],
				p_dt_sigma[5], p_dt_sigma[6], p_dt_sigma[7], p_dt_sigma[8], p_dt_sigma[9],
				p_dt_sigma[10], p_dt_sigma[11], p_dt_sigma[12], p_dt_sigma[13], p_dt_sigma[14],
				p_dt_sigma[15], p_dt_sigma[16], p_dt_sigma[17], p_dt_sigma[18], p_dt_sigma[19], p_dt_sigma[20]);
			printf("ROCV:RANGE %d 1 %g %g\n", n_frame_id, (v_last_pose - m_buoy_list[0]).f_Length(), f_range_sigma00);
			printf("ROCV:RANGE %d 2 %g %g\n", n_frame_id, (v_last_pose - m_buoy_list[1]).f_Length(), f_range_sigma00);
			printf("ROCV:RANGE %d 3 %g %g\n", n_frame_id, (v_last_pose - m_buoy_list[2]).f_Length(), f_range_sigma00);
			printf("ROCV:RANGE %d 4 %g %g\n", n_frame_id, (v_last_pose - m_buoy_list[3]).f_Length(), f_range_sigma00);

			fprintf(stderr, "%g ", ((v_last_pose + v_last_velo * f_dt) - m_buoy_list.back()).f_Length());
		}
		f_last_time = f_time;
		v_last_pose = m_buoy_list.back();
		++ n_frame_id;*/
		// print (ground truth) receiver xyz and ranges to the buoys
	}
};

#ifdef MAINDEMO
static CDemo *p_demo = 0;
#endif

class CSplineDemo {
protected:
	CColorShader m_color_shader;
	CPointShader m_point_shader;
	typedef CKochanekBartelsSpline<Vector2f> _TySpline;
	_TySpline m_spline;
	CGLVertexArrayObject m_vao;
	CGLArrayBufferObject m_vbo;
	size_t m_n_selected_kp, m_n_selected_slider;
	Vector2f m_v_selection_off;
	CGUIWidget m_gui;

public:
	CSplineDemo()
		:m_n_selected_kp(size_t(-1)), m_n_selected_slider(size_t(-1)), m_gui(Vector2f(-.9f, -1), .5f)
	{
		m_color_shader.Compile();
		m_point_shader.Compile();

		m_vao.Bind();
		{
			m_vbo.Bind();
			m_vbo.BufferData(sizeof(Vector2f)); // ??
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, 2, GL_FLOAT, false, 0, m_vbo.p_OffsetPointer(0));
		}
		CGLVertexArrayObject::Release();

		m_gui.Set_Title("tens\ncont\nbias\ntime", .2f);
	}

	void onMouseClick(int n_button, int n_state, Vector2f v_position)
	{
		if(n_state == 0) { // pushing down
			Vector2f v_min = m_gui.v_Min(), v_max = m_gui.v_Max();
			m_n_selected_slider = size_t(-1);
			if(v_position.x >= v_min.x && v_position.x <= v_max.x &&
			   v_position.y >= v_min.y && v_position.y <= v_max.y) {
				if(m_n_selected_kp == size_t(-1))
					return;
				// a point needs to be selected

				Vector2f p_pos[slider_Num];
				Get_AbsSlider_Positions(p_pos);
				// get slider positions

				size_t n_closest_point = size_t(-1);
				float f_closest_dist = .1f;
				for(size_t i = 0; i < slider_Clickable_Num; ++ i) {
					float f_dist = (v_position - p_pos[i]).f_Length();
					if(f_closest_dist > f_dist) {
						f_closest_dist = f_dist;
						n_closest_point = i;
					}
				}
				// find the closest slider

				if(n_button == GLUT_LEFT_BUTTON) {
					m_n_selected_slider = n_closest_point;
					if(n_closest_point != size_t(-1))
						m_v_selection_off = p_pos[n_closest_point] - v_position;
				} else
					Set_Slider_Default(n_closest_point);
				// grab the slider or set default
			} else {
				m_n_selected_kp = size_t(-1);

				size_t n_closest_point = size_t(-1);
				float f_closest_dist = .1f;
				for(size_t i = 0, n = m_spline.n_Point_Num(); i < n; ++ i) {
					float f_dist = (v_position - m_spline.r_Point(i).v_position).f_Length();
					if(f_closest_dist > f_dist) {
						f_closest_dist = f_dist;
						n_closest_point = i;
					}
				}
				if(n_button == GLUT_LEFT_BUTTON) {
					if(n_closest_point == size_t(-1)) {
						m_n_selected_kp = m_spline.n_Point_Num();
						m_v_selection_off = Vector2f(0, 0);

						_TySpline::_TyKeyPoint kp(v_position, (m_spline.b_Empty())? 0 :
							m_spline.r_Point(m_spline.n_Point_Num() - 1).f_time + 1);
						m_spline.PushBack(kp);
						// add a new point
					} else {
						m_n_selected_kp = n_closest_point;
						m_v_selection_off = m_spline.r_Point(n_closest_point).v_position - v_position;
						// select an existing point
					}
				} else {
					if(n_closest_point != size_t(-1))
						m_spline.Erase(n_closest_point);
					// delete the point from the spline
				}
			}
		}
	}

	void onMouseMotion(int n_button, Vector2f v_position)
	{
		if(n_button == GLUT_LEFT_BUTTON && m_n_selected_slider != size_t(-1)) {
			Vector2f v_new_slider_pos = m_v_selection_off + v_position;
			std::pair<Vector3f, Vector4f> t_pos = t_Sliders();
			float f_left = t_pos.first.x, f_width = t_pos.first.y, f_slider_height = t_pos.first.z;
			Vector4f p_y = t_pos.second;
			float f_new_relvalue = (v_new_slider_pos.x - f_left) / f_width;
			Set_Slider_RelValue(m_n_selected_slider, f_new_relvalue);
		} else if(n_button == GLUT_LEFT_BUTTON && m_n_selected_kp != size_t(-1))
			m_spline.r_Point(m_n_selected_kp).v_position = m_v_selection_off + v_position;
		// move the selected keypoint
	}

	std::pair<Vector3f, Vector4f> t_Sliders() const
	{
		Vector2f v_min = m_gui.v_Min(), v_max = m_gui.v_Max();
		Vector2f v_size = v_max - v_min;
		v_min.y += v_size.y * .05f;
		v_size.y *= .85f;
		const float f_left = v_min.x + v_size.x * .2f, f_width = v_size.x * .6f;
		const float p_y[4] = {v_min.y + v_size.y * .2f, v_min.y + v_size.y * .4f,
			v_min.y + v_size.y * .6f, v_min.y + v_size.y * .8f};
		const float f_slider_height = v_size.y * .05f;
		return std::make_pair(Vector3f(f_left, f_width, f_slider_height), Vector4f(p_y[0],
			p_y[1], p_y[2], p_y[3]));
	}

	enum {
		slider_Tension = 0,
		slider_Continuity,
		slider_Bias,
		slider_Time,
		slider_Time_Min,
		slider_Time_Max,
		slider_Num,
		slider_Clickable_Num = slider_Time_Min // slider_Time_Min and above are "read only"
	};

	void Get_RelSlider_Positions(float p_pos[slider_Num]) const
	{
		if(m_n_selected_kp != size_t(-1)) {
			float f_time = m_spline.b_Empty()? 0 :
				m_spline.r_Point(m_spline.n_Point_Num() - 1).f_time;
			p_pos[slider_Tension] = m_spline.r_Point(m_n_selected_kp).f_tension * .5f + .5f;
			p_pos[slider_Continuity] = m_spline.r_Point(m_n_selected_kp).f_continuity * .5f + .5f;
			p_pos[slider_Bias] = m_spline.r_Point(m_n_selected_kp).f_bias * .5f + .5f;
			p_pos[slider_Time] = m_spline.r_Point(m_n_selected_kp).f_time / f_time;
			if(m_n_selected_kp)
				p_pos[slider_Time_Min] = m_spline.r_Point(m_n_selected_kp - 1).f_time / f_time;
			else
				p_pos[slider_Time_Min] = 0;
			if(m_n_selected_kp + 1 < m_spline.n_Point_Num())
				p_pos[slider_Time_Max] = m_spline.r_Point(m_n_selected_kp + 1).f_time / f_time;
			else
				p_pos[slider_Time_Max] = 1;
		}
	}

	void Set_Slider_Default(size_t n_slider)
	{
		_ASSERTE(m_n_selected_kp != size_t(-1));
		_ASSERTE(n_slider >= 0 && n_slider < slider_Clickable_Num);

		switch(n_slider) {
		case slider_Tension:
			m_spline.r_Point(m_n_selected_kp).f_tension = 0;
			break;
		case slider_Continuity:
			m_spline.r_Point(m_n_selected_kp).f_continuity = 0;
			break;
		case slider_Bias:
			m_spline.r_Point(m_n_selected_kp).f_bias = 0;
			break;
		case slider_Time:
			break; // no default for time
		}
	}

	void Set_Slider_RelValue(size_t n_slider, float f_pos)
	{
		_ASSERTE(m_n_selected_kp != size_t(-1));
		_ASSERTE(n_slider >= 0 && n_slider < slider_Clickable_Num);

		float f_time = m_spline.b_Empty()? 0 :
			m_spline.r_Point(m_spline.n_Point_Num() - 1).f_time;

		if(n_slider == slider_Time)
			f_pos = f_time * f_pos;
		else
			f_pos = max(-1, min(1, f_pos * 2 - 1)); // clamp others to +-1

		switch(n_slider) {
		case slider_Tension:
			m_spline.r_Point(m_n_selected_kp).f_tension = f_pos;
			break;
		case slider_Continuity:
			m_spline.r_Point(m_n_selected_kp).f_continuity = f_pos;
			break;
		case slider_Bias:
			m_spline.r_Point(m_n_selected_kp).f_bias = f_pos;
			break;
		case slider_Time:
			m_spline.r_Point(m_n_selected_kp).f_time = f_pos;
			break;
		}
	}

	void Get_AbsSlider_Positions(Vector2f p_pos[slider_Num]) const
	{
		float p_relpos[slider_Num];
		Get_RelSlider_Positions(p_relpos);

		std::pair<Vector3f, Vector4f> t_pos = t_Sliders();
		float f_left = t_pos.first.x, f_width = t_pos.first.y, f_slider_height = t_pos.first.z;
		Vector4f p_y = t_pos.second;

		for(int i = 0; i < slider_Num; ++ i)
			p_pos[i] = Vector2f(f_left + f_width * p_relpos[i], p_y[3 - min(3, i)]);
	}

	void Get_SliderBar_Positions(Vector2f p_pos[slider_Num * 2]) const
	{
		Vector2f p_abspos[slider_Num];
		Get_AbsSlider_Positions(p_abspos);

		std::pair<Vector3f, Vector4f> t_pos = t_Sliders();
		float f_left = t_pos.first.x, f_width = t_pos.first.y, f_slider_height = t_pos.first.z;
		Vector4f p_y = t_pos.second;

		for(int i = 0; i < slider_Num; ++ i) {
			p_pos[i * 2 + 0] = p_abspos[i] - Vector2f(0, ((i > 3)? .5f : 1) * f_slider_height);
			p_pos[i * 2 + 1] = p_abspos[i] + Vector2f(0, ((i > 3)? .5f : 1) * f_slider_height);
		}
	}

	void onDisplay()
	{
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

		Matrix4f t_mvp;
		t_mvp.Identity();

		m_gui.Draw();

		glDisable(GL_DEPTH_TEST);
		glEnable(GL_LINE_SMOOTH);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glEnable(GL_BLEND);
		// will draw antialiassed lines and points

		{
			std::pair<Vector3f, Vector4f> t_pos = t_Sliders();
			float f_left = t_pos.first.x, f_width = t_pos.first.y, f_slider_height = t_pos.first.z;
			Vector4f p_y = t_pos.second;

			Vector2f p_line_list[(4 + slider_Num) * 2] = {
				Vector2f(f_left, p_y[0]), Vector2f(f_left + f_width, p_y[0]),
				Vector2f(f_left, p_y[1]), Vector2f(f_left + f_width, p_y[1]),
				Vector2f(f_left, p_y[2]), Vector2f(f_left + f_width, p_y[2]),
				Vector2f(f_left, p_y[3]), Vector2f(f_left + f_width, p_y[3])
			};

			Get_SliderBar_Positions(p_line_list + 8);

			CGLArraySetup geom(p_line_list, sizeof(p_line_list), sizeof(Vector2f), GL_FLOAT, 2, 0, 0, 0, 0, GL_LINES);
			m_color_shader.Bind(t_mvp, Vector4f(1, 1, 1, 1));
			geom.Draw();
		}
		// draw the sliders

		std::vector<Vector2f> control_point_list(m_spline.n_Point_Num());
		for(size_t i = 0, n = m_spline.n_Point_Num(); i < n; ++ i)
			control_point_list[i] = m_spline.r_Point(i).v_position;
		// get a list of control points

		const size_t n_point_num = (m_spline.n_Arc_Num())? size_t(m_spline.f_Length() * 20) + 1 : 0;
		// calculate spline length

		std::vector<Vector2f> point_list(n_point_num);
		CSplineSampler_ExplicitTime_UniformStep<_TySpline> sampler(m_spline);
		const float f_step = m_spline.f_Spline_Time() / (n_point_num - 1);
		for(size_t i = 0; i < n_point_num; ++ i)
			point_list[i] = sampler.v_Position(i * f_step);
		// interpolate the spline

		m_vao.Bind();
		{
			m_vbo.Bind();
			m_vbo.BufferData((control_point_list.size() + point_list.size()) * sizeof(Vector2f));
			if(!control_point_list.empty())
				m_vbo.BufferSubData(0, control_point_list.size() * sizeof(Vector2f), &control_point_list[0]);
			if(!point_list.empty()) {
				m_vbo.BufferSubData(control_point_list.size() * sizeof(Vector2f),
					point_list.size() * sizeof(Vector2f), &point_list[0]);
			}
		}
		{
			m_point_shader.Bind(t_mvp, Vector4f(0, 1, 0, 1), 7);
			if(m_n_selected_kp != size_t(-1)) {
				_ASSERTE(m_n_selected_kp < control_point_list.size());
				glDrawArrays(GL_POINTS, 0, m_n_selected_kp);
				glDrawArrays(GL_POINTS, m_n_selected_kp + 1, control_point_list.size() - (m_n_selected_kp + 1));
				m_point_shader.Bind(t_mvp, Vector4f(1, 0, 0, 1), 7);
				glDrawArrays(GL_POINTS, m_n_selected_kp, 1);
			} else
				glDrawArrays(GL_POINTS, 0, control_point_list.size());
			m_color_shader.Bind(t_mvp, Vector4f(1, 1, 1, 1));
			glDrawArrays(GL_LINE_STRIP, control_point_list.size(), point_list.size());
			m_point_shader.Bind(t_mvp, Vector4f(1, 1, 1, 1), 5);
			glDrawArrays(GL_POINTS, control_point_list.size(), point_list.size());
		}
		CGLVertexArrayObject::Release();
	}
};

#ifndef MAINDEMO
static CSplineDemo *p_demo = 0;
#endif

/**
 *	@brief called by GLUT when the window needs to repaint
 */
void onDisplay()
{
	if(p_demo)
		p_demo->onDisplay();

	glutSwapBuffers();
	// display results

	glutPostRedisplay();
	// tell glut we want to draw the next frame as soon as possible
}

static Vector2f v_old_gl_mouse(0, 0);

int n_button;

void onMouseClick(int _n_button, int n_state, int n_x, int n_y)
{
	Vector2f v_gl_coord(float(n_x) / n_width * 2 - 1, 1 - float(n_y) / n_height * 2);
	// calculate normalized mouse coordinates

	if(p_demo)
		p_demo->onMouseClick(_n_button, n_state, v_gl_coord);

	v_old_gl_mouse = v_gl_coord;
	n_button = (n_state)? -1 : _n_button;
}

void onMouseMotion(int n_x, int n_y)
{
	Vector2f v_gl_coord(float(n_x) / n_width * 2 - 1, 1 - float(n_y) / n_height * 2);
	// calculate normalized mouse coordinates

	if(p_demo)
		p_demo->onMouseMotion(n_button, v_gl_coord);

	if(n_button == GLUT_LEFT_BUTTON) { // "left" mouse button - looking
		f_angle_y += v_gl_coord.x - v_old_gl_mouse.x;
#ifdef __GL_WINDOW_TRACKBALL_STYLE_CAMERA
		f_angle_x -= v_gl_coord.y - v_old_gl_mouse.y;
#else // __GL_WINDOW_TRACKBALL_STYLE_CAMERA
		f_angle_x += v_gl_coord.y - v_old_gl_mouse.y;
#endif // __GL_WINDOW_TRACKBALL_STYLE_CAMERA
	} else if(n_button == GLUT_MIDDLE_BUTTON) { // rolling
#ifdef __GL_WINDOW_TRACKBALL_STYLE_CAMERA
		f_angle_z += v_gl_coord.x - v_old_gl_mouse.x;
#endif // __GL_WINDOW_TRACKBALL_STYLE_CAMERA
	} else if(n_button == GLUT_RIGHT_BUTTON) { // rolling or zooming
#ifdef __GL_WINDOW_TRACKBALL_STYLE_CAMERA
		v_camera_pos.z += v_gl_coord.x - v_old_gl_mouse.x;
#else // __GL_WINDOW_TRACKBALL_STYLE_CAMERA
		f_angle_z += v_gl_coord.x - v_old_gl_mouse.x;
#endif // __GL_WINDOW_TRACKBALL_STYLE_CAMERA
	}

	v_old_gl_mouse = v_gl_coord;
}

static bool b_wireframe = false;

#include "MT_PNG_Codec.h"

/**
 *	@brief called by GLUT when a key is pressed
 */
void onKeyPressed(unsigned char n_key_code, int n_x, int n_y)
{
	if(n_key_code == 27)
		exit(0);
	else {
		switch(n_key_code) {
		case 'v':
			{
				void (GLApi *glBlitFramebufferEXT)(int srcX0, int srcY0, int srcX1, int srcY1,
                    int dstX0, int dstY0, int dstX1, int dstY1,
                    GLbitfield mask, GLenum filter) = 0;
				glBlitFramebufferEXT = (void (GLApi*)(int, int, int, int, int, int, int, int, GLbitfield,
					GLenum))wglGetProcAddress("glBlitFramebufferEXT");
				// get glBlitFramebufferEXT

				double f_length = p_demo->f_AnimationLength(), f_fps = 23.976;
				size_t n_frame_num = size_t(ceil(f_length * f_fps));

				CMTEncoder encoder(1280, 720); // HD
				TBmp *p_bitmap = encoder.p_Get_EmptyBitmap(); //TBmp::p_Alloc(1280, 720);

				{
					int n_old_width = n_width, n_old_height = n_height;
					onResize(p_bitmap->n_width, p_bitmap->n_height);
					// resize viewport

					p_demo->onPlayAnimation();
					// start playing the animation

					for(size_t n_frame = 0; n_frame < n_frame_num; ++ n_frame) {
						printf("rendering frame " PRIsize " / " PRIsize "\r", n_frame, n_frame_num);
						double f_time = n_frame / f_fps;

						if(glBlitFramebufferEXT) {
							CGLFrameBufferObject fbo(p_bitmap->n_width, p_bitmap->n_height, 1, false, GL_RGBA,
								0, true, false, GL_DEPTH_COMPONENT24_OES, 0, false, false, 0, 0);
							fbo.Bind();
							p_demo->onDisplay(f_time);
							fbo.Release();
						}
						// need to prime the reflection textures with correct resolution

						CGLFrameBufferObject fbo(p_bitmap->n_width, p_bitmap->n_height, 1, false, GL_RGBA,
							(glBlitFramebufferEXT)? 8 : 0, true, false, GL_DEPTH_COMPONENT24_OES,
							(glBlitFramebufferEXT)? 8 : 0, false, false, 0, 0);
						{
							fbo.Bind();
							if(glBlitFramebufferEXT)
								b_inhibit_texture_referesh = true;
							// texture readback would cause invalid operation,
							// multisampled textures can't be easily read in shader

							p_demo->onDisplay(f_time);

							if(glBlitFramebufferEXT)
								b_inhibit_texture_referesh = false;
							if(!glBlitFramebufferEXT) {
								glReadPixels(0, 0, fbo.n_Width(), fbo.n_Height(),
									GL_RGBA, GL_UNSIGNED_BYTE, p_bitmap->p_buffer);
							}
							fbo.Release();
						}
						// render the scene with multisampled FBO

						if(glBlitFramebufferEXT) {
							CGLFrameBufferObject fbo_noms(fbo.n_Width(), fbo.n_Height(), 1, false, GL_RGBA,
								0, false, false, 0, 0, false, false, 0, 0); // no need for depth buffer on this one
							fbo_noms.Bind(0x8CA9/*GL_DRAW_FRAMEBUFFER_EXT*/);
							fbo.Bind(0x8CA8/*GL_READ_FRAMEBUFFER_EXT*/); // after creating fbo_noms!
							glBlitFramebufferEXT(0, 0, fbo.n_Width()/* - 1*/, fbo.n_Height()/* - 1*/, 0, 0, fbo.n_Width()/* - 1*/,
								fbo.n_Height()/* - 1*/, GL_COLOR_BUFFER_BIT, GL_NEAREST);
							// The lower bounds of the rectangle are inclusive, while the upper bounds are exclusive. (from http://www.opengl.org/sdk/docs/man3/xhtml/glBlitFramebuffer.xml)
							fbo_noms.Release(0x8CA9/*GL_DRAW_FRAMEBUFFER_EXT*/);
							fbo.Release(0x8CA8/*GL_READ_FRAMEBUFFER_EXT*/);
							// create a dummy and blit

							fbo_noms.Bind();
							glReadPixels(0, 0, fbo.n_Width(), fbo.n_Height(), GL_RGBA, GL_UNSIGNED_BYTE, p_bitmap->p_buffer);
							fbo_noms.Release();
							// bind for reading and read back
						}
						// use framebuffer blit

						char p_s_filename[64];
						stl_ut::Format(p_s_filename, sizeof(p_s_filename),
							"g:/oseo/frame_%05" _PRIsize ".png", n_frame);
#if 0
						p_bitmap->Flip(true); // it is upside down
						CPngCodec::Save_PNG(p_s_filename, *p_bitmap, true);
#else // 0
						if(!encoder.Enqueue_Bitmap(p_s_filename, p_bitmap))
							fprintf(stderr, "error: encoder.Enqueue_Bitmap() failed\n");
#endif // 0
						// save hires antialiassed screenshots
					}
					// for each frame

					onResize(n_old_width, n_old_height);
					// change the viewport back

					if(!encoder.Synchronize())
						fprintf(stderr, "error: encoder.Synchronize() failed\n");
					//p_bitmap->Delete();
					// cleanup

					printf("\ndone\n");
				}
			}
			break;
		case 'p':
			{
				void (GLApi *glBlitFramebufferEXT)(int srcX0, int srcY0, int srcX1, int srcY1,
                    int dstX0, int dstY0, int dstX1, int dstY1,
                    GLbitfield mask, GLenum filter) = 0;
				glBlitFramebufferEXT = (void (GLApi*)(int, int, int, int, int, int, int, int, GLbitfield,
					GLenum))wglGetProcAddress("glBlitFramebufferEXT");
				// get glBlitFramebufferEXT

				TBmp *p_bitmap = TBmp::p_Alloc(4096, 3072);

				{
					int n_old_width = n_width, n_old_height = n_height;
					onResize(p_bitmap->n_width, p_bitmap->n_height);
					// resize viewport

					if(glBlitFramebufferEXT) {
						CGLFrameBufferObject fbo(p_bitmap->n_width, p_bitmap->n_height, 1, false, GL_RGBA,
							0, true, false, GL_DEPTH_COMPONENT24_OES, 0, false, false, 0, 0);
						fbo.Bind();
						onDisplay();
						fbo.Release();
					}
					// need to prime the reflection textures with correct resolution

					CGLFrameBufferObject fbo(p_bitmap->n_width, p_bitmap->n_height, 1, false, GL_RGBA,
						(glBlitFramebufferEXT)? 8 : 0, true, false, GL_DEPTH_COMPONENT24_OES,
						(glBlitFramebufferEXT)? 8 : 0, false, false, 0, 0);
					{
						fbo.Bind();
						if(glBlitFramebufferEXT)
							b_inhibit_texture_referesh = true;
						// texture readback would cause invalid operation,
						// multisampled textures can't be easily read in shader

						onDisplay();

						if(glBlitFramebufferEXT)
							b_inhibit_texture_referesh = false;
						if(!glBlitFramebufferEXT) {
							glReadPixels(0, 0, fbo.n_Width(), fbo.n_Height(),
								GL_RGBA, GL_UNSIGNED_BYTE, p_bitmap->p_buffer);
						}
						fbo.Release();
					}
					// render the scene with multisampled FBO

					if(glBlitFramebufferEXT) {
						CGLFrameBufferObject fbo_noms(fbo.n_Width(), fbo.n_Height(), 1, false, GL_RGBA,
							0, false, false, 0, 0, false, false, 0, 0); // no need for depth buffer on this one
						fbo_noms.Bind(0x8CA9/*GL_DRAW_FRAMEBUFFER_EXT*/);
						fbo.Bind(0x8CA8/*GL_READ_FRAMEBUFFER_EXT*/); // after creating fbo_noms!
						glBlitFramebufferEXT(0, 0, fbo.n_Width() /*- 1*/, fbo.n_Height() /*- 1*/, 0, 0, fbo.n_Width() /*- 1*/,
							fbo.n_Height() /*- 1*/, GL_COLOR_BUFFER_BIT, GL_NEAREST);
						// The lower bounds of the rectangle are inclusive, while the upper bounds are exclusive. (from http://www.opengl.org/sdk/docs/man3/xhtml/glBlitFramebuffer.xml)
						fbo_noms.Release(0x8CA9/*GL_DRAW_FRAMEBUFFER_EXT*/);
						fbo.Release(0x8CA8/*GL_READ_FRAMEBUFFER_EXT*/);
						// create a dummy and blit

						fbo_noms.Bind();
						glReadPixels(0, 0, fbo.n_Width(), fbo.n_Height(), GL_RGBA, GL_UNSIGNED_BYTE, p_bitmap->p_buffer);
						fbo_noms.Release();
						// bind for reading and read back
					}
					// use framebuffer blit

					onResize(n_old_width, n_old_height);
					// change the viewport back
				}

				p_bitmap->Flip(true); // it is upside down
				char p_s_filename[64];
				for(int i = 0; i < 10000; ++ i) {
					stl_ut::Format(p_s_filename, sizeof(p_s_filename),
						"screenshots/screenshot_%03d.tga", i);
					if(!TFileInfo(p_s_filename).b_exists)
						break;
				}
				CTgaCodec::Save_TGA(p_s_filename, *p_bitmap, true);
				p_bitmap->Delete();
				// save hires antialiassed screenshots
			}
			break;
		case 'f':
			b_wireframe = !b_wireframe;
			if(b_wireframe)
				glesemuEnableWireframe();
			else
				glesemuDisableWireframe();
			break;
		case 'n':
			if(p_demo)
				p_demo->onPushSplinePose();
			break;
		case 'm':
			if(p_demo)
				p_demo->onPlayAnimation();
			break;
		case 'j':
			if(p_demo)
				p_demo->onResetAnimation();
			break;
		case 'k':
			if(p_demo)
				p_demo->onSaveAnimation();
			break;
		case 'l':
			if(p_demo)
				p_demo->onLoadAnimation();
			break;
		};
#ifndef __GL_WINDOW_TRACKBALL_STYLE_CAMERA
		const float k = .5f;
		switch(n_key_code) {
		case 'w':
			v_camera_pos += Vector3f(-k * float(sin(f_angle_y)) * float(cos(f_angle_x)),
				k * float(sin(f_angle_x)), float(cos(f_angle_y)) * k * float(cos(f_angle_x)));
			break;
		case 's':
			v_camera_pos -= Vector3f(-k * float(sin(f_angle_y)) * float(cos(f_angle_x)),
				k * float(sin(f_angle_x)), float(cos(f_angle_y)) * k * float(cos(f_angle_x)));
			break;
		case 'a':
			v_camera_pos += Vector3f(k * float(cos(f_angle_y)), 0, k * float(sin(f_angle_y)));
			break;
		case 'd':
			v_camera_pos -= Vector3f(k * float(cos(f_angle_y)), 0, k * float(sin(f_angle_y)));
			break;
		default:
			return; // no glPostRedisplay()
		}
		glutPostRedisplay();
#endif // !__GL_WINDOW_TRACKBALL_STYLE_CAMERA
	}
}

int main(int n_arg_num, char **p_arg_list)
{
	glutInit(&n_arg_num, p_arg_list);
	glutInitDisplayMode(GLUT_RGBA | GLUT_DEPTH | GLUT_DOUBLE);
	glutInitWindowSize(n_width, n_height);
	glutInitWindowPosition(200, 200);
	glutCreateWindow("Underwater Demo");
	// init GLUT

	CGLES20ExtensionHandler::n_GetGLES20FuncPointers();
	CGLES20ExtensionHandler::n_GetARBVertexArrayObjectFuncPointers();
	// ...

	printf("controls:\n"
		"n: onPushSplinePose()\n"
		"m: onPlayAnimation()\n"
		"j: onResetAnimation()\n"
		"k: onSaveAnimation()\n"
		"l: onLoadAnimation()\n");

	//glTexEnvi(/*GL_POINT_SPRITE*/0x8861, /*COORD_REPLACE_NV*/ 0x8862, GL_TRUE); // this is only needed on MAC OS X
    glEnable(/*GL_POINT_SPRITE*/0x8861); // this is deprecated in forward-compatible contexts, only needed when emulating using OpenGL 2.x or lower
	glEnable(/*GL_PROGRAM_POINT_SIZE_EXT*/0x8642); // todo - is this deprecated? if so, why is it in the OpenGL ES emulator then? it has forward compatible contexts, right?
	// enable point sprites (GLES doesn't have glPointSize())
	// todo - clarify how this works

#ifdef MAINDEMO
	CDemo d;
	p_demo = &d;
#else
	CSplineDemo dd;
	p_demo = &dd;
#endif
	// ...

	glutReshapeFunc(onResize);
	glutKeyboardFunc(onKeyPressed);
	glutMouseFunc(onMouseClick);
	glutMotionFunc(onMouseMotion);
	glutDisplayFunc(onDisplay);

	glutMainLoop();

	return 0;
}
