#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include <assert.h>
#define _ASSERTE(expr) assert(expr)

#include "NMEA.h"
#include "SIRF_Binary.h"

/*
 *								=== globals ===
 */

const uint8_t *p_Find_SIRF_MessageStartMarker(const uint8_t *p_buffer, const uint8_t *p_end)
{
	_ASSERTE(p_buffer && p_end);
	_ASSERTE(p_buffer <= p_end);
	if(p_buffer == p_end || p_buffer == p_end - 1)
		return 0;
	// skip too small messages

	-- p_end;
	// so that we can compare without addition

	for(; p_buffer < p_end; ++ p_buffer) { // need to step one at a time to find misaligned marker
		if(p_buffer[0] == 0xa0 && p_buffer[1] == 0xa2)
			return p_buffer;
	}
	// find the message start marker

	return 0;
}

uint16_t n_SIRF_GetWord(const uint8_t *p_buffer, const uint8_t *p_end)
{
	_ASSERTE(p_buffer && p_end);
	_ASSERTE(p_buffer < p_end);
	_ASSERTE(p_buffer + 2 <= p_end);

	return (uint16_t(p_buffer[0]) << 8) | p_buffer[1];
}

bool SIRF_MessageChecksum(uint16_t &r_n_checksum, const uint8_t *p_buffer, const uint8_t *p_end)
{
	_ASSERTE(p_buffer && p_end);
	_ASSERTE(p_buffer <= p_end);
	if(p_Find_SIRF_MessageStartMarker(p_buffer, p_end) != p_buffer || p_buffer + 4 > p_end)
		return false;
	_ASSERTE(p_buffer[0] == 0xa0 && p_buffer[1] == 0xa2);
	// assert the start header

	uint16_t n_length = n_SIRF_GetWord(p_buffer + 2, p_end);
	_ASSERTE(!(n_length & 0x8000)); // the bit 15 should not be set
	// get length

	const uint8_t *p_checksum = p_buffer + (4 + n_length); // 2B start marker, 2B length, payload
	const uint8_t *p_end_marker = p_checksum + 2;
	if(p_end_marker + 1 >= p_end || p_end_marker[0] != 0xb0 || p_end_marker[1] != 0xb3)
		return false;
	// message not entirely in the buffer

	uint16_t n_checksum = 0; 
	for(const uint8_t *p_data = p_buffer + 4; p_data != p_checksum; ++ p_data)
		n_checksum = (n_checksum + *p_data) & 0x7fff;
	r_n_checksum = n_checksum;

	return true;
}

const uint8_t *p_Find_SIRF_MessageChecksum(const uint8_t *p_buffer, const uint8_t *p_end)
{
	_ASSERTE(p_buffer && p_end);
	_ASSERTE(p_buffer <= p_end);
	if(p_Find_SIRF_MessageStartMarker(p_buffer, p_end) != p_buffer || p_buffer + 4 > p_end)
		return 0;
	_ASSERTE(p_buffer[0] == 0xa0 && p_buffer[1] == 0xa2);
	// assert the start header

	uint16_t n_length = n_SIRF_GetWord(p_buffer + 2, p_end);
	//_ASSERTE(!(n_length & 0x8000)); // the bit 15 should not be set
	// get length

	const uint8_t *p_checksum = p_buffer + (4 + n_length); // 2B start marker, 2B length, payload
	const uint8_t *p_end_marker = p_checksum + 2;
	if(p_end_marker + 1 >= p_end)
		return 0;
	// message not entirely in the buffer

	return p_checksum;
}

bool b_Validate_SIRF_Message(uint16_t &r_n_length, const uint8_t *p_buffer, const uint8_t *p_end)
{
	r_n_length = 0;

	const uint8_t *p_cheksum_recv;
	if(!(p_cheksum_recv = p_Find_SIRF_MessageChecksum(p_buffer, p_end)))
		return false;
	// validates the markers and such

	uint16_t n_checksum;
	if(!SIRF_MessageChecksum(n_checksum, p_buffer, p_end))
		return false;
	// gets the checksum

	if(n_SIRF_GetWord(p_cheksum_recv, p_end) != n_checksum)
		return false;
	// make sure the checksums match

	r_n_length = n_SIRF_GetWord(p_buffer + 2, p_end);
	// get length

	return true;
}

void Calculate_SIRF_MessageChecksum(uint8_t *p_buffer, const uint8_t *p_end)
{
	_ASSERTE(p_buffer && p_end);
	_ASSERTE(p_buffer <= p_end);
	_ASSERTE(p_buffer[0] == 0xa0 && p_buffer[1] == 0xa2); // start marker
	// assert the start header

	uint16_t n_length = n_SIRF_GetWord(p_buffer + 2, p_end);
	_ASSERTE(!(n_length & 0x8000)); // the bit 15 should not be set
	// get length

	uint8_t *p_checksum = p_buffer + (4 + n_length); // 2B start marker, 2B length, payload
	const uint8_t *p_end_marker = p_checksum + 2;
	_ASSERTE(p_end_marker + 1 < p_end);
	_ASSERTE(p_end_marker[0] == 0xb0 && p_end_marker[1] == 0xb3);
	// get pointers

	uint16_t n_checksum;
	bool b_result = SIRF_MessageChecksum(n_checksum, p_buffer, p_end);
	_ASSERTE(b_result);
	// calculate checksum of the payload

	_ASSERTE(!p_checksum[0] && !p_checksum[1]);
	// there should be zeros prepared in place of the checksum

	p_checksum[0] = uint8_t(n_checksum >> 8);
	p_checksum[1] = uint8_t(n_checksum);
	_ASSERTE(n_SIRF_GetWord(p_checksum, p_end) == n_checksum);
	// store the checksum
}

void  SIRF_Finalize_Message(uint8_t *p_buffer, const uint8_t *p_end)
{
	size_t n_length = p_end - p_buffer; // size of the array
	_ASSERTE(n_length >= 8); // 2B start marker, 2B length, 2B checksum, 2B end marker
	_ASSERTE(!(n_length & ~0x7ffffU)); // make sure length is 15-bit
	_ASSERTE(p_buffer[0] == 0xa0 && p_buffer[1] == 0xa2 &&
		p_end[-2] == 0xb0 && p_end[-1] == 0xb3); // check the markers
	uint8_t n_length_hi = uint8_t((n_length - 8) >> 8),
		n_length_lo = uint8_t(n_length - 8);
	p_buffer[2] = n_length_hi;
	p_buffer[3] = n_length_lo;
	_ASSERTE(n_SIRF_GetWord(p_buffer + 2, p_end) == n_length - 8);
	Calculate_SIRF_MessageChecksum(p_buffer, p_end);
}

bool SIRF_InitializeDataSource(FT_HANDLE h_device, int n_reset_flags,
	int32_t n_X_meters /*= 0*/, int32_t n_Y_meters /*= 0*/, int32_t n_Z_meters /*= 0*/,
	int32_t n_clock_drift_Hz /*= 75000*/, uint32_t n_time_of_week_100sec /*= 0*/,
	uint16_t n_week_number /*= 0*/, uint8_t n_channel_num /*= 8*/)
{
	_ASSERTE(n_reset_flags <= 0xff);
	uint8_t n_reset_flags8 = uint8_t(n_reset_flags);

	uint8_t p_message[] = {0xa0, 0xa2, // start marker
		0x00, 0x00, // space for message length
		0x80, // Message ID
		uint8_t(n_X_meters >> 24),
		uint8_t(n_X_meters >> 16),
		uint8_t(n_X_meters >> 8),
		uint8_t(n_X_meters), // ECEF X [meters]
		uint8_t(n_Y_meters >> 24),
		uint8_t(n_Y_meters >> 16),
		uint8_t(n_Y_meters >> 8),
		uint8_t(n_Y_meters), // ECEF Y [meters]
		uint8_t(n_Z_meters >> 24),
		uint8_t(n_Z_meters >> 16),
		uint8_t(n_Z_meters >> 8),
		uint8_t(n_Z_meters), // ECEF Z [meters]
		uint8_t(n_clock_drift_Hz >> 24),
		uint8_t(n_clock_drift_Hz >> 16),
		uint8_t(n_clock_drift_Hz >> 8),
		uint8_t(n_clock_drift_Hz), // Clock Drift [Hz]
		uint8_t(n_time_of_week_100sec >> 24),
		uint8_t(n_time_of_week_100sec >> 16),
		uint8_t(n_time_of_week_100sec >> 8),
		uint8_t(n_time_of_week_100sec), // Time of Week * 100 [sec]
		uint8_t(n_week_number >> 8),
		uint8_t(n_week_number), // Week Number (Extended week number (0 - no limit))
		n_channel_num, // Channels (Range 1 - 12)
		n_reset_flags8, // Reset Configuration Bit Map
		0x00, 0x00, // space for checksum
		0xb0, 0xb3}; // end marker
	SIRF_Finalize_StaticArrayMessage(p_message);
	// set to SIRF binary protocol, configure port 0

	DWORD n_written, n_sent = 0;
	do {
		FT_STATUS n_result = FT_Write(h_device, p_message + n_sent,
			sizeof(p_message) - n_sent, &n_written);
		if(n_result != FT_OK)
			return false;
		n_sent +=  n_written;
	} while(n_sent < sizeof(p_message));
	// send it

	return true;
}

bool SIRF_ConfigureUART0(FT_HANDLE h_device, int n_baud_rate, int n_bit_num /*= 8*/,
	int n_stop_bit_num /*= 1*/, int n_parity /*= FT_PARITY_NONE*/)
{
	uint8_t p_message[] = {0xa0, 0xa2, // start marker
		0x00, 0x00, // space for message length
		0xA5, // Message ID
		0x00, // Port
		0x00, // In Protocol
		0x00, // Out Protocol (same as In Protocol)
		uint8_t(n_baud_rate >> 24),
		uint8_t(n_baud_rate >> 16),
		uint8_t(n_baud_rate >> 8),
		uint8_t(n_baud_rate), // Bit Rate
		n_bit_num, // Data Bits
		n_stop_bit_num, // Stop Bits
		(n_parity == FT_PARITY_NONE)? 0 : (n_parity == FT_PARITY_ODD)? 1 : 2, // Parity
		0x00, // Reserved
		0x00, // Reserved
		0xff, // Port 1
		0x05, 0x05, // Dummy Protocols
		0x00, 0x00, 0x00, 0x00, // Dummy Bit Rate
		0x00, // Dummy Data Bits
		0x00, // Dummy Stop Bits
		0x00, // Dummy Parity
		0x00, // Dummy Reserved
		0x00, // Dummy Reserved
		0xff, // Port 2
		0x05, 0x05, // Dummy Protocols
		0x00, 0x00, 0x00, 0x00, // Dummy Bit Rate
		0x00, // Dummy Data Bits
		0x00, // Dummy Stop Bits
		0x00, // Dummy Parity
		0x00, // Dummy Reserved
		0x00, // Dummy Reserved
		0xff, // Port 3
		0x05, 0x05, // Dummy Protocols
		0x00, 0x00, 0x00, 0x00, // Dummy Bit Rate
		0x00, // Dummy Data Bits
		0x00, // Dummy Stop Bits
		0x00, // Dummy Parity
		0x00, // Dummy Reserved
		0x00, // Dummy Reserved
		0x00, 0x00, // space for checksum
		0xb0, 0xb3}; // end marker
	SIRF_Finalize_StaticArrayMessage(p_message);
	// set to SIRF binary protocol, configure port 0

	DWORD n_written, n_sent = 0;
	do {
		FT_STATUS n_result = FT_Write(h_device, p_message + n_sent,
			sizeof(p_message) - n_sent, &n_written);
		if(n_result != FT_OK)
			return false;
		n_sent +=  n_written;
	} while(n_sent < sizeof(p_message));
	// send it

	return true;
}

bool SIRF_SetMessageRate(FT_HANDLE h_device, int n_message_id, int n_rate_sec /*= 1*/, int n_mode /*= 0*/)
{
	uint8_t p_message[] = {0xa0, 0xa2, // start marker
		0x00, 0x00, // space for message length
		0xA6, // Message ID
		n_mode, // Mode: 00 = enable/disable one message
		n_message_id, // Message ID to be set
		n_rate_sec, // Update Rate [sec]
		0x00, // Reserved
		0x00, // Reserved
		0x00, // Reserved
		0x00, // Reserved
		0x00, 0x00, // space for checksum
		0xb0, 0xb3}; // end marker
	SIRF_Finalize_StaticArrayMessage(p_message);

	DWORD n_written, n_sent = 0;
	do {
		FT_STATUS n_result = FT_Write(h_device, p_message + n_sent,
			sizeof(p_message) - n_sent, &n_written);
		if(n_result != FT_OK)
			return false;
		n_sent +=  n_written;
	} while(n_sent < sizeof(p_message));
	// send it

	return true;
}

/*
 *								=== ~globals ===
 */

/*
 *								=== CSIRFProtocolSanitizer ===
 */

bool CSIRFProtocolSanitizer::b_Validate(FT_HANDLE h_device,
	int n_baud_rate, int n_bit_num, int n_stop_bit_num, int n_parity)
{
	char p_s_init[256];
	sprintf(p_s_init, "$PSRF100,0,%d,%d,%d,%d*__\r\n", n_baud_rate, n_bit_num,
		n_stop_bit_num, (n_parity == FT_PARITY_NONE)? 0 : (n_parity == FT_PARITY_ODD)? 1 : 2);
	Calculate_NMEA_MessageChecksum(p_s_init); // !!
	// set from NMEA to SIRF binary protocol

	uint8_t p_init2[] = {0xa0, 0xa2, // start marker
		0x00, 0x00, // space for message length
		0xA5, // Message ID
		0x00, // Port
		0x00, // In Protocol
		0x00, // Out Protocol (same as In Protocol)
		uint8_t(n_baud_rate >> 24),
		uint8_t(n_baud_rate >> 16),
		uint8_t(n_baud_rate >> 8),
		uint8_t(n_baud_rate), // Bit Rate
		n_bit_num, // Data Bits
		n_stop_bit_num, // Stop Bits
		(n_parity == FT_PARITY_NONE)? 0 : (n_parity == FT_PARITY_ODD)? 1 : 2, // Parity
		0x00, // Reserved
		0x00, // Reserved
		0xff, // Port 1
		0x05, 0x05, // Dummy Protocols
		0x00, 0x00, 0x00, 0x00, // Dummy Bit Rate
		0x00, // Dummy Data Bits
		0x00, // Dummy Stop Bits
		0x00, // Dummy Parity
		0x00, // Dummy Reserved
		0x00, // Dummy Reserved
		0xff, // Port 2
		0x05, 0x05, // Dummy Protocols
		0x00, 0x00, 0x00, 0x00, // Dummy Bit Rate
		0x00, // Dummy Data Bits
		0x00, // Dummy Stop Bits
		0x00, // Dummy Parity
		0x00, // Dummy Reserved
		0x00, // Dummy Reserved
		0xff, // Port 3
		0x05, 0x05, // Dummy Protocols
		0x00, 0x00, 0x00, 0x00, // Dummy Bit Rate
		0x00, // Dummy Data Bits
		0x00, // Dummy Stop Bits
		0x00, // Dummy Parity
		0x00, // Dummy Reserved
		0x00, // Dummy Reserved
		0x00, 0x00, // space for checksum
		0xb0, 0xb3}; // end marker
	SIRF_Finalize_StaticArrayMessage(p_init2);
	// set to SIRF binary protocol, configure port 0

	char p_s_buffer[1024];
	int n_switch_num = 0;
	for(int i = 0; i < 4; ++ i) { // try a few times
		DWORD n_written;
		FT_STATUS n_result;
		n_result = FT_Write(h_device, p_init2, sizeof(p_init2), &n_written); // set SIRF baud mode and such
		// write the request

		DWORD n_read;
		n_result = FT_Read(h_device, p_s_buffer, sizeof(p_s_buffer) - sizeof(char), &n_read);
		_ASSERTE(n_read < sizeof(p_s_buffer) / sizeof(p_s_buffer[0]));
		p_s_buffer[n_read] = 0; // zero terminate
		// read a message

		const uint8_t *p_begin = (const uint8_t*)p_s_buffer, *p_end = (const uint8_t*)(p_s_buffer + 1024);
		while((p_begin = p_Find_SIRF_MessageStartMarker(p_begin, p_end)) != 0) {
			uint16_t n_len;
			if(b_Validate_SIRF_Message(n_len, p_begin, p_end))
				return true; // found a valid SIRF binary packet
			++ p_begin;
		}
		// see if it is in binary mode

		char *p_s_message = p_s_buffer;
		while(p_s_message = strchr(p_s_message, '$')) {
			uint8_t n_checksum;
			if(!NMEA_MessageChecksum(n_checksum, p_s_message)) {
				++ p_s_message;
				continue;
			}
			// see if the message has a terminator and we can get a checksum

			const char *p_s_checksum;
			if(!(p_s_checksum = p_s_Find_NMEA_MessageChecksum(p_s_message))) {
				++ p_s_message;
				continue;
			}
			// try to find checksum chars

			int n_recv_checksum;
			sscanf(p_s_checksum, "%x\r\n", &n_recv_checksum);
			if(n_checksum != n_recv_checksum) {
				++ p_s_message;
				continue;
			}
			// compare checksum

			n_result = FT_Write(h_device, p_s_init, DWORD(strlen(p_s_init) * sizeof(char)), &n_written); // in NMEA
			// we received a valid NMEA message with matching
			// checksum, tell the GPS to switch to SIRF

			if(++ n_switch_num < 5) // avoid infinite looping
				-- i; // might have to retry couple of times before the GPS empties its buffers
			break;
		}
		// see if it is in ascii mode
	}

	return false;
	// despite many tries, we are only receiving gibberish
}

/*
 *								=== ~CSIRFProtocolSanitizer ===
 */

/*
 *		end-of-file
 */
