-- buf_fsm.vhd: FSM of buf part of CGMII OBUF
-- Copyright (C) 2012 CESNET
-- Author(s): Vaclav Hummel <xhumme00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use work.math_pack.all;

-- -----------------------------------------------------------------------------
--                             Entity
-- -----------------------------------------------------------------------------
entity obuf_cgmii_buf_fsm is
   port(
      -- Clock signal
      CLK               : in std_logic;
      -- Synchronous reset
      RESET             : in std_logic;

      -- Input interface
      REG_OBUF_EN       : in std_logic;
      TX_SOP            : in std_logic;
      TX_EOP            : in std_logic;
      TRANSACTION_ON    : in std_logic;
      TX_SRC_RDY        : in std_logic;   --SRC_RDY from FIFO
      TX_DST_RDY        : in std_logic;   --DST_RDY from component

      -- Output interface
      MASK_SOP_EN       : out std_logic;  --Enable/disable SOP masking
      MASK_EOP_EN       : out std_logic;  --Enable/disable EOP masking

      TX_SRC_ENABLE     : out std_logic;  --Enable/disable SRC_RDY on TX
      TX_DST_ENABLE     : out std_logic   --Enable/disable DST_RDY on TX
   );
end entity obuf_cgmii_buf_fsm;


architecture obuf_cgmii_buf_fsm_arch of obuf_cgmii_buf_fsm is

   -- Type definition
   type T_FSM is (st_off_no_mask,st_off_on_mask,st_on);

   -- Signals declaration
   signal curr_state : T_FSM := st_off_no_mask;
   signal next_state : T_FSM;
begin

   -- --------------------------------------------------------------------------

   fsmp: process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            curr_state <= st_off_no_mask;
         else
            curr_state <= next_state;
         end if;
      end if;
   end process;

   -- --------------------------------------------------------------------------
   next_state_logic:process(curr_state,REG_OBUF_EN,TRANSACTION_ON,TX_EOP,TX_SRC_RDY,TX_DST_RDY,TX_SOP)
   begin
      next_state <= curr_state;

      case curr_state is
         when st_off_no_mask =>
            if(REG_OBUF_EN = '1')then
               next_state <= st_on;
            end if;

         when st_off_on_mask =>
            if(REG_OBUF_EN = '1')then
               next_state <= st_on;
            end if;

         when st_on  => 
            if(REG_OBUF_EN = '0' and TRANSACTION_ON = '0')then
               next_state <= st_off_no_mask;
            elsif(REG_OBUF_EN = '0' and TX_SOP = '1' and TX_EOP = '1' and TRANSACTION_ON = '1' 
                  and TX_SRC_RDY = '1' and TX_DST_RDY = '1')then
               next_state <= st_off_on_mask;
            end if;

         when others => null;
      end case;
   end process next_state_logic;

   -- --------------------------------------------------------------------------
   output_logic: process(curr_state,REG_OBUF_EN,TRANSACTION_ON,TX_EOP,TX_SOP,TX_SRC_RDY,TX_DST_RDY)
   begin
      --Default values
      TX_SRC_ENABLE  <= TX_SRC_RDY;
      TX_DST_ENABLE  <= TX_DST_RDY;      
      MASK_SOP_EN    <= '0';
      MASK_EOP_EN    <= '0';

      case curr_state is
         when st_off_no_mask  =>
            TX_SRC_ENABLE <= '0';
            TX_DST_ENABLE <= '0';

         when st_off_on_mask  =>
            if(TX_DST_RDY = '1' and TX_SRC_RDY = '1' and REG_OBUF_EN = '1')then
               --Mask EOP
               MASK_EOP_EN <= '1';
            else
               --Output isn't enabled
               TX_SRC_ENABLE <= '0';
               TX_DST_ENABLE <= '0';
            end if;

         when st_on  =>
            if(REG_OBUF_EN = '0' and TRANSACTION_ON = '1' and TX_SOP = '1' and TX_EOP = '1' and TX_SRC_RDY = '1' and TX_DST_RDY = '1')then
               --Mask SOP, stop FIFO, confirm TX
               MASK_SOP_EN <= '1';
               TX_DST_ENABLE <= '0';
               TX_SRC_ENABLE <= '1';
            elsif(REG_OBUF_EN = '0' and TRANSACTION_ON = '0')then
               TX_DST_ENABLE <= '0';
               TX_SRC_ENABLE <= '0';
           end if;

         when others => null;
      end case;
   end process output_logic;

end architecture obuf_cgmii_buf_fsm_arch;
