-- obuf_cgmii_arch.vhd: CGMII Output Buffer full architecture
-- Copyright (C) 2012 CESNET
-- Author(s): Vaclav Hummel <xhumme00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

use work.cgmii_obuf_pkg.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of obuf_cgmii is

   --OBUF FLU RX ------------------------------------------
   signal input_rx_data       : std_logic_vector(511 downto 0);
   signal input_rx_sop_pos    : std_logic_vector(2 downto 0);
   signal input_rx_eop_pos    : std_logic_vector(5 downto 0);
   signal input_rx_sop        : std_logic;
   signal input_rx_eop        : std_logic;
   signal input_rx_src_rdy    : std_logic;
   signal input_rx_dst_rdy    : std_logic;

   --OBUF MI32 --------------------------------------------
   signal input_mi_dwr     : std_logic_vector(31 downto 0); -- Input Data
   signal input_mi_addr    : std_logic_vector(31 downto 0); -- Address
   signal input_mi_rd      : std_logic;                     -- Read Request
   signal input_mi_wr      : std_logic;                     -- Write Request
   signal input_mi_be      : std_logic_vector(3  downto 0); -- Byte Enable
   signal input_mi_drd     : std_logic_vector(31 downto 0); -- Output Data
   signal input_mi_ardy    : std_logic;                     -- Address Ready
   signal input_mi_drdy    : std_logic;                     -- Data Ready

   --PROCESS RX -------------------------------------------
   signal process_rx_data       	: std_logic_vector(511 downto 0);
   signal process_rx_sop_pos    	: std_logic_vector(2 downto 0);
   signal process_rx_eop_pos    	: std_logic_vector(5 downto 0);
   signal process_rx_sop        	: std_logic;
   signal process_rx_eop        	: std_logic;
   signal process_rx_src_rdy    	: std_logic;
   signal process_rx_dst_rdy    	: std_logic;

   --CGMII_ENC RX ----------------------------------------
   signal enc_rx_data      : std_logic_vector(511 downto 0);
   signal enc_rx_sop_v     : std_logic_vector(7 downto 0);
   signal enc_rx_eop_v     : std_logic_vector(7 downto 0);
   signal enc_rx_eop_pos   : std_logic_vector(2 downto 0);

   signal enc_crc32_data_in   : std_logic_vector(31 downto 0);
   signal enc_crc32_read_req  : std_logic;

   --OBUF CGMII TX ---------------------------------------
   signal output_cgmii_txd   : std_logic_vector(511 downto 0);   
   signal output_cgmii_txc   : std_logic_vector(63 downto 0);

   --Other signals ----------------------------------------
   --! Register for information about outgoing packet
   signal outgoing_packet_reg : std_logic := '0';

begin
   --------------------------------------------------------
   --! Input signal map
   --------------------------------------------------------
   --FLU ------------------------------
   --! FLU Data map
   input_rx_data     <= RX_DATA;
   --! FLU SOP_POS map
   input_rx_sop_pos  <= RX_SOP_POS;
   --! FLU EOP_POS map
   input_rx_eop_pos  <= RX_EOP_POS;
   --! FLU SOP map
   input_rx_sop      <= RX_SOP;
   --! FLU EOP map
   input_rx_eop      <= RX_EOP;
   --! FLU SRC_RDY map
   input_rx_src_rdy  <= RX_SRC_RDY;
   --! FLU DST_RDY map
   RX_DST_RDY        <= input_rx_dst_rdy;

   --MI32 -----------------------------
   --! Input data
   input_mi_dwr   <= MI_DWR;
   --! Address
   input_mi_addr  <= MI_ADDR;
   --! Read request
   input_mi_rd    <= MI_RD;
   --! Write request
   input_mi_wr    <= MI_WR;
   --! Byte enable
   input_mi_be    <= MI_BE;
   --! Output data
   MI_DRD         <= input_mi_drd;
   --! Address ready
   MI_ARDY        <= input_mi_ardy;
   --! Data ready
   MI_DRDY        <= input_mi_drdy;


   --------------------------------------------------------
   --! Buf component
   --------------------------------------------------------
   BUF_I:entity work.buf 
   generic map(
      DFIFO_ITEMS => DFIFO_ITEMS,
      HFIFO_ITEMS => HFIFO_ITEMS,
      CNT_DSP     => CNT_DSP
   )
   port map(
      -----------------------------------------------------
      --! \name Clock & Reset 
      -----------------------------------------------------
      FLU_CLK        => RX_CLK,
      FLU_RESET      => RX_RESET,

      CGMII_CLK      => CGMII_TXCLK,
      CGMII_RESET    => CGMII_TXRESET,

      MI_CLK         => MI_CLK,
      MI_RESET       => MI_RESET,
      
      -----------------------------------------------------
      --! \name FLU RX 
      -----------------------------------------------------
      RX_DATA        => input_rx_data,
      RX_SOP_POS    	=> input_rx_sop_pos,
      RX_EOP_POS    	=> input_rx_eop_pos,
      RX_SOP        	=> input_rx_sop,
      RX_EOP        	=> input_rx_eop,
      RX_SRC_RDY    	=> input_rx_src_rdy,
      RX_DST_RDY    	=> input_rx_dst_rdy,
      
      -----------------------------------------------------
      --! \name FLU TX 
      -----------------------------------------------------
      TX_DATA        => process_rx_data,
      TX_SOP_POS     => process_rx_sop_pos,
      TX_EOP_POS     => process_rx_eop_pos,
      TX_SOP         => process_rx_sop,
      TX_EOP         => process_rx_eop,
      TX_SRC_RDY     => process_rx_src_rdy,
      TX_DST_RDY     => process_rx_dst_rdy,

      -----------------------------------------------------
      --! \name MI32 interface 
      -----------------------------------------------------
      MI_DWR      	   => input_mi_dwr,     -- Input Data
      MI_ADDR     	   => input_mi_addr,    -- Address
      MI_RD       	   => input_mi_rd,      -- Read Request
      MI_WR       	   => input_mi_wr,      -- Write Request
      MI_BE       	   => input_mi_be,      -- Byte Enable
      MI_DRD      	   => input_mi_drd,     -- Output Data
      MI_ARDY     	   => input_mi_ardy,    -- Address Ready
      MI_DRDY     	   => input_mi_drdy     -- Data Ready
   );

   --------------------------------------------------------
   --! Process component for CGMII OBUF
   --------------------------------------------------------
   PROCESS_I:entity work.cgmii_obuf_process
   port map(
      -----------------------------------------------------
      --! \ name Common interface
      -----------------------------------------------------
      --! Input clock
      CLK         => CGMII_TXCLK,
      --! Synchronous reset
      RESET       => CGMII_TXRESET,

      -----------------------------------------------------
      --! \name FLU RX interface
      -----------------------------------------------------
      --! Write side data
      RX_DATA       	=> process_rx_data, 
      --! Start of packet position
      RX_SOP_POS    	=> process_rx_sop_pos,
      --! End of packet position
      RX_EOP_POS    	=> process_rx_eop_pos,
      --! Start of packet valid
      RX_SOP        	=> process_rx_sop,
      --! End of packet valid
      RX_EOP        	=> process_rx_eop,
      --! Source ready
      RX_SRC_RDY    	=> process_rx_src_rdy,
      --! Destination ready
      RX_DST_RDY    	=> process_rx_dst_rdy,
      
      -----------------------------------------------------
      --! \name TX output interface 
      -----------------------------------------------------
      --! Read side ready
      TX_DATA        => enc_rx_data,
      --! SOP block indication (one-hot)
      TX_SOP_V       => enc_rx_sop_v,
      --! EOP block indication (one-hot)
      TX_EOP_V       => enc_rx_eop_v,
      --! End in block position
      TX_EOP_POS     => enc_rx_eop_pos,

      -----------------------------------------------------
      --! \name CRC32 FIFO interface
      -----------------------------------------------------
      --! CRC32 FIFO output
      CRC32_DATA_OUT => enc_crc32_data_in,
      --! CRC32 FIFO read request
      CRC32_READ_REQ => enc_crc32_read_req
   );

   --! \brief This register holds information about outgoing packet.
   --! This signal is mapped to CGMII_TXCLK clock domain.
   outgoing_packet_regp:process(CGMII_TXCLK)
   begin
      if(CGMII_TXCLK = '1' and CGMII_TXCLK'event)then
         if(CGMII_TXRESET = '1')then
            outgoing_packet_reg <= '0';
         elsif(process_rx_eop = '1' and process_rx_src_rdy = '1')then
            outgoing_packet_reg <= '0';
         elsif(process_rx_sop = '1' and process_rx_src_rdy = '1')then
            outgoing_packet_reg <= '1';
         end if;
      end if;
   end process;

   --------------------------------------------------------
   --! CGMII encoder component for CGMII OBUF
   --------------------------------------------------------
   CGMII_ENC_I:entity work.cgmii_obuf_enc
   port map(
      -----------------------------------------------------
      --! \ name Common interface
      -----------------------------------------------------
      --! Input clock
      CLK         => CGMII_TXCLK,
      --! Synchronous reset
      RESET       => CGMII_TXRESET,
      
      -----------------------------------------------------
      --! \name RX input interface 
      -----------------------------------------------------
      --! Read side ready
      RX_DATA        => enc_rx_data,
      --! SOP block indication (one-hot)
      RX_SOP_V       => enc_rx_sop_v,
      --! EOP block indication (one-hot)
      RX_EOP_V       => enc_rx_eop_v,
      --! End in block position
      RX_EOP_POS     => enc_rx_eop_pos,

      -----------------------------------------------------
      --! \name CRC32 FIFO interface
      -----------------------------------------------------
      --! CRC32 FIFO input
      CRC32_DATA_IN  => enc_crc32_data_in,
      --! CRC32 FIFO read request
      CRC32_READ_REQ => enc_crc32_read_req,

      -----------------------------------------------------
      --! \name CGMII output interface
      -----------------------------------------------------
      --! CGMII data bus
      CGMII_TXD     => output_cgmii_txd,
      --! CGMII control bus
      CGMII_TXC     => output_cgmii_txc
   );

   --------------------------------------------------------
   --! Output signal map
   --------------------------------------------------------
   --! CGMII TXD map
   CGMII_TXD        <= output_cgmii_txd;
   --! CGMII TXC map
   CGMII_TXC        <= output_cgmii_txc;
   --! Outgoing packet map
   OUTGOING_PACKET   <= outgoing_packet_reg;

end architecture;
