-- mi32_async_handshake.vhd: Component for clock domain crossing on MI32 bus.
--                           Architecture utilizing async_bus_handshake
--                           component.
-- Copyright (C) 2014 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;


-- ----------------------------------------------------------------------------
--                            Entity declaration
-- ----------------------------------------------------------------------------

--! !!!!!!!!!! WATCH OUT !!!!!!!!!!
--! This component does NOT support more than 1 read transaction in progress.

entity MI32_ASYNC_HANDSHAKE is
port(
   --! \name Master MI32 interface
   -- -------------------------------------------------------------------------
   --! \brief Clock
   CLK_M     : in  std_logic;
   --! \brief Reset
   RESET_M   : in  std_logic;
   --! \brief Address bus
   MI_M_ADDR : in  std_logic_vector(31 downto 0);
   --! \brief Data to be written
   MI_M_DWR  : in  std_logic_vector(31 downto 0);
   --! \brief Byte enable signals
   MI_M_BE   : in  std_logic_vector(3  downto 0);
   --! \brief Read request
   MI_M_RD   : in  std_logic;
   --! \brief Write request
   MI_M_WR   : in  std_logic;
   --! \brief Address ready
   MI_M_ARDY : out std_logic;
   --! \brief Read data ready
   MI_M_DRDY : out std_logic;
   --! \brief Read data
   MI_M_DRD  : out std_logic_vector(31 downto 0);

   --! \name Slave interface MI32 interface
   -- -------------------------------------------------------------------------
   --! \brief Clock
   CLK_S     : in  std_logic;
   --! \brief Reset
   RESET_S   : in  std_logic;
   --! \brief Address bus
   MI_S_ADDR : out std_logic_vector(31 downto 0);
   --! \brief Data to be written
   MI_S_DWR  : out std_logic_vector(31 downto 0);
   --! \brief Byte enable signals
   MI_S_BE   : out std_logic_vector(3  downto 0);
   --! \brief Read request
   MI_S_RD   : out std_logic;
   --! \brief Write request
   MI_S_WR   : out std_logic;
   --! \brief Address ready
   MI_S_ARDY : in  std_logic;
   --! \brief Read data ready
   MI_S_DRDY : in  std_logic;
   --! \brief Read data
   MI_S_DRD  : in  std_logic_vector(31 downto 0)
);
end entity MI32_ASYNC_HANDSHAKE;


-- ----------------------------------------------------------------------------
--                         Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of MI32_ASYNC_HANDSHAKE is

   --! Constants declaration
   -- -------------------------------------------------------------------------
   
   --! Data width of m2s interface (ADDR + DWR + BE + RD + WR)
   constant C_M2S_DATA_WIDTH : integer := 32 + 32 + 4 + 1 + 1;

   --! Signals declaration
   -- -------------------------------------------------------------------------
   
   --! m2s input side
   signal m2s_data_in       : std_logic_vector(C_M2S_DATA_WIDTH-1 downto 0);
   signal m2s_send          : std_logic;
   signal m2s_ready         : std_logic;
   
   --! m2s output side
   signal m2s_data_out      : std_logic_vector(C_M2S_DATA_WIDTH-1 downto 0);
   signal m2s_load          : std_logic;
   signal m2s_valid         : std_logic;
   signal mi_s_ardy_in      : std_logic;
   signal wait_for_ardy     : std_logic := '0';

   --! s2m input side
   signal s2m_ready         : std_logic;
   
   --! s2m output side
   signal s2m_valid         : std_logic;
   signal reg_s2m_valid     : std_logic;
   signal s2m_load          : std_logic;


-- ----------------------------------------------------------------------------
--                            Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --                      Master to slave interface
   -- -------------------------------------------------------------------------

   --! Input side logic
   -- -------------------------------------------------------------------------

   --! Composition of inputs to CDC component
   m2s_data_in <= MI_M_ADDR & -- (69 downto 38)
                  MI_M_DWR  & -- (37 downto  6)
                  MI_M_BE   & -- ( 5 downto  2)
                  MI_M_RD   & -- (1)
                  MI_M_WR;    -- (0)
   m2s_send <= MI_M_RD OR MI_M_WR;

   --! Driving output ports
   MI_M_ARDY <= m2s_send AND m2s_ready;

   --! Instantiation of component implementing CDC using handshake
   -- -------------------------------------------------------------------------

   m2s_handshake_i : entity work.ASYNC_BUS_HANDSHAKE
   generic map (
      DATA_WIDTH => C_M2S_DATA_WIDTH
   )
   port map ( 
      --! A clock domain
      ACLK       => CLK_M,
      ARST       => RESET_M,
      ADATAIN    => m2s_data_in,
      ASEND      => m2s_send,
      AREADY     => m2s_ready,
     
      --! B clock domain
      BCLK       => CLK_S,
      BRST       => RESET_S,
      BDATAOUT   => m2s_data_out,
      BLOAD      => m2s_load,
      BVALID     => m2s_valid
   );

   --! Output side logic
   -- -------------------------------------------------------------------------

   --! Internal representation of port value (just because of simulation)
   mi_s_ardy_in <= MI_S_ARDY;

   --! Wait for ARDY flag logic
   wait_for_ardy_p : process(CLK_S)
   begin
      if (rising_edge(CLK_S)) then
         if (RESET_S = '1') then
            wait_for_ardy <= '0';
         elsif (m2s_load = '1') then
            wait_for_ardy <= '1';
         elsif (mi_s_ardy_in = '1') then
            wait_for_ardy <= '0';
         end if;
      end if;
   end process wait_for_ardy_p;

   --! Load signal composition
   m2s_load <= m2s_valid AND         --! new transaction is available
               NOT wait_for_ardy AND --! ARDY for previous transaction has been received
               s2m_ready;            --! read data has been properly synchronized

   --! Driving output ports
   MI_S_ADDR <= m2s_data_out(C_M2S_DATA_WIDTH- 1 downto C_M2S_DATA_WIDTH-32);
   MI_S_DWR  <= m2s_data_out(C_M2S_DATA_WIDTH-33 downto C_M2S_DATA_WIDTH-64);
   MI_S_BE   <= m2s_data_out(5 downto 2);
   MI_S_RD   <= m2s_data_out(1) AND wait_for_ardy;
   MI_S_WR   <= m2s_data_out(0) AND wait_for_ardy;


   -- -------------------------------------------------------------------------
   --                      Slave to master interface
   -- -------------------------------------------------------------------------

   --! Instantiation of component implementing CDC using handshake
   -- -------------------------------------------------------------------------

   s2m_handshake_i : entity work.ASYNC_BUS_HANDSHAKE
   generic map (
      DATA_WIDTH => 32
   )
   port map ( 
      --! A clock domain
      ACLK       => CLK_S,
      ARST       => RESET_S,
      ADATAIN    => MI_S_DRD,
      ASEND      => MI_S_DRDY,
      AREADY     => s2m_ready,
     
      --! B clock domain
      BCLK       => CLK_M,
      BRST       => RESET_M,
      BDATAOUT   => MI_M_DRD,
      BLOAD      => s2m_load,
      BVALID     => s2m_valid
   );

   --! Output side logic
   -- -------------------------------------------------------------------------

   --! Load signal composition
   s2m_load <= s2m_valid;

   --! Aligning s2m_valid with output data
   reg_s2m_valid_p : process(CLK_M)
   begin
      if (rising_edge(CLK_M)) then
         if (RESET_M = '1') then
            reg_s2m_valid <= '0';
         else
            reg_s2m_valid <= s2m_valid;
         end if;
      end if;
   end process reg_s2m_valid_p;

   --! Driving output port
   MI_M_DRDY <= reg_s2m_valid;

end architecture full;
