-- dic_block_table.vhd: DIC block table
-- Copyright (C) 2012 CESNET
-- Author(s): Pavel Benacek <benacek@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
--! Block shift lookup table
entity DIC_BLOCK_TABLE is
   port (
      -----------------------------------------------------
      --! \name Input interface
      -----------------------------------------------------
      --! Actual DIC value
      ACT_DIC_VAL       : in std_logic_vector(2 downto 0);
      --! Inter-frame gap between packets
      FLU_GAP           : in std_logic_vector(4 downto 0);

      -----------------------------------------------------
      --! \name Output interface
      -----------------------------------------------------
      --! SOP block shift with respect to ACT_DIC_VAL, SOP_VECT and FLU_GAP
      SOP_BLOCK_SHIFT   : out std_logic_vector(1 downto 0);
      --! New DIC value with respect to ACT_DIC_VAL, SOP_VECT and FLU_GAP
      NEW_DIC_VAL       : out std_logic_vector(2 downto 0)
   );
end entity DIC_BLOCK_TABLE;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of DIC_BLOCK_TABLE is
   --! DIC value enable/disable signal
   signal dic_en  : std_logic;

   --! Alias for two MSB bits
   alias UP is FLU_GAP(4 downto 3);
begin

   --! \brief This process counts SOP block shift.
   --! \details This process calculate new SOP position with respect 
   --! to actual DIC value and measured FLU gap between two FLU frames.
   block_shiftp:process(dic_en,FLU_GAP)
   begin
      --Default values
      SOP_BLOCK_SHIFT <= "00";

      if(FLU_GAP < 8)then
         --Measured gap is between 0-7
         if(dic_en = '1')then
            SOP_BLOCK_SHIFT <= "10";
         else
            SOP_BLOCK_SHIFT <= "11";
         end if;

      elsif(FLU_GAP < 16)then
         --Measured gap is between 8-15
         if(dic_en = '1')then
            SOP_BLOCK_SHIFT <= "01";
         else
            SOP_BLOCK_SHIFT <= "10";
         end if;

      elsif(FLU_GAP < 24)then
         --Measured gap is between 16-23
         if(dic_en = '1')then
            SOP_BLOCK_SHIFT <= "00"; 
         else
            SOP_BLOCK_SHIFT <= "01";
         end if;

      end if;
   end process;

   --! \brief This process counts new DIC value.
   --! \details This process calculate new DIC value with respect
   --! to actual DIC value and measured FLU gap between two FLU frames.
   --! It also indicates if DIC could be used.
   dic_countp:process(FLU_GAP,ACT_DIC_VAL)
   begin
      --Default values
      dic_en         <= '0'; --DIC is disabled by default
      NEW_DIC_VAL    <= ACT_DIC_VAL;

      case FLU_GAP(2 downto 0) is
         when "001" =>
            if(ACT_DIC_VAL = 0)then
               dic_en <= '1';
               NEW_DIC_VAL <= "111";
            else
               NEW_DIC_VAL <= ACT_DIC_VAL - 1;
            end if;
         
         when "010" =>
            if(ACT_DIC_VAL < 2)then
               if(UP /= "11")then
                  dic_en <= '1';
                  NEW_DIC_VAL <= ACT_DIC_VAL + 6;
               else
                  NEW_DIC_VAL <= "000";
               end if;
            else
               NEW_DIC_VAL <= ACT_DIC_VAL - 2;
            end if;

         when "011" => 
            if(ACT_DIC_VAL < 3)then
               if(UP /= "11")then
                  dic_en <= '1';
                  NEW_DIC_VAL <= ACT_DIC_VAL + 5;
               else    
                  NEW_DIC_VAL <= "000";
               end if;
            else
               NEW_DIC_VAL <= ACT_DIC_VAL - 3;
            end if;

         when "100" =>
            if(ACT_DIC_VAL < 4)then
               if(UP /= "11")then
                  dic_en <= '1';
                  NEW_DIC_VAL <= ACT_DIC_VAL + 4;
               else
                  NEW_DIC_VAL <= "000";
               end if;  
            else
               NEW_DIC_VAL <= ACT_DIC_VAL - 4;
            end if;

         when "101" =>
            if(ACT_DIC_VAL < 5)then
               if(UP /= "11")then
                  dic_en <= '1';
                  NEW_DIC_VAL <= ACT_DIC_VAL + 3;
               else
                  NEW_DIC_VAL <= "000";
               end if;
            else
               NEW_DIC_VAL <= ACT_DIC_VAL - 5;
            end if;

         when "110" =>
            if(ACT_DIC_VAL < 6)then
               if(UP /= "11")then
                  dic_en <= '1';
                  NEW_DIC_VAL <= ACT_DIC_VAL + 2;
               else
                  NEW_DIC_VAL <= "000";
               end if;
            else
               NEW_DIC_VAL <= ACT_DIC_VAL - 6;
            end if;

         when "111" =>
            --In this state, check counter saturation.
            --FLU_GAP(4 downto 3) = "11" & CASE statement "111" ==> FLU_GAP is 31. So, new DIC value is 0
            if(ACT_DIC_VAL < 7 and UP /= "11")then
               dic_en <= '1';
               NEW_DIC_VAL <= ACT_DIC_VAL + 1;
            else
               NEW_DIC_VAL <= "000";
            end if;
         
         when others => null;
      end case;
   end process;

end architecture full;
