-- descrambler_gen.vhd : Parellel descrambler for 10/40/100G Ethernet with 
--                       generics data width (polynomial: 1 + X^39 + X^58)
--                      
-- Copyright (C) 2010 CESNET
-- Author(s): Stepan Friedl <friedl@liberouter.org>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: comboi10g4_arch.vhd 13948 2010-06-04 15:49:43Z xfried00 $
--
-- NOTES: 
-- Polynomial 
-- +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
-- |Sr0 Sr1 Sr2          ....       Sr38                   Sr57|
-- +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
--  ^                                |                       |
--  |                    Din -+---> XOR <--------------------+  
--  |                         |      |
--  +-------------------------+      +---> Dout

-- Sr0 = D0
-- Sr1 = D1
-- ...
-- Sr57 = D57
-- 
-- Dout0 = D0 + Sr38 + Sr57    
-- Dout1 = D1 + Sr37 + Sr56
-- ...
-- Dout38 = D38 + Sr0 + Sr19  => Dout(i) = Di + Sr(38-i) + Sr(57-i)
-- Dout39 = D39 + D0 + Sr18
-- ...
-- Dout57 = D57 + D(57-39) + Sr0 => Dout(i) = Di + D(i-39) + Sr(57-i)
-- Dout58 = D58 + D(58-39) + D0 
-- Dout59 = D59 + D(59-39) + D(59-58) => Dout(i) = Di + D(i-39) + Sr(i-58)
-- 

library ieee;
use ieee.std_logic_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity descrambler_gen is
   generic (
      WIDTH : natural range 58 to 2048 := 256
      );
   port (
      RESET     : in std_logic;
      CLK       : in std_logic; -- TX clock, 156.25MHz
      EN        : in std_logic; -- Clock enable
      BYPASS    : in std_logic := '0'; -- 
      SEED      : in std_logic_vector(57 downto 0);  -- initial seed
      D         : in std_logic_vector(WIDTH-1 downto 0);  -- Input data
      Q         : out std_logic_vector(WIDTH-1 downto 0)  -- Output data
      
   );
end descrambler_gen;

architecture behavioral of descrambler_gen is

signal sr   : std_logic_vector(57 downto 0);
signal dout : std_logic_vector(WIDTH-1 downto 0);

begin

GEN_S0_S38: for i in 0 to 38 generate
   dout(i) <= D(i) xor sr(38-i) xor sr(57-i); -- Dout(i) = Di + Sr(38-i) + Sr(57-i)
end generate; 

GEN_S39_S57: for i in 39 to 57 generate
   dout(i) <= D(i) xor D(i-39) xor sr(57-i); -- Dout(i) = Di + D(i-39) + Sr(57-i) 
end generate;

GEN_S58: for i in 58 to dout'high generate
   dout(i) <= D(i) xor D(i-39) xor D(i-58); -- Dout(i) = Di + D(i-39) + D(i-58)???
end generate;

S_SEQ: process(clk, reset)
begin
   if CLK'event and CLK = '1' then
      if RESET = '1' then
         sr <= SEED;
      elsif (EN = '1') then
         for i in 0 to sr'high loop
            sr(i) <= D(dout'high-i);
         end loop;
      end if;
   end if;
end process;

Q <= dout when BYPASS = '0' else D;

end behavioral;
