-- buf_arch.vhd: buf full architecture.
-- Copyright (C) 2012 CESNET
-- Author(s): Vaclav Hummel <xhumme00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use work.math_pack.all;

use work.cgmii_obuf_pkg.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of buf is
   --Constants --------------------------------------------
   -- addresses
   constant REG_CNT_PACKETS_ADDR_LOW   : std_logic_vector(5 downto 2) := "0000";
   constant REG_CNT_PACKETS_ADDR_HIGH  : std_logic_vector(5 downto 2) := "0100";
   constant REG_CNT_BYTES_ADDR_LOW     : std_logic_vector(5 downto 2) := "0001";
   constant REG_CNT_BYTES_ADDR_HIGH    : std_logic_vector(5 downto 2) := "0101";
   constant REG_OBUF_EN_ADDR           : std_logic_vector(5 downto 2) := "1000";
   constant REG_CTRL_ADDR              : std_logic_vector(5 downto 2) := "1011";
   constant REG_STATUS_ADDR            : std_logic_vector(5 downto 2) := "1100";

   --input into FIFO -------------------------------------
   signal rx_data_in       : std_logic_vector(511 downto 0);
   signal rx_sop_pos_in    : std_logic_vector(2 downto 0);
   signal rx_eop_pos_in    : std_logic_vector(5 downto 0);
   signal rx_sop_in        : std_logic;
   signal rx_eop_in        : std_logic;
   signal rx_src_rdy_in    : std_logic;
   signal rx_dst_rdy_in    : std_logic;

   signal wait_sop         : std_logic;
   signal cnt_words        : std_logic_vector(log2(DFIFO_ITEMS) - 1 downto 0);

   --PIPELINE STAGE 1 -------------------------------------
   signal pipe_st1_rx_data       : std_logic_vector(511 downto 0);
   signal pipe_st1_rx_sop_pos    : std_logic_vector(2 downto 0);
   signal pipe_st1_rx_eop_pos    : std_logic_vector(5 downto 0);
   signal pipe_st1_rx_sop        : std_logic;
   signal pipe_st1_rx_eop        : std_logic;
   signal pipe_st1_rx_src_rdy    : std_logic;
   signal pipe_st1_rx_dst_rdy    : std_logic;
   
   --PIPELINE STAGE 2 -------------------------------------
   signal pipe_st2_tx_data       : std_logic_vector(511 downto 0);
   signal pipe_st2_tx_sop_pos    : std_logic_vector(2 downto 0);
   signal pipe_st2_tx_eop_pos    : std_logic_vector(5 downto 0);
   signal pipe_st2_tx_sop        : std_logic;
   signal pipe_st2_tx_eop        : std_logic;
   signal pipe_st2_tx_src_rdy    : std_logic;
   signal pipe_st2_tx_dst_rdy    : std_logic;

   --Buf signals ------------------------------------------
   signal flu_tx_data         : std_logic_vector(511 downto 0);
   signal flu_tx_dst_rdy      : std_logic;
   signal flu_tx_src_rdy      : std_logic;
   signal flu_tx_sop_pos      : std_logic_vector(2 downto 0);
   signal flu_tx_sop_pos_rep  : std_logic_vector(5 downto 0);
   signal flu_tx_eop_pos      : std_logic_vector(5 downto 0);
   signal flu_tx_sop          : std_logic;
   signal flu_tx_sop_out      : std_logic;
   signal flu_tx_eop          : std_logic;
   signal flu_tx_eop_out      : std_logic;
   signal output_flu_src_rdy  : std_logic;
   signal output_flu_dst_rdy  : std_logic;
   signal transaction_on      : std_logic := '0';
   signal wait_for_sop        : std_logic := '0';
   signal reg_wait_for_sop    : std_logic;

   --Mask signals
   signal mask_sop_en      : std_logic;
   signal mask_eop_en      : std_logic;

   --Statistical counters
   signal reset_counters      : std_logic;
   signal cnt_packets         : std_logic_vector(63 downto 0) := (others => '0');
   signal cnt_packets_en      : std_logic;
   signal cnt_bytes           : std_logic_vector(63 downto 0) := (others => '0');
   signal cnt_bytes_in0       : std_logic_vector(6 downto 0);
   signal cnt_bytes_in1       : std_logic_vector(6 downto 0);
   signal cnt_bytes_in2       : std_logic_vector(6 downto 0);
   signal cnt_bytes_in3       : std_logic_vector(6 downto 0);
   signal cnt_bytes_in4       : std_logic_vector(6 downto 0);
   signal cnt_bytes_in5       : std_logic_vector(6 downto 0);
   signal reg_cnt_bytes_in    : std_logic_vector(6 downto 0);
   signal mux_cnt_bytes_out   : std_logic_vector(6 downto 0);
   signal mux_cnt_bytes_sel   : std_logic_vector(2 downto 0);
   signal count_dsp_reset     : std_logic;
   signal count_dsp_a         : std_logic_vector(63 downto 0);
   signal cnt_packets_a       : std_logic_vector(63 downto 0);

   --MI32 -------------------------------------------------
   --Output registers (for snapshots)
   signal reg_counters_we  : std_logic; --Signal to take register values
   signal reg_cnt_packets  : std_logic_vector(63 downto 0);
   signal reg_cnt_bytes    : std_logic_vector(63 downto 0);
   signal reg_ctrl_we      : std_logic;
   signal reg_obuf_en      : std_logic := '0';
   signal reg_obuf_en_we   : std_logic;
   signal reg_status_we    : std_logic; 
   signal reg_status       : std_logic_vector(6 downto 0) := "1010000"; 

   --MI32 
   signal mi_int_dwr             : std_logic_vector(31 downto 0); 
   signal mi_int_addr            : std_logic_vector(31 downto 0); 
   signal mi_int_rd              : std_logic; 
   signal mi_int_wr              : std_logic; 
   signal mi_int_be              : std_logic_vector(3 downto 0); 
   signal mi_int_drd             : std_logic_vector(31 downto 0); 
   signal mi_int_ardy            : std_logic; 
   signal mi_int_drdy            : std_logic; 
   
   signal mi_master_dwr          : std_logic_vector(31 downto 0); 
   signal mi_master_addr         : std_logic_vector(31 downto 0); 
   signal mi_master_rd           : std_logic; 
   signal mi_master_wr           : std_logic; 
   signal mi_master_be           : std_logic_vector(3 downto 0); 
   signal mi_master_drd          : std_logic_vector(31 downto 0); 
   signal mi_master_ardy         : std_logic; 
   signal mi_master_drdy         : std_logic; 
   
   -- Decoder
   signal mx_decoder_mi_drd      : std_logic_vector(31 downto 0);
   signal reg_obuf_en_cs         : std_logic;
   signal reg_ctrl_cs            : std_logic;
   signal reg_status_cs          : std_logic;
   signal reg_obuf_en32          : std_logic_vector(31 downto 0);
   signal reg_status32           : std_logic_vector(31 downto 0);

begin

   --FIFO stuck protection

   wait_sopp:process(FLU_CLK)
   begin
      if(FLU_CLK = '1' and FLU_CLK'event)then
         if(FLU_RESET = '1')then
            cnt_words <= (others=>'0');
            wait_sop <= '0';
         elsif( rx_src_rdy_in = '1' and rx_dst_rdy_in = '1')then
            if(wait_sop = '0')then
               --sop was not detected before
               if(RX_SOP = '1' and RX_EOP = '0')then
                  cnt_words <= conv_std_logic_vector(1,log2(DFIFO_ITEMS));
                  wait_sop <= '1';	
               end if;
            else
               --sop was detected before
               if(RX_SOP = '1' and RX_EOP = '1')then
                  cnt_words <= conv_std_logic_vector(1,log2(DFIFO_ITEMS));
               elsif(RX_EOP = '1')then
                  cnt_words <= (others => '0');
               else
                  cnt_words <= cnt_words + 1;
               end if;
            end if;
         end if;
      end if;
   end process;

  --OUTPUT from stuck protection

   rx_src_rdy_in <= RX_SRC_RDY;
   RX_DST_RDY <= rx_dst_rdy_in;
   rx_data_in <= RX_DATA;

   rx_sop_in <= '1' when (rx_src_rdy_in = '1' and rx_dst_rdy_in = '1' and (conv_integer(cnt_words) = (DFIFO_ITEMS - 1)) and RX_EOP = '0')else
                RX_SOP;
   
   rx_eop_in <= '1' when (rx_src_rdy_in = '1' and rx_dst_rdy_in = '1' and (conv_integer(cnt_words) = (DFIFO_ITEMS - 1)) and RX_EOP = '0')else
                RX_EOP;

   rx_sop_pos_in <= "001" when (rx_src_rdy_in = '1' and rx_dst_rdy_in = '1' and (conv_integer(cnt_words) = (DFIFO_ITEMS - 1)) and RX_EOP = '0')else
                RX_SOP_POS;

   rx_eop_pos_in <= "000111" when (rx_src_rdy_in = '1' and rx_dst_rdy_in = '1' and (conv_integer(cnt_words) = (DFIFO_ITEMS - 1)) and RX_EOP = '0')else
                RX_EOP_POS;

  --------------------------------------------------------
  -- FIFO part
  --------------------------------------------------------
   PACKET_FIFO_I:entity work.FLU_PFIFO 
   generic map(
      DATA_WIDTH     => 512,
      SOP_POS_WIDTH  => 3,
      DFIFO_ITEMS    => DFIFO_ITEMS,
      HFIFO_ITEMS    => HFIFO_ITEMS
   )
   port map(
      -----------------------------------------------------
      --! \name Clocking & Reset interface
      -----------------------------------------------------
      RX_CLK            => FLU_CLK,
      RX_RESET          => FLU_RESET,
      TX_CLK            => CGMII_CLK,
      TX_RESET          => CGMII_RESET,

      -----------------------------------------------------
      --! \name Frame Link Unaligned input interface
      -----------------------------------------------------
      RX_DATA       => rx_data_in,
      RX_SOP_POS    => rx_sop_pos_in,
      RX_EOP_POS    => rx_eop_pos_in,
      RX_SOP        => rx_sop_in,
      RX_EOP        => rx_eop_in,
      RX_SRC_RDY    => rx_src_rdy_in,
      RX_DST_RDY    => rx_dst_rdy_in,
      RX_STATUS     => open,
      
      -----------------------------------------------------
      --! \name Frame Link Unaligned output interface
      -----------------------------------------------------
      TX_DATA       => pipe_st1_rx_data,
      TX_SOP_POS    => pipe_st1_rx_sop_pos,
      TX_EOP_POS    => pipe_st1_rx_eop_pos,
      TX_SOP        => pipe_st1_rx_sop,
      TX_EOP        => pipe_st1_rx_eop,
      TX_SRC_RDY    => pipe_st1_rx_src_rdy,
      TX_DST_RDY    => pipe_st1_rx_dst_rdy,

      -----------------------------------------------------
      --! \name Output statistical interface
      -----------------------------------------------------
      PACKET_COUNT   => open

   );

   --FLU pipeline stage 1
   PIPE_ST1_I:entity work.FLU_PIPE
   generic map(
      -- FrameLinkUnaligned Data Width
      DATA_WIDTH     => 512,
      SOP_POS_WIDTH  => 3, 
      USE_OUTREG     => false,
      FAKE_PIPE      => false
   )   
   port map(
      -- Common interface 
      CLK            => CGMII_CLK,
      RESET          => CGMII_RESET,
      
      -- Input interface
      RX_DATA        => pipe_st1_rx_data,
      RX_SOP_POS     => pipe_st1_rx_sop_pos,
      RX_EOP_POS     => pipe_st1_rx_eop_pos,
      RX_SOP         => pipe_st1_rx_sop,
      RX_EOP         => pipe_st1_rx_eop,
      RX_SRC_RDY     => pipe_st1_rx_src_rdy,
      RX_DST_RDY     => pipe_st1_rx_dst_rdy,
 
      -- Output interface
      TX_DATA        => flu_tx_data,
      TX_SOP_POS     => flu_tx_sop_pos,
      TX_EOP_POS     => flu_tx_eop_pos,
      TX_SOP         => flu_tx_sop,
      TX_EOP         => flu_tx_eop,
      TX_SRC_RDY     => flu_tx_src_rdy,
      TX_DST_RDY     => flu_tx_dst_rdy
   );

   flu_tx_sop_out <= flu_tx_sop and not(mask_sop_en);
   flu_tx_eop_out <= flu_tx_eop and not(mask_sop_en);

   --FLU pipeline stage 1
   PIPE_ST2_I:entity work.FLU_PIPE
   generic map(
      -- FrameLinkUnaligned Data Width
      DATA_WIDTH     => 512,
      SOP_POS_WIDTH  => 3, 
      USE_OUTREG     => false,
      FAKE_PIPE      => false
   )   
   port map(
      -- Common interface 
      CLK            => CGMII_CLK,
      RESET          => CGMII_RESET,
      
      -- Input interface
      RX_DATA        => flu_tx_data,
      RX_SOP_POS     => flu_tx_sop_pos,
      RX_EOP_POS     => flu_tx_eop_pos,
      RX_SOP         => flu_tx_sop_out,
      RX_EOP         => flu_tx_eop_out,
      RX_SRC_RDY     => output_flu_src_rdy,
      RX_DST_RDY     => output_flu_dst_rdy,
 
      -- Output interface
      TX_DATA        => pipe_st2_tx_data,
      TX_SOP_POS     => pipe_st2_tx_sop_pos,
      TX_EOP_POS     => pipe_st2_tx_eop_pos,
      TX_SOP         => pipe_st2_tx_sop,
      TX_EOP         => pipe_st2_tx_eop,
      TX_SRC_RDY     => pipe_st2_tx_src_rdy,
      TX_DST_RDY     => pipe_st2_tx_dst_rdy
   );

   --Output map -----------------------
   TX_DATA <= pipe_st2_tx_data;
   TX_SOP_POS  <= pipe_st2_tx_sop_pos;
   TX_EOP_POS  <= pipe_st2_tx_eop_pos;
   TX_SOP      <= pipe_st2_tx_sop;
   TX_EOP      <= pipe_st2_tx_eop;
   TX_SRC_RDY  <= pipe_st2_tx_src_rdy;
   pipe_st2_tx_dst_rdy <= TX_DST_RDY;
   --Repaired sop_pos
   flu_tx_sop_pos_rep <= flu_tx_sop_pos & "000";

   --Transaction detector(for transactions which takes 2 clocks or more)

   trans_detp:process(CGMII_CLK)
   begin
      if(CGMII_CLK = '1' and CGMII_CLK'event)then
         if(CGMII_RESET = '1')then
            transaction_on <= '0';
         else
            if(flu_tx_src_rdy = '1' and flu_tx_dst_rdy = '1')then
               if(flu_tx_sop_pos_rep > flu_tx_eop_pos and flu_tx_sop = '1' and flu_tx_eop = '1' and reg_obuf_en = '1')then
                  --Transaction begin
                  transaction_on <= '1';
               elsif(transaction_on = '1' and flu_tx_eop = '1')then
                  --Transaction has been finished
                  transaction_on <= '0';
               end if;
            end if;
         end if;
      end if;
   end process;

   --------------------------------------------------------
   --FSM
   --------------------------------------------------------

   FSM_I:entity work.obuf_cgmii_buf_fsm
   port map(
      -- Clock signal
      CLK               => CGMII_CLK,
      -- Synchronous reset
      RESET             => CGMII_RESET,

      -- Input interface
      REG_OBUF_EN       => reg_obuf_en,
      TX_SOP            => flu_tx_sop,
      TX_EOP            => flu_tx_eop,
      TRANSACTION_ON    => transaction_on,
      TX_SRC_RDY        => flu_tx_src_rdy,   --SRC_RDY from FIFO
      TX_DST_RDY        => output_flu_dst_rdy,  --DST_RDY from component input

      -- Output interface
      MASK_SOP_EN       => mask_sop_en, --Enable/disable SOP masking
      MASK_EOP_EN       => mask_eop_en, --Enable/disable EOP masking

      TX_SRC_ENABLE     => output_flu_src_rdy,  --Enable/disable SRC_RDY on TX (output from component)
      TX_DST_ENABLE     => flu_tx_dst_rdy    --Enable/disable DST_RDY on TX (input to FIFO)
   );

   --------------------------------------------------------
   --Statistical counters
   --------------------------------------------------------
   --Enable signals
   --Count up if source and destination is ready and the end of the packet has been detedted
   cnt_packets_en <= '1' when(output_flu_dst_rdy = '1' and output_flu_src_rdy = '1' and  flu_tx_eop_out = '1')
                     else '0';

  --Total sent packets
   cnt_packetsg: if NOT CNT_DSP generate
      cnt_packetsp:process(CGMII_CLK)      
      begin     
         if(CGMII_CLK = '1' and CGMII_CLK'event)then      
            if((CGMII_RESET = '1') or reset_counters = '1')then     
               cnt_packets <= (others=>'0');     
            else     
               if(cnt_packets_en = '1')then     
                  cnt_packets <= cnt_packets + 1;     
               end if;     
            end if;     
         end if;     
      end process;
   end generate;

   cnt_dsp_packetsg: if CNT_DSP generate
      cnt_packets_a <= (63 downto 1 => '0') & cnt_packets_en;

      cnt_packets_dspi:entity work.COUNT_DSP
      generic map(
         DATA_WIDTH => 64,
         --! Input pipeline registers
         REG_IN => 1,
         --! Reset when MAX == P ( 0 => "NO_RESET", 1 => "RESET_MATCH") 
         AUTO_RESET => 0
      )
      port map(
         --! Clock input
         CLK => CGMII_CLK,
         --! Enable input
         ENABLE => '1',
         --! Reset input
         RESET => count_dsp_reset,
         --! Data input 
         A => cnt_packets_a,
         --! Data input (must MAX % A = 0), Maximum value 
         MAX => (others => 'X'),
         --! Data output
         P => cnt_packets
      );
   end generate;

   --Total sent bytes mux selection signal
   mux_cnt_bytes_selp:process(output_flu_src_rdy, output_flu_dst_rdy, wait_for_sop, flu_tx_sop_out, flu_tx_eop_out)
   begin
      mux_cnt_bytes_sel <= "000";
      if(output_flu_src_rdy = '1' and output_flu_dst_rdy = '1')then
         if(wait_for_sop = '0')then
            --waiting for sop
            if(flu_tx_sop_out = '1' and flu_tx_eop_out = '1') then
               --packet starts and ends in the same flu word, add length of packet
               mux_cnt_bytes_sel <= "001";
            elsif(flu_tx_sop_out = '1')then
               --sop detected, add length to the end of the flu word
               mux_cnt_bytes_sel <= "010";
            end if;
         else
            --waiting for eop
            if(flu_tx_sop_out = '1' and flu_tx_eop_out = '1')then
               --one packet ends, another starts
               mux_cnt_bytes_sel <= "011";
            elsif(flu_tx_eop_out = '1')then
               --packet ends
               mux_cnt_bytes_sel <= "100";
            else
               --packet continues, add length of the flu word
               mux_cnt_bytes_sel <= "101";
            end if;
         end if;               
      end if;
   end process;

   --Total sent bytes - multiplexor
   cnt_bytes_in0 <= (others => '0');
   cnt_bytes_in1 <= conv_std_logic_vector(((64 - conv_integer(flu_tx_sop_pos&"000")) - (63 - conv_integer(flu_tx_eop_pos))), 7);
   cnt_bytes_in2 <= conv_std_logic_vector((64 - conv_integer(flu_tx_sop_pos&"000")), 7);
   cnt_bytes_in3 <= conv_std_logic_vector((conv_integer(flu_tx_eop_pos) + 1) + (64 - conv_integer(flu_tx_sop_pos&"000")), 7);
   cnt_bytes_in4 <= conv_std_logic_vector((conv_integer(flu_tx_eop_pos) + 1), 7);
   cnt_bytes_in5 <= conv_std_logic_vector(64, 7);

   mux_cnt_bytes_outp:process(cnt_bytes_in0,cnt_bytes_in1, cnt_bytes_in2, cnt_bytes_in3, cnt_bytes_in4,
                              cnt_bytes_in5, mux_cnt_bytes_sel)
   begin
      if (mux_cnt_bytes_sel = "000") then
         mux_cnt_bytes_out <= cnt_bytes_in0;
      elsif (mux_cnt_bytes_sel = "001") then
         mux_cnt_bytes_out <= cnt_bytes_in1;
      elsif (mux_cnt_bytes_sel = "010") then
         mux_cnt_bytes_out <= cnt_bytes_in2;
      elsif (mux_cnt_bytes_sel = "011") then
         mux_cnt_bytes_out <= cnt_bytes_in3;
      elsif (mux_cnt_bytes_sel = "100") then
         mux_cnt_bytes_out <= cnt_bytes_in4;
      else
         mux_cnt_bytes_out <= cnt_bytes_in5;
      end if;
   end process;

   --Total sent bytes - wait_for_sop control signal

   wait_for_sopp:process(reg_wait_for_sop, output_flu_src_rdy, output_flu_dst_rdy, flu_tx_sop_out,
                         flu_tx_eop_out)
   begin
      wait_for_sop <= reg_wait_for_sop;
      if(output_flu_src_rdy = '1' and output_flu_dst_rdy = '1')then
         if(reg_wait_for_sop = '0')then
            --waiting for sop
            if(flu_tx_sop_out = '1' and flu_tx_eop_out = '0')then
               wait_for_sop <= '1';
            end if;
         else
            if(flu_tx_eop_out = '1' and flu_tx_sop_out = '0')then
               --packet ends
               wait_for_sop <= '0';
            end if;
         end if;               
      end if;
   end process;

   reg_wait_for_sopp:process(CGMII_CLK)
   begin
      if (CGMII_CLK'event and CGMII_CLK = '1') then
         if (CGMII_RESET = '1') then
            reg_wait_for_sop <= '0';
         else
            reg_wait_for_sop <= wait_for_sop;
         end if;
      end if;
   end process;

   cnt_bytesg: if NOT CNT_DSP generate
      --Total sent bytes input register
      reg_cnt_bytes_inp:process(CGMII_CLK)
      begin 
         if (CGMII_CLK'event and CGMII_CLK = '1') then  
            if (CGMII_RESET = '1') then 
               reg_cnt_bytes_in <= (others => '0');  
            else
               reg_cnt_bytes_in <= mux_cnt_bytes_out;
            end if; 
         end if; 
      end process;

      --Total sent bytes
      cnt_bytesp:process(CGMII_CLK)     
      begin    
         if (CGMII_CLK'event and CGMII_CLK = '1') then    
            if (CGMII_RESET = '1' or reset_counters = '1') then  
               cnt_bytes <= (others => '0');    
            else   
               cnt_bytes <= cnt_bytes + reg_cnt_bytes_in;
            end if;
         end if;     
      end process;
   end generate;

   cnt_dsp_bytesg: if CNT_DSP generate
      --Total sent bytes
      count_dsp_reset <= CGMII_RESET or reset_counters;
      count_dsp_a <= (63 downto 7 => '0') & mux_cnt_bytes_out;

      count_dspi:entity work.COUNT_DSP
      generic map(
         DATA_WIDTH => 64,
         --! Input pipeline registers
         REG_IN => 1,
         --! Reset when MAX == P ( 0 => "NO_RESET", 1 => "RESET_MATCH") 
         AUTO_RESET => 0
      )
      port map(
         --! Clock input
         CLK => CGMII_CLK,
         --! Enable input
         ENABLE => '1',
         --! Reset input
         RESET => count_dsp_reset,
         --! Data input 
         A => count_dsp_a,
         --! Data input (must MAX % A = 0), Maximum value 
         MAX => (others => 'X'),
         --! Data output
         P => cnt_bytes
      );
   end generate;

   --------------------------------------------------------
   --SW readable/writable registers 
   --------------------------------------------------------
   -- register reg_cnt_packets
   reg_cnt_packetsp: process(CGMII_CLK)
   begin
      if (CGMII_CLK'event AND CGMII_CLK = '1') then
         if (reg_counters_we = '1') then
            reg_cnt_packets <= cnt_packets;
            reg_cnt_bytes <= cnt_bytes;
         end if;
      end if;
   end process;

   -- register reg_obuf_en
   reg_obuf_enp: process(CGMII_CLK)
   begin
      if (CGMII_CLK'event AND CGMII_CLK = '1') then
         if (CGMII_RESET = '1') then
            reg_obuf_en <= '0'; 
         elsif (reg_obuf_en_we = '1') then
            reg_obuf_en <= mi_int_DWR(0);
         end if;
      end if;
   end process;

   -- control register
   reg_counters_wep: process(mi_int_DWR(7 downto 0), reg_ctrl_we)
   begin
      if ((mi_int_DWR(7 downto 0) = OBUFCMD_STROBE_COUNTERS) and (reg_ctrl_we = '1')) then
         reg_counters_we <= '1';
      else
         reg_counters_we <= '0';
      end if;
   end process;

   reset_countersp: process(mi_int_DWR(7 downto 0), reg_ctrl_we)
   begin
      if ((mi_int_DWR(7 downto 0) = OBUFCMD_RESET_COUNTERS) and (reg_ctrl_we = '1')) then
         reset_counters <= '1';
      else
         reset_counters <= '0';
      end if;
   end process;

   -- register reg_obuf_status
   reg_status_p: process(CGMII_CLK)
   begin
      if (CGMII_CLK'event AND CGMII_CLK = '1') then
         if (CGMII_RESET = '1') then
            -- new third bit added
            reg_status <= "1010000";
         elsif (reg_status_we = '1') then
            reg_status(0) <= mi_int_DWR(0);
         end if;
      end if;
   end process reg_status_p;

   --------------------------------------------------------
   --MI32 
   --------------------------------------------------------

   mi_int_DRDY <= mi_int_RD;
   mi_int_ARDY <= '1';
   mi_int_DRD  <= mx_decoder_mi_drd;

   -- write part
   reg_obuf_en_we    <= reg_obuf_en_cs and mi_int_WR;
   reg_ctrl_we       <= reg_ctrl_cs and mi_int_WR;
   reg_status_we     <= reg_status_cs and mi_int_WR;

   -- Set CS signals according to mi_int_ADDR
   chip_selectp: process(mi_int_ADDR(5 downto 2))
   begin
      reg_obuf_en_cs  <= '0';
      reg_ctrl_cs     <= '0';
      reg_status_cs   <= '0';

      case (mi_int_ADDR(5 downto 2)) is
         when REG_OBUF_EN_ADDR      => reg_obuf_en_cs       <= '1';
         when REG_CTRL_ADDR         => reg_ctrl_cs          <= '1';
         when REG_STATUS_ADDR       => reg_status_cs        <= '1';
         when others =>
      end case;
   end process;

   -- Reading Part
   -- 32bit-wide signals
   reg_obuf_en32  <= (31 downto 1 => '0') & reg_obuf_en;
   reg_status32   <= (31 downto 7 => '0') & reg_status;

   -- Output register multiplexer
   mx_decoder_mi_drdp: process(mi_int_ADDR(5 downto 2), reg_cnt_packets, reg_cnt_bytes, reg_obuf_en32, reg_status32)
   begin
      --Default value
      mx_decoder_mi_drd <= (others=>'0');
      
      case (mi_int_ADDR(5 downto 2)) is
         when REG_CNT_PACKETS_ADDR_LOW =>
            mx_decoder_mi_drd <= reg_cnt_packets(31 downto 0);
         when REG_CNT_PACKETS_ADDR_HIGH =>
            mx_decoder_mi_drd <= reg_cnt_packets(63 downto 32);
         when REG_CNT_BYTES_ADDR_LOW =>
            mx_decoder_mi_drd <= reg_cnt_bytes(31 downto 0);
         when REG_CNT_BYTES_ADDR_HIGH =>
            mx_decoder_mi_drd <= reg_cnt_bytes(63 downto 32);
         when REG_OBUF_EN_ADDR =>
            mx_decoder_mi_drd <= reg_obuf_en32;
         when REG_STATUS_ADDR =>
            mx_decoder_mi_drd <= reg_status32;
         when others =>
            mx_decoder_mi_drd <= (others => '0');
      end case;
   end process;

   -- -------------------------------------------------------------------------
   --                       Asynchronous MI32 connection
   -- -------------------------------------------------------------------------

   -- Asynchronous MI32 interface
   mi32_asynci: entity work.mi32_async_handshake
      port map(
         CLK_M             => MI_CLK,
         RESET_M           => MI_RESET,
         MI_M_DWR          => mi_master_DWR,
         MI_M_ADDR         => mi_master_ADDR,
         MI_M_RD           => mi_master_RD,
         MI_M_WR           => mi_master_WR,
         MI_M_BE           => mi_master_BE,
         MI_M_DRD          => mi_master_DRD,
         MI_M_ARDY         => mi_master_ARDY,
         MI_M_DRDY         => mi_master_DRDY,

         CLK_S             => CGMII_CLK,
         RESET_S           => CGMII_RESET,
         MI_S_DWR          => mi_int_DWR,
         MI_S_ADDR         => mi_int_ADDR,
         MI_S_RD           => mi_int_RD,
         MI_S_WR           => mi_int_WR,
         MI_S_BE           => mi_int_BE,
         MI_S_DRD          => mi_int_DRD,
         MI_S_ARDY         => mi_int_ARDY,
         MI_S_DRDY         => mi_int_DRDY
      );

   mi_master_DWR <= MI_DWR;
   mi_master_ADDR <= MI_ADDR;
   mi_master_RD <= MI_RD;
   mi_master_WR <= MI_WR;
   mi_master_BE <= MI_BE;
   MI_DRD <= mi_master_DRD;
   MI_ARDY <= mi_master_ARDY;
   MI_DRDY <= mi_master_DRDY;

end architecture full;

