-- addr_dec_ent.vhd: Address decoder for CGMII Input buffer -
--                   entity declaration
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                           Entity declaration
-- ----------------------------------------------------------------------------

entity addr_dec is
   generic(
      -- Number of MAC addresses that can be placed into CAM (at most 16)
      MAC_COUNT                     : integer := 16;
      -- Synthesise design without resets, default values of registers are set
      -- by initialization of signals
      RESET_BY_INIT                 : boolean := false
   );
   port(
      -- Common signals -------------------------------------------------------
      -- Internal clock sigal
      INT_CLK                       : in std_logic;
      -- Reset for INT_CLK domain, active in '1'
      INT_RESET                     : in std_logic;
      -- MI32 clock signal
      MI_CLK                        : in std_logic;
      -- Reset for MI_CLK domain, active in '1'
      MI_RESET                      : in std_logic;


      -- ----------------------------------------------------------------------
      --                           User side input
      -- ----------------------------------------------------------------------

      -- MI32 input interface -------------------------------------------------
      -- Input Data
      MI_DWR                        : in  std_logic_vector(31 downto 0);
      -- Address
      MI_ADDR                       : in  std_logic_vector(31 downto 0);
      -- Read Request
      MI_RD                         : in  std_logic;
      -- Write Request
      MI_WR                         : in  std_logic;
      -- Byte Enable
      MI_BE                         : in  std_logic_vector(3  downto 0);
      -- Output Data
      MI_DRD                        : out std_logic_vector(31 downto 0);
      -- Address Ready
      MI_ARDY                       : out std_logic;
      -- Data Ready
      MI_DRDY                       : out std_logic;


      -- ----------------------------------------------------------------------
      --                    Interface to Memory unit
      -- ----------------------------------------------------------------------

      -- General data signals (mem <-> address decoder) -----------------------
      -- Data to be written
      DATA_WR                       : out std_logic_vector(31 downto 0);
      -- Read data
      DATA_RD                       : in std_logic_vector(31 downto 0);
      -- Read data are valid
      DATA_VLD                      : in std_logic;

      -- Registers write interface (address decoder -> mem) -------------------
      -- Write enable for IBUF enable register
      IBUF_EN_WE                    : out std_logic;
      -- Write enable for Error mask register
      ERROR_MASK_WE                 : out std_logic;
      -- Write enable for status register
      STATUS_WE                     : out std_logic;
      -- Write enable for Minimum frame length register
      MIN_FRAME_LEN_WE              : out std_logic;
      -- Write enable for Frame MTU register
      MAX_FRAME_LEN_WE              : out std_logic;
      -- Write enable for MAC check mode register
      MAC_CHECK_MODE_WE             : out std_logic;

      -- Registers read interface (mem -> address decoder) --------------------
      -- Read from IBUF enable register
      IBUF_EN_RD                    : out std_logic;
      -- Read from Error mask register
      ERROR_MASK_RD                 : out std_logic;
      -- Read from status register
      STATUS_RD                     : out std_logic;
      -- Read from Minimum frame length register
      MIN_FRAME_LEN_RD              : out std_logic;
      -- Read from Frame MTU register
      MAX_FRAME_LEN_RD              : out std_logic;
      -- Read from MAC check mode register
      MAC_CHECK_MODE_RD             : out std_logic;

      -- CAM read/write interface (mem <-> address decoder) -------------------
      -- Address within CAM + low/high part of MAC selector (the lowest bit)
      CAM_ADDR                      : out std_logic_vector(log2(MAC_COUNT) downto 0);
      -- Write enable signal for CAM
      CAM_WE                        : out std_logic;
      -- Read enable signal for CAM
      CAM_RE                        : out std_logic;
      -- Busy signal for CAM
      CAM_BUSY                      : in std_logic;


      -- ----------------------------------------------------------------------
      --                  Interface to Statistical unit
      -- ----------------------------------------------------------------------

      -- Control interface ----------------------------------------------------
      -- Software reset (resets active couters but preserves actual snapshot)
      SW_RESET                      : out std_logic;
      -- Enable read counter values (= make a snapshot)
      READ_EN                       : out std_logic;
      -- Release current snapshot
      LAST_ADDR_EN                  : out std_logic;

      -- Statistics inputs ----------------------------------------------------
      -- Counter of all frames
      TOTAL_FRAMES_CNT              : in std_logic_vector(63 downto 0);
      -- Counter of correct frames
      CORRECT_FRAMES_CNT            : in std_logic_vector(63 downto 0);
      -- Counter of discarded frames
      DISCARDED_FRAMES_CNT          : in std_logic_vector(63 downto 0);
      -- Counter of frames discarded due to buffer overflow
      DISCARDED_BUF_OVF_FRAMES_CNT  : in std_logic_vector(63 downto 0);
      -- Counter of data octets in frames that were successfully received
      OCTETS_RECEIVED_OK_CNT        : in std_logic_vector(63 downto 0);
      -- Number of transfered bytes
      TRANSFERED_OCTETS_CNT         : in std_logic_vector(63 downto 0); 
      -- Counter of frames with bad CRC
      CRC_ERROR_CNT                 : in std_logic_vector(63 downto 0); 
      -- Counter of frames over MTU
      OVER_MTU_CNT                  : in std_logic_vector(63 downto 0); 
      -- Counter of frames below minimal allowed length
      BELOW_MIN_CNT                 : in std_logic_vector(63 downto 0); 
      -- Counter of frames with broadcast MAC address
      BCAST_FRAMES_CNT              : in std_logic_vector(63 downto 0); 
      -- Counter of frames with multicast MAC address
      MCAST_FRAMES_CNT              : in std_logic_vector(63 downto 0); 
      -- Counter of frames which are known as fragment frames
      FRAGMENT_FRAMES_CNT           : in std_logic_vector(63 downto 0); 
      -- Counter of frames which are known as jabber frames
      JABBER_FRAMES_CNT             : in std_logic_vector(63 downto 0); 
      -- "Histogram" length counters
      FRAMES_64_CNT                 : in std_logic_vector(63 downto 0); 
      FRAMES_65_127_CNT             : in std_logic_vector(63 downto 0); 
      FRAMES_128_255_CNT            : in std_logic_vector(63 downto 0); 
      FRAMES_256_511_CNT            : in std_logic_vector(63 downto 0); 
      FRAMES_512_1023_CNT           : in std_logic_vector(63 downto 0); 
      FRAMES_1024_1518_CNT          : in std_logic_vector(63 downto 0); 
      -- Counter values validity signal
      CNT_VALID                     : in std_logic
   );
end entity addr_dec;

