-- transformer.vhd: Implementation of DMA Transformer. Based on FL Transformers
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use work.math_pack.all;
use work.dma_pkg.all;

-- ------------------------------------------------------------------------
--                        Entity declaration
-- ------------------------------------------------------------------------
entity DMA_TRANSFORMER is
   generic(
      RX_DATA_WIDTH  : integer := 256;
      TX_DATA_WIDTH  : integer := 512;
      HDR_WIDTH      : integer :=  32
   );
   port(
      CLK            : in  std_logic;
      RESET          : in  std_logic;

      -- RX interface
      RX_DATA        : in  std_logic_vector(RX_DATA_WIDTH-1 downto 0);
      RX_HDR         : in  std_logic_vector(HDR_WIDTH-1 downto 0);
      RX_SOP         : in  std_logic;
      RX_EOP         : in  std_logic;
      RX_SRC_RDY     : in  std_logic;
      RX_DST_RDY     : out std_logic;

      -- TX interface
      TX_DATA        : out std_logic_vector(TX_DATA_WIDTH-1 downto 0);
      TX_HDR         : out std_logic_vector(HDR_WIDTH-1 downto 0);
      TX_SOP         : out std_logic;
      TX_EOP         : out std_logic;
      TX_SRC_RDY     : out std_logic;
      TX_DST_RDY     : in  std_logic
   );
end entity DMA_TRANSFORMER;

-- ------------------------------------------------------------------------
--                      Architecture declaration
-- ------------------------------------------------------------------------
architecture full of DMA_TRANSFORMER is

   -- Common transformer signals
   signal data_strobe   : std_logic;
   signal lblk          : std_logic;
   signal flag_sop      : std_logic;

   -- UP transformer signals
   signal act_byte      : std_logic_vector(log2(TX_DATA_WIDTH/RX_DATA_WIDTH) downto 0);
   signal reg_updata    : std_logic_vector(TX_DATA_WIDTH-RX_DATA_WIDTH-1 downto 0) := (others => '0');

   -- DOWN transformer signals
   signal dst_rdy          : std_logic;
   signal valid_bytes      : std_logic_vector(log2(RX_DATA_WIDTH/8)-1 downto 0);
   signal real_bytes       : std_logic_vector(log2(RX_DATA_WIDTH/8)-1 downto 0);
   signal reg_valid_bytes  : std_logic_vector(log2(RX_DATA_WIDTH/8)-1 downto 0);
   signal valid_count      : std_logic_vector(log2(RX_DATA_WIDTH/8)-1 downto 0);

   signal flag_eop         : std_logic;
   signal eop              : std_logic;
   signal sig_eop          : std_logic;
   signal got_valid        : std_logic;
   signal cnt_byte         : std_logic_vector(log2(RX_DATA_WIDTH/8)-1 downto 0);
   signal cnt_data         : std_logic_vector(log2(RX_DATA_WIDTH/TX_DATA_WIDTH)-1 downto 0);

   signal reg_downdata     : std_logic_vector(RX_DATA_WIDTH-1 downto 0) := (others => '0');
   signal gen_mux_in       : std_logic_vector(RX_DATA_WIDTH-1 downto 0);

   signal src_rdy          : std_logic;
   signal first_word       : std_logic;

begin

gen_none : if (RX_DATA_WIDTH = TX_DATA_WIDTH) generate
   TX_DATA        <= RX_DATA;
   TX_HDR         <= RX_HDR;
   TX_SOP         <= RX_SOP;
   TX_EOP         <= RX_EOP;
   TX_SRC_RDY     <= RX_SRC_RDY;
   RX_DST_RDY     <= TX_DST_RDY;
end generate;

gen_up : if (RX_DATA_WIDTH < TX_DATA_WIDTH) generate

   RX_DST_RDY     <= TX_DST_RDY;

   TX_HDR         <= RX_HDR;
   TX_EOP         <= RX_EOP and RX_SRC_RDY;
   TX_SOP         <= lblk and (RX_SOP or flag_sop);
   TX_SRC_RDY     <= lblk;

   data_strobe    <= TX_DST_RDY and RX_SRC_RDY;
   lblk           <= RX_SRC_RDY when RX_EOP = '1' or conv_integer(act_byte) = (TX_DATA_WIDTH/RX_DATA_WIDTH) else '0';

   act_bytep : process(RESET, CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(RESET = '1') then
            act_byte       <= conv_std_logic_vector(1, log2(TX_DATA_WIDTH/RX_DATA_WIDTH)+1);
         else
            if(TX_DST_RDY = '1' and RX_SRC_RDY = '1') then
               if(lblk = '1') then
                  act_byte <= conv_std_logic_vector(1, log2(TX_DATA_WIDTH/RX_DATA_WIDTH)+1);
               else
                  act_byte <= act_byte + 1;
               end if;
            end if;
         end if;
      end if;
   end process;

   GEN_TX_DATA : for i in (TX_DATA_WIDTH/RX_DATA_WIDTH)-1 downto 0 generate
      process(reg_updata, RX_DATA, lblk, act_byte)
      begin
         if(lblk = '1') then
            if(conv_integer(act_byte) = i+1) then
               TX_DATA((RX_DATA_WIDTH*(i+1))-1 downto RX_DATA_WIDTH*i)  <= RX_DATA;
            elsif(i = (TX_DATA_WIDTH/RX_DATA_WIDTH)-1) then
               TX_DATA((RX_DATA_WIDTH*(i+1))-1 downto RX_DATA_WIDTH*i)  <= (others => '0');
            else
               TX_DATA((RX_DATA_WIDTH*(i+1))-1 downto RX_DATA_WIDTH*i)  <= reg_updata((RX_DATA_WIDTH*(i+1))-1 downto RX_DATA_WIDTH*i);
            end if;
         else
            TX_DATA((RX_DATA_WIDTH*(i+1))-1 downto RX_DATA_WIDTH*i)     <= (others => '0');
         end if;
      end process;
   end generate;

   -- ---------------------------------------------------------------------
   --                   Registers
   -- ---------------------------------------------------------------------

   flag_sopp : process(RESET, CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(RESET = '1') then
            flag_sop    <= '0';
         else
            if(lblk = '1' and TX_DST_RDY = '1') then
               flag_sop <= '0';
            elsif(RX_SOP = '1' and RX_SRC_RDY = '1') then
               flag_sop <= '1';
            end if;
         end if;
      end if;
   end process;

   GEN_DATA : for i in 0 to (TX_DATA_WIDTH/RX_DATA_WIDTH)-2 generate
      process(CLK)
      begin
         if(CLK'event AND CLK = '1') then
            if(data_strobe = '1' and conv_integer(act_byte)-1 = i) then
               reg_updata(((i+1)*RX_DATA_WIDTH)-1 downto i*RX_DATA_WIDTH) <= RX_DATA;
            end if;
         end if;
      end process;
   end generate;

end generate;

gen_down : if (RX_DATA_WIDTH > TX_DATA_WIDTH) generate

   TX_HDR      <= RX_HDR;

   TX_SOP      <= RX_SOP and dst_rdy;
   TX_EOP      <= eop;
   TX_SRC_RDY  <= src_rdy;
   RX_DST_RDY  <= TX_DST_RDY and first_word;

   eop         <= lblk and (sig_eop or flag_eop);
   sig_eop     <= dst_rdy and RX_EOP and RX_SRC_RDY;
   src_rdy     <= '0' when RX_SRC_RDY = '0' and dst_rdy = '1' else RX_SRC_RDY or flag_sop or flag_eop;
   dst_rdy     <= first_word;
   data_strobe <= dst_rdy and RX_SRC_RDY;
   gen_mux_in  <= RX_DATA when data_strobe = '1' else reg_downdata;
   first_word  <= '1' when conv_integer(cnt_data) = 0 else '0';

   got_valid   <= '0' when valid_count > cnt_byte else '1';
   real_bytes  <= (RX_HDR(3 downto 0) & "00") - 1 when RX_EOP = '1' and TX_DST_RDY = '1' else conv_std_logic_vector(RX_DATA_WIDTH/8-1, log2(RX_DATA_WIDTH/8));
   valid_bytes <= conv_std_logic_vector(1, log2(RX_DATA_WIDTH/8)) when RX_SOP = '1' and RX_HDR(DMA_REQUEST_TYPE) /= DMA_TYPE_WRITE else real_bytes;
   valid_count <= valid_bytes when dst_rdy = '1' else reg_valid_bytes;
   lblk        <= got_valid when RX_SRC_RDY = '1' or flag_sop = '1' else '1';

   GEN_MUX_U : entity work.GEN_MUX
      generic map(
         DATA_WIDTH  => TX_DATA_WIDTH,
         MUX_WIDTH   => RX_DATA_WIDTH/TX_DATA_WIDTH
      )
      port map(
         DATA_IN  => gen_mux_in,
         SEL      => cnt_data,
         DATA_OUT => TX_DATA
      );

   cnt_bytep : process(RESET, CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(RESET = '1') then
            cnt_byte <= conv_std_logic_vector((TX_DATA_WIDTH/8)-1, log2(RX_DATA_WIDTH/8));
         else
            if(TX_DST_RDY = '1' and src_rdy = '1') then
               if(lblk = '1') then
                  cnt_byte <= conv_std_logic_vector((TX_DATA_WIDTH/8)-1, log2(RX_DATA_WIDTH/8));
               else
                  cnt_byte <= cnt_byte + conv_std_logic_vector((TX_DATA_WIDTH/8), log2(RX_DATA_WIDTH/8));
               end if;
            end if;
         end if;
      end if;
   end process;

   cnt_datap : process(RESET, CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(RESET = '1') then
            cnt_data <= (others => '0');
         else
            if(TX_DST_RDY = '1' and src_rdy = '1') then
               if(lblk = '1') then
                  cnt_data <= (others => '0');
               else
                  cnt_data <= cnt_data + 1;
               end if;
            end if;
         end if;
      end if;
   end process;

   -- ---------------------------------------------------------------------
   --                   Registers
   -- ---------------------------------------------------------------------

   reg_valid_bytesp : process(RESET, CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(RESET = '1') then
            reg_valid_bytes <= (others => '0');
         else
            if(dst_rdy = '1' and (RX_SRC_RDY = '1' or flag_sop = '1')) then
               reg_valid_bytes <= valid_bytes;
            end if;
         end if;
      end if;
   end process;

   reg_datap : process(CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(data_strobe = '1') then
            reg_downdata <= RX_DATA;
         end if;
      end if;
   end process;

   flag_sopp : process(RESET, CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(RESET = '1') then
            flag_sop <= '0';
         else
            if(RX_SOP = '1' and RX_SRC_RDY = '1') then
               flag_sop <= '1';
            elsif(eop = '1') then
               flag_sop <= '0';
            end if;
         end if;
      end if;
   end process;

   flag_eopp : process(RESET, CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(RESET = '1') then
            flag_eop <= '0';
         else
            if(lblk = '1' and TX_DST_RDY = '1') then
               flag_eop <= '0';
            elsif(sig_eop = '1') then
               flag_eop <= '1';
            end if;
         end if;
      end if;
   end process;

end generate;

end architecture full;
