-- rq_int.vhd: Requester Request Interface connection - architecture
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                       Architecture declaration
-- ----------------------------------------------------------------------------

architecture rq_int_arch of rq_int is

   -- Types declaration -------------------------------------------------------

   type state_type is (S_IDLE, S_WRITE_TRANSACTION,
                       S_FINISH_WRITE_TRANSACTION); 

   -- Signals declaration -----------------------------------------------------

   -- Register for second half of data word
   signal reg_data_second_half              : std_logic_vector(127 downto 0);

   -- Register for 3 lowest bits of Dwords count minus 1
   signal dma_hdr_dwords_minus_1            : std_logic_vector(10 downto 0);
   signal reg_dma_hdr_dwords_minus_1_3_bits : std_logic_vector(2 downto 0);

   -- Auxiliary signal with value for output port
   signal dma_dst_rdy_sig                   : std_logic;

   -- FSM signals
   signal present_state                     : state_type;
   signal next_state                        : state_type;


-- ----------------------------------------------------------------------------
--                             Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --       Registers for delaying some values on DMA-like interface
   -- -------------------------------------------------------------------------

   -- register for second half of DMA_DATA ------------------------------------
   reg_data_second_half_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (DMA_SRC_RDY = '1' AND dma_dst_rdy_sig = '1') then
            reg_data_second_half <= DMA_DATA(255 downto 128);
         end if;
      end if;
   end process reg_data_second_half_p;

   -- register for dma_hdr_dwords_minus_1 value -------------------------------
   -- input value
   dma_hdr_dwords_minus_1 <= DMA_HDR_DWORDS - 1 when (DMA_HDR_DWORDS /= 0) else
                             DMA_HDR_DWORDS;

   -- register
   reg_dma_hdr_dwords_minus_1_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (DMA_SRC_RDY = '1' AND dma_dst_rdy_sig = '1') then
            if (DMA_SOP = '1') then
               reg_dma_hdr_dwords_minus_1_3_bits <=
                     dma_hdr_dwords_minus_1(2 downto 0);
            end if;
         end if;
      end if;
   end process reg_dma_hdr_dwords_minus_1_p;


   -- -------------------------------------------------------------------------
   --                                 FSM
   -- -------------------------------------------------------------------------

   -- present state register --------------------------------------------------
   present_state_reg_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            present_state <= S_IDLE;
         else
            present_state <= next_state;
         end if;        
      end if;
   end process present_state_reg_p;
 
   -- next state logic --------------------------------------------------------
   next_state_logic_p : process(present_state, DMA_SRC_RDY, RQ_READY, DMA_SOP,
                                DMA_EOP, DMA_HDR_TYPE, dma_hdr_dwords_minus_1,
                                reg_dma_hdr_dwords_minus_1_3_bits)
   begin
      case (present_state) is

         when S_IDLE =>
            if (DMA_SRC_RDY = '1' AND RQ_READY = '1' AND
                DMA_SOP = '1') then
               -- read transaction
               if (DMA_HDR_TYPE = '0') then
                  if (DMA_EOP = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_IDLE;
                  end if;
               -- write transaction
               elsif (DMA_HDR_TYPE = '1') then
                  if (DMA_EOP = '1') then
                     if (dma_hdr_dwords_minus_1 < 4) then
                        next_state <= S_IDLE;
                     else
                        next_state <= S_FINISH_WRITE_TRANSACTION;
                     end if;
                  else
                     next_state <= S_WRITE_TRANSACTION;
                  end if;
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_IDLE;
            end if;

         when S_WRITE_TRANSACTION =>
            if (DMA_SRC_RDY = '1' AND RQ_READY = '1') then
               if (DMA_EOP = '1') then
                  if (reg_dma_hdr_dwords_minus_1_3_bits < 4) then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_FINISH_WRITE_TRANSACTION;
                  end if;
               else
                  next_state <= S_WRITE_TRANSACTION;
               end if;
            else
               next_state <= S_WRITE_TRANSACTION;
            end if;

         when S_FINISH_WRITE_TRANSACTION =>
            if (RQ_READY = '1') then
               next_state <= S_IDLE;
            else
               next_state <= S_FINISH_WRITE_TRANSACTION;
            end if;

--         when others =>
--            next_state <= S_IDLE;

      end case;      
   end process next_state_logic_p;
 
   -- output logic ------------------------------------------------------------
   output_logic_p : process (present_state, DMA_SRC_RDY, RQ_READY, DMA_SOP,
                             DMA_EOP, DMA_HDR_TYPE, DMA_HDR_DWORDS,
                             dma_hdr_dwords_minus_1, DMA_HDR_ADDR, PCIE_TAG,
                             DMA_DATA,reg_dma_hdr_dwords_minus_1_3_bits,
                             reg_data_second_half)
   begin
      -- default values
      RQ_DATA          <= (others => '0');
      RQ_LAST          <= '0';
      RQ_KEEP          <= (others => '0');
      RQ_VALID         <= '0';
      dma_dst_rdy_sig  <= '0';

      case (present_state) is

         when S_IDLE =>
            if (DMA_SRC_RDY = '1' AND RQ_READY = '1' AND
                DMA_SOP = '1') then
               -- read transaction
               if (DMA_HDR_TYPE = '0') then
                  if (DMA_EOP = '1') then
                     RQ_DATA  <=
                        (255 downto 128 => '0')   & -- empty space in transaction payload
                        '0'                       & -- force ECRC
                        "000"                     & -- attributes
                        "000"                     & -- transaction class
                        '0'                       & -- requester ID enable
                        X"0000"                   & -- completer ID
                        "000" & PCIE_TAG          & -- tag
                        X"0000"                   & -- requester ID
                        '0'                       & -- poisoned request
                        "000" & DMA_HDR_TYPE      & -- request type
                        DMA_HDR_DWORDS            & -- Dword count
                        DMA_HDR_ADDR(63 downto 2) & -- address
                        "00";                       -- address type
                     RQ_LAST          <= '1';
                     RQ_KEEP          <= "00001111";
                     RQ_VALID         <= '1';
                     dma_dst_rdy_sig  <= '1';
                  end if;
               -- write transaction
               elsif (DMA_HDR_TYPE = '1') then
                  if (DMA_EOP = '1') then
                     if (DMA_HDR_DWORDS = 0) then
                        RQ_DATA  <=
                           (255 downto 128 => '0')   & -- empty space in transaction payload
                           '0'                       & -- force ECRC
                           "000"                     & -- attributes
                           "000"                     & -- transaction class
                           '0'                       & -- requester ID enable
                           X"0000"                   & -- completer ID
                           X"00"                     & -- tag
                           X"0000"                   & -- requester ID
                           '0'                       & -- poisoned request
                           "000" & DMA_HDR_TYPE      & -- request type
                           DMA_HDR_DWORDS            & -- Dword count
                           DMA_HDR_ADDR(63 downto 2) & -- address
                           "00";                       -- address type
                        RQ_LAST         <= '1';
                        RQ_KEEP         <= "00001111";
                        RQ_VALID        <= '1';
                        dma_dst_rdy_sig <= '1';
                     elsif (dma_hdr_dwords_minus_1 < 4) then
                        RQ_DATA  <=
                           DMA_DATA(127 downto 0)    & -- empty space in transaction payload
                           '0'                       & -- force ECRC
                           "000"                     & -- attributes
                           "000"                     & -- transaction class
                           '0'                       & -- requester ID enable
                           X"0000"                   & -- completer ID
                           X"00"                     & -- tag
                           X"0000"                   & -- requester ID
                           '0'                       & -- poisoned request
                           "000" & DMA_HDR_TYPE      & -- request type
                           DMA_HDR_DWORDS            & -- Dword count
                           DMA_HDR_ADDR(63 downto 2) & -- address
                           "00";                       -- address type
                        RQ_LAST         <= '1';
                        case (dma_hdr_dwords_minus_1(2 downto 0)) is
                           when "000"  => RQ_KEEP <= "00011111";
                           when "001"  => RQ_KEEP <= "00111111";
                           when "010"  => RQ_KEEP <= "01111111";
                           when "011"  => RQ_KEEP <= "11111111";
                           when others => RQ_KEEP <= "00001111";
                        end case;
                        RQ_VALID        <= '1';
                        dma_dst_rdy_sig <= '1';
                     else
                        RQ_DATA  <=
                           DMA_DATA(127 downto 0)    & -- empty space in transaction payload
                           '0'                       & -- force ECRC
                           "000"                     & -- attributes
                           "000"                     & -- transaction class
                           '0'                       & -- requester ID enable
                           X"0000"                   & -- completer ID
                           X"00"                     & -- tag
                           X"0000"                   & -- requester ID
                           '0'                       & -- poisoned request
                           "000" & DMA_HDR_TYPE      & -- request type
                           DMA_HDR_DWORDS            & -- Dword count
                           DMA_HDR_ADDR(63 downto 2) & -- address
                           "00";                       -- address type
                        RQ_KEEP         <= "11111111";
                        RQ_VALID        <= '1';
                        dma_dst_rdy_sig <= '1';
                     end if;
                  else
                     RQ_DATA  <=
                        DMA_DATA(127 downto 0)    & -- empty space in transaction payload
                        '0'                       & -- force ECRC
                        "000"                     & -- attributes
                        "000"                     & -- transaction class
                        '0'                       & -- requester ID enable
                        X"0000"                   & -- completer ID
                        X"00"                     & -- tag
                        X"0000"                   & -- requester ID
                        '0'                       & -- poisoned request
                        "000" & DMA_HDR_TYPE      & -- request type
                        DMA_HDR_DWORDS            & -- Dword count
                        DMA_HDR_ADDR(63 downto 2) & -- address
                        "00";                       -- address type
                     RQ_KEEP         <= "11111111";
                     RQ_VALID        <= '1';
                     dma_dst_rdy_sig <= '1';
                  end if;
               end if;
            end if;

         when S_WRITE_TRANSACTION =>
            if (DMA_SRC_RDY = '1' AND RQ_READY = '1') then
               if (DMA_EOP = '1') then
                  if (reg_dma_hdr_dwords_minus_1_3_bits < 4) then
                     RQ_DATA     <= DMA_DATA(127 downto 0) & reg_data_second_half;
                     RQ_LAST     <= '1';
                     case (reg_dma_hdr_dwords_minus_1_3_bits) is
                        when "000"  => RQ_KEEP <= "00011111";
                        when "001"  => RQ_KEEP <= "00111111";
                        when "010"  => RQ_KEEP <= "01111111";
                        when "011"  => RQ_KEEP <= "11111111";
                        when others  => RQ_KEEP <= "00001111";
                     end case;
                     RQ_VALID        <= '1';
                     dma_dst_rdy_sig <= '1';
                  else
                     RQ_DATA         <= DMA_DATA(127 downto 0) & reg_data_second_half;
                     RQ_KEEP         <= "11111111";
                     RQ_VALID        <= '1';
                     dma_dst_rdy_sig <= '1';
                  end if;
               else
                  RQ_DATA         <= DMA_DATA(127 downto 0) & reg_data_second_half;
                  RQ_KEEP         <= "11111111";
                  RQ_VALID        <= '1';
                  dma_dst_rdy_sig <= '1';
               end if;
            end if;

         when S_FINISH_WRITE_TRANSACTION =>
            if (RQ_READY = '1') then
               RQ_DATA     <= (255 downto 128 => '0') & reg_data_second_half;
               RQ_LAST     <= '1';
               case (reg_dma_hdr_dwords_minus_1_3_bits) is
                  when "100"  => RQ_KEEP <= "00000001";
                  when "101"  => RQ_KEEP <= "00000011";
                  when "110"  => RQ_KEEP <= "00000111";
                  when "111"  => RQ_KEEP <= "00001111";
                  when others  => RQ_KEEP <= "00000000";
               end case;
               RQ_VALID        <= '1';
            end if;

--         when others =>
--            null;

      end case;      
   end process output_logic_p;


   -- -------------------------------------------------------------------------
   --                    Assigning values to output ports
   -- -------------------------------------------------------------------------

   DMA_DST_RDY <= dma_dst_rdy_sig;
   RQ_USER     <= X"00000000000000F" when ((DMA_SOP = '1') AND
                                           (DMA_HDR_DWORDS = 1)) else
                  X"0000000000000FF" when ((DMA_SOP = '1') AND
                                           (DMA_HDR_DWORDS /= 0)) else
                  (others => '0');

end architecture rq_int_arch;
