-- rc_int.vhd: Requester Completion Interface connection - architecture
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                       Architecture declaration
-- ----------------------------------------------------------------------------

architecture rc_int_arch of rc_int is

   -- Types declaration -------------------------------------------------------

   type state_type is (S_IDLE, S_READ_COMPLETION, S_FINISH_READ_COMPLETION); 

   -- Signals declaration -----------------------------------------------------

   -- Values from completion descriptor
   signal error_code                : std_logic_vector(3 downto 0);
   signal request_completed         : std_logic;
   signal dword_count               : std_logic_vector(10 downto 0);
   signal completion_status         : std_logic_vector(2 downto 0);
   signal tag                       : std_logic_vector(4 downto 0);

   -- Registers for descriptor values propagated to the DMA-like interface
   signal reg_request_completed     : std_logic;
   signal reg_dword_count           : std_logic_vector(10 downto 0);
   signal reg_tag                   : std_logic_vector(4 downto 0);

   -- Register for storing last 5 Dwords of RC_DATA
   signal reg_rc_data_last_5_dwords : std_logic_vector(159 downto 0);

   -- Register for RC_USER_SOF
   signal reg_rc_user_sof           : std_logic;

   -- Auxiliary signal with values of some ports
   signal rc_ready_sig              : std_logic;
   signal rc_valid_sig              : std_logic;
   signal rc_data_sig               : std_logic_vector(255 downto 0);
   signal rc_user_sof_sig           : std_logic;

   -- FSM signals
   signal present_state             : state_type;
   signal next_state                : state_type;

-- ----------------------------------------------------------------------------
--                             Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --        Assigning values to internal representation of input ports
   -- -------------------------------------------------------------------------

   rc_valid_sig    <= RC_VALID;
   rc_data_sig     <= RC_DATA;
   rc_user_sof_sig <= RC_USER_SOF;

   
   -- -------------------------------------------------------------------------
   --               Decomposition of completion descriptor
   -- -------------------------------------------------------------------------

   -- signals contain valid values only when the first word of the completion
   -- is currently received
   error_code        <= RC_DATA(15 downto 12);
   request_completed <= RC_DATA(30);
   dword_count       <= RC_DATA(42 downto 32);
   completion_status <= RC_DATA(45 downto 43);
   tag               <= RC_DATA(68 downto 64);


   -- -------------------------------------------------------------------------
   --                   Registers for some input ports
   -- -------------------------------------------------------------------------

   -- registering descriptor values propagated to then DMA-like interface 
   regs_descriptor_values_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (rc_valid_sig = '1' AND rc_ready_sig = '1') then
            reg_request_completed <= request_completed;
            reg_dword_count       <= dword_count;
            reg_tag               <= tag;
         else
            reg_request_completed <= '0';
            reg_dword_count       <= (others => '0');
            reg_tag               <= (others => '0');
         end if;
      end if;
   end process regs_descriptor_values_p;

   -- register for storing last 5 Dwords of RC_DATA
   reg_rc_data_last_5_dwords_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (rc_valid_sig = '1' AND rc_ready_sig = '1') then
            reg_rc_data_last_5_dwords <= rc_data_sig(255 downto 96);
         else
            reg_rc_data_last_5_dwords <= (others => '0');
         end if;
      end if;
   end process reg_rc_data_last_5_dwords_p;

   -- register for storing RC_USER_SOF
   reg_rc_user_sof_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (rc_valid_sig = '1' AND rc_ready_sig = '1') then
            reg_rc_user_sof <= rc_user_sof_sig;
         else
            reg_rc_user_sof <= '0';
         end if;
      end if;
   end process reg_rc_user_sof_p;
  
 
   -- -------------------------------------------------------------------------
   --                                 FSM
   -- -------------------------------------------------------------------------

   -- present state register --------------------------------------------------
   present_state_reg_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            present_state <= S_IDLE;
         else
            present_state <= next_state;
         end if;        
      end if;
   end process present_state_reg_p;
 
   -- next state logic --------------------------------------------------------
   next_state_logic_p : process(present_state, DMA_DST_RDY, RC_VALID, RC_KEEP,
                                RC_USER_SOF, RC_LAST, error_code,
                                completion_status)
   begin
      case (present_state) is

         when S_IDLE =>
            if (RC_VALID = '1' AND DMA_DST_RDY = '1' AND
                RC_USER_SOF = '1') then
               -- correct completion
               if (error_code = "0000" AND completion_status = "000") then
                  if (RC_LAST = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_READ_COMPLETION;
                  end if;
               -- completion with an error
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_IDLE;
            end if;

         when S_READ_COMPLETION =>
            if (RC_VALID = '1' AND DMA_DST_RDY = '1') then
               if (RC_LAST = '1') then
                  if (RC_KEEP < "00001000") then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_FINISH_READ_COMPLETION;
                  end if;
               else
                  next_state <= S_READ_COMPLETION;
               end if;
            else
               next_state <= S_READ_COMPLETION;
            end if;

         when S_FINISH_READ_COMPLETION =>
            if (DMA_DST_RDY = '1') then
               next_state <= S_IDLE;
            else
               next_state <= S_FINISH_READ_COMPLETION;
            end if;

--         when others =>
--            next_state <= S_IDLE;

      end case;      
   end process next_state_logic_p;
 
   -- output logic ------------------------------------------------------------
   output_logic_p : process (present_state, DMA_DST_RDY, RC_VALID, RC_KEEP,
                             RC_USER_SOF, RC_LAST, error_code,
                             completion_status, RC_DATA, dword_count, tag,
                             request_completed, reg_rc_data_last_5_dwords,
                             reg_dword_count, reg_tag, reg_request_completed,
                             reg_rc_user_sof)
   begin
      -- default values
      rc_ready_sig     <= '0';
      DMA_DATA         <= (others => '0');
      DMA_HDR_DWORDS   <= (others => '0');
      DMA_SOP          <= '0';
      DMA_EOP          <= '0';
      DMA_SRC_RDY      <= '0';
      PCIE_TAG         <= (others => '0');
      PCIE_TAG_RELEASE <= '0';

      case (present_state) is

         when S_IDLE =>
            if (RC_VALID = '1' AND DMA_DST_RDY = '1' AND
                RC_USER_SOF = '1') then
               -- correct completion
               if (error_code = "0000" AND completion_status = "000") then
                  if (RC_LAST = '1') then
                     rc_ready_sig     <= '1';
                     DMA_DATA         <= (255 downto 160 => '0') & RC_DATA(255 downto 96);
                     DMA_HDR_DWORDS   <= dword_count;
                     DMA_SOP          <= '1';
                     DMA_EOP          <= '1';
                     DMA_SRC_RDY      <= '1';
                     PCIE_TAG         <= tag;
                     PCIE_TAG_RELEASE <= request_completed;
                  else
                     rc_ready_sig <= '1';
                  end if;
               -- completion with an error
               else
                  rc_ready_sig     <= '1';
                  PCIE_TAG         <= tag;
                  PCIE_TAG_RELEASE <= request_completed;
               end if;
            end if;

         when S_READ_COMPLETION =>
            if (RC_VALID = '1' AND DMA_DST_RDY = '1') then
               if (RC_LAST = '1') then
                  if (RC_KEEP < "00001000") then
                     rc_ready_sig     <= '1';
                     DMA_DATA         <= RC_DATA(95 downto 0) & reg_rc_data_last_5_dwords;
                     DMA_HDR_DWORDS   <= reg_dword_count;
                     DMA_SOP          <= reg_rc_user_sof;
                     DMA_EOP          <= '1';
                     DMA_SRC_RDY      <= '1';
                     PCIE_TAG         <= reg_tag;
                     PCIE_TAG_RELEASE <= reg_request_completed;
                  else
                     rc_ready_sig     <= '1';
                     DMA_DATA         <= RC_DATA(95 downto 0) & reg_rc_data_last_5_dwords;
                     DMA_HDR_DWORDS   <= reg_dword_count;
                     DMA_SOP          <= reg_rc_user_sof;
                     DMA_SRC_RDY      <= '1';
                     PCIE_TAG         <= reg_tag;
                     PCIE_TAG_RELEASE <= reg_request_completed;
                  end if;
               else
                  rc_ready_sig     <= '1';
                  DMA_DATA         <= RC_DATA(95 downto 0) & reg_rc_data_last_5_dwords;
                  DMA_HDR_DWORDS   <= reg_dword_count;
                  DMA_SOP          <= reg_rc_user_sof;
                  DMA_SRC_RDY      <= '1';
                  PCIE_TAG         <= reg_tag;
                  PCIE_TAG_RELEASE <= reg_request_completed;
               end if;
            end if;

         when S_FINISH_READ_COMPLETION =>
            if (DMA_DST_RDY = '1') then
               DMA_DATA         <= (255 downto 160 => '0') & reg_rc_data_last_5_dwords;
               DMA_EOP          <= '1';
               DMA_SRC_RDY      <= '1';
            end if;

--         when others =>
--            null;

      end case;      
   end process output_logic_p;


   -- -------------------------------------------------------------------------
   --                    Assigning values to output ports
   -- -------------------------------------------------------------------------

   RC_READY <= rc_ready_sig;

end architecture rc_int_arch;
