-- clkgen.vhd : clock generator for Fiberblaze FB1CG card
--!
--! \file
--! \brief clock generator for Fiberblaze FB1CG card
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;
use ieee.numeric_std.all;

library unisim;
use unisim.vcomponents.all;


-- ----------------------------------------------------------------------------
--                           Entity declaration
-- ----------------------------------------------------------------------------

entity clkgen is
generic (
   --! \brief CLK0 divider - REAL number (from 1400 MHz)
   CLK0_DIV : real := 7.0;
   --! \brief CLK1 divider (from 1400 MHz)
   CLK1_DIV : integer := 7;
   --! \brief CLK2 divider (from 1400 MHz)
   CLK2_DIV : integer := 7;
   --! \brief CLK3 divider (from 1400 MHz)
   CLK3_DIV : integer := 7;
   --! \brief CLK4 divider (from 1400 MHz)
   CLK4_DIV : integer := 7;
   --! \brief CLK5 divider (from 1400 MHz)
   CLK5_DIV : integer := 7
);
port (
   --! \brief Clock in (125 MHz)
   CLKIN    : in  std_logic;
   --! \brief Input reset
   RESET    : in  std_logic;
   --! \brief 200 MHz clock output
   CLK200   : out std_logic;
   --! \brief Configurable clock output
   CLK0     : out std_logic;
   --! \brief Configurable clock output
   CLK1     : out std_logic;
   --! \brief Configurable clock output
   CLK2     : out std_logic;
   --! \brief Configurable clock output
   CLK3     : out std_logic;
   --! \brief Configurable clock output
   CLK4     : out std_logic;
   --! \brief Configurable clock output
   CLK5     : out std_logic;
   --! \brief Output clocks are ready
   LOCKED   : out std_logic
);
end clkgen;


-- ----------------------------------------------------------------------------
--                          Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of clkgen is

   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! Output clock signals
   signal clkfbout  : std_logic;
   signal clkout0   : std_logic;
   signal clkout1   : std_logic;
   signal clkout2   : std_logic;
   signal clkout3   : std_logic;
   signal clkout4   : std_logic;
   signal clkout5   : std_logic;
   signal clkout6   : std_logic;


-- ----------------------------------------------------------------------------
--                            Architecture body
-- ----------------------------------------------------------------------------

begin

   --! MMCM instantiation
   -- -------------------------------------------------------------------------
   mmcm_i : MMCME2_BASE
   generic map (
      DIVCLK_DIVIDE    => 5, --! Divide input 125 to 25 MHz
      CLKFBOUT_MULT_F  => 56.000, --! Multiply 25 to 1400 MHz
      CLKOUT0_DIVIDE_F => CLK0_DIV,
      CLKOUT1_DIVIDE   => 7, --! Static to make 200 from 1400 MHz
      CLKOUT2_DIVIDE   => CLK1_DIV,
      CLKOUT3_DIVIDE   => CLK2_DIV,
      CLKOUT4_DIVIDE   => CLK3_DIV,
      CLKOUT5_DIVIDE   => CLK4_DIV,
      CLKOUT6_DIVIDE   => CLK5_DIV,
      CLKIN1_PERIOD    => 8.0 --! Suppose 125 MHz input
   )
   port map (
      CLKFBOUT            => clkfbout,
      CLKFBOUTB           => open,
      CLKOUT0             => clkout0,
      CLKOUT0B            => open,
      CLKOUT1             => clkout1,
      CLKOUT1B            => open,
      CLKOUT2             => clkout2,
      CLKOUT2B            => open,
      CLKOUT3             => clkout3,
      CLKOUT3B            => open,
      CLKOUT4             => clkout4,
      CLKOUT5             => clkout5,
      CLKOUT6             => clkout6,
      CLKFBIN             => clkfbout,
      CLKIN1              => CLKIN,
      LOCKED              => LOCKED,
      PWRDWN              => '0',
      RST                 => RESET
   );

   --! Output clock buffers
   -- -------------------------------------------------------------------------
   --! MMCM Port 0 (CLK0 output)
   clk0_buf : BUFG
   port map (
      O   => CLK0,
      I   => clkout0
   );

   --! MMCM Port 1 (CLK200 output)
   clk200_buf : BUFG
   port map (
      O   => CLK200,
      I   => clkout1
   );

   --! MMCM Port 2 (CLK1 output)
   clk1_buf : BUFG
   port map (
      O   => CLK1,
      I   => clkout2
   );

   --! MMCM Port 3 (CLK2 output)
   clk2_buf : BUFG
   port map (
      O   => CLK2,
      I   => clkout3
   );

   --! MMCM Port 4 (CLK3 output)
   clkout3_buf : BUFG
   port map (
      O   => CLK3,
      I   => clkout4
   );

   --! MMCM Port 5 (CLK4 output)
   clkout4_buf : BUFG
   port map (
      O   => CLK4,
      I   => clkout5
   );

   --! MMCM Port 6 (CLK5 output)
   clkout5_buf : BUFG
   port map (
      O   => CLK5,
      I   => clkout6
   );

end architecture full;
