/*!
 * \file write_crc.c
 * \brief Write functions
 * \author Andrej Hank <xhanka00@liberouter.org>
 * \date 2008-2011
 */
/*! Copyright (C) 20082011 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <libsze2.h>
#include <commlbr.h>

/*!
 * \brief Prepare packet with CRC32 on last 4 B - for debugging purposes
 * 
 * \return
 * 	- sze->ct_tx_buffer - packet prepared with CRC32 on last 4 B in network byte order
 * 	- NULL - error
 */
unsigned char *szedata_prepare_packet_with_crc(struct szedata *sze, unsigned char *hw_data, unsigned int hw_data_len, unsigned char *sw_data, unsigned int sw_data_len, unsigned int *len) {
	*(u_int32_t *)(sw_data + sw_data_len) = cl_crc32(sw_data, sw_data_len, sw_data_len, CL_ETHERNET_CRC32_SIZE);

	return(szedata_prepare_packet(sze, hw_data, hw_data_len, sw_data, sw_data_len + CL_ETHERNET_CRC32_SIZE, len));
}

/*!
 * \brief Burst write operation - add last 4B CRC32.
 * 
 * \param sze Handle
 * \param hw_data 
 * \param hw_data_len 
 * \param sw_data 
 * \param sw_data_len 
 * \param iface Interface to send on
 * 
 * \return
 * 	- 0 - OK
 * 	- 1 - try again later - can't lock enough memory
 * 	- -1 - szedata_tx_unlock_data error
 *
 * Szedata2 packet is prepared and stored directly in szedata2 driver's buffer.
 * Packets are stored one after another until buffer is full, then prepared
 * packets are sent at once. You can explicitly flush buffer, send all prepared
 * and not yet sent packets, by burst_write_flush functions.
 * Flush is performed also during szedata_close function.
 */
int szedata_burst_write_next_with_crc(struct szedata *sze, unsigned char *hw_data, unsigned int hw_data_len, unsigned char *sw_data, unsigned int sw_data_len, unsigned short iface) {
	u_int32_t crc = cl_crc32(sw_data, sw_data_len, sw_data_len, CL_ETHERNET_CRC32_SIZE);
	MSG(CL_VERBOSE_LIBRARY, "%s : CRC %08x", __func__, crc);
	*(u_int32_t *)(sw_data + sw_data_len) = crc;

	return szedata_burst_write_next(sze, hw_data, hw_data_len, sw_data, sw_data_len + CL_ETHERNET_CRC32_SIZE, iface);
}

