/*!
 * \file sze2write_main.c
 * \brief Sends data from pcap dump file to sze2 interface
 * \author Andrej Hank <xhanka00@liberouter.org>
 * \date 2008
 */
/*! Copyright (C) 2008 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */
#include <err.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#include <sys/stat.h>

#include <libsze2.h>
#include <commlbr.h>

#include "sze2write.h"
#include "pcaplbr.h"

#include <pcap.h>

__RCSID("$Id$");
#define VERSION "$Id$"

#define POLL_TIMEOUT -1

#define DUMP_FILE_NAME_MAX_CHARS	255
#define DUMP_FILE_NAME_MAX_DIGITS	3

/* this much driver won't provide */
#define BURST_BUFF_SIZE_MAX (100000)
/* 100 B for netcope alignments and hwdata */
#define BURST_BUFF_SIZE_MIN (1518 + 100)

static void usage(void);
void handSigInt();

bool STOP = false;
int ret = 0;

/* macro used solely in the main function; depends on variable ret and label free_res */
#define ERRX(level, message)	{ret = level; warnx(message); goto free_res;} while (0)

int main(int argc, char *argv[])
{
	unsigned long read_packet_c = 0;	/*!< counter of read packet */
	struct pcap_file_header file_hdr;	/*!< dump file header */
	void *pkt_ptr = NULL;			/*!< packet data pointer */
	struct pcap_pkthdr pkt_hdr;		/*!< packet header */
	int inf = -1;				/*!< dump file descriptor */
	char *infilename = NULL;		/*!< file name */
	char *binary_filename = NULL;		/*!< file name */
	unsigned short interface = 0;		/*!< TX interface number */
	char *sze_dev = "/dev/szedataII0";	/*!< path to device */
	unsigned int rx = 0x00, tx = SZE2_ALL_INTERFACES;
	struct timeval start, stop;

	struct szedata *sze = NULL;
	unsigned char *data = NULL;
	unsigned char * precomputed = NULL;
	unsigned int sze_packet_len = 0;
	bool SILENT = false;
	bool COMPUTE_CRC = false;
	bool PRECOMPUTE_CRC = false;
	bool BINARY_PRECOMPUTED = false;

	int opt;
	struct sigaction sigact;
	short events;
	int r = 0;
	bool got_next = false;

	long ltemp;
	unsigned long ultemp;

	int LOOPS = 1;
	int RANDOM_INTERFACES = false;
	int interfaces_cnt = 4;		/* number of sze interfaces */

	debug = -1;
	verbose = -1;

	char * dump_file_base_name = NULL;
	u_int dump_file_cnt = 0;
	FILE *dump_file = NULL;
	bool DUMP_PACKETS = false;
	u_int dump_size = 0;
	bool BUFFERED_WRITE = false;
	u_int buffer_packet_c = 0;
	u_int buffer_size = 0;

	sze_header_t HEADER_TYPE = NIC;
	uint32_t nific_bitmap=0;


/*! \def ARGUMENTS Acceptable command line arguments */
#ifdef DEBUG
	#define ARGUMENTS ":b:B:cd:D:f:Fhi:l:nN:PRsv:VX:"
#else
	#define ARGUMENTS ":b:B:cd:f:Fhi:l:nN:PRsv:VX:"
#endif

	while ((opt = getopt(argc, argv, ARGUMENTS)) >= 0) {
		switch (opt) {
		case 'b':
			BUFFERED_WRITE = true;
			if ( (cl_xstrtol(optarg, &ltemp, 10)) ||
					(ltemp < BURST_BUFF_SIZE_MIN) ||
					(ltemp > BURST_BUFF_SIZE_MAX)) {
				errx(EXIT_FAILURE, "wrong argument of the -b option, use values between %d and %d",
						BURST_BUFF_SIZE_MIN, BURST_BUFF_SIZE_MAX);
			}
			buffer_size = ltemp;
			SZE2_BURST_WRITE_SIZE = ltemp;
			VERBOSE(CL_VERBOSE_OFF, "WARNING: Driver might not provide big enough buffer size - your application may freeze in a loop");
			VERBOSE(CL_VERBOSE_OFF, "Using burst write buffer size of length: %u", buffer_size);
			break;
		case 'c':
			COMPUTE_CRC = true;
			break;
		case 'd':
			sze_dev = optarg;
			break;
#ifdef DEBUG
		case 'D':	/* debug level */
			if ( (cl_xstrtoul(optarg, &ultemp, 10)) || (ultemp > 2) ) {
				errx(EXIT_FAILURE, "wrong value of -D attribute");
			}
			debug = ultemp;
			break;
#endif
		case 'i':
			if (cl_xstrtoul(optarg, &ultemp, 10)) {
				errx(EXIT_FAILURE, "wrong value of -i attribute");
			}
			interface = ultemp;
			tx = 1 << interface;
			break;
		case 'f':
			infilename = optarg;
			break;
		case 'B':
			binary_filename = optarg;
			break;
		case 'l':
			if( (cl_xstrtoul(optarg, &ultemp, 10)) || (ultemp < 1) ) {
				errx(EXIT_FAILURE, "wrong value of %c attribute", opt);
			}
			LOOPS = ultemp;
			break;
		case 'F':
			LOOPS = -1;
			break;
		case 'n':
			HEADER_TYPE = NIFIC;
			break;
		case 'N':
			HEADER_TYPE = NIFIC4;
			if( (cl_xstrtoul(optarg, &ultemp, 10)) || (ultemp < 1) ) {
				errx(EXIT_FAILURE, "wrong value of %c attribute", opt);
			}
			nific_bitmap = ultemp;
			break;
		case 'V':
			printf("%s\n", VERSION);
			return EXIT_SUCCESS;
		case '?':
		case 'h':
			usage();
			return EXIT_SUCCESS;
		case 'R':
			RANDOM_INTERFACES = true;
			break;
		case 's':
			SILENT = true;
			break;
		case 'P':
			PRECOMPUTE_CRC = true;
			break;
		case 'v':	/* verbose mode */
			if( (cl_xstrtoul(optarg, &ultemp, 10)) || (ultemp > 2) ) {
				errx(EXIT_FAILURE, "wrong value of -v attribute");
			}
			verbose = ultemp;
			break;
		case 'X':
			DUMP_PACKETS = true;
			dump_file_base_name = optarg;
			if(strlen(dump_file_base_name) > DUMP_FILE_NAME_MAX_CHARS - DUMP_FILE_NAME_MAX_DIGITS)
				errx(EXIT_FAILURE, "File name too long");
			break;
		case ':':
			errx(EXIT_FAILURE, "Missing parameter for argument '%c'", optopt);
		default:
			errx(EXIT_FAILURE, "Unknown error");
		}
	}
	argc -= optind;
	argv += optind;

	if (argc != 0)
		errx(EXIT_FAILURE, "Stray arguments.");

	if(COMPUTE_CRC)
		VERBOSE(CL_VERBOSE_OFF, "Computing and adding software CRC");

	if(RANDOM_INTERFACES) 
		VERBOSE(CL_VERBOSE_OFF, "Sending on random interfaces");

	/* seed random generator */
	srand((unsigned int)clock());

	/* register SIGINT handler */
	sigact.sa_handler = handSigInt;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;
	if (sigaction(SIGINT, &sigact, NULL) == -1) {
		errx(EXIT_FAILURE, "sigaction()");
	}

	/* Process packet dump file to sze2 dump file */
	if(PRECOMPUTE_CRC) {
		if(binary_filename && infilename) {
			if (dump2sze2packet(infilename, binary_filename)!=0)
				errx(EXIT_FAILURE, "Can't convert dump %s to binary %s", infilename, binary_filename);
			MSG(CL_VERBOSE_BASIC, "%s file created", binary_filename);
			return EXIT_SUCCESS;
		} else {
			errx(EXIT_FAILURE, "Specify -B and -f");
		}
		/* program ends here */
	}

	if(!binary_filename && !infilename) {
		errx(EXIT_FAILURE, "Specify dump file");
	}

	if(binary_filename) {
		/* Read sze2dump file to memory */
		if (dump2sze_read(binary_filename, &precomputed)!=0)
			errx(EXIT_FAILURE, "Can't read precomputed packets to memory");

		data = precomputed;

		MSG(CL_VERBOSE_BASIC, "%s file buffered", binary_filename);

		BINARY_PRECOMPUTED = true;
	} else {
		/* open pcap dump file and read header */
		if ((inf = cl_pcap_open_dump_file(infilename, &file_hdr)) == -1) {
			return EXIT_FAILURE;
		}
		MSG(CL_VERBOSE_BASIC, "%s opened", infilename);

		/* allocate space for buffer */
		pkt_ptr = malloc(SZE2_MAX_FRAME_SIZE);
		if (!pkt_ptr) {
			ERRX(EXIT_FAILURE, "Can't allocate memory");
		}
	}

	if(DUMP_PACKETS) {
		sze = szedata_open_dummy();
		r = dump_file_open_and_increment(&dump_file_cnt, &dump_file, dump_file_base_name);
		if ((sze == NULL) || (r == false)) {
			ERRX(EXIT_FAILURE, "unable to dump packets");
		}
	} else {
		sze = szedata_open(sze_dev);
		if (sze == NULL) {
			ERRX(3, "szedata_open failed");
		}

		interfaces_cnt = szedata_ifaces_available(sze, SZE2_DIR_TX);
		if (RANDOM_INTERFACES && (tx != SZE2_ALL_INTERFACES)) {
			ERRX(EXIT_FAILURE, "combination of parameters '-R' and '-i' detected");
		}
		if (!RANDOM_INTERFACES && (interface >= interfaces_cnt)) {
			ERRX(EXIT_FAILURE, "selected interface does not exist");
		}

		ret = szedata_subscribe3(sze, &rx, &tx);
		if (ret) {
			goto free_res;
		}

		ret = szedata_start(sze);
		if (ret) {
			goto free_res;
		}

		MSG(CL_VERBOSE_ADVANCED, "Szedata initialized");
	}

	events = SZEDATA_POLLTX;

	gettimeofday(&start, NULL);
	ret = 0;
	while(LOOPS && !STOP) {
		got_next = false;

		/* read and prepare packet */
		if(BINARY_PRECOMPUTED) {
			if(BUFFERED_WRITE) {
				if((sze_packet_len = precached_prepare_burst(data, buffer_size, &buffer_packet_c)) > 0) {
					MSG(CL_VERBOSE_BASIC, "buffer_packet_c %d", buffer_packet_c);
					read_packet_c += buffer_packet_c;
					got_next = true;
				}
			} else { /* not buffered */
				if((sze_packet_len = precached_got_next(data)) != 0) {
					got_next = true;
					read_packet_c++;
				}
			}
		} else {
			if(!cl_pcap_read_pkt(inf, pkt_ptr, &pkt_hdr)) {
				got_next = true;
				read_packet_c++;
				if(!BUFFERED_WRITE) {
					data = prepare_transformed(sze, COMPUTE_CRC, HEADER_TYPE, pkt_ptr, pkt_hdr.caplen, &sze_packet_len, nific_bitmap);
				}
			}
		}

		if(RANDOM_INTERFACES) {
			interface = rand() % interfaces_cnt;
			MSG(CL_VERBOSE_BASIC, "interface %d\n", interface);
		}

		if(got_next) { /* packet read and prepared OK */
			/* print packet contents */
			if(!SILENT) {
				unsigned short print_options = 0;
				if(COMPUTE_CRC || BINARY_PRECOMPUTED) {
					print_options |= SZE2_PRINT_OPTION_ALL | SZE2_PRINT_OPTION_CRC;
				} else {
					print_options |= SZE2_PRINT_OPTION_ALL;
				}

				if(BUFFERED_WRITE) {
					/* TODO - check */
					if(BINARY_PRECOMPUTED) {
						uint32_t k;
						u_char *print_ptr = data;

						for(k = 0; k < buffer_packet_c; k++) {
							printf("\nPacket %-8ld", read_packet_c - buffer_packet_c + k + 1);
							szedata_print_packet(print_ptr, print_options);
							print_ptr += precached_got_next(print_ptr);
						}
					}
				} else {
					printf("\nPacket %-8ld", read_packet_c);
					szedata_print_packet(data, print_options);
				}
			}

			if(DUMP_PACKETS) { /* only dump packets to file, don't send through card */
				if (sze_packet_len > DUMP_FILE_SIZE) {
					ERRX(3, "Can't dump packet, packet size too long");
				}

				dump_packet(data, sze_packet_len, &dump_size, &dump_file, &dump_file_cnt, dump_file_base_name); 
			} else { /* send through card */
				do {
					MSG(CL_VERBOSE_ADVANCED, "try_write %p, %p, %d, %d", sze, data, sze_packet_len, interface);
					if(BUFFERED_WRITE && !BINARY_PRECOMPUTED)
						r = send_burst_transformed(sze,COMPUTE_CRC,HEADER_TYPE, pkt_ptr, pkt_hdr.caplen, interface, nific_bitmap);
					else
						r = szedata_try_write_next(sze, data, sze_packet_len, interface);

					if(r < 0) {
						ERRX(EXIT_FAILURE, "szedata_prepare_and_try_write_next error");
					} else if(r == 1) {
						/* wait */
						events = SZEDATA_POLLTX;
						ret = szedata_poll(sze, &events, POLL_TIMEOUT);
						if (ret < 0)
							goto free_res;
					}
				} while(r == 1);

				/* move to next packet in buffer if precomputed */
				if(BINARY_PRECOMPUTED)
					data += sze_packet_len;
			}

		} else { /* packet read failed - probably no more packets - rewind */
			if (LOOPS == 1) {
				break;
			} else {
				if (BINARY_PRECOMPUTED) {
					data = precomputed;
				} else if (lseek(inf, sizeof(struct pcap_file_header), SEEK_SET) != sizeof(struct pcap_file_header)) {
					ERRX(EXIT_FAILURE, "Can't loop");
				}
				if (LOOPS > 1) {
					LOOPS--;
				}
			}
		}
	}

	gettimeofday(&stop, NULL);
	unsigned long timeval_diff = cl_timeval_diff(start, stop);

	/* close file and free header */
	if(DUMP_PACKETS) {
		printf("%ld packets dumped in %lu.%06lu s. Ending...\n", read_packet_c, timeval_diff / 1000000, timeval_diff % 1000000);
	} else {
		printf("%ld packets sent in %lu.%06lu s. Ending...\n", read_packet_c, timeval_diff / 1000000, timeval_diff % 1000000);
	}

free_res:
	if (sze) {
		if (DUMP_PACKETS) {
			szedata_close_dummy(sze);
		} else {
			szedata_close(sze);
		}
	}

	if (dump_file) {
		fclose(dump_file);
	}

	if(BINARY_PRECOMPUTED) {
		free(precomputed);
	} else {
		cl_pcap_close_dump_file(inf);
		free(pkt_ptr);
	}

	return ret;
}

/*!
 * \brief	Display usage of program
 */
static void usage()
{
#ifdef DEBUG
	printf("Usage: %s [-cFhnPRsV] [-b buf_size] [-B file]\n"
		"                [-d path] [-D level] [-f file] [-i number] [-l loops]\n"
		"                [-N bitmap] [-v level] [-X dump_file]\n", getprogname());
#else
	printf("Usage: %s [-cFhnPRsV] [-b buf_size] [-B file]\n"
		"                [-d path] [-f file] [-i number] [-l loops] [-N bitmap]\n"
		"                [-v level] [-X dump_file]\n", getprogname());
#endif
	printf("-b buf_size    Fast buffered write, buf_size in Bytes (only with -B)\n");
	printf("-B file        Name of a binary file with szedata2 packets and precomputed CRC\n");
	printf("-d path        Path to a device file to use\n");
#ifdef DEBUG
	printf("-D level       Debug level - 0|1|2\n");
#endif
	printf("-f file        Name of a PCAP dump file\n");
	printf("-i number      Interface number\n");
	printf("-l loops       Repeat more times\n");
	printf("-v level       Verbosity level - 0|1|2\n");
	printf("-X dump_file   Dump packets to files (max %dB each) in 8B numbers per line format, don't send through the card (requires -f)\n", DUMP_FILE_SIZE);
	printf("-c             Compute and append CRC32 at the end of szedata2 packets (does not work with -B)\n");
	printf("-F             Loop forever\n");
	printf("-h             Show this text\n");
	printf("-n             Add NIFIC hw header to each packet(random sw interface id)\n");
	printf("-N bitmap      Same as -n, for NIFIC v.4 and set bitmap of output interfaces\n");
	printf("-P             Transform PCAP dump to szedata2 binary dump with precomputed CRC, don't send through card (requires -f -B)\n");
	printf("-R             Send each packet to a random interface\n");
	printf("-s             Silent - don't print packet contents\n");
	printf("-V             Show version\n");
}

/*! 
 * \brief SIGINT signal handler
 * 
 * \param sig 
 */
void handSigInt() {
	STOP = true;
	ret = 0;
}
