/*!
 * \file sze2write.c
 * \brief Sends data from pcap dump file to sze2 interface
 * \author Andrej Hank <xhanka00@liberouter.org>
 * \date 2008
 */
/*! Copyright (C) 2008 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */
#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#include <sys/stat.h>

#include <libsze2.h>
#include <commlbr.h>

#include <pcap.h>

#include "sze2write.h"
#include "write_crc.h"
#include "pcaplbr.h"

__RCSID("$Id$");
#define VERSION "$Id$"

#define DUMP_FILE_NAME_MAX_CHARS	255
#define DATA_IN_INIT_SIZE 10240
#define BUF_SIZE 1024
/*! maximum size of project hw data */
#define MAX_HW_DATA_SIZE		16
/*! nic hw lenght */
#define NIC_HW_DATA_SIZE		0
/*! nific hw lenght */
#define NIFIC_HW_DATA_SIZE		16
#define NIFIC4_HW_DATA_SIZE		5

void project_hw_data(sze_header_t HEADER_TYPE, u_char *hw_data, u_int *hw_len, uint32_t packet_len, uint32_t nific_bitmap);

/*!
 * \brief Open dump file and increment dump file counter.
 * 
 * \param dump_file_cnt Dump file counter
 * \param *file Return parameter - open file 
 * \param dump_file_base_name 
 * 
 * \return 
 * 	false - failed
 * 	true - success
 */
bool dump_file_open_and_increment(u_int *dump_file_cnt, FILE **file, char *dump_file_base_name) {

	char dump_file_name[DUMP_FILE_NAME_MAX_CHARS + 1];

	snprintf(dump_file_name, DUMP_FILE_NAME_MAX_CHARS, "%s.%03d", dump_file_base_name, *dump_file_cnt);

	*file = fopen(dump_file_name, "w");
	if(!(*file))
		return false;
	else
		*dump_file_cnt=*dump_file_cnt + 1;

	return true;

}

/*!
 * \brief Dump packet to file - 8 B numbers per line, hex format.
 * 
 * \param data Packet
 * \param len Lenght
 * \param dump_file File to dump
 */
void dump_packet_to_file(u_char *data, u_int32_t len, FILE * dump_file) {
	int i, j, loops;

	if(len % 8)
		errx(2, "Can't devide len by 8");

	loops = len / 8;
	for(i = 0; i < loops; i++) {
		for(j = 7; j >= 0; j--) {
			fprintf(dump_file, "%02x", *(data + i * 8 + j));
		}
		fprintf(dump_file,"\n");
	}
}


/*!
 * \brief Transform PCAP dump file to szedata2 dump file - szedata2 packets each
 * after another, end with zero segsize.
 * 
 * \param in_file_name Input PCAP dump file name
 * \param out_file_name Output szedata2 dump file name
 * 
 * \return 
 * 	0 - OK
 * 	-1 - failed
 */
int dump2sze2packet(char *in_file_name, char *out_file_name)
{
	struct pcap_file_header file_hdr;	/*!< dump file header */
	void *pkt_ptr = NULL;			/*!< packet data pointer */
	struct pcap_pkthdr pkt_hdr;		/*!< packet header */
	int inf = -1;				/*!< dump file descriptor */
	int pcap_ret;

	struct szedata *sze = NULL;
	unsigned char *data;
	unsigned int new_len;
	u_int pkt_cnt = 0;

	FILE *in_dump_file = NULL;
	FILE *out_file = NULL;
	u_int32_t dump_bytes = 0;

	/* open dummy sze */
	sze = szedata_open_dummy();

	/* open dump file and read header */
	if((inf = cl_pcap_open_dump_file(in_file_name, &file_hdr)) == -1)
		exit(EXIT_FAILURE);
	
	MSG(CL_VERBOSE_BASIC, "%s opened", in_file_name);

	/* allocate space for buffer */
	pkt_ptr = malloc(SZE2_MAX_FRAME_SIZE * sizeof(u_char));
	if(!pkt_ptr)
		errx(1, "Can't allocate memory");


	/* open output file */
	out_file=fopen(out_file_name, "wb");
	if (out_file==NULL)
	{
		VERBOSE(-1, "Failed to open file for output dump: %s", out_file_name);
		free(pkt_ptr);
		return -1;
	}


	pcap_ret=0;
	while(pcap_ret==0) 
	{
		/* read packet */
		pcap_ret = cl_pcap_read_pkt(inf, pkt_ptr, &pkt_hdr);
		if(pcap_ret == 0) /* packet read OK */
		{
			pkt_cnt++;
			data = szedata_prepare_packet_with_crc(sze, NULL, 0,
				pkt_ptr, pkt_hdr.caplen, &new_len);

			/* print packet contents */
			VERBOSE(1, "dump2sze2packet: new packet of len %d:", new_len);
			if (verbose>0)
				szedata_print_packet(data,
					SZE2_PRINT_OPTION_ALL | SZE2_PRINT_OPTION_CRC);

			fwrite(data, 1, new_len, out_file);
			dump_bytes += new_len;
		}
	}

	/* Write terminating zero packet */
	unsigned char zero_packet[2]={0};
	fwrite(zero_packet, 2, 1, out_file);

	/* close files and print summary */
	if(in_dump_file) { fclose(in_dump_file); }
	if(out_file) { fclose(out_file); }

	VERBOSE(1, "%d packets dumped in %d Bytes\n", pkt_cnt, dump_bytes);

	szedata_close_dummy(sze);
	cl_pcap_close_dump_file(inf);
	free(pkt_ptr);

	return 0;
}


/*!
 *  Returns size of packet which is first on pointed address, aligned to 8
 */
uint32_t precached_got_next(unsigned char *data)
{
	return CL_ALIGN((data[1] << 8) | data[0], 8);
}


/*!
 * \brief Read sze2 dump file to memory
 */
int dump2sze_read(char *file_name, unsigned char **data)
{
	FILE *sze_dump_file = NULL;
	uint32_t data_max_size=DATA_IN_INIT_SIZE*sizeof(unsigned char);
	uint32_t i=0;

	sze_dump_file=fopen(file_name, "rb");
	if (sze_dump_file==NULL)
	{
		VERBOSE(-1, "Failed to open file with sze2 dump: %s", file_name);
		return -1;
	}

	*data=malloc(data_max_size);
	if (*data==NULL)
	{
		VERBOSE(-1, "Can't allocate memory");
		return -1;
	}

	uint32_t have_read=0, read=0;

	/* Read all data to buffer */
	while ((read = fread(*data + i*BUF_SIZE, 1, BUF_SIZE, sze_dump_file))!=0)
	{
		have_read += read;

		if (have_read+BUF_SIZE>=data_max_size) /* need bigger memory */
		{
			*data=realloc(*data, data_max_size*2);

			if (data==NULL)
			{
				VERBOSE(-1, "Not enough memory for dump file.");
				return -1;
			}

			data_max_size *= 2;
		}
		i++;
	}

	if(sze_dump_file) { fclose(sze_dump_file); };

	return 0;
}

/*!
 * \brief Prepare BIG szedata2 packet from precached buffer - more packets after each other. For faster transfers.
 * 
 * \param data Data
 * \param buffer_size Maximum size of buffer - maximum size of BIG packet
 * \param buffer_packet_c Return parameter - number of buffered packets
 * 
 * \return 
 * 	Lenght of BIG packet
 */
u_int precached_prepare_burst(u_char *data, u_int buffer_size, u_int *buffer_packet_c) {
	bool buffer_ready = false;
	uint32_t buffered_len = 0, packet_len = 0;
	u_char *buffer_ptr = data;

	*buffer_packet_c = 0;

	do {
		if((packet_len = precached_got_next(buffer_ptr)) != 0) {  /* got next */

			if ((buffered_len + CL_ALIGN(packet_len, 8)) <=
					buffer_size) {
				/* add next packet to buffer */
				buffered_len += CL_ALIGN(packet_len, 8);
				(*buffer_packet_c) += 1;
				buffer_ptr += CL_ALIGN(packet_len, 8);
			} else { /* buffer full */
				buffer_ready = true;
			}
		} else { /* no next */
			buffer_ready = true;
		}
	} while (!buffer_ready);

	MSG(CL_VERBOSE_BASIC, "buffered bytes  : %8d B", buffered_len);
	MSG(CL_VERBOSE_BASIC, "buffered packets: %8d ", *buffer_packet_c);

	return buffered_len;
}

/*!
 * \brief Prepare szedata2 packet to sze tx_buffer. Add CRC if specified, add
 * it, add hwdata according header type.
 * 
 * \param sze Handle
 * \param COMPUTE_CRC Add 4B CRC
 * \param HEADER_TYPE Specify type
 * \param sw_data Data
 * \param sw_len Lenght
 * \param packet_len Return parameter - szedata2 packet lenght
 * 
 * \return
 * 	Pointer to szedata2 packet stored in tx_buffer
 */
u_char * prepare_transformed(struct szedata *sze, bool COMPUTE_CRC, sze_header_t HEADER_TYPE, u_char *sw_data, u_int sw_len, u_int *packet_len, uint32_t nific_bitmap) {
	u_char *data;

	u_char hw_data[MAX_HW_DATA_SIZE];
	u_int hw_data_len = 0;

	project_hw_data(HEADER_TYPE, hw_data, &hw_data_len, sw_len, nific_bitmap);

	if(COMPUTE_CRC) {
		data = szedata_prepare_packet_with_crc(sze, hw_data, hw_data_len, sw_data, sw_len, packet_len);
	} else {
		data = szedata_prepare_packet(sze, hw_data, hw_data_len, sw_data, sw_len, packet_len);
	}

	return data;

}

/*!
 * \brief Create hw_data and hw_len according to specified HEADER_TYPE
 * \param HEADER_TYPE
 * \param hw_data Buffer of lenght MAX_HW_DATA_SIZE
 * \param hw_len Return parameter - lenght of hw data
 * \param packet_len Needed with nific HEADER_TYPE
 */
void project_hw_data(sze_header_t HEADER_TYPE, u_char *hw_data, u_int *hw_len, uint32_t packet_len, uint32_t nific_bitmap) {

	uint8_t tmp = 0;
	memset(hw_data, 0, MAX_HW_DATA_SIZE);

	switch(HEADER_TYPE) {
	case NIC:
		*hw_len = NIC_HW_DATA_SIZE;
		break;
    
	case NIFIC:
		*hw_len = NIFIC_HW_DATA_SIZE;
		MSG(CL_VERBOSE_LIBRARY, "sw_len: 0x%04x", packet_len);

		/* 4b interface ID */
		tmp = (rand() % 14) + 2;
		MSG(CL_VERBOSE_LIBRARY, "interface ID: 0x%02x", tmp);
		/* LSB 4b of packet lenght */
		tmp += (packet_len & 0xF) << 4;
		*((uint8_t *)hw_data) = tmp;
		MSG(CL_VERBOSE_LIBRARY, "interface ID (0-3 @ 0-3) and sw_len(0-3 @ 4-7): 0x%02x", tmp);

		tmp = (uint8_t)(packet_len >> 4);
		*((uint8_t *)(hw_data + 1)) = tmp;
		MSG(CL_VERBOSE_LIBRARY, "sw_len (4-11 @ 0-7): 0x%02x", tmp);

		tmp = (uint8_t)(packet_len >> 12);
		*((uint8_t *)(hw_data + 2)) = tmp;
		MSG(CL_VERBOSE_LIBRARY, "sw_len (12-15 @ 0-3): 0x%02x", tmp);
    
		break;
    
	case NIFIC4:
		*hw_len = NIFIC4_HW_DATA_SIZE;
		MSG(CL_VERBOSE_LIBRARY, "sw_len: 0x%04x", packet_len);

		/* 4b interface ID */
		tmp = 15;
		MSG(CL_VERBOSE_LIBRARY, "interface ID: 0x%02x", tmp);
		/* LSB 4b of packet lenght */
		tmp += (packet_len & 0xF) << 4;
		MSG(CL_VERBOSE_LIBRARY, "interface ID (0-3 @ 0-3) and sw_len(0-3 @ 4-7): 0x%02x", tmp);
		*((uint8_t *)hw_data) = tmp;

		tmp = (uint8_t)(packet_len >> 4);
		*((uint8_t *)(hw_data + 1)) = tmp;
		MSG(CL_VERBOSE_LIBRARY, "sw_len (4-11 @ 0-7): 0x%02x", tmp);

		tmp = (uint8_t)(packet_len >> 12);
		/* LSB 4b of bitmap */
		tmp += (nific_bitmap & 0xF) << 4;
		*((uint8_t *)(hw_data + 2)) = tmp;
		MSG(CL_VERBOSE_LIBRARY, "sw_len (12-15 @ 0-3) and bitmap (0-3 @ 4-7): 0x%02x", tmp);

		tmp = (uint8_t)(nific_bitmap >> 4);
		*((uint8_t *)(hw_data + 3)) = tmp;
		MSG(CL_VERBOSE_LIBRARY, "bitmap (4-11 @ 0-7): 0x%02x", tmp);

		tmp = (uint8_t)(nific_bitmap >> 12);
		*((uint8_t *)(hw_data + 4)) = tmp;
		MSG(CL_VERBOSE_LIBRARY, "bitmap (12-15 @ 0-3): 0x%02x", tmp);

		break;
	}
}

/*!
 * \brief Send burst packet transformed according COMPUTE_CRC and HEADER_TYPE.
 *
 * \return
 *	0 - OK
 *	-1 - failed
 *	1 - try again later
 */
int send_burst_transformed(struct szedata *sze, bool COMPUTE_CRC, sze_header_t HEADER_TYPE, u_char *sw_data, u_int sw_len, u_short iface, uint32_t nific_bitmap) {
	u_char hw_data[MAX_HW_DATA_SIZE];
	u_int hw_data_len = 0;

	project_hw_data(HEADER_TYPE, hw_data, &hw_data_len, sw_len, nific_bitmap);

	if(COMPUTE_CRC)
		return szedata_burst_write_next_with_crc(sze, hw_data, hw_data_len, sw_data, sw_len, iface);
	else
		return szedata_burst_write_next(sze, hw_data, hw_data_len, sw_data, sw_len, iface);
}

/*!
 * \brief Add packet to dump file. If necessary close, increment and open new
 * one.
 * 
 * \param data Szedata2 packet
 * \param sze_packet_len Packet len
 * \param dump_size Current size of file
 * \param *dump_file Current open file
 * \param dump_file_cnt Current file number
 * \param dump_file_base_name File base name
 */
void dump_packet(u_char *data, u_int sze_packet_len, u_int *dump_size, FILE **dump_file, u_int *dump_file_cnt, char *dump_file_base_name) {
	u_int32_t first_size = 0, second_size = 0;
	(*dump_size) += sze_packet_len;

	if(*dump_size >= DUMP_FILE_SIZE) {
		second_size = *dump_size - DUMP_FILE_SIZE;
		first_size = sze_packet_len - second_size;

		MSG(CL_VERBOSE_ADVANCED, "ending file: %03d first: %d second: %d whole: %d\n", *dump_file_cnt - 1, first_size, second_size, sze_packet_len);
		/* write first data */
		dump_packet_to_file(data, first_size, *dump_file);

		/* close and open another file */
		if(fclose(*dump_file) != 0)
			errx(1, "Can't close file");

		dump_file_open_and_increment(dump_file_cnt, dump_file, dump_file_base_name);

		/* write second data */
		if(second_size > 0)
			dump_packet_to_file(data + first_size, second_size, *dump_file);

		*dump_size = second_size;
	} else {
		dump_packet_to_file(data, sze_packet_len, *dump_file);
	}
}

