/*!
 * \file pcaplbr.c
 * \brief Functions for common tasks with PCAP structures, library
 * \author Andrej Hank <xhanka00@liberouter.org>
 * \date 2008
 */
/*!
 * Copyright (C) 2008 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#ifndef CL_PCAP_DISABLE

#include <fcntl.h>
#include <pcap.h>
#include <time.h>
#include <unistd.h>

#include <commlbr.h>

#include "pcaplbr.h"
/** \brief CVS ID tag for ident program */
__RCSID("$Id$");

/*!
 * \brief Print PCAP packet header data, timestamps, or content. Format can be set by CL_PCAP_PRINT macros.
 *
 * \param packet Frame contents
 * \param header PCAP header
 * \param print_options Format options
 *
 * Print options can be combined by logical OR.
 */
void cl_pcap_print_packet(const unsigned char *packet, struct pcap_pkthdr header, unsigned short print_options) {
	int print_head = 0, print_sw = 0, print_crc = 0;

	if(print_options & CL_PCAP_PRINT_RICH) {
		print_head = 1;
		print_sw = 1;
	} else if(print_options & CL_PCAP_PRINT_BRIEF) {
		print_head = 1;
	}

	if(print_options & CL_PCAP_PRINT_CONTENT)
		print_sw = 1;

	if(print_options & CL_PCAP_PRINT_CRC)
		print_crc = 1;


	if(print_head)
		printf("%s%.6ld us - %.5d B, captured %.5d B\n",
		ctime(&(header.ts.tv_sec)), header.ts.tv_usec, header.len,
		header.caplen);

	if(print_sw)
		cl_dump_buffer(stdout, (unsigned char *)packet,
			print_crc ? header.caplen - CL_ETHERNET_CRC32_SIZE
			: header.caplen, "sw ", CL_DUMP_CHARS_PER_LINE,
			CL_DUMP_CHARS_PER_WORD, 0x0);

	if(print_crc)
		cl_dump_buffer(stdout,
			(unsigned char *)(packet + header.caplen -
			CL_ETHERNET_CRC32_SIZE),
			CL_ETHERNET_CRC32_SIZE,
			"crc", CL_DUMP_CHARS_PER_LINE,
			CL_DUMP_CHARS_PER_WORD, 0x0);

	fflush(stdout);
}

/*!
 * \brief Read PCAP file header
 * 
 * \param fd File handle
 * \param filehdr Return parameter - read file header
 * 
 * \return
 * 	0 - OK
 OK* 	-1 - failed
 */
int cl_pcap_read_file_hdr(int fd, struct pcap_file_header *filehdr) {

	int result = 0;
	ssize_t datalen;

	datalen = read(fd, filehdr, sizeof(struct pcap_file_header));
	if (datalen < (ssize_t)sizeof(struct pcap_file_header)) {
		MSG(CL_VERBOSE_LIBRARY, "%s : Can't read file header", __func__);
		result = -1;
	}
	return result;
}


/*!
 * \brief Read PCAP packet header from file
 * 
 * \param fd File handle
 * \param pkt_hdr Return parameter - read header
 *
 * \return
 * 	0 - OK
 * 	-1 - failed
 */
int cl_pcap_read_pkt_hdr(int fd, struct pcap_pkthdr *pkt_hdr) {
	int datalen;
/*Have to read into special struct, due to pcap allways save timestamp as 32bit (mentioned in pcap.h)*/
	struct {
		uint32_t sec;
		uint32_t usec;
		uint32_t caplen;
		uint32_t len;
	} pcap_hdr;

	datalen = read(fd, &pcap_hdr, sizeof(pcap_hdr));

	if (datalen < 16)  {
		MSG(CL_VERBOSE_LIBRARY, "%s : Can't read packet header", __func__);
		return -1;
	}

	pkt_hdr->ts.tv_sec	= pcap_hdr.sec;
	pkt_hdr->ts.tv_usec	= pcap_hdr.usec;
	pkt_hdr->caplen		= pcap_hdr.caplen;
	pkt_hdr->len		= pcap_hdr.len;

	return 0;
}

/*!
 * \brief Read frame data from file
 * 
 * \param fd File handle
 * \param pkt_ptr Pointer to store data to - must be preallocated (say to MAX
 * ETHERNET FRAME SIZE)
 * \param pkt_len Lenght to read
 * 
 * \return
 * 	0 - OK
 * 	-1 - failed
 */
int cl_pcap_read_pkt_data(int fd, void *pkt_ptr, u_int32_t pkt_len) {

	ssize_t datalen;
	int result = 0;

	datalen = read(fd, pkt_ptr, pkt_len);

	if (datalen < (ssize_t)pkt_len) {
		result = -1;
	};

	return result;
}

/*!
 * \brief Read one packet from pcap dump
 * 
 * \param inf File descriptor
 * \param pkt_ptr Space for packet content to be copiet, min MAX_ETH_FRAME_SIZE
 * must be provided
 * \param pkt_hdr Packet header - return parameter - must be allocated (size does not change), will be changed
 * 
 * \return
 * 	-1 - failed 
 * 	0 - OK - packet read into pkt_ptr
 */
int cl_pcap_read_pkt(int inf, void *pkt_ptr, struct pcap_pkthdr *pkt_hdr) {

	/* read header */
	if (cl_pcap_read_pkt_hdr(inf, pkt_hdr))
		return -1;

	/* read data */
	if(cl_pcap_read_pkt_data(inf, pkt_ptr, pkt_hdr->caplen))
		return -1;

	return 0;
}

/*!
 * \brief Open pcap dump file and read file header
 * 
 * \param in_file_name File name
 * \param file_hdr Return parameter - read header
 * 
 * \return
 * 	file descriptor of open dump file - success
 * 	-1 - failed
 */
int cl_pcap_open_dump_file(char *in_file_name, struct pcap_file_header *file_hdr) {

	int inf; /* file descriptor */

	if ((inf = open(in_file_name,O_RDONLY)) < 0) {
		VERBOSE(CL_VERBOSE_OFF, "%s : Cannot open input dump file: %s", __func__, in_file_name);
		return -1;
	}


	if(cl_pcap_read_file_hdr(inf, file_hdr)) {
		VERBOSE(CL_VERBOSE_OFF,"%s: Error reading input file header.\n", __func__);
		return -1;
	}

	return inf;
}

/*!
 * \brief Close file
 * 
 * \param inf File descriptor
 * 
 * \return
 * 	0 - success
 * 	-1 - failed
 */
int cl_pcap_close_dump_file(int inf) {

	if(close(inf) == -1) {
		VERBOSE(CL_VERBOSE_OFF,"%s: Can't close dump file.\n", __func__);
		return -1;
	}

	return 0;
}

#endif
