/*!
 * Copyright (C) 2010 CESNET
 * Author(s): Andrej Hank
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <stdbool.h>
#include <err.h>
#include <errno.h> 
#include <pcap.h>

#include <commlbr.h>
#include <libsze2.h>

/* let pool for 256 KB */

bool stop_flag = false;
int ret, ret_val = -1;
unsigned long long cnt = 0, bytes = 0;

void handSigInt();

int main(int argc, char **argv) {

	bool USE_PCAP = false;


	/* COMMON structures */
	unsigned int len;
	u_char *data;
	int seconds = 0;
	int ret;

	/* SZE2 structures */
	struct szedata *sze = NULL;
	char *sze_dev = "/dev/szedataII0";
	unsigned int rx = 0xFF, tx = 0x00;

	/* PCAP structures */
	pcap_t *pcap_handle = NULL;
	struct pcap_pkthdr header;
	char errbuf[PCAP_ERRBUF_SIZE];      /* Error string */
	char *pcap_interface = NULL;

/*! \def ARGUMENTS Acceptable command line arguments */
#ifdef DEBUG
	#define ARGUMENTS "d:i:hP:s:"
#else
	#define ARGUMENTS "d:i:hD:P:s:"
#endif


	int opt;
	long ltemp; /*!< getopt */

	while ((opt = getopt(argc, argv, ARGUMENTS)) >= 0) {
		switch (opt) {
		case 'P':
			MSG(CL_VERBOSE_OFF, "using PCAP functions");
			USE_PCAP = true;
			pcap_interface = optarg;
			MSG(CL_VERBOSE_OFF, "sniffing on %s", pcap_interface);
			break;
		case 'd':
			sze_dev = optarg;
			break;
		case 'i':
			if( (cl_strtol(optarg, &ltemp)) ||  (ltemp < 0) ) {
				errx(1, "wrong value of -i attribute");
			}
			rx =  1 << ltemp;
			MSG(CL_VERBOSE_OFF, "will read on %ld", ltemp);
			break;
		case 's':
			if( (cl_strtol(optarg, &ltemp)) ||  (ltemp <= 0) ) {
				errx(1, "wrong value of -s attribute");
			}
			seconds = ltemp;
			MSG(CL_VERBOSE_OFF, "will count with %d seconds", seconds);
			break;
#ifdef DEBUG
		case 'D':	/* debug level */
			if( (cl_strtol(optarg, &ltemp)) || (ltemp > 2) || (ltemp < 0) ) {
				errx(1, "wrong value of -D attribute");
                        }
			debug = ltemp;
			break;
#endif
		case '?':
		case 'h':
			printf("usage ./sze2read [-s seconds] [-P iface] [-Dh]\n"
				   "      ./sze2read [-s seconds] [-d path] [-i iface] [-Dh]\n");
			exit(0);
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 0)
		errx(EXIT_FAILURE, "Stray arguments.");

	struct sigaction sigact;

	/* register SIGINT handler */
	sigact.sa_handler = handSigInt;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;
	if (sigaction(SIGINT, &sigact, NULL) == -1) {
		errx(1, "sigaction()");
	}
	


	if(USE_PCAP) {
		/* initialize PCAP */
		pcap_handle = pcap_open_live(pcap_interface, 1000, 1, 0, errbuf);
		if(pcap_handle == NULL) {
			errx(1, "Can't initialize pcap - %s", errbuf);
		}
	} else {
		/* initialize SZE2 */
		sze = szedata_open(sze_dev);
		if (sze == NULL)
			errx(3, "szedata_open failed");

		MSG(CL_VERBOSE_BASIC, "subscribing: rx-0x%02hx; tx-0x%02hx",
				rx, tx);
		MSG(CL_VERBOSE_BASIC, "POLL TIMEOUTs RX: %u; TX: %u", SZE2_RX_POLL_TIMEOUT, SZE2_TX_POLL_TIMEOUT);
		ret = szedata_subscribe3(sze, &rx, &tx);
		if (ret)
			abort();

		MSG(CL_VERBOSE_BASIC, "subscribed: rx-0x%02hx tx-0x%02hx", rx, tx);

		ret = szedata_start(sze);
		if (ret)
			abort();
	}

	while(!stop_flag) {
		if(USE_PCAP) {
			data = (unsigned char *)pcap_next(pcap_handle, &header);
			if(data != NULL) {
				cnt++;
				bytes+=header.len;
			} else {
				MSG(CL_VERBOSE_ADVANCED, "nothing to read");
			}

		} else {
			if((data = szedata_read_next(sze, &len)) != NULL) {
				cnt++;
				bytes+=len;
			} else {
				MSG(CL_VERBOSE_ADVANCED, "nothing to read");
			}
		}
	}

	MSG(CL_VERBOSE_OFF, "%lld received packets", cnt);
	MSG(CL_VERBOSE_OFF, "%lld bytes", bytes);
	MSG(CL_VERBOSE_OFF, "%lld B average", cnt == 0 ? 0 : bytes/cnt);
	if(seconds) {
		MSG(CL_VERBOSE_OFF, "%lf Mpkt/s", (cnt >> 20) / (double)seconds);
		MSG(CL_VERBOSE_OFF, "%lf Gpkt/s", (cnt >> 30) / (double)seconds);
		MSG(CL_VERBOSE_OFF, "%lf MB/s", (bytes >> 20) / (double)seconds);
		MSG(CL_VERBOSE_OFF, "%lf Gb/s", (bytes >> 27) / (double)seconds);
	}
#ifdef PERF_DEBUG
	MSG(CL_VERBOSE_OFF, "%d split cnt", split_cnt);
	MSG(CL_VERBOSE_OFF, "%d lock cnt", lock_cnt);
	MSG(CL_VERBOSE_OFF, "%d unlock cnt", unlock_cnt);

	MSG(CL_VERBOSE_OFF, "average lock %d B", unlock_cnt == 0 ? 0 : bytes / unlock_cnt);
#endif

	if(!USE_PCAP)
		szedata_close(sze);

	return ret_val;
}

void handSigInt() { 
	stop_flag = true;
	ret_val = 0;
}

