/*!
 * \file	obufctl.c
 * \brief	Tool for controlling OBUF component
 * \author	Andrej Hank <xhanka00@liberouter.org>
 * \author	Miroslav Vadkerti <thrix@liberouter.org>
 * \author	Peter Stranak <stranak@liberouter.org>
 * \author	Kramolis Petr <kramolis@cesnet.cz>
 * \date	2006-2011
 */

/*
 * Copyright (C) 2006-2009  CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <unistd.h>

#include <combo.h>
#include <commlbr.h>

__RCSID ("$Id$");
#define VERSION "1.3"

/* \brief acceptable command line arguments */
#define ARGUMENTS	":b:d:e:i:m:R:s:v:x:AhrV"

/* EMAC component name */
#define EMAC_COMP_NAME	"EMAC"
/* EMAC Mode Configuration offset */
#define EMAC_CTRL_OFFS	0x100
/* EMAC Mode Configuration data for 10/100/1000 Mbps */
#define EMAC_SPEED_10		0x14000000
#define EMAC_SPEED_100		0x54000000
#define EMAC_SPEED_1000		0x94000000

/* ID component base addres */
#define CSID_COMPONENT_BASE_ADDR 0x00000000
/* ID component repeater */
#define CSID_ID_MINOR_ADDR     	 0x48
/* ID component repeater */
#define CSID_DESIGN_NAME         0x20
/* ID component repeater */
#define CSID_REPEATER_ADDR       0x70

/* ID component and NETCOPE versions with repeater */
#define ID_MINOR_REPEATER 3
#define ID_MAJOR_REPEATER 1
#define NETCOPE_MINOR_REPEATER 1
#define NETCOPE_MAJOR_REPEATER 2

/*! COMBO6_2XFP2 add on card name */
#define COMBO6_2XFP2	"xfp2"
/*! COMBOI_10G1 add on card name */
#define COMBOI_10G2	"10G1"
/*! COMBOI_10G2 add on card name */
#define COMBOI_10G4	"10G2"
/*! COMBOI_1G4 add on card name */
#define COMBOI_1G4	"1G4"
/*! UNKNOWN add on card name */
#define COMBO_UNKNOWN	"unknown"

/*!
 * \def	OBUF_COMP_NAME
 * \brief	OBUF component name
 */
#define OBUF_COMP_NAME	"OBUF"


/* 2XFP2 flag */
bool xfp = false;

/* 1G4 card flag */
bool c_1g4 = false;

/*
 * \var Control verbosity
 */
extern int verbose;

/*!
 * \def	MAC_BUFSIZE
 * \brief	MAC buffer size
 */
#define MAC_BUFSIZE	20

/*!
 * \enum	commands
 * \brief	List of available commands
 */
enum commands {
	CMD_ENABLE,
	CMD_RESET,
	CMD_SET_SPEED,
	CMD_SET_MAC,
	CMD_PRINT_STATUS,
	CMD_SET_REPEATER
};

/*!
 * \brief	Convert hex number to integer
 *
 * \param number	Converted hex number
 *
 * \retval	converted value on success, -1 on error
 */
int
hex2num(char number)
{
	if (number >= '0' && number <= '9') {
		return number - '0';
	}
	if (number >= 'a' && number <= 'f') {
		return number - 'a' + 10;
	}
	if (number >= 'A' && number <= 'F') {
		return number - 'A' + 10;
	}
	return -1;
}

/*!
 * \brief	Convert HW address string (MAC) to 2 32bit integer.
 *		Lo contains lower 32 bits, hi higher 16 bits
 *
 * \param txt	Converted MAC string (aa:bb:cc:dd:ee:ff)
 * \param hi	Higher 16 bits
 * \param lo	Lower 32bits of MAC
 *
 * \retval	0 on success, -1 on error
 */
int
str2hwaddr(char *txt, u_int32_t *hi, u_int32_t *lo)
{
	int byte, a, b;

	for (byte = 0; byte < 6; byte++) {
		a = hex2num(*txt++);
		if (a < 0) {
			return -1;
		}
		b = hex2num(*txt++);
		if (b < 0) {
			return -1;
		}
		if (byte > 1) {
			*lo = *lo << 8;
			*lo |= (a << 4) | b;
		} else {
			*hi = *hi << 8;
			*hi |= (a << 4) | b;
		}
		if (byte < 5 && *txt++ != ':') {
			return -1;
		}
	}

	return 0;
}

/*!
 * \brief 	Convert 48bit MAC address stored in lower and higher
 *		integer into MAC address string. Txt must be an > 18 bit array.
 *
 * \param txt	MAC string (aa:bb:cc:dd:ee:ff) from conversion
 * \param hi	Higher 16 bits
 * \param lo	Lower 32bits of MAC
 */
void
hwaddr2str(char *txt, u_int32_t hi, u_int32_t lo)
{
	char str[MAC_BUFSIZE] = {0};	/* temp string */
	int i;	/* cycle variable */

	/* print */
	sprintf(str, "%04x%08x", hi, lo);
	VERBOSE(CL_VERBOSE_LIBRARY, "Extracted HW address: %s\n", str);

	/* insert colons */
	for (i = 0; i < 13; i++) {
		if (i % 2 == 0 && i != 0 && i != 12) {
			*txt++ = ':';
		}
		*txt++ = str[i];
	}
}

/*!
 * \brief 	Check if there is repeater implemented in firmware
 *
 * \param dev		COMBO device
 * \param id_space	COMBO space with mapped ID component
 *
 * \retval	0 firmware without repeater
 * \retval	1 firmware with repeater
 */
int
repeater_check(cs_device_t *dev, cs_space_t *id_space){
	int id_major,id_minor;
	int netcope_major, netcope_minor;
	int versions;
	char * version;
	char *board, *card, *chip;

	/* read ID component and netcope version */
	versions = cs_space_read_4(dev, id_space, CSID_ID_MINOR_ADDR);
	version = (char *)&versions;
	/* get ID component version*/
	id_minor= *version;
	id_major= *(version+1);
	/* get Netcope version */
	netcope_minor = *(version+2);
	netcope_major = *(version+3);
	/* identify board */
	if (cs_identify(dev, &board, &card, &chip))
		board = NULL;

	/* check if id commponent supports repeater */
	if(id_major > ID_MAJOR_REPEATER || (id_major == ID_MAJOR_REPEATER
		&& (id_minor >= ID_MINOR_REPEATER ))){
	} else {
		return 0;
	}
	/* check if netcope supports repeater */
	if(netcope_major > NETCOPE_MAJOR_REPEATER || (netcope_major == NETCOPE_MAJOR_REPEATER
		&& netcope_minor>= NETCOPE_MINOR_REPEATER) 
		|| (board && (strcmp(board, "FB8XG") == 0 || strcmp(board, "FB1CG") == 0 ))) {
	} else {
		return 0;
	}
	return 1;
}

/*!
 * \brief Print OBUF speed
 *
 * \param speed Speed flag in OBUF status register
 */
void
print_speed(speed)
{
	switch (speed) {
	case 0x1:
		printf("100 Mb/s\n");
		break;
	case 0x2:
		printf("1 Gb/s\n");
		break;
	case 0x3:
		printf("10 Gb/s\n");
		break;
	case 0x4:
		printf("40 Gb/s\n");
		break;
	case 0x5:
		printf("100 Gb/s\n");
		break;
	default:
		printf("10 Mb/s\n");
		break;
	}
}

/*!
 * \brief	Print OBUF counters
 *
 * \param data	Data to print in obuf_t structure
 * \param emac_data	EMAC Mode Configuration data for speed info
 * \param ifc	Printed interface
 */
void
print_obuf(cs_obuf_t data, u_int32_t emac_data, int ifc)
{
	char	mac_string[MAC_BUFSIZE] = {0};	/* MAC string */

	printf("------------------ OBUF Status ----\n");
	/* OBUF was probably selected by its base adress */
	if (ifc < 0) {
		printf("Interface    : unknown\n");
	} else {
		printf("Interface    : %d\n", ifc);
	}
	printf("OBUF         : %s\n", (data.reg_en != 1) ? "DISABLED" : "ENABLED");
	printf("OBUF speed   : ");

	if (c_1g4) {
		/* OBUF was probably selected by its base adress - in emac_data is a garbage*/
		if (ifc < 0) {
			printf("unknown\n");
		} else {
			/* Speed is set in two highests bits */
			print_speed(emac_data >> 30);
		}
	} else {
		print_speed((data.obuf_status & 0x70) >> 4);
	}

	printf("---- OBUF Packets/Frames Stats ----\n");
	printf("Packets      : %llu\n", (long long unsigned) data.cnt_packets);
	printf("Transmitted  : %llu\n", (long long unsigned) data.cnt_recv);
	printf("Err packets  : %llu\n", (long long unsigned) data.cnt_error);
	printf("---------------- OBUF Settings ----\n");
	hwaddr2str(mac_string, data.mac_high, data.mac_low);
	printf("MAC address  : %s\n", mac_string);

}


/*!
 * \brief	Print repeater counters
 *
 * \param dev		COMBO device
 * \param id_space	COMBO space with mapped ID component
 * \param ifc	Printed interface
 */
void
print_repeater(cs_device_t *dev, cs_space_t *id_space, int ifc){
	uint32_t repeater;

	/* check if this is firmware with repeater */
	if(!repeater_check(dev,id_space)){
		return;
	}

	/* read repeater settings */
	repeater = cs_space_read_4(dev, id_space, CSID_REPEATER_ADDR);
	/* get repeater settings for specified interface */
	repeater = ((repeater >> (ifc*2)) & 3);
	printf("\n");
	printf("-------------- REPEATER Status ----\n");
	printf("Interface   : %d\n",ifc);
	if(repeater == 1){
		printf("REPEATER    : 2 (idle)\n");
	} else if (repeater  == 3){
		printf("REPEATER    : 1 (repeater output)\n");
	} else {
		printf("REPEATER    : 0 (OBUF output)\n");
	}
}

#define BOARD_UNKNOWN 0
#define BOARD_FB8XG   1
/*!
 * \brief	Identify FPGA board.
 * \param path	Path to device file to use
 * \return		One of BOARD_? values.
 */
int
identify_board(char *path)
{
	int board_id = BOARD_UNKNOWN;
	char *board, *card, *chip;
	cs_device_t *device;
	if (cs_attach_noex(&device, path) == 0) {
		if (cs_identify(device, &board, &card, &chip) == 0) {
			if (strcmp(board,"FB8XG") == 0)
				board_id = BOARD_FB8XG;
		}
		cs_detach(&device);
	}
	return board_id;
}

/*!
 * \brief	Display usage of program
 */
void
usage(int board_id)
{
	printf("Usage: %s [-ehrvV] [-b address] [-d path] [-i interface]\n"
			"               [-m address] [-s speed] [-v level] [-x file] \n",
			getprogname());
	printf("Only one command may be used at a time.\n");
	printf("-b address     OBUF component base address (hex)\n");
	printf("-d path        Path to device file to use\n");
	printf("-e 0|1         Enable(1)/disable(0) interface\n");
	printf("-i interface   Select interface (default 0)\n");
	printf("-R state       Set repeater status (0 - OBUF, 1 - repeater, 2 - idle)\n");
	printf("-m address     Set OBUF MAC address (format aa:bb:cc:dd:ee:ff)\n");
	if(board_id == BOARD_UNKNOWN)
		printf("-s speed       Set speed to 10|100|1000 Mb/s (usable only for 1 Gb/s copper interfaces)\n");
	printf("-v level       Verbose level (0-2)\n");
	printf("-x file        File design.xml used for component space mapping\n");
	printf("-A             Print status of all interfaces or affect all interfaces if used\n"
			"               with other option\n");
	printf("-h             Show this text\n");
	printf("-r             Reset OBUF frame counters\n");
	printf("-V             Show version\n");
}

/*!
 * \brief	Map OBUF (also EMAC, if necessary) component
 *
 * \param dev		COMBO device
 * \param obuf_space	COMBO space with mapped OBUF component
 * \param emac_space	COMBO space with mapped EMAC component
 * \param emac_data	EMAC Mode Configuration data for speed info
 * \param ifc		Used interface
 *
 * \retval	0 on success
 * \retval	-1 on error
 * \retval	1 when there is no OBUF on selected interface
 */
int
map_ifc(cs_device_t *dev, cs_space_t **obuf_space, cs_space_t **emac_space,
		u_int32_t *emac_data, int ifc)
{
	cs_component_t	*comp_id;

	/* find information about OBUF component */
	if (cs_component_find(dev, &comp_id, NULL, OBUF_COMP_NAME, ifc) != 0) {
		/* component not found */
		return 1;
	}

	/* map OBUF space */
	if (cs_component_space(comp_id, obuf_space) != 0) {
		warnx("Failed to map OBUF component.");
		return -1;
	}

	/* on 1G4 cards map EMAC for speed control and get EMAC Mode Configuration data */
	if (c_1g4 == true) {
		if (cs_component_find(dev, &comp_id, NULL, EMAC_COMP_NAME, ifc) != 0) {
			warnx("Component EMAC with index %d not found in your design.", ifc);
			return -1;
		}
		if (cs_component_space(comp_id, emac_space) != 0) {
			warnx("Failed to map EMAC component.");
			return -1;
		}
		*emac_data = (u_int32_t) cs_space_read_4(dev, *emac_space, EMAC_CTRL_OFFS);
	}
	return 0;
}

/*!
 * \brief	Load design
 *
 * \param dev		COMBO device
 * \param xmlfile	Path to file
 *
 * \retval	0 on success
 * \retval	-1 on error
 */
int
load_design(cs_device_t *dev, char *xmlfile)
{
	int	xerr = 0;

	/* load design xml from file */
	if (xmlfile != NULL) {
		/* load data about design from specified design.xml */
		xerr = cs_design_reload(dev, CS_DESIGN_XML, xmlfile);
		if (xerr < 0) {
			warnx("Loading file %s failed : %s.", xmlfile, strerror(-xerr));
		}
	} else {
		/* load data about design from corresponding design.xml */
		xerr = cs_design_reload(dev, CS_DESIGN_NONE, NULL);
		if (xerr < 0) {
			warnx("Loading design.xml failed. Please check your installation.");
		}
	}
	return xerr;
}

/*!
 * \brief	Do requested operation on selected device
 *
 * \param command	Requested operation
 * \param param		Parameters for the operation
 * \param dev		COMBO device
 * \param obuf_space	COMBO space with mapped OBUF component
 * \param emac_space	COMBO space with mapped EMAC component
 * \param id_space	COMBO space with mapped ID component
 * \param data		OBUF data structure
 * \param emac_data	EMAC Mode Configuration data for speed info
 * \param ifc		Used interface
 *
 * \retval	0 on success
 * \retval	-1 on error
 */
int
execute_operation(enum commands command, long param, cs_device_t *dev,
		cs_space_t *obuf_space, cs_space_t *emac_space, cs_space_t *id_space,
		cs_obuf_t data, u_int32_t emac_data, int ifc , long rep_setting )
{
	int		ret;
	u_int32_t	hi, lo;		/* HI, LOW MAC */
	cs_phy_t	*phy;		/* phyter */
	unsigned int phy_speed;
	uint32_t repeater=0;
	int rep_value=0; //repeater value (default OBUF)



	switch (command) {
		/* If enable/disable OBUF required */
		case CMD_ENABLE:
			if (param == 0) {
				VERBOSE(CL_VERBOSE_BASIC, "Disabling OBUF.");
				cs_obuf_disable(dev, obuf_space);
			} else {
				VERBOSE(CL_VERBOSE_BASIC, "Enabling OBUF.");
				cs_obuf_enable(dev, obuf_space);
			}
			break;

		/* If reset frame counters required */
		case CMD_RESET:
			VERBOSE(CL_VERBOSE_BASIC, "Resetting frame counters.");
			cs_obuf_reset(dev, obuf_space);
			break;

		/* If set OBUF speed required */
		case CMD_SET_SPEED:
			VERBOSE(CL_VERBOSE_BASIC, "Set Speed to: %ld\n", param);
			/* if other than 1G4 card detected */
			if (!c_1g4) {
				warnx("You can change speed only on ComboI-1G4 cards.");
				return -1;
			}
			if (ifc < 0) {
				warnx("You can't address OBUF with '-b address'"
						" parameter while setting its speed on ComboI-1G4"
						" cards. Use parameter '-i interface' instead.");
				return -1;
			}
			/* attach phyter */
			phy = cs_phy_attach(dev, NULL, NULL, NULL, ifc);
			/* phy->dbytes == 1	--> SFP		--> Optical
			   else			--> GBIC	--> Copper
			   according to phyterctl
			*/
			if (phy != NULL) {
				if (phy->dbytes != 1) {
					/* advertise speed in phyter */
					phy_speed = (unsigned)param;
					ret = phy->phy_service(phy, CS_PHY_ADVSPEED, &phy_speed, NULL, NULL);
					if (ret != 0) {
						cs_phy_detach(phy);
						warnx("Unsupported operation for interface %d.", ifc);
						return -1;
					}

					/* advertise speed in EMAC */
					if (param == 1000) {
						cs_space_write_4(dev, emac_space, EMAC_CTRL_OFFS, EMAC_SPEED_1000);
					} else if (param == 100) {
						cs_space_write_4(dev, emac_space, EMAC_CTRL_OFFS, EMAC_SPEED_100);
					} else if (param == 10) {
						cs_space_write_4(dev, emac_space, EMAC_CTRL_OFFS, EMAC_SPEED_10);
					}
				} else {
					printf("Optical phyter found on interface %d. Skipping ...\n", ifc);
				}
				cs_phy_detach(phy);
			} else {
				printf("No compatible phyter found on interface %d\n", ifc);
			}
			break;

		/* If MAC address specified */
		case CMD_SET_MAC:
			hi = lo = 0;
			if (str2hwaddr((char *)param, &hi, &lo) == -1) {
				warnx("Bad MAC address %s. Please use the format aa:bb:cc:dd:ee:ff.",
						(char *)param);
				return -1;
			}
			/* disable OBUF before proceeding */
			if (data.reg_en) {
				cs_obuf_disable(dev, obuf_space);
			}

			VERBOSE(CL_VERBOSE_LIBRARY, "Writing MAC string %s > HI: 0x%x  LO: 0x%x ",
					(char *)param, hi, lo);
			cs_obuf_set_mac(dev, obuf_space, hi, lo);

			/* enable OBUF if was disabled */
			if (data.reg_en) {
				cs_obuf_enable(dev, obuf_space);
			}
			break;

		/* Set repeater */
		case CMD_SET_REPEATER:

			//check repeater support
			if(!repeater_check(dev, id_space)){
				errx(EXIT_FAILURE, "There is no repeater in loaded firmware\n");
			}

			//map repeater values
			if(rep_setting==1){
				rep_value=3;
			}
			else if(rep_setting==2){
				rep_value=1;
			}

			rep_value =  rep_value << ifc*2;
			//read repeater setting
			repeater = cs_space_read_4(dev, id_space, CSID_REPEATER_ADDR);
			//clean repeater setting for 'ifc' interface
			repeater = repeater & (~(3 << (ifc*2)));

			//set repeater setting for 'ifc' interface
			rep_value = repeater | rep_value;
			//write repeater setting
			cs_space_write_4(dev, id_space, CSID_REPEATER_ADDR,rep_value);
			break;

		/* If print OBUF status required */
		case CMD_PRINT_STATUS:
			print_obuf(data, emac_data, ifc);
			print_repeater(dev,id_space,ifc);
			break;

		default:
			break;
	}
	return 0;
}

/*!
 * \brief	Detach device (also unload design) and exit program
 *
 * \param ret		Exit value
 * \param dev		COMBO device
 */
void
detach_and_exit(int ret, cs_device_t *dev)
{
	cs_design_free(dev);
	cs_detach(&dev);
	exit(ret);
}

/*!
 * \brief	Program main function.
 *
 * \param	argc	number of arguments
 * \param	*argv[]	array with the arguments
 *
 * \retval	EXIT_SUCCESS on success
 * \retval	EXIT_FAILURE on error
 */
int
main(int argc, char *argv[])
{
	cs_device_t	*dev;			/* COMBO device */
	cs_space_t	*obuf_space;		/* OBUF component space */
	cs_space_t	*emac_space;		/* EMAC component space */
	cs_space_t      *id_space;		/* ID component space */
	u_int32_t	base_address = 0;	/* component address in design */
	cs_obuf_t	data;			/* OBUF data structure */
	u_int32_t	emac_data = 0;		/* EMAC Mode Configuration data */
	long		ifc = 0;		/* interface number */
	char		*file = CS_PATH_DEV(0);	/* default COMBO device */
	char		*xmlfile = NULL;	/* design.xml file with path */
	int		c;			/* temp variable */
	long		tmp;			/* temp variable */
	char		*card;			/* cs_identify param */
	bool		all = false;		/* indicates to affect all ifaces */
	char		cmds = 0;		/* how many operations requested */
	char		interfaces = 0;
	enum commands	command = CMD_PRINT_STATUS;
	long		param = 0;
	long		rep_setting = 0;		/* repeater setting*/
	u_int32_t	ifc_count = 0;
	char 		name_buf[32];
	int 		i;
	bool		help = false;
	char 		char_tmp;

	/* process command line arguments */
	opterr = 0;
	while ((c = getopt(argc, argv, ARGUMENTS)) != -1) {
		switch(c) {
			case 'b':
				if (cl_strtol(optarg, &tmp) ||
						tmp <= 0) {
					errx(EXIT_FAILURE, "Wrong base address.");
				}
				base_address = tmp;
				interfaces++;
				break;
			case 'd':
				file = optarg;
				break;
			case 'e':
				if (cl_strtol(optarg, &param) ||
						(param != 0 && param != 1)) {
					errx(EXIT_FAILURE, "Wrong enable value (0|1).");
				}
				command = CMD_ENABLE;
				cmds++;
				break;
			case 'i':
				if (cl_strtol(optarg, &ifc) || ifc < 0) {
					errx(EXIT_FAILURE, "Wrong interface number.");
				}
				interfaces++;
				break;
			case 'm':
				param = (long)optarg;
				command = CMD_SET_MAC;
				cmds++;
				break;
			case 's':
				if (cl_strtol(optarg, &param) ||
						(param != 10 && param != 100 && param != 1000)) {
					errx(EXIT_FAILURE, "Wrong speed. Must be one of 10|100|1000.");
				}
				command = CMD_SET_SPEED;
				cmds++;
				break;
			case 'v':
				if (cl_strtol(optarg, &tmp) ||
						tmp < 0) {
					errx(EXIT_FAILURE, "Invalid verbosity level attribute. Please specify number "
							"between 0 and 2.");
				}
				verbose = (tmp <= 2)? tmp : 2;
				break;
			case 'x':
				xmlfile = optarg;
				break;
			case 'A':
				all = true;
				interfaces++;
				break;
			case 'h':
				help = true;
				break;
			case 'R':
				if (cl_strtol(optarg, &rep_setting) ||
						(rep_setting < 0 || rep_setting > 2)) {
					errx(EXIT_FAILURE, "Wrong repeater setting. Must be one of 0, 1, 2.");
				}
				command = CMD_SET_REPEATER;
				cmds++;
				break;
			case 'r':
				command = CMD_RESET;
				cmds++;
				break;
			case 'V':
				printf("OBUF control tool - version %s\n", VERSION);
				return EXIT_SUCCESS;
			case '?':
				errx(EXIT_FAILURE, "Unknown argument '%c'", optopt);
			case ':':
				errx(EXIT_FAILURE, "Missing parameter for argument '%c'", optopt);
			default:
				errx(EXIT_FAILURE, "Unknown error");
		}
	}
	argc -= optind;
	argv += optind;

	if (argc != 0) {
		errx(EXIT_FAILURE, "stray arguments");
	}

	if (help) {
		usage(identify_board(file));
		return EXIT_SUCCESS;
	}

	if (cmds > 1) {
		errx(EXIT_FAILURE, "More than one operation requested. Please select just one.");
	}

	if (interfaces > 1) {
		errx(EXIT_FAILURE, "Combination of parameters '-b', '-A', '-i' detected. "
				"Please don't combine them.");
	}

	/* attach device and map address spaces */
	if (cs_attach_noex(&dev, file) != 0) {
		err(EXIT_FAILURE, "cs_attach_noex failed");
	}

	/* find information about component (depending on Combo card) */
	if (cs_identify(dev, NULL, &card, NULL) != 0) {
		warnx("cs_identify() failed");
		detach_and_exit(EXIT_FAILURE, dev);
	}
	VERBOSE(CL_VERBOSE_BASIC, "Addon: %s", card);
	if (!strncmp(card, COMBO6_2XFP2, sizeof(COMBO6_2XFP2) - 1) ||
			!strncmp(card, COMBO_UNKNOWN, sizeof(COMBO_UNKNOWN) - 1)) {
		xfp = true;
	} else if (!strncmp(card, COMBOI_1G4, sizeof(COMBOI_1G4) - 1)) {
		xfp = true;
		c_1g4 = true;
	} else if (!strncmp(card, COMBOI_10G2, sizeof(COMBOI_10G2) - 1)) {
		xfp = true;
	} else if (!strncmp(card, COMBOI_10G4, sizeof(COMBOI_10G4) - 1)) {
		xfp = true;
	}


	/* map space of id component */
	if (cs_space_map(dev, &id_space, CS_SPACE_FPGA, 0x100,
		CSID_COMPONENT_BASE_ADDR, 0)) {
		err(EXIT_FAILURE, "cs_space_map failed to map space at"
			"address 0x%X!", CSID_COMPONENT_BASE_ADDR);
	}

	/*get design name */
        cs_space_read_multi_4(dev,id_space, CSID_DESIGN_NAME,8,(uint32_t *)name_buf);
	for (i=0;i<32;i=i+2){
		char_tmp=name_buf[i];
		name_buf[i]=name_buf[i+1];
		name_buf[i+1]=char_tmp;
	}

	/* if all interfaces requested */
	if (all) {
		if (load_design(dev, xmlfile) < 0) {
			detach_and_exit(EXIT_FAILURE, dev);
		}

		if (cs_component_number(dev, NULL, OBUF_COMP_NAME, &ifc_count) != 0) {
			warnx("cs_component_number() failed");
			detach_and_exit(EXIT_FAILURE, dev);
		}

		/* go through all OBUFs */
		for (ifc = 0; ifc < ifc_count; ifc++) {


			if(command!=CMD_SET_REPEATER){
				tmp = map_ifc(dev, &obuf_space, &emac_space, &emac_data, ifc);
				if (tmp == 1) {
					if(command == CMD_PRINT_STATUS){
						print_repeater( dev, id_space, ifc);
					}
					continue;
				} else if (tmp < 0) {
					detach_and_exit(EXIT_FAILURE, dev);
				}

			/* read all data in OBUF registers */
			data = cs_obuf_read(dev, obuf_space, xfp);
			}

			if (execute_operation(command, param, dev, obuf_space, emac_space, id_space, data, emac_data, ifc, rep_setting) != 0) {
				detach_and_exit(EXIT_FAILURE, dev);
			}

			if (command == CMD_PRINT_STATUS) {
				printf("\n");
			}
		}
	} else {
		if (base_address != 0) {
			/* Map space of the component specified by base address */
			VERBOSE(CL_VERBOSE_BASIC, "Mapping space at base 0x%x", base_address);
			if (cs_space_map(dev, &obuf_space, CS_SPACE_FPGA, CS_MAP_ALL, base_address, 0) != 0) {
				warnx("cs_space_map failed to map obuf_space");
				detach_and_exit(EXIT_FAILURE, dev);
			}
			/* Interface number is now invalid */
			ifc = -1;

			/* map space of id component */
			if (cs_space_map(dev, &id_space, CS_SPACE_FPGA, 0x100,
				CSID_COMPONENT_BASE_ADDR, 0)) {
				err(EXIT_FAILURE, "cs_space_map failed to map space at"
					"address 0x%X!", CSID_COMPONENT_BASE_ADDR);
			}
		} else {
			/* Map space of the component specified by design */
			if (load_design(dev, xmlfile) < 0) {
				detach_and_exit(EXIT_FAILURE, dev);
			}

			if (cs_component_number(dev, NULL, OBUF_COMP_NAME, &ifc_count) != 0) {
				warnx("cs_component_number() failed");
				detach_and_exit(EXIT_FAILURE, dev);
			}

			if (ifc_count == 0) {
				warnx("No interface found in your design.");
				detach_and_exit(EXIT_FAILURE, dev);
			} else if(ifc >= ifc_count){
				warnx("Interface %ld is out of range (0-%d).", ifc, ifc_count - 1);
				detach_and_exit(EXIT_FAILURE, dev);
			}

			if(command == CMD_SET_REPEATER){
				/*skip obuf_space nad emac_space mapiing*/
			} else if (map_ifc(dev, &obuf_space, &emac_space, &emac_data, ifc) != 0) {
				if(command == CMD_PRINT_STATUS){
					print_repeater( dev, id_space, ifc);
					detach_and_exit(EXIT_SUCCESS, dev);
				} else {
					warnx("Component OBUF with index %ld not found in your design.", ifc);
					detach_and_exit(EXIT_FAILURE, dev);
				}
			}
		}

		/* read all data in OBUF registers */
		if(command != CMD_SET_REPEATER){
			data = cs_obuf_read(dev, obuf_space, xfp);
		}

		if (execute_operation(command, param, dev, obuf_space, emac_space, id_space, data, emac_data, ifc, rep_setting) != 0) {
			if (base_address != 0) {
				cs_space_unmap(dev, &obuf_space);
			}
			detach_and_exit(EXIT_FAILURE, dev);
		}
	}

	if (base_address != 0) {
		cs_space_unmap(dev, &obuf_space);
	}
	detach_and_exit(EXIT_SUCCESS, dev);
	return EXIT_SUCCESS;
}
