/*!
 * \file afilterctl_main.c
 * \brief Tool for controlling advanced filter - configure filter
 * \author Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: afilterctl_main.c 3122 2013-02-06 01:41:51Z xkekel00 $
 *
 */

#include "afilterctl_main.h"
#include "afilterctl_input.h"
#include "hwio.h"
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <arpa/inet.h>

/// offset of counter registers
#define COUNTERS_OFFSET  0x200

/// filter registers addresses: read from HW
#define REGRD_DISABLE     0x000
#define REGRD_IGNORE      0x004
#define REGRD_FULL        0x008
#define REGRD_BUSY        0x00C
#define REGRD_RULE_TYPES  0x010
#define REGRD_IV_NUMBER   0x014
#define REGRD_DATA        0x018
#define REGRD_PROTO       0x01C
#define REGRD_PORTS       0x020
#define REGRD_SRCIP       0x024
#define REGRD_DSTIP       0x034
#define REGRD_IV          0x044
#define _REGRD_RULES_INFO(a)  (REGRD_IV+((a)<<2))
#define REGRD_RULES_INFO  _REGRD_RULES_INFO(flt_get_config(comp)->iv_number)

/// filter registers addresses: write to HW
#define REGWR_STOPCMD     0x000
#define REGWR_IGNORECMD   0x004
#define REGWR_ADDCMD      0x008
#define REGWR_DELCMD      0x00C
#define REGWR_DIRRD       0x010
#define REGWR_DIRCLR      0x014
#define REGWR_DATA        0x018
#define REGWR_PROTO       0x01C
#define REGWR_PORTS       0x020
#define REGWR_SRCIP       0x024
#define REGWR_DSTIP       0x034
#define REGWR_IV          0x044

/// read counters of given interface
inline int flt_read_ifc_counters(struct hwio_comp *comp, int iface, flt_cnt_status *s) {
    uint32_t al,ah,dl,dh;
    if(hwio_comp_read(comp,COUNTERS_OFFSET+(iface<<4)+0x4,&ah)!=HWIO_EOK ||
       hwio_comp_read(comp,COUNTERS_OFFSET+(iface<<4)    ,&al)!=HWIO_EOK ||
       hwio_comp_read(comp,COUNTERS_OFFSET+(iface<<4)+0xC,&dh)!=HWIO_EOK ||
       hwio_comp_read(comp,COUNTERS_OFFSET+(iface<<4)+0x8,&dl)!=HWIO_EOK)
        return 1;
    s->allow=(((uint64_t)ah)<<32)+al;
    s->deny =(((uint64_t)dh)<<32)+dl;
    return 0;
}

/// reset counters of given interface
inline int flt_reset_ifc_counters(struct hwio_comp *comp, int iface) {
    return hwio_comp_write(comp,COUNTERS_OFFSET+(iface<<4),0)!=HWIO_EOK;
}

/// read counters of each interface of HW filter
int flt_read_counters(struct hwio_comp *comp, int reset, int ints, flt_cnt_status *res) {
    if(ints<=0)
        return 0;
    for(int i=0; i<ints; i++) {
        if(flt_read_ifc_counters(comp,i,res+i)!=0)
            return 1;
        if(reset)
            if(flt_reset_ifc_counters(comp,i)!=0)
                return 1;
    }
    return 0;
}

/// read status of HW filter rules
int flt_read_status(struct hwio_comp *comp, flt_status_flags *res) {
    if(hwio_comp_read(comp,REGRD_DISABLE,&(res->disable))!=HWIO_EOK ||
       hwio_comp_read(comp,REGRD_IGNORE, &(res->ignore) )!=HWIO_EOK ||
       hwio_comp_read(comp,REGRD_FULL,   &(res->full)   )!=HWIO_EOK ||
       hwio_comp_read(comp,REGRD_BUSY,   &(res->busy)   )!=HWIO_EOK)
        return 1;
    return 0;
}

/// parse rule from string into rule structure
int flt_parse_rule(struct hwio_comp *comp, char *str, flt_rule *res) {
    unsigned e;
    if(!flt_get_rule(str,res,&e))
        return PARSE_RULE_ERROR;
    flt_config *fc=flt_get_config(comp);
    if(fc==NULL)
        return PARSE_RULE_HWERROR;
    if(res->type>=fc->rule_types)
        return PARSE_RULE_ERROR;
    if((fc->cfg[res->type].flags&e) ||
       (EMPTY_DATA&e))
        return PARSE_RULE_EMPTY;
    return 0;
}

/// fills register with source IP address
int flt_write_srcip(struct hwio_comp *comp, uint32_t *ip) {
    uint32_t addr=REGWR_SRCIP;
    for(int i=0; i<4; i++, addr+=4)
        if(hwio_comp_write(comp,addr,ip[i])!=HWIO_EOK)
            return 1;
    return 0;
}
/// fills register with destination IP address
int flt_write_dstip(struct hwio_comp *comp, uint32_t *ip) {
    uint32_t addr=REGWR_DSTIP;
    for(int i=0; i<4; i++, addr+=4)
        if(hwio_comp_write(comp,addr,ip[i])!=HWIO_EOK)
            return 1;
    return 0;
}
/// fills register with port numbers
int flt_write_ports(struct hwio_comp *comp, uint16_t sp, uint16_t dp) {
    return hwio_comp_write(comp,REGWR_PORTS,(((uint32_t)dp)<<16)+sp)!=HWIO_EOK;
}
/// fills register with protocol number
int flt_write_proto(struct hwio_comp *comp, uint8_t p) {
    return hwio_comp_write(comp,REGWR_PROTO,p)!=HWIO_EOK;
}
/// fills register with rule identificator
int flt_write_data(struct hwio_comp *comp, uint32_t id) {
    return hwio_comp_write(comp,REGWR_DATA,id)!=HWIO_EOK;
}

/// reads register with source IP address
int flt_read_srcip(struct hwio_comp *comp, uint32_t *ip) {
    uint32_t addr=REGRD_SRCIP;
    for(int i=0; i<4; i++, addr+=4)
        if(hwio_comp_read(comp,addr,ip+i)!=HWIO_EOK)
            return 1;
    return 0;
}
/// reads register with destination IP address
int flt_read_dstip(struct hwio_comp *comp, uint32_t *ip) {
    uint32_t addr=REGRD_DSTIP;
    for(int i=0; i<4; i++, addr+=4)
        if(hwio_comp_read(comp,addr,ip+i)!=HWIO_EOK)
            return 1;
    return 0;
}
/// reads register with port numbers
int flt_read_ports(struct hwio_comp *comp, uint16_t *sp, uint16_t *dp) {
    uint32_t ports;
    if(hwio_comp_read(comp,REGRD_PORTS,&ports)!=HWIO_EOK)
        return 1;
    if(sp) *sp=ntohs(ports&0xFFFF);
    if(dp) *dp=ntohs((ports>>16)&0xFFFF);
    return 0;
}
/// reads register with protocol number
int flt_read_proto(struct hwio_comp *comp, uint8_t *p) {
    uint32_t proto;
    if(hwio_comp_read(comp,REGRD_PROTO,&proto)!=HWIO_EOK)
        return 1;
    *p=proto&0xFF;
    return 0;
}
/// reads register with rule identificator
int flt_read_data(struct hwio_comp *comp, uint32_t *id) {
    return hwio_comp_read(comp,REGRD_DATA,id)!=HWIO_EOK;
}

/// fills data registers of HW filter
int flt_write_registers(struct hwio_comp *comp, flt_rule *r, int write_data) {
    flt_config *fc=flt_get_config(comp);
    if(fc==NULL)
        return 1;
    int ret=0;
    uint8_t f=fc->cfg[r->type].flags;
    if(f&FLT_RULE_TYPE_FLAGS_SRCIP)
        ret|=flt_write_srcip(comp,r->srcip);
    if(f&FLT_RULE_TYPE_FLAGS_DSTIP)
        ret|=flt_write_dstip(comp,r->dstip);
    if((f&FLT_RULE_TYPE_FLAGS_SRCPORT) ||
       (f&FLT_RULE_TYPE_FLAGS_DSTPORT))
        ret|=flt_write_ports(comp,r->srcport,r->dstport);
    if(f&FLT_RULE_TYPE_FLAGS_PROTO)
        ret|=flt_write_proto(comp,r->proto);
    if(write_data)
        ret|=flt_write_data(comp,r->id);
    return ret;
}

/// reads data registers of HW filter
int flt_read_registers(struct hwio_comp *comp, flt_rule *r) {
    flt_config *fc=flt_get_config(comp);
    if(fc==NULL)
        return 1;
    int ret=0;
    uint8_t f=fc->cfg[r->type].flags;
    if(f&FLT_RULE_TYPE_FLAGS_SRCIP)
        ret|=flt_read_srcip(comp,r->srcip);
    if(f&FLT_RULE_TYPE_FLAGS_DSTIP)
        ret|=flt_read_dstip(comp,r->dstip);
    if((f&FLT_RULE_TYPE_FLAGS_SRCPORT) ||
       (f&FLT_RULE_TYPE_FLAGS_DSTPORT))
        ret|=flt_read_ports(comp,&(r->srcport),&(r->dstport));
    if(f&FLT_RULE_TYPE_FLAGS_PROTO)
        ret|=flt_read_proto(comp,&(r->proto));
    ret|=flt_read_data(comp,&(r->id));
    return ret;
}

/// calls add command of HW filter
int flt_add_rule_cmd(struct hwio_comp *comp, unsigned mask) {
    return hwio_comp_write(comp,REGWR_ADDCMD,mask)!=HWIO_EOK;
}

/// calls remove command of HW filter
int flt_del_rule_cmd(struct hwio_comp *comp, unsigned mask) {
    return hwio_comp_write(comp,REGWR_DELCMD,mask)!=HWIO_EOK;
}

/// wait for filter to be ready
int flt_wait_ready(struct hwio_comp *comp, unsigned mask) {
    int loops=0,err=HWIO_EOK;
    uint32_t stat=0;
    while((err=hwio_comp_read(comp,REGRD_BUSY,&stat))==HWIO_EOK &&
          (stat&(MASK_GLOBAL_BUSY|(mask<<1)))) {
        loops++;
        if(loops>=1024)
            return 0;
        usleep(100);
    }
    if(err!=HWIO_EOK)
        return 0;
    return 1;
}

/// adds rule into HW filter
int flt_add_rule(struct hwio_comp *comp, flt_rule *r) {
    uint32_t mask=MASK_RULE_TYPE(r->type),stat;
    int loops=0,err=HWIO_EOK;
    // wait for filter to be ready
    if(!flt_wait_ready(comp,mask))
        return ADD_RULE_BUSY;
    // fill registers
    if(flt_write_registers(comp,r,1)!=0)
        return ADD_RULE_BUSY;
    // wait for end of last add
    while((err=hwio_comp_read(comp,REGRD_FULL,&stat))==HWIO_EOK && (stat&mask)) {
        loops++;
        if(loops>=1024)
            return ADD_RULE_FULL;
        usleep(100);
    }
    if(err!=HWIO_EOK)
        return ADD_RULE_BUSY;
    // add rule
    if(flt_add_rule_cmd(comp,mask)!=0)
        return ADD_RULE_BUSY;
    return 0;
}

/// removes rule from HW filter
int flt_del_rule(struct hwio_comp *comp, flt_rule *r) {
    uint32_t mask=MASK_RULE_TYPE(r->type);
    // wait for filter to be ready
    if(!flt_wait_ready(comp,mask))
        return DEL_RULE_BUSY;
    // fill registers
    if(flt_write_registers(comp,r,0)!=0)
        return DEL_RULE_BUSY;
    // remove rule
    if(flt_del_rule_cmd(comp,mask)!=0)
        return DEL_RULE_BUSY;
    return 0;
}

/// reads init vector register from HW filter
int flt_read_initv(struct hwio_comp *comp, uint32_t *iv) {
    uint32_t addr=REGRD_IV;
    int iv_words=flt_get_config(comp)->iv_number;
    for(int i=0; i<iv_words; i++, addr+=4)
        if(hwio_comp_read(comp,addr,iv+i)!=HWIO_EOK)
            return 1;
    return 0;
}

/// fills init vector register inside HW filter
int flt_write_initv(struct hwio_comp *comp, uint32_t *iv) {
    uint32_t addr=REGWR_IV;
    int iv_words=flt_get_config(comp)->iv_number;
    for(int i=0; i<iv_words; i++, addr+=4)
        if(hwio_comp_write(comp,addr,iv[i])!=HWIO_EOK)
            return 1;
    return 0;
}

/// parse init vector register from string representation
int flt_parse_initv(struct hwio_comp *comp, char *str, uint32_t *iv) {
    unsigned e;
    int iv_words=flt_get_config(comp)->iv_number;
    if(flt_get_initv(str,iv,&e)!=iv_words)
        return PARSE_INITV_ERROR;
    if(e==((1<<iv_words)-1))
        return PARSE_INITV_NOEFF;
    for(int i=0; i<iv_words; i++)
        if(e&(1<<i))
            if(hwio_comp_read(comp,REGRD_IV+4*i,iv+i)!=HWIO_EOK)
                return PARSE_INITV_ERROR;
    return 0;
}

/// calls stop command of HW filter
int flt_stop_cmd(struct hwio_comp *comp, unsigned mask) {
    return hwio_comp_write(comp,REGWR_STOPCMD,mask)!=HWIO_EOK;
}

/// calls ignore command of HW filter
int flt_ignore_cmd(struct hwio_comp *comp, unsigned mask) {
    return hwio_comp_write(comp,REGWR_IGNORECMD,mask)!=HWIO_EOK;
}

/// initialize direct iterator into HW filter hash tables
void flt_dir_iterator_init(int rt, flt_dir_iterator *it) {
    it->type=rt;
    it->table=0;
    it->item=0;
}

/// next iterator, should be empty field
void flt_dir_iterator_next(struct hwio_comp *comp, flt_dir_iterator *it) {
    it->item++;
    if(flt_get_config(comp)->cfg[it->type].items<=it->item) {
        it->item=0;
        it->table++;
    }
}

/// transforms dirrect iterator into dirrect address
uint32_t flt_dir_iterator_2addr(flt_config *fc, flt_dir_iterator *it) {
    return (it->type<<(32-fc->rule_types_width))|(it->table<<fc->cfg[it->type].items_width)|(it->item);
}

/// finds and reads next valid rule from HW filter hash tables
int flt_dir_iterator_nextrule(struct hwio_comp *comp, flt_dir_iterator *it, flt_rule *r) {
    flt_config *fc=flt_get_config(comp);
    if(fc==NULL)
        return 1;
    uint32_t data;
    while(!(it->table>=fc->cfg[it->type].tables&&it->item>0)) {
        if(hwio_comp_write(comp,REGWR_DIRRD,flt_dir_iterator_2addr(fc,it))!=HWIO_EOK ||
           flt_read_data(comp,&data)!=0)
            return 0;
        if(data!=FILTER_NOMATCH) {
            r->type=it->type;
            return flt_read_registers(comp,r)==0;
        }
        flt_dir_iterator_next(comp,it);
    }
    return 0;
}

/// clears field inside HW filter hash tables at address given by iterator
int flt_dir_iterator_clear(struct hwio_comp *comp, flt_dir_iterator *it) {
    return hwio_comp_write(comp,REGWR_DIRCLR,flt_dir_iterator_2addr(flt_get_config(comp),it))!=HWIO_EOK;
}

/// clears all rules inside HW filter hash tables of type given by iterator
int flt_dir_iterator_clearall(struct hwio_comp *comp, flt_dir_iterator *it) {
    flt_config *fc=flt_get_config(comp);
    uint32_t items=fc->cfg[it->type].items;
    int l2items=fc->cfg[it->type].items_width;
    uint32_t tables=fc->cfg[it->type].tables;
    uint32_t start=it->type<<(32-fc->rule_types_width);
    uint32_t table_start;
    for(uint32_t t=0; t<tables; t++) {
        table_start=start|(t<<l2items);
        for(uint32_t i=0; i<items; i++)
            if(hwio_comp_write(comp,REGWR_DIRCLR,table_start|i)!=HWIO_EOK)
                return 1;
    }
    return hwio_comp_write(comp,REGWR_DIRCLR,start|(tables<<l2items))!=HWIO_EOK;
}

int flt_read_config(struct hwio_comp *comp, uint32_t *cfg, int *iv) {
    uint32_t rt;
    if(hwio_comp_read(comp,REGRD_IV_NUMBER,&rt)!=HWIO_EOK)
        return 0;
    *iv=rt;
    if(hwio_comp_read(comp,REGRD_RULE_TYPES,&rt)!=HWIO_EOK)
        return 0;
    for(unsigned i=0; i<rt; i++)
        if(hwio_comp_read(comp,REGRD_RULES_INFO+(i<<2),cfg+i)!=HWIO_EOK)
            return 0;
    return rt;
}

