/*!
 * \file afilterctl_input.h
 * \brief Tool for controlling advanced filter - read configuration module
 * \author Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: afilterctl_input.h 3122 2013-02-06 01:41:51Z xkekel00 $
 *
 */

#ifndef AFILTERCTL_INPUT_H_INCLUDED
#define AFILTERCTL_INPUT_H_INCLUDED

#include "afilterctl_main.h"

#define SUPPORTED_CCCI_VERSION 0x1

#define NETOFFSETSTATUS (999) //TODO!!!

#define NETOFFSETTYPE    (4)
#define NETOFFSETIPVER   (5)
#define NETOFFSETSRCIPMASK (6)
#define NETOFFSETPROTO     (7)
#define NETOFFSETSRCIP0  (8)
#define NETOFFSETSRCIP1 (12)
#define NETOFFSETSRCIP2  (16)
#define NETOFFSETSRCIP3  (20)
#define NETOFFSETDSTIP0  (24)
#define NETOFFSETDSTIP1  (28)
#define NETOFFSETDSTIP2  (32)
#define NETOFFSETDSTIP3  (36)
#define NETOFFSETSRCPORT   (40)
#define NETOFFSETDSTPORT   (42)
#define NETOFFSETID        (44)
#define NETOFFSETSID       (48)

#define CCCIRESPONSESIZE (52)
#define CCCIREQUESTSIZE (52)

#define NETACTIONADD    0
#define NETACTIONDELETE 1

#define ACTIONSTATUSFAIL    0
#define ACTIONSTATUSSUCCESS 1

#define CCCIHEADERLEN (sizeof(ccci_header_t))

typedef struct  {
	char version;
	char reserved;
	uint32_t probe_id ;
} ccci_header_t;


typedef struct {
    uint8_t action;   /*!< Action to perform with FRule */
    uint8_t ipver;    /*!< IP protocol version */
    uint16_t status;  /*!< Status code after performing Action with Rule*/
    flt_rule frule;   /*!< FRule */
    uint32_t sid;     /*!< SID of FRule */
} net_rule_t;

void parse_netrule2netbuf(net_rule_t * netrule, char * netbuf);
void parse_netbuf2netrule(char * netbuf, net_rule_t * netrule);
void receive_ccci_header(int fd, ccci_header_t * ccci_header);
void send_ccci_header(int servicefd, ccci_header_t * ccci_header);


/**
* Read one filter rule from given text.
* @param str String with rule representation.
* @param res Pointer to parsed rule structure.
* @param empty Pointer to empty flags of parsed rule values.
* @return Returns 1 on success, or 0 on parsing error.
*/
int flt_get_rule(char *str, flt_rule *res, unsigned *empty);
#define EMPTY_SRCIP     FLT_RULE_TYPE_FLAGS_SRCIP
#define EMPTY_DSTIP     FLT_RULE_TYPE_FLAGS_DSTIP
#define EMPTY_SRCPORT   FLT_RULE_TYPE_FLAGS_SRCPORT
#define EMPTY_DSTPORT   FLT_RULE_TYPE_FLAGS_DSTPORT
#define EMPTY_PROTO     FLT_RULE_TYPE_FLAGS_PROTO
#define EMPTY_DATA      0x8000
#define EMPTY_IPV4_ONLY FLT_RULE_TYPE_FLAGS_IPV4_ONLY

/**
* Read filter init vector from given text.
* @param str String with rule representation.
* @param res Pointer to parsed result.
* @param empty Pointer to empty flags of parsed values.
* @return Returns number of parsed words on success, or 0 on parsing error.
*/
int flt_get_initv(char *str, uint32_t *res, unsigned *empty);
/**
* Correctly converts IP address from binary to string
* @param ip IPv4 or IPv6 address in binary.
* @param str String with result, must be allocated.
*/
void ip2string(unsigned char *ip, char *str);
#define IPSTR_LEN 46

#endif // FILTERCTL_INPUT_H_INCLUDED
