/*!
 * \file afilterctl_cfg.c
 * \brief Tool for controlling advanced filter - configuration of HW filter
 * \author Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
 * \date 2013
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: afilterctl_cfg.c 3122 2013-02-06 01:41:51Z xkekel00 $
 *
 */

#include "afilterctl_cfg.h"

/// ///////////////////////////////////////////////////////////////////////////
/// CONFIGURABLE PART 1/1
//     if FILTER_STATIC_CONFIG is defined, static afilter configuration specified in this file is used
//     else afilter configuration is red from HW on its first usage
#define FILTER_STATIC_CONFIG

// static configuration of filter
//    ignored when FILTER_STATIC_CONFIG is not defined
//    must be consistent with filter_core_config.vhd when FILTER_STATIC_CONFIG is defined
static flt_config fc = {
        // rule_type - number of different supported rule types
        //    values: 1-30
        4,
        // iv_number - equal or higher than maximum number of tables used in @cfg
        3,
        // cfg - @rule_type records of format:
        //     items - number of items inside one hash table
        //     tables - number of parallel hash tables
        //     flags - which fields from 5-tuple will be used as rule key
        {
            {   512,    3, FLT_RULE_TYPE_FLAGS_SRCIP|FLT_RULE_TYPE_FLAGS_IPV4_ONLY},
            {   512,    3, FLT_RULE_TYPE_FLAGS_SRCIP},
            {   512,    3, FLT_RULE_TYPE_FLAGS_SRCIP},
            {   512,    3, FLT_RULE_TYPE_FLAGS_SRCIP},

        }
    };
/// ///////////////////////////////////////////////////////////////////////////

#define CFG_MASK_TABLES(a)  ((a) & 0x000000FF)
#define CFG_MASK_ITEMS(a)   (((a) & 0x03FFFF00) >> 8)
#define CFG_MASK_FLAGS(a)   (((a) & 0xFC000000) >> 26)

int my_log2(uint32_t x) {
    uint32_t check = 1;
    for(int i = 0; i < 32; i++, check <<= 1)
        if(check >= x)
            return i;
    return 32;
}

flt_config *flt_get_config(struct hwio_comp *comp) {
    static int have_config = 0;
    if(!have_config) {
#ifndef FILTER_STATIC_CONFIG
        uint32_t cfg[31];
        fc.rule_type = flt_read_config(comp, cfg, &(fc.iv_number));
        if(!fc.rule_type)
            return NULL;
        for(unsigned i = 0; i < fc.rule_types; i++) {
            fc.cfg[i].tables = CFG_MASK_TABLES(cfg[i]);
            fc.cfg[i].items  = CFG_MASK_ITEMS(cfg[i]);
            fc.cfg[i].flags  = CFG_MASK_FLAGS(cfg[i]);
        }
#endif
        fc.rule_types_width=my_log2(fc.rule_types);
        for(int i = 0; i < fc.rule_types; i++)
            fc.cfg[i].items_width=my_log2(fc.cfg[i].items);
        have_config = 1;
    }
    return &fc;
}
