/*!
 * \file afilterctl.c
 * \brief Tool for controlling advanced filter
 * \author Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: afilterctl.c 3122 2013-02-06 01:41:51Z xkekel00 $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include "hwio.h"
#include "hwio_tool.h"
#include "afilterctl_input.h"

#define INTERFACES    2    // number of interfaces

#define TOOL_OPTS "CD:GRST:VW:r"

struct hwio_tool_custom {
    int reset;
};

struct hwio_comp_spec tool_compat[]={
    {"ug4-150", "axi_advanced_filter", HWIO_VER3(1, 0, HWIO_VERSION_NA)},
    {"combov2", "Advanced_filter",     HWIO_VER2(1, 0)},
    HWIO_COMPAT_END
};

/// print help message
void hwio_tool_help(const char *toolname, const struct hwio_tool_params *params) {
    printf("Usage: %s [options | commands]\n", toolname);
    printf("Options:\n");
    printf("  -h       print this help message\n");
    printf("  -v       verbosity level\n");
    printf("  -q       no verbose (negative verbosity level)\n");
    printf("  -r       reset mode of commands (work with 'C' and 'R')\n");
    printf("  -x pth   path to design.xml file\n");
    printf("  -d pth   path to device file\n");
    printf("  -A       perform action on all advanced filters in design\n");
    printf("Commands:\n");
    printf("  -I       informations about all advanced filters in design\n");
    printf("  -L       list all advanced filters in design\n");
    printf("  -C       show packet counters\n");
    printf("  -D t;k   delete items with specified type and key\n");
    printf("  -G       show configuration of rule types inside filter\n");
    printf("  -R       show all rules inside filter\n");
    printf("  -S       show status of tables with rules.\n");
    printf("  -T t;k;d try to add specified item\n");
    printf("  -V       show currently used hash initialization vector\n");
    printf("  -W v     set initialization vector to given value (also clear rules)\n");
    printf("Where:\n");
    printf("  t = rule type, number in <0, RULE_TYPES)\n");
    printf("  k = rule key, 5-tuple \"srcip;dstip;srcport;dstport;proto\"\n");
    printf("      some parts of 5-tuple are ignored depending on rule type\n");
    printf("      ignored parts of 5-tuple can be blank\n");
    printf("  d = rule data (id), 32-bit unsined integer\n");
    printf("  v = init vectors, sequence of 32-bit unsigned integers\n");
}

/// read HW filter counters
int filter_counters(struct hwio_comp *c, hwio_tool_comp_err_t err, const struct hwio_tool_params *params) {
    flt_cnt_status s[INTERFACES];
    if(flt_read_counters(c,params->custom->reset,INTERFACES,s)) {
        fprintf(stderr,"Error reading data from HW!\n");
        return EXIT_FAILURE;
    }
    printf(" *** PACKET COUNTERS *** \n");
    for(unsigned int i=0; i<INTERFACES; i++) {
        printf("Interface %d:\n",i);
        printf("    Denied : %10" PRIu64 "\n",s[i].deny);
        printf("    Allowed: %10" PRIu64 "\n",s[i].allow);
        printf("    TOTAL  : %10" PRIu64 "\n",s[i].allow+s[i].deny);
    }
    printf("\n");
    return EXIT_SUCCESS;
}

/// read HW filter static configuration
int filter_config(struct hwio_comp *c, hwio_tool_comp_err_t err, const struct hwio_tool_params *params) {
    static const char *srcip   = " SrcIP";
    static const char *dstip   = " DstIP";
    static const char *srcport = " SrcPort";
    static const char *dstport = " DstPort";
    static const char *proto   = " Proto";
    static const char *empty   = "";
    static const char *v4      = "v4";
    flt_config *fc = flt_get_config(c);
    if(fc == NULL) {
        fprintf(stderr,"Error reading data from HW!\n");
        return EXIT_FAILURE;
    }
    printf(" *** FILTER STATIC CONFIGURATION *** \n");
    printf("Supported rule types: %10d\n", fc->rule_types);
    printf("Words in init vector: %10d\n", fc->iv_number);
    for(unsigned int i = 0; i < fc->rule_types; i++) {
        printf("Rule type %d:\n",i);
        printf("    Tables: %10" PRIu8  "\n", fc->cfg[i].tables);
        printf("    Items : %10" PRIu32 "\n", fc->cfg[i].items);
        printf("    Key   :%s%s%s%s%s%s%s\n",
               (fc->cfg[i].flags & FLT_RULE_TYPE_FLAGS_SRCIP)       ? srcip : empty,
               ((fc->cfg[i].flags & FLT_RULE_TYPE_FLAGS_SRCIP) &&
                (fc->cfg[i].flags & FLT_RULE_TYPE_FLAGS_IPV4_ONLY)) ? v4 : empty,
               (fc->cfg[i].flags & FLT_RULE_TYPE_FLAGS_DSTIP)       ? dstip : empty,
               ((fc->cfg[i].flags & FLT_RULE_TYPE_FLAGS_DSTIP) &&
                (fc->cfg[i].flags & FLT_RULE_TYPE_FLAGS_IPV4_ONLY)) ? v4 : empty,
               (fc->cfg[i].flags & FLT_RULE_TYPE_FLAGS_SRCPORT)     ? srcport : empty,
               (fc->cfg[i].flags & FLT_RULE_TYPE_FLAGS_DSTPORT)     ? dstport : empty,
               (fc->cfg[i].flags & FLT_RULE_TYPE_FLAGS_PROTO)       ? proto : empty
               );
    }
    printf("\n");
    return EXIT_SUCCESS;
}

/// add rule to HW filter
int filter_add(struct hwio_comp *c, hwio_tool_comp_err_t err, const struct hwio_tool_params *params) {
    const char *str=params->cmd.arg;
    int size=strlen(str)+1,x;
    char sstr[size];
    flt_rule r;
    memcpy(sstr,str,size);
    if((x=flt_parse_rule(c,sstr,&r))!=0) {
        if(x==PARSE_RULE_ERROR)
            fprintf(stderr,"Wrong rule format, syntax error!\n");
        if(x==PARSE_RULE_EMPTY)
            fprintf(stderr,"Wrong rule format, required value left empty!\n");
        if(x==PARSE_RULE_HWERROR)
            fprintf(stderr,"Error in HW filter, unable to read filter config!\n");
        return EXIT_FAILURE;
    }
    if((x=flt_add_rule(c,&r))!=0) {
        if(x==ADD_RULE_BUSY)
            fprintf(stderr,"Error in HW filter, possible infinite busy state!\n");
        if(x==ADD_RULE_FULL)
            fprintf(stderr,"Table with desired rule type already full!\n");
        return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}


/// delete rule from HW filter
int filter_del(struct hwio_comp *c, hwio_tool_comp_err_t err, const struct hwio_tool_params *params) {
    const char *str=params->cmd.arg;
    int size=strlen(str)+1,x;
    char sstr[size+2];
    flt_rule r;
    memcpy(sstr,str,size);
    sstr[size-1]=';';
    sstr[size]='0';
    sstr[size+1]='\0';
    if((x=flt_parse_rule(c,sstr,&r))!=0) {
        if(x==PARSE_RULE_ERROR)
            fprintf(stderr,"Wrong rule format, syntax error!\n");
        if(x==PARSE_RULE_EMPTY)
            fprintf(stderr,"Wrong rule format, required value left empty!\n");
        if(x==PARSE_RULE_HWERROR)
            fprintf(stderr,"Error in HW filter, unable to read filter config!\n");
        return EXIT_FAILURE;
    }
    if((x=flt_del_rule(c,&r))!=0) {
        if(x==DEL_RULE_BUSY)
            fprintf(stderr,"Error in HW filter, possible infinite busy state!\n");
        return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}


/// read HW filter status
int filter_status(struct hwio_comp *c, hwio_tool_comp_err_t err, const struct hwio_tool_params *params) {
    static const char *busy="BUSY";
    static const char *ready="READY";
    static const char *full="FULL";
    static const char *ignore=" (IGNORED)";
    static const char *notsup="NOT SUPPORTED";
    static const char *empty="";
    flt_config *fc = flt_get_config(c);
    if(fc == NULL) {
        fprintf(stderr,"Error reading data from HW!\n");
        return EXIT_FAILURE;
    }
    flt_status_flags s;
    flt_read_status(c,&s);
    printf(" *** FILTER STATUS *** \n");
    printf("Status flag vectors:\n");
    printf("    Disable: %08x\n",s.disable);
    printf("    Ignore : %08x\n",s.ignore);
    printf("    Full   : %08x\n",s.full);
    printf("    Busy   : %08x\n",s.busy);
    printf("Rule types:\n");
    for(unsigned int i=0, mask=1; i<fc->rule_types; i++,mask<<=1) {
        printf("    Type %d: %s%s\n", i,
               (s.disable&mask)?notsup:(s.busy&mask)?busy:(s.full&mask)?full:ready,
               ((s.ignore&mask)&&(!(s.disable&mask)))?ignore:empty);
    }
    printf("\n");
    return EXIT_SUCCESS;
}

/// read/reset HW filter rules
int filter_rules(struct hwio_comp *comp, int read, int reset) {
    if(!read&&!reset)
        return EXIT_SUCCESS;
    flt_config *fc = flt_get_config(comp);
    if(fc == NULL) {
        fprintf(stderr,"Error reading data from HW!\n");
        return EXIT_FAILURE;
    }
    if(!flt_wait_ready(comp,0xFFFFFFFF)) {  // wait for filter ready
        fprintf(stderr,"Error in HW filter, possible infinite busy state!\n");
        return EXIT_FAILURE;
    }
    flt_stop_cmd(comp,0xFFFFFFFF); // stop filter reconfiguration operations
    if(reset)                // and possibly soft clear tables (ignore results)
        flt_ignore_cmd(comp,0xFFFFFFFF);
    if(!flt_wait_ready(comp,0xFFFFFFFF)) { // wait until stop and ignore commands are finished
        fprintf(stderr,"Error in HW filter, possible infinite busy state!\n");
        return EXIT_FAILURE;
    }
    flt_dir_iterator it;
    flt_rule r;
    char str[IPSTR_LEN];
    if(read)
        printf(" *** RULES INSIDE FILTER *** \n");
    flt_status_flags s;
    flt_read_status(comp,&s);
    for(unsigned type=0; type<fc->rule_types; type++) { // each rule type
        if(s.disable&MASK_RULE_TYPE(type)) // skip disabled rule types
            continue;
        flt_dir_iterator_init(type,&it);
        if(!read)
            flt_dir_iterator_clearall(comp,&it); // fast clean when no read
        else
            while(flt_dir_iterator_nextrule(comp,&it,&r)) {
                if(reset)
                    flt_dir_iterator_clear(comp,&it); // clear rule
                printf("%d;",type);                                      // [type]; ...
                if(fc->cfg[type].flags&FLT_RULE_TYPE_FLAGS_SRCIP) {      // ... [srcip]; ...
                    ip2string((unsigned char *)r.srcip,str);
                    printf("%s",str);
                }
                putchar(';');
                if(fc->cfg[type].flags&FLT_RULE_TYPE_FLAGS_DSTIP) {      // ... [dstip]; ...
                    ip2string((unsigned char *)r.dstip,str);
                    printf("%s",str);
                }
                putchar(';');
                if(fc->cfg[type].flags&FLT_RULE_TYPE_FLAGS_SRCPORT)      // ... [srcport]; ...
                    printf("%u",r.srcport);
                putchar(';');
                if(fc->cfg[type].flags&FLT_RULE_TYPE_FLAGS_DSTPORT)      // ... [dstport]; ...
                    printf("%u",r.dstport);
                putchar(';');
                if(fc->cfg[type].flags&FLT_RULE_TYPE_FLAGS_PROTO)        // ... [proto]; ...
                    printf("%u",r.proto);
                putchar(';');
                printf("%u\n",r.id);                                     // [id]\n
                flt_dir_iterator_next(comp,&it);
            }
    }
    if(read)
        printf("\n");
    flt_stop_cmd(comp,0); // enable all
    flt_ignore_cmd(comp,0);
    return EXIT_SUCCESS;
}
int filter_rules_call(struct hwio_comp *c, hwio_tool_comp_err_t err, const struct hwio_tool_params *params) {
    return filter_rules(c,1,params->custom->reset);
}

/// read HW filter init vectors
int filter_read_iv(struct hwio_comp *c, hwio_tool_comp_err_t err, const struct hwio_tool_params *params) {
    flt_config *fc = flt_get_config(c);
    if(fc == NULL) {
        fprintf(stderr,"Error reading data from HW!\n");
        return EXIT_FAILURE;
    }
    uint32_t iv[fc->iv_number];
    flt_read_initv(c,iv);
    printf(" *** INITIALIZATION VECTOR *** \n");
    for(int i=0; i<fc->iv_number; i++)
        printf("Seed %d: 0x%08x\n",i,iv[i]);
    printf("\n");
    return EXIT_SUCCESS;
}


/// set HW filter init vectors
int filter_write_iv(struct hwio_comp *c, hwio_tool_comp_err_t err, const struct hwio_tool_params *params) {
    flt_config *fc = flt_get_config(c);
    if(fc == NULL) {
        fprintf(stderr,"Error reading data from HW!\n");
        return EXIT_FAILURE;
    }
    uint32_t iv[fc->iv_number];
    const char *str=params->cmd.arg;
    int size=strlen(str)+1,x;
    char sstr[size];
    memcpy(sstr,str,size);
    if((x=flt_parse_initv(c,sstr,iv))!=0) {
        if(x==PARSE_INITV_ERROR)
            fprintf(stderr,"Wrong init vector format, expected format '%%u;%%u;...'!\n");
        if(x==PARSE_INITV_NOEFF)
            fprintf(stderr,"Wrong init vector format, no effect!\n");
        return EXIT_FAILURE;
    }
    if(filter_rules(c,0,1)!=EXIT_SUCCESS) // silently reset rules
        return EXIT_FAILURE;
    flt_write_initv(c,iv);
    return EXIT_SUCCESS;
}


int tool_custom_parse(int opt, const char *optarg, struct hwio_tool_params *params) {
    switch(opt) {
        case 'C': case 'G': case 'R': case 'S': case 'V':
            if(hwio_tool_params_has_cmd(params))
                return hwio_tool_error_another_cmd(opt,params);
            params->cmd.code = opt;
            params->cmd.arg = NULL;
            return HWIO_TOOL_CUSTOM_OK;
        case 'D': case 'T': case 'W':
            if(hwio_tool_params_has_cmd(params))
                return hwio_tool_error_another_cmd(opt,params);
            params->cmd.code = opt;
            params->cmd.arg = optarg;
            return HWIO_TOOL_CUSTOM_OK;
        case 'r':
            params->custom->reset=1;
            return HWIO_TOOL_CUSTOM_OK;
        default:
            return HWIO_TOOL_CUSTOM_UNKNOWN;
    }
}


/// MAIN //////////////////////////////////////////////////////
int main(int argc, char *argv[]) {
    struct hwio_tool_custom tool_params={0};
    struct hwio_tool_params params = hwio_tool_params_new(&tool_params,argc,argv);
    switch(hwio_tool_params_parse(&params,&tool_custom_parse,TOOL_OPTS)) {
        case HWIO_TOOL_PARAMS_FAILED:
            return HWIO_TOOL_EARGS;
        case HWIO_TOOL_PARAMS_EXIT:
            return 0;
        default:
            break;
    }
    hwio_tool_verbose(&params, 1, "Arguments parsed successfully");
    hwio_tool_verbose(&params, 1, "Command: -%c", params.cmd.code);
    hwio_tool_verbose(&params, 2, "Argument: %s", params.cmd.arg);
    switch(params.cmd.code) {
        case 'I':
            return hwio_tool_exec_generic(&params,tool_compat,&hwio_tool_comp_info,&hwio_tool_error_handler);
        case 'L':
            return hwio_tool_exec_generic(&params,tool_compat,&hwio_tool_comp_name,&hwio_tool_error_handler);
        case 'C':
            return hwio_tool_exec_generic(&params,tool_compat,&filter_counters,&hwio_tool_error_handler);
        case 'D':
            return hwio_tool_exec_generic(&params,tool_compat,&filter_del,&hwio_tool_error_handler);
        case 'G':
            return hwio_tool_exec_generic(&params,tool_compat,&filter_config,&hwio_tool_error_handler);
        case 'R':
            return hwio_tool_exec_generic(&params,tool_compat,&filter_rules_call,&hwio_tool_error_handler);
        case 'S':
            return hwio_tool_exec_generic(&params,tool_compat,&filter_status,&hwio_tool_error_handler);
        case 'T':
            return hwio_tool_exec_generic(&params,tool_compat,&filter_add,&hwio_tool_error_handler);
        case 'V':
            return hwio_tool_exec_generic(&params,tool_compat,&filter_read_iv,&hwio_tool_error_handler);
        case 'W':
            return hwio_tool_exec_generic(&params,tool_compat,&filter_write_iv,&hwio_tool_error_handler);
        case '\0':
            return hwio_tool_error(HWIO_TOOL_EARGS,"No command was specified");
        default:
            return hwio_tool_error(HWIO_TOOL_EARGS,"Command -%c is not implemented",params.cmd.code);
    }
}
