/*
 * \file util.c
 * \brief Miscellaneous utility routines.
 * \author Jachym Holecek <freza@liberouter.org>
 * \author Jaroslav Kysela <perex@perex.cz>
 * \author Radek Krejci <rkrejci@cesnet.cz>
 * \date 2003-2012
 *
 * Copyright (C) 2003-2012 CESNET
 */
/* LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/shm.h>
#include <fcntl.h>
#include <semaphore.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <commlbr.h>
#include <unistd.h>
#include <errno.h>

#include "combosix.h"

__RCSID("$Id$");

static int
decode_byte(FILE *file, u_int8_t *dst)
{
	int c;

	if ((c = fgetc(file)) == EOF)
		return -EINVAL;
	if (c >= '0' && c <= '9')
		*dst = (c - '0') << 4;
	else if (c >= 'a' && c <= 'f')
		*dst = ((c - 'a') + 10) << 4;
	else if (c >= 'A' && c <= 'F')
		*dst = ((c - 'A') + 10) << 4;
	else
		return -EINVAL;
	if ((c = fgetc(file)) == EOF)
		return -EINVAL;
	if (c >= '0' && c <= '9')
		*dst |= (c - '0');
	else if (c >= 'a' && c <= 'f')
		*dst |= ((c - 'a') + 10);
	else if (c >= 'A' && c <= 'F')
		*dst |= ((c - 'A') + 10);
	else
		return -EINVAL;
	return 0;
}

#define decode_uint8(dst) \
	do { \
		if (decode_byte(file, &(dst)) < 0) \
			goto __error; \
	} while (0)

#define decode_uint16(dst) \
	do { \
		if (decode_byte(file, &x) < 0) \
			goto __error; \
		(dst) = x << 8; \
		if (decode_byte(file, &x) < 0) \
			goto __error; \
		(dst) |= x; \
	} while (0)

/**
 * \brief Decode MCS. We ignore features that are not essential for us.
 * \param file File with MCS contents
 * \param data decoded data (mallocated)
 * \param num data length
 * \return zero on success otherwise a negative error code
 *
 * Note: \e file is not closed.
 */
int
cs_mcs_decode(FILE *file, u_int8_t **data, u_int32_t *num)
{
	u_int16_t lsb, msb = 0;
	u_int8_t bytes, type, x, val;
	int xerr = -EINVAL;

	*data 	= NULL;
	*num 	= 0;

	/*
	 * Every line begins with ':'.
	 */
	if (fgetc(file) != ':')
		return -EINVAL;

	for(;;) {
		decode_uint8(bytes);	/* Line length */
		decode_uint16(lsb);	/* Address LSB */
		decode_uint8(type);	/* Record type */

		switch (type) {
		case 0x00:
			/*
			 * Data item.
			 */
			break;
		case 0x01:
			/*
			 * Last record.
			 */
			return 0;
		case 0x04:
			/*
			 * Address MSB.
			 */
			decode_uint16(msb);
			bytes -= 2;
			break;
		}

		*data = (u_int8_t *) realloc(*data, *num + bytes);
		if (*data == NULL) {
			xerr = -ENOMEM;
			goto __error;
		}

		while (bytes-- > 0) {
			/*
			 * Read a byte of data.
			 */
			decode_uint8(val);
			(*data)[*num] = val;
			(*num)++;
		}

#if 0
		/*
		 * Each line has a CRC.
		 */
		decode_uint8(crc);
#endif

		/*
		 * Skip newline (may be DOS-ish).
		 */
		do {
			x = fgetc(file);
		} while (x != ':');
	}

      __error:
      	if (*data) {
      		free(*data);
	      	*data = NULL;
	}
	*num = 0;
      	return xerr;
}

/* structure describing particular libcombo's lock types */
static struct lock_type {
	char *desc;
	sem_t *locker;
	int level;
	int mask;
} cs_locks[] = {
	{"/COMBO-I2C", NULL, 0, CS_LOCK_I2C}, /* i2c - for CS_LOCK_I2C cs_lock_type */
	{"/COMBO-MDIO", NULL, 0, CS_LOCK_MDIO}, /* mdio - for CS_LOCK_MDIO cs_lock_type */
	{"/COMBO-IBUF", NULL, 0, CS_LOCK_IBUF}, /* ibuf - for CS_LOCK_IBUF cs_lock_type */
	{"/COMBO-OBUF", NULL, 0, CS_LOCK_OBUF} /* obuf - for CS_LOCK_OBUF cs_lock_type */
};
/* max index in cs_locks[] array - KEEP IT CONSISTENT!!! */
#define CS_LOCKS_MAX_INDEX 3

/* number of currently held libcombo's lock types (I2C, MDIO, ...) */
static int cs_locks_count = 0;

/* signal sets for blocking signals during holding a lock */
static sigset_t sigset, oldsigset;

/*
 * Internal function to lock single type of libcombo's locker, public API
 * function to this is cs_lock() which locks a set of lockers (set using OR).
 * Note that lock_type here is different to lock_type parameter in cs_lock() -
 * cs_lock()'s one is a mask, while cs_single_lock's is index to cs_locks[]
 * array.
 *
 * \return 0 on success, -1 on error, EAGAIN if the semaphore is blocked
 */
int cs_single_lock(int lock_type)
{
	int ret;
	int oldmask;

	if (lock_type > CS_LOCKS_MAX_INDEX) {
		return EXIT_FAILURE;
	}

	/*
	 * Case of nested locking - the process tries to lock already locked locker
	 */
	if (cs_locks[lock_type].level != 0) {
		cs_locks[lock_type].level++;
		return EXIT_SUCCESS;
	}

	/*
	 * Create or open named semaphore
	 */
	if (cs_locks[lock_type].locker == NULL) {
		oldmask = umask(0); /* a workaround, which solves wrong permission of newly created semaphore */
		cs_locks[lock_type].locker = sem_open(cs_locks[lock_type].desc, O_CREAT, S_IRWXU|S_IRWXG|S_IRWXO, 1);
		umask(oldmask);

		if (cs_locks[lock_type].locker == NULL) {
			VERBOSE(CL_VERBOSE_OFF, "Opening semaphore with %s locker failed: %s", cs_locks[lock_type].desc, strerror(errno));
			return EXIT_FAILURE;
		}

	}

	/*
	 * Finally, do the job.
	 */
	ret = sem_trywait(cs_locks[lock_type].locker);
	if (ret != -1) {
		cs_locks[lock_type].level++; /* note that I'm the owner (for nested locking) */
		cs_locks_count++;
	} else {
		if (errno == EAGAIN) {
			ret = EAGAIN;
		}
	}
	return ret;
}

/*
 * Internal function to unlock single type of libcombo's locker, public API
 * function to this is cs_unlock() which unlocks a set of lockers (set using OR).
 * Note that lock_type here is different to lock_type parameter in cs_unlock() -
 * cs_unlock()'s one is a mask, while cs_single_unlock's is index to cs_locks[]
 * array.
 */
int cs_single_unlock(int lock_type)
{
	int ret;

	if (lock_type > CS_LOCKS_MAX_INDEX) {
		return EXIT_FAILURE;
	}

	if (cs_locks[lock_type].locker == NULL || cs_locks[lock_type].level == 0) {
		return EXIT_FAILURE;
	}

	cs_locks[lock_type].level--;
	/* do unlock only if we are on the base level (all nested lockers were unlocked) */
	if (cs_locks[lock_type].level == 0) {
		/* release the lock */
		ret = sem_post(cs_locks[lock_type].locker);
		sem_close(cs_locks[lock_type].locker);
		cs_locks[lock_type].locker = NULL;
		cs_locks_count--;

		return ret;
	} else {
		return EXIT_SUCCESS;
	}
}

/**
 * \brief Lock serves to bound a set of functions accessing the card that could not be interrupted.
 *
 * \param[in] lock_type Bitmap of lockers to lock. Value is OR of CS_LOCK_* values.
 *
 * \return 0 on success, -1 on error
 */
int cs_lock(int lock_type)
{
#define LOCK_USLEEP 50

	int lock_list = 0;
	int ret, i;
	static void (*exit_clean)(void) = NULL;

	/* prepare full sigset for blocking all signals */
	if (sigfillset(&sigset) != 0) {
		VERBOSE(CL_VERBOSE_OFF, "Preparing signal set failed.");
		return EXIT_FAILURE;
	}

again:
	lock_list = 0;

	/*
	 * block all signals
	 */
	if (cs_locks_count == 0) {
		if (sigprocmask(SIG_SETMASK, &sigset, &oldsigset) != 0) {
			VERBOSE(CL_VERBOSE_OFF, "Blocking signals inside the libcombo locker failed.");
			return EXIT_FAILURE;
		}
	}

	for (i = 0; i <= CS_LOCKS_MAX_INDEX; i++) {
		/* note that we check for CS_LOCK_ macros but cs_single_lock must be called with index in cs_locks[] array */
		if (lock_type & cs_locks[i].mask) {
			ret = cs_single_lock(i);
			if (ret == 0) {
				lock_list |= cs_locks[i].mask;
			} else {
				cs_unlock(lock_list);
				/* lock was not received due to blocking - try it to lock again later */
				if (ret == EAGAIN) {
					usleep(LOCK_USLEEP);
					goto again;
				}
				return EXIT_FAILURE;
			}
		}
	}

        /*
         * Register cs_unlock_all() function to be executed at program
         * termination. This ensures that lockers will be released at program
         * (normal) exit. The function is registered only once (when cs_lock is
         * called for the first time).
         */
	if (exit_clean == NULL) {
		exit_clean = cs_unlock_all;
		atexit (exit_clean);
	}

	return EXIT_SUCCESS;
}

/**
 * \brief Unlock ends the area of a set of functions accessing the card that could not be interrupted.
 *
 * \param[in] lock_type Bitmap of lockers to unlock. Value is OR of CS_LOCK_* values.
 *
 * \return 0 on success, -1 on error
 */
int cs_unlock(int lock_type)
{
	int i;

	/*
	 * note that CS_LOCK_ macros must be converted to the index of
	 * cs_locks[] array. Also remember that order of unlocking must be
	 * reverse in comparison to locking procedure
	 */
	for (i = CS_LOCKS_MAX_INDEX; i >= 0; i--) {
		if (lock_type & cs_locks[i].mask) {
			cs_single_unlock(i);
		}
	}

	/* unblock signals */
	if (cs_locks_count == 0) {
		sigprocmask(SIG_SETMASK, &oldsigset, NULL);
	}

	return EXIT_SUCCESS;
}

/**
 * \brief Unlock all currently held lockers.
 *
 * \return 0 on success
 */
void cs_unlock_all()
{
	int i;

	/* start with the highest index to follow order from cs_unlock */
	for (i = CS_LOCKS_MAX_INDEX; i >= 0; i--) {
		while (cs_locks[i].level != 0) {
			cs_single_unlock(i);
		}
	}
}
