/*!
 * \file pcippc.c
 * \brief Combo6 PCR PPC booting
 * \author Jaroslav Kysela <perex@perex.cz>
 * \author Jachym Holecek <freza@liberouter.org>
 * \date 2006
 *
 * Copyright (C) 2006  CESNET
 *
 * The design and component functions work with the FPGA designs.
 * For more details, please, look to \ref design page.
 */
/*
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <sys/mman.h>
#include <sys/stat.h>

#ifdef __linux__
#include <endian.h>
#else
#include <sys/endian.h>
#endif

#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <ctype.h>
#include <fcntl.h>
#include <string.h>
#include <elf.h>

#include <commlbr.h>
#include "combosix.h"
#include "cs_local.h"

__RCSID("$Id$");

/* PowerPC's "internal" memory. */
#define PCR_PPC_IBASE		0x020000	/* Instruction memory */
#define PCR_PPC_DBASE		0x040000	/* Data memory */
#define PCR_PPC_ISIZE		(16 * 1024)
#define PCR_PPC_DSIZE		(8 * 1024)
#define PCR_PPC_HSIZE		256		/* Header size */

#define PCR_ENTRY		0x023ffc	/* First insn fetch addr. */
#define PCR_IMEM_LOCALLY	0xff000000	/* PPC sees IMEM here. */
#define PCR_DMEM_LOCALLY	0x00040000	/* PPC sees DMEM here. */

struct eppc_id {
	u_int8_t		h_magic[8];	/* PPCESNET */
	u_int32_t		h_format;	/* 32-bit format ID */
	u_int8_t		h_text[116];	/* Cesnet PCI Controller PowerPC Code */
	u_int32_t		i_version;	/* code version */
	u_int8_t		i_text[124];	/* code text id */
} __attribute__((packed));

#define CONV2(val)		(little_endian ? le16toh(val) : be16toh(val))
#define CONV4(val)		(little_endian ? le32toh(val) : be32toh(val))
#define VNOC4(val)		(little_endian ? htole32(val) : htobe32(val))
#define VNOC4N(val)		(!little_endian ? htole32(val) : htobe32(val))

/* This is already big endian. */
#define PPC_BRANCH_OPC		0x48000000	/* Branch insn opcode. */
#define PPC_BRANCH_AA		0x00000002	/* Absolute address jump. */
#define PPC_BRANCH_LK		0x00000001	/* Load PC to LR before jump. */
#define PPC_BRANCH_LI		0x03fffffc	/* Immediate value mask. */
#define PPC_BRANCH(addr)	(((addr) & PPC_BRANCH_LI) | \
					PPC_BRANCH_AA | PPC_BRANCH_OPC)
/**
 * \brief Get verbosity level from the environment
 * \return Verbosity level
 * \retval 0 No verbosity
 */
static int
elf_verbose(void)
{
	const char *s = getenv("LIBCOMBO_ELF_VERBOSE");

	if (s == NULL)
		return 0;
	if (isdigit((int)*s))
		return atol(s);
	return 1;
}

/**
 * \brief Load ELF segment to memory
 * \param ppc_imem Instruction memory
 * \param ppc_dmem Data memory
 * \param data Segment's data
 * \param padr Segment's physical address
 * \param msiz Segment's memory size
 * \param fsiz Segment's file size
 * \param little_endian Segment's endianity
 * \return zero on success otherwise a negative error code
 */
static int
load_elf_segment(u_int32_t *ppc_imem, u_int32_t *ppc_dmem, u_char *data,
		 u_long padr, u_long msiz, u_long fsiz, int little_endian)
{
	int verbose = elf_verbose();
	u_int32_t *dest;
	u_long i;

	if (padr >= PCR_IMEM_LOCALLY - PCR_PPC_HSIZE) {
		if (padr + fsiz > (PCR_IMEM_LOCALLY + PCR_PPC_ISIZE + PCR_PPC_HSIZE)) {
			if (verbose)
				fprintf(stderr, "address %08lx out of bounds\n", padr);
			return -EINVAL;
		}

		padr -= PCR_IMEM_LOCALLY - PCR_PPC_HSIZE;
		dest = ppc_imem;
	} else {
		if (padr < PCR_PPC_DBASE || padr + fsiz > (PCR_PPC_DBASE + PCR_PPC_DSIZE)) {
			if (verbose)
				fprintf(stderr, "address %08lx out of bounds", padr);
			return -EINVAL;
		}

		padr -= PCR_PPC_DBASE;
		dest = ppc_dmem;
	}

	if (msiz > fsiz)
		memset(dest + padr, 0, msiz - fsiz);

	if (fsiz > 0)
		for (i = 0; i < fsiz; i += 4, padr += 4) {
			if (padr < PCR_PPC_HSIZE) {
				dest[i / 4] = VNOC4N(*((u_int32_t *)(data + i)));
			} else {
				dest[i / 4] = CONV4(*((u_int32_t *)(data + i)));
			}
		}

	return 0;
}

/**
 * \brief Load PPC program for Cesnet PCI bridge
 * \param elf_file Statically linked PPC program in ELF format
 * \param boot Returned boot descriptor
 * \return zero on success otherwise a negative error code
 */
int
cs_load_pcippc(const char *elf_file, cs_boot_v1_t **boot)
{
	int verbose = elf_verbose();
	struct stat st;
	Elf32_Ehdr *elf;
	Elf32_Phdr *phd;
	u_char *mm = (u_char*)MAP_FAILED;
	u_int32_t val4;
	u_int16_t val2;
	u_int32_t *imem, *dmem;
	u_int phcnt, phsiz;
	int fd, err, little_endian;
	struct eppc_id *eppc_id;

	*boot = (cs_boot_v1_t*)malloc(sizeof(cs_boot_v1_t));
	if (*boot == NULL)
		return -ENOMEM;
	(*boot)->size = PCR_PPC_ISIZE + PCR_PPC_DSIZE + PCR_PPC_HSIZE;
	(*boot)->data = (uint8_t*)malloc((*boot)->size);
	if ((*boot)->data == NULL) {
		free(*boot);
		*boot = NULL;
		return -ENOMEM;
	}
	memset((*boot)->data, 0, (*boot)->size);
	(*boot)->addr = 1024;
	(*boot)->next = NULL;

	imem = (u_int32_t *)((*boot)->data);
	dmem = (u_int32_t *)((*boot)->data + PCR_PPC_ISIZE + PCR_PPC_HSIZE);

	/*
	 * Grok ELF file for loadable sections.
	 */
	fd = open(elf_file, O_RDONLY, 0);
	if (fd == -1) {
		err = -errno;
		if (verbose)
			fprintf(stderr, "could not open pcippc program '%s'\n", elf_file);
		goto __end;
	}

	if (fstat(fd, &st) == -1) {
		err = -errno;
		if (verbose)
			fprintf(stderr, "could not stat program '%s'\n", elf_file);
		goto __end;
	}

	mm = (u_char *)mmap(NULL, st.st_size, PROT_READ, MAP_SHARED, fd, 0);
	if (mm == MAP_FAILED) {
		err = -errno;
		if (verbose)
			fprintf(stderr, "mmap failed\n");
		goto __end;
	}

	elf = (Elf32_Ehdr *) mm;
	if (elf->e_ident[EI_MAG0] != ELFMAG0 ||
	    elf->e_ident[EI_MAG1] != ELFMAG1 ||
	    elf->e_ident[EI_MAG2] != ELFMAG2 ||
	    elf->e_ident[EI_MAG3] != ELFMAG3) {
		if (verbose)
			fprintf(stderr, "invalid magic %02x %02x %02x %02x, not an ELF file\n",
				(int)elf->e_ident[EI_MAG0], (int)elf->e_ident[EI_MAG1],
				(int)elf->e_ident[EI_MAG2], (int)elf->e_ident[EI_MAG3]);
		err = -EINVAL;
		goto __end;
	}

	if (elf->e_ident[EI_CLASS] != ELFCLASS32) {
		if (verbose)
			fprintf(stderr, "cannot handle class %02x\n", (int)elf->e_ident[EI_CLASS]);
		close(fd);
		err = -EINVAL;
		goto __end;
	}

	switch (elf->e_ident[EI_DATA]) {
	case ELFDATA2LSB:
		if (verbose)
			fprintf(stderr, "ENCODING: LE two's complement\n");
		little_endian = 1;
		break;

	case ELFDATA2MSB:
		if (verbose)
			fprintf(stderr, "ENCODING: BE two's complement\n");
		little_endian = 0;
		break;

	default:
		fprintf(stderr, "invalid data encoding %02x\n",
		    (int)elf->e_ident[EI_DATA]);
		err = -EINVAL;
		goto __end;
	}

	if ((val2 = CONV2(elf->e_type)) != ET_EXEC) {
		fprintf(stderr, "cannot handle ELF type %04x\n", val2);
		err = -EINVAL;
		goto __end;
	}

	if ((val2 = CONV2(elf->e_machine)) != EM_PPC) {
		fprintf(stderr, "cannot handle machine type %04x\n", val2);
		err = -EINVAL;
		goto __end;
	}

	phsiz = CONV2(elf->e_phentsize);
	phcnt = CONV2(elf->e_phnum);
	phd = (Elf32_Phdr *) (mm + CONV4(elf->e_phoff));

	for (; phcnt-- > 0; phd += phsiz) {
		u_long offs, padr, vadr, fsiz, msiz, flag;

		val4 = CONV4(phd->p_type);
		if (val4 == PT_LOAD) {
			offs = CONV4(phd->p_offset);
			padr = CONV4(phd->p_paddr);
			vadr = CONV4(phd->p_vaddr);
			fsiz = CONV4(phd->p_filesz);
			msiz = CONV4(phd->p_memsz);
			flag = CONV4(phd->p_flags);

			if (verbose)
				fprintf(stderr,
				"LOAD[%d]: pa 0x%08lx va 0x%08lx mode %c%c%c "
				"sz %luB clr %luB\n",
				phcnt, padr, vadr,
				(flag & PF_R ? 'r' : '-'),
				(flag & PF_W ? 'w' : '-'),
				(flag & PF_X ? 'x' : '-'),
				msiz, (msiz - fsiz));

			if (padr != vadr) {
				if (verbose)
					fprintf(stderr, "^^^ padr != vadr\n");
				continue;
			}

			if (padr != PCR_IMEM_LOCALLY - 0x100 &&
			    padr != PCR_DMEM_LOCALLY) {
				if (verbose)
					fprintf(stderr, "^^^ address 0x%lx not known\n", padr);
				continue;
			}

			if (padr == PCR_IMEM_LOCALLY - PCR_PPC_HSIZE) {
				if (fsiz >= PCR_PPC_ISIZE + PCR_PPC_HSIZE) {
					fprintf(stderr, "instruction segment overflow\n");
					err = -EINVAL;
					goto __end;
				}
				eppc_id = (struct eppc_id *)(mm + offs);
				if (memcmp(eppc_id->h_magic, "PPCESNET", 8)) {
					fprintf(stderr, "PPCESNET magic check failed\n");
					err = -EINVAL;
					goto __end;
				}
				if (eppc_id->h_format != CONV4(0x18ec0001)) {
					fprintf(stderr, "header format check failed");
					err = -EINVAL;
					goto __end;
				}
				if (strcmp((char *)eppc_id->h_text, "Cesnet PCI Controller PowerPC Code")) {
					fprintf(stderr, "header text check failed\n");
					err = -EINVAL;
					goto __end;
				}
				if (verbose) {
					printf("Version: 0x%x\n", CONV4(eppc_id->i_version));
					printf("Text: %s\n", eppc_id->i_text);
				}
			} else if (padr == PCR_DMEM_LOCALLY) {
				if (msiz >= PCR_PPC_DSIZE) {
					fprintf(stderr, "data segment overflow\n");
					err = -EINVAL;
					goto __end;
				}
			} else {
				fprintf(stderr, "unknown segment 0x%08lx\n", padr);
				err = -EINVAL;
				goto __end;
			}

			err = load_elf_segment(imem, dmem, mm + offs, padr,
				msiz, fsiz, little_endian);
			if (err < 0)
				goto __end;
		} else if (val4 == PT_NOTE) {
			offs = CONV4(phd->p_offset);
			padr = CONV4(phd->p_paddr);
			vadr = CONV4(phd->p_vaddr);
			fsiz = CONV4(phd->p_filesz);
			msiz = CONV4(phd->p_memsz);
			flag = CONV4(phd->p_flags);

			if (verbose)
				fprintf(stderr,
				"NOTE[%d]: pa 0x%08lx va 0x%08lx mode %c%c%c "
				"sz %luB clr %luB",
				phcnt, padr, vadr,
				(flag & PF_R ? 'r' : '-'),
				(flag & PF_W ? 'w' : '-'),
				(flag & PF_X ? 'x' : '-'),
				msiz, (msiz - fsiz));
		} else {
			if (verbose)
				fprintf(stderr, "IGNORE[%d]: type %08x\n", phcnt, val4);
		}
	}

	val4 = CONV4(elf->e_entry);
	if (verbose)
		fprintf(stderr, "ENTRY: 0x%08x [target endianess]\n", val4);

	imem[(PCR_ENTRY - PCR_PPC_IBASE + PCR_PPC_HSIZE) / 4] = PPC_BRANCH(val4);

#if 0
	{
	FILE *f;
	f = fopen("pcippc-dump.bin", "w+");
	fwrite((*boot)->data, (*boot)->size, 1, f);
	fclose(f);
	}
#endif

	eppc_id = (struct eppc_id *)imem;
	eppc_id->h_format = CONV4(eppc_id->h_format);
	eppc_id->i_version = CONV4(eppc_id->i_version);

	err = 0;

      __end:
	if (err < 0)
		cs_free_pcippc(boot);
	if (mm != MAP_FAILED)
		munmap(mm, st.st_size);
	if (fd >= 0)
		close(fd);
	return err;
}

/**
 * \brief Free boot descriptor with PPC program for Cesnet PCI bridge
 * \param boot Boot descriptor to free
 */
void
cs_free_pcippc(cs_boot_v1_t **boot)
{
	if (*boot) {
		if ((*boot)->data)
			free((*boot)->data);
		free(*boot);
		*boot = NULL;
	}
}
