/*! 
 * \file tcamctrl.h
 * \brief Function portotypes and important constants for controlling CS_TCAM_CTRL
 * component
 * \author Andrej Hank <xhanka00@liberouter.org>
 * \date 2007-07-12
 */
/*
 * Copyright (C) 2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#ifndef _LIBEROUTER_TCAMCTRL_H_
#define _LIBEROUTER_TCAMCTRL_H_

#include <stdbool.h>

#include "../combosix.h"

/*!
 * \defgroup tcam_ctrl_component TCAM_CTRL component
 * \ingroup components
 * 
 * \brief Module intended to control TCAM_CTRL component.   
 * \{*/

/* registers */
#define CS_TCAM_CTRL_WIDTH	0x3000

#define CS_TCAM_CTRL 	        0x0000
#define CS_TCAM_STAT 	        0x0004
#define CS_TCAM_CMD	        0x0008
#define CS_TCAM_SEARCH	        0x000C
#define CS_TCAM_ADR		0x2000
#define CS_TCAM_DATA		0x2010

#define CS_TCAM_GMASK_REG	0x180020
#define CS_TCAM_MASK_OFFSET	0x080000
#define CS_TCAM_BWRITE_REG	0x18003B
#define CS_TCAM_BREAD_REG	0x18003A
/* registers */


/*! 
 * \brief Maximum burst address 
 */
#define CS_TCAM_MAX_BURST_ADDR 0x1FFF

/*! 
 * \brief Maximum burst count 
 */
#define CS_TCAM_MAX_BURST_COUNT 511

/*! 
 * \brief Minimum burst count 
 */
#define CS_TCAM_MIN_BURST_COUNT 4

/*! tcam row structure - 68b long */
typedef struct cs_tcam_ctrl_row {
	u_int32_t first; /*!< LS */
	u_int32_t second; /*!< middle */
	u_int32_t third; /*!< MS */

} cs_tcam_ctrl_row_t;

/*! 
 * \brief Command number written into command register
 */
typedef enum cs_tcam_ctrl_command{
	CS_TCAM_CTRL_READ_COMMAND,
	CS_TCAM_CTRL_WRITE_COMMAND,
	CS_TCAM_CTRL_SEARCH_COMMAND,
	CS_TCAM_CTRL_BURST_READ_COMMAND = 4,
	CS_TCAM_CTRL_BURST_WRITE_COMMAND
} cs_tcam_ctrl_command_t;

/*! 
 * \brief Type of command to distinguish data/mask operations
 */
typedef enum cs_tcam_ctrl_operation{
	CS_TCAM_CTRL_OPERATION_DATA,
	CS_TCAM_CTRL_OPERATION_MASK
} cs_tcam_ctrl_operation_t;

/* basic operations */
void cs_tcam_ctrl_init (cs_device_t * dev, cs_space_t * space);
int cs_tcam_ctrl_init_inst_cache (cs_device_t * dev, cs_space_t * space, char *src_file_name); 
void cs_tcam_ctrl_clear_search_reg (cs_device_t * dev, cs_space_t * space);
void cs_tcam_ctrl_get_unit_control (cs_device_t * dev, cs_space_t * space);
void cs_tcam_ctrl_give_unit_control (cs_device_t * dev, cs_space_t * space);

/* performing commands */
void cs_tcam_ctrl_write_command_with_params (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_command_t command_type);
void cs_tcam_ctrl_wait_for_command_completion(cs_device_t * dev, cs_space_t * space);

/* IO rows */
bool cs_tcam_ctrl_get_row_from_file (cs_tcam_ctrl_row_t *row, FILE *file);
bool cs_tcam_ctrl_get_multiple_rows_from_file (cs_tcam_ctrl_row_t *rows, int count, FILE *file);
bool cs_tcam_ctrl_get_row_from_string (char *buffer, cs_tcam_ctrl_row_t *row);
void cs_tcam_ctrl_print_row (cs_tcam_ctrl_row_t row, FILE *output);
void cs_tcam_ctrl_print_multiple_rows(cs_tcam_ctrl_row_t *rows, int count, FILE * output);

/* write */
void cs_tcam_ctrl_write_row (cs_device_t * dev, cs_space_t * space, u_int32_t addr, cs_tcam_ctrl_row_t row, u_int32_t addr_offset);
void cs_tcam_ctrl_single_write (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t data, cs_tcam_ctrl_row_t addr, cs_tcam_ctrl_operation_t operation);
void cs_tcam_ctrl_single_write_gmask (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t data, cs_tcam_ctrl_row_t addr, int reg_num);

/* search */
bool cs_tcam_ctrl_search (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t addr, cs_tcam_ctrl_row_t *data, u_int32_t *match_addr);

/* read */
cs_tcam_ctrl_row_t cs_tcam_ctrl_read_row (cs_device_t * dev, cs_space_t * space, u_int32_t addr);
cs_tcam_ctrl_row_t cs_tcam_ctrl_single_read (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t addr, cs_tcam_ctrl_operation_t operation);
cs_tcam_ctrl_row_t cs_tcam_ctrl_single_read_gmask (cs_device_t * dev, cs_space_t * space, int cmd_params, int reg_num);

/* bursts */
bool cs_tcam_ctrl_burst_init (cs_device_t * dev, cs_space_t * space, int burst_count, int burst_addr, u_int32_t reg);
bool cs_tcam_ctrl_burst_write (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t *rows, int burst_count, int burst_addr, cs_tcam_ctrl_operation_t operation);
bool cs_tcam_ctrl_burst_read (cs_device_t * dev, cs_space_t * space, int cmd_params, int burst_count, int burst_addr, cs_tcam_ctrl_row_t *rows, cs_tcam_ctrl_operation_t operation);

/*! \}*/

#endif
