/*!
 * \file sdramctrl.c
 * \brief SDRAM_CTRL component control routines.
 * \author Miroslav Vadkerti <thrix@liberouter.org>
 * \author Petr Springl <xsprin01@liberouter.org>
 * \date 2006, 2007
 */

/* Copyright (C) 2006, 2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */


#include <commlbr.h>

#include "../combosix.h"
#include "sdramctrl.h"

__RCSID ("$Id$");

/* registers */
/*! Status register */
#define SDRAM_LB_STATUS		0x0000
/*! Command register */
#define SDRAM_LB_COMMAND	0x0004
/*! Address register */
#define SDRAM_LB_ADDR		0x0008
/*! Data register */
#define SDRAM_LB_DATA		0x000C

/* command constants */
/*! Get control command */
#define SDRAM_CMD_GET_CONT	0x01
/*! Give control command */
#define SDRAM_CMD_GIVE_CONT	0x02
/*! Read command */
#define SDRAM_CMD_WRITE		0x04
/*! Write command */
#define SDRAM_CMD_READ		0x08
/*! Load mode register command */
#define SDRAM_CMD_LMR		0x10

/*!
 * \brief Get control over component 
 *        --------------------------
 * \param dev		device file
 * \param space		address space specification
 * \return	0	OK<BR>
 *		-1	error - cant get control 
 */
int
cs_sdramctrl_get_control(cs_device_t *dev, cs_space_t *space)
{
	uint32_t	count = 0; 	/* wait cycle counter */

	/* set bit for getting control over the component */
	cs_space_write_4(dev, space, SDRAM_LB_COMMAND, SDRAM_CMD_GET_CONT);
	VERBOSE(CL_VERBOSE_LIBRARY, "Write to register 0x%X: value 0x%X (get control)", 
		SDRAM_LB_COMMAND, SDRAM_CMD_GET_CONT);

	/* waiting until control is set */ 
	while((cs_space_read_4(dev, space, SDRAM_LB_STATUS) & 0x1) != 0x1) {
		count++;
		if(count >= 1000){
			return -1;  /* failed - not set bit GOT_CONTROL */
		}
	}

	/* return ret */
	return 0;
}


/*!
 * \brief Give control over component
 *        ----------------------------
 * \param dev           device file
 * \param space         address space specification
 * \return	0       OK
 * \return      -1      load mode reg failed
 */
void
cs_sdramctrl_give_control(cs_device_t *dev, cs_space_t *space)
{
	/* set bit for giving control over the component */
	cs_space_write_4(dev, space, SDRAM_LB_COMMAND, SDRAM_CMD_GIVE_CONT);
	VERBOSE(CL_VERBOSE_LIBRARY, "Write to command register 0x%X: value 0x%X (give control)", 
		SDRAM_LB_COMMAND, SDRAM_CMD_GIVE_CONT);
}

/*!
 * \brief Write operation into memory
 *        ----------------------------
 * \param dev           Device file
 * \param space         Address space specification
 * \param address	Address for writing
 * \param data          Pointer to data array to write (4 or 5 unsigned ints)
 * \param length	Length of written data (should be 128 or 148)
 * \return      0       OK<BR>
 *		-1 	error	
 */
int
cs_sdramctrl_write(cs_device_t *dev, cs_space_t *space, u_int32_t address, 
		u_int32_t *data, u_int32_t length)
{

	int	count = 0;	/* wait counter */
	int	give_back = 0; /* give back flag */
	int	k;			/* cycle variable */

	/* get control over component - if just not getted */	
	if((cs_space_read_4(dev, space, SDRAM_LB_STATUS)&0x1) != 0x1){
		if(cs_sdramctrl_get_control(dev, space) == -1) {
			return -1;
		}
		/* flag for giving control back to component */
		give_back = 1;
	}

	/* write length wide data */ 
	count = length / 32;
	if(length % 32 != 0) count++;
	for(k = 0; k < count; k++) {	
		/* write data to the data register */ 
		cs_space_write_4(dev, space, SDRAM_LB_DATA + 4*k, data[k]); 
		VERBOSE(CL_VERBOSE_LIBRARY, "Write to data register 0x%x: value 0x%x", SDRAM_LB_DATA + 4*k, data[k]);
	}
		
	/* write address to the address register */ 
	cs_space_write_4(dev, space, SDRAM_LB_ADDR, address);
	VERBOSE(CL_VERBOSE_LIBRARY, "Write to address register 0x%X: value 0x%X", 
		SDRAM_LB_ADDR, address);
	/* set bit - WRITE REQUEST in command register */ 
	cs_space_write_4(dev, space, SDRAM_LB_COMMAND, SDRAM_CMD_WRITE);	
	VERBOSE(CL_VERBOSE_LIBRARY, "Write to command register 0x%X: value 0x%X (give control)", 
		SDRAM_LB_COMMAND, SDRAM_CMD_WRITE);

	/* give control back */
	if(give_back){
		cs_sdramctrl_give_control(dev, space);
	}

	/* return */ 
    return 0;
}

/*!
 * \brief Read operation from memory
 *        --------------------------
 * \param dev           device file
 * \param space         address space specification
 * \param address       address for reading 
 * \param data          returning data from reading (should be 4 or 5 uint array)
 * \param length	data length (should be 128 or 144)
 * \return      0       OK<BR>
 *              -1      error
 */
int
cs_sdramctrl_read(cs_device_t *dev, cs_space_t *space, u_int32_t address, 
		u_int32_t *data, u_int32_t length)
{

	int     give_back = 0;	/* give back flag */
	int 	count = 0;	/* wait counter */
	int 	k;			/* cycle variable */

	/* get control over component - if just not getted */
	if((cs_space_read_4(dev, space, SDRAM_LB_STATUS) & 0x1) != 0x1) {
		if(cs_sdramctrl_get_control(dev, space) == -1) {
			return -1;
		}
		/* flag for giving control back to component */
		give_back = 1;
	}

	/* write address to the address register */
	cs_space_write_4(dev, space, SDRAM_LB_ADDR, address);
	VERBOSE(CL_VERBOSE_LIBRARY, "Write to address register 0x%X: value 0x%X", 
		SDRAM_LB_ADDR, address);

	/* set bit - READ REQUEST in command register */
	cs_space_write_4(dev, space, SDRAM_LB_COMMAND, SDRAM_CMD_READ);
	VERBOSE(CL_VERBOSE_LIBRARY, "Write to command register 0x%X: value 0x%X (read command)", 
		SDRAM_LB_COMMAND, SDRAM_CMD_READ);

	/* waiting until data are prepared */
	while((cs_space_read_4(dev, space, SDRAM_LB_STATUS) & 0x2) != 0x2) {
		count++;
		if(count >= 1000) {
			return -1;  /* failed - not set bit GOT_CONTROL */
		}
	}
	
	/* read length wide data */ 
	count = length / 32;
	if(length % 32 != 0) count++;
	for(k = 0; k < count; k++) {	
		data[k] = cs_space_read_4(dev, space, SDRAM_LB_DATA + k*4);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read data from data register 0x%X: value 0x%X", 
			SDRAM_LB_DATA + k*4, data[k]);
	}
	
	/* give control back */
	if(give_back) {
		cs_sdramctrl_give_control(dev, space);
	}

	/* success */
	return 0;
}


/*!
 * \brief Load mode register with specified value \e reg
 *        ----------------------------------------------------------------
 * \param dev           device file
 * \param space         address space specification
 * \param reg         	content of the SDRAM Mode register 
 * \return      0       OK<BR>
 *              -1      error
 */
int
cs_sdramctrl_load_mode_reg(cs_device_t *dev, cs_space_t *space, u_int32_t reg)
{
	int     give_back = 0;	/* give back control flag */

	/* get control over component - if just not getted */
	if((cs_space_read_4(dev, space, SDRAM_LB_STATUS)&0x1) != 0x1) {
		if(cs_sdramctrl_get_control(dev, space) == -1) {
			return -1;
		}
		/* flag for giving control back to component */
		give_back = 1;
	}

	/* write valid content of the SDRAM Mode register into lower bits of
    	   ADDR register (seven lowest) */ 
	cs_space_write_4(dev, space, SDRAM_LB_ADDR, reg & 0x7F);
	VERBOSE(CL_VERBOSE_LIBRARY, "Write to address register 0x%X: value 0x%X", 
		SDRAM_LB_ADDR, reg & 0x7F);

	/* load mode register command is sent to SDRAM */	
	cs_space_write_4(dev, space, SDRAM_LB_COMMAND, SDRAM_CMD_LMR);
	VERBOSE(CL_VERBOSE_LIBRARY, 
		"Write to command register 0x%X: value 0x%X (load mode register)", 
		SDRAM_LB_COMMAND, SDRAM_CMD_LMR);

	/* give back control after get control command */
	if(give_back) {
		cs_sdramctrl_give_control(dev, space);
	}

	/* success */
	return 0;
}



