/*!
 * \file 	obuf.c
 * \brief 	Functions for controlling OBUF component
 * \author 	Andrej Hank <xhanka00@liberouter.org>
 * \author	Miroslav Vadkerti <thrix@liberouter.org>
 * \date 	2006, 2007
 */

/* Copyright (C) 2006 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <commlbr.h>

#include "../combosix.h"
#include "obuf.h"

__RCSID ("$Id$");

/*!
 * \def		OBUFCMD_STROBE_COUNTERS
 * \brief	Constant for strobe counters command
 */
#define OBUFCMD_STROBE_COUNTERS		0x01

/*!
 * \def		OBUFCMD_RESET_COUNTERS
 * \brief	Constant for strobe counters command
 */
#define OBUFCMD_RESET_COUNTERS		0x02

/*!
 * \def		OBUFCMD_SET_SPEED10
 * \brief	Constant for set 10Mb/s command
 */
#define OBUFCMD_SET_SPEED10		0x04

/*!
 * \def		OBUFCMD_SET_SPEED100
 * \brief	Constant for set 100Mb/s command
 */
#define OBUFCMD_SET_SPEED100		0x05

/*!
 * \def		OBUFCMD_SET_SPEED1000
 * \brief	Constant for set 1000Mb/s command
 */
#define OBUFCMD_SET_SPEED1000		0x06

/*! All packets count register address */
#define OBUF_CNT_PACKETS	0x0000
/*! All received packets count register address */
#define OBUF_CNT_RECV		0x0004
/*! All error packets count register address */
#define OBUF_CNT_RECVERR	0x0008
/*! All packets count register address */
#define OBUF_CNT_PACKETS_HI	0x0010
/*! All received packets count register address */
#define OBUF_CNT_RECV_HI	0x0014
/*! All error packets count register address */
#define OBUF_CNT_RECVERR_HI	0x0018
/*! All received packets count register address */
#define OBUF_EN			0x0020
/*! MAC lower 32bits register address */
#define OBUF_MAC_LOW		0x0024
/*! MAC high 16 bits register address */
#define OBUF_MAC_HIGH		0x0028
/*! Control register address */
#define OBUF_CTRL		0x002C
/*! Status register address */
#define OBUF_STATUS		0x0030

/*!
 *\brief 	Return OBUF state (enabled/disabled)
 *\param dev	Device
 *\param space	Space
 *\return	True value if OBUF enabled
 */
int
cs_obuf_is_enabled(cs_device_t *dev, cs_space_t *space)
{
	return cs_space_read_4(dev, space, OBUF_EN);
}

/*!
 *\brief 	Read OBUF counters
 *\param dev	Device
 *\param space	Space
 *\param xgmii	XGMII OBUF flag
 *\return	Read data in cs_obuf_t structure
 */
cs_obuf_t
cs_obuf_read (cs_device_t *dev, cs_space_t *space, bool xgmii)
{
	cs_obuf_t	data;

	cs_lock (CS_LOCK_OBUF);

	cs_space_write_4(dev, space, OBUF_CTRL, OBUFCMD_STROBE_COUNTERS);

	/* Clear counters */
	data.cnt_packets = data.cnt_recv = data.cnt_error = 0;


	/* Read counters and registers */
	if(xgmii) {
		data.cnt_packets = cs_space_read_4(dev, space, OBUF_CNT_PACKETS_HI);
		data.cnt_recv    = cs_space_read_4(dev, space, OBUF_CNT_RECV_HI);
		data.cnt_error   = cs_space_read_4(dev, space, OBUF_CNT_RECVERR_HI);
	}
	data.cnt_packets = (data.cnt_packets << 32) + cs_space_read_4(dev, space, OBUF_CNT_PACKETS);
	data.cnt_recv    = (data.cnt_recv << 32) + cs_space_read_4(dev, space, OBUF_CNT_RECV);
	data.cnt_error   = (data.cnt_error << 32) + cs_space_read_4(dev, space, OBUF_CNT_RECVERR);

	data.reg_en 	 = cs_space_read_4(dev, space, OBUF_EN);
	data.mac_low     = cs_space_read_4(dev, space, OBUF_MAC_LOW);
	data.mac_high    = cs_space_read_4(dev, space, OBUF_MAC_HIGH);
	data.obuf_status = cs_space_read_4(dev, space, OBUF_STATUS);

	cs_unlock(CS_LOCK_OBUF);

	return data;
}

/*!
 *\brief 	Disable OBUF unit
 *\param dev	Device
 *\param space	Space
 */
void
cs_obuf_disable(cs_device_t *dev, cs_space_t *space)
{
	cs_space_write_4(dev, space, OBUF_EN, 0);
	/*VERBOSE(CL_VERBOSE_LIBRARY, "OBUF disable register: %d  addr: 0x%x\n", 1, OBUF_EN);*/
}
/*!
 *\brief 	Enable OBUF unit
 *\param dev	Device
 *\param space	Space
 */
void
cs_obuf_enable(cs_device_t *dev, cs_space_t *space)
{
	cs_space_write_4(dev, space, OBUF_EN, 1);
	/*VERBOSE(CL_VERBOSE_LIBRARY, "OBUF enable register: %d  addr: 0x%x\n", 0, OBUF_EN);*/
}

/*!
 *\brief 	Set OBUF unit speed.
 *\param dev	Device
 *\param space	Space
 *\param speed	Requested unit speed - 10|100|1000
 */
void
cs_obuf_set_speed (cs_device_t *dev, cs_space_t *space, int speed)
{
	int cmd;	/* command */

	/* select speed */
	switch(speed) {
		case 10: cmd = OBUFCMD_SET_SPEED10; break;
		case 100: cmd = OBUFCMD_SET_SPEED100; break;
		case 1000:
		default: cmd = OBUFCMD_SET_SPEED1000; break;
	}

	/* be verbose */
	VERBOSE (CL_VERBOSE_LIBRARY, "Set Speed to: %d    command : %d \n", speed, cmd);

	/* write to control register */
	cs_space_write_4(dev, space, OBUF_CTRL, cmd);
}

/*!
 *\brief 	Reset all frame counters to 0.
 *\param dev	Device
 *\param space	Space
 */
void
cs_obuf_reset (cs_device_t *dev, cs_space_t *space)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "Resetting frame counters.");
	cs_space_write_4(dev, space, OBUF_CTRL, OBUFCMD_RESET_COUNTERS);
}

/*!
 * \brief 	Set mac address. Write lower and higher MAC address data to apropriate registers.
 * \param dev	Device
 * \param space	Space
 * \param lo	Lower 32 bits of MAC
 * \param hi	Higher 32 bits of MAC
 */
void
cs_obuf_set_mac (cs_device_t *dev, cs_space_t *space, u_int32_t hi, u_int32_t lo)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "Writing MAC HI: 0x%x  LO: 0x%x.", hi, lo);

	cs_lock(CS_LOCK_OBUF);

	cs_space_write_4(dev, space, OBUF_MAC_LOW, lo);
	cs_space_write_4(dev, space, OBUF_MAC_HIGH, hi);

	cs_unlock(CS_LOCK_OBUF);
}

