/**
 * \file i2c.h
 * \brief Basic i2c communication routines for i2c_sw component.
 * \author Miroslav Vadkerti <thrix@liberouter.org>
 * \date 2007
 */

/*
 * Copyright (C) 2007  CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#ifndef _LIBEROUTER_I2C_H_
#define _LIBEROUTER_I2C_H_

/* combo include */
#include "../combosix.h"

/*!
 * \defgroup i2c_component I2C component
 * \ingroup components 
 * \{
 *
 * \brief This module implements basic routines for reading/writing registers
 * of devices connected to I2C bus. 
 *
 * Before using cs_i2c_read/write functions cs_i2c_init function has to be called.
 * This function detects the type of the controller (see below) and ensures that
 * correct communication type is used. If this function is not called, 
 * the cs_i2c_read/write functions use communication protocol for the elder
 * SW I2C controller. Also commonly for correct behvior you will have to
 * call cs_i2c_set_reg to set the correct offset of the I2C control signals
 * from the mapped COMBO space.
 * 
 * The module supports two I2C controllers:
 *
 * SW I2C CONTROLLER
 * Elder simple controller managed only by SW. This is used for communication
 * with all devices connected to I2C bus on Combo6x card and it's interface cards. * 
 * 
 * The default address (0x0) of the register with attached I2C controller
 * can be overrided with the function cs_i2c_set_reg.
 *
 * The default I2C address (0xAC) of the connected device can be overrided
 * with the function cs_i2c_set_addr.
 *
 * HW I2C CONTROLLER
 * Newer inteligent controller with simple SW interdace. This controller fixes 
 * problems with elder SW controller which suffered from timing problems during 
 * communication - this caused incorrect behavior - read/write errors. 
 *
 */

void 
cs_i2c_set_addr(u_int32_t);

void 
cs_i2c_set_reg(u_int32_t);

void 
cs_i2c_set_data_bytes(u_int32_t);

void 
cs_i2c_enable_clk_stretch(u_int32_t);

int 
cs_i2c_detect_dlength(cs_device_t * dev, cs_space_t * phy, u_int32_t sel, u_int32_t addr);

void
cs_i2c_init(cs_device_t * dev, cs_space_t *i2c);

int cs_i2c_write(cs_device_t * dev, cs_space_t * phy, u_int32_t sel,
    u_int32_t reg, u_int32_t data);

int cs_i2c_read(cs_device_t * dev, cs_space_t * phy, u_int32_t sel,
    u_int32_t reg, u_int32_t *data);

int
cs_i2c_read_array(cs_device_t * dev, cs_space_t * i2c, u_int8_t *cmd, 
	u_int8_t *data);

int
cs_i2c_write_array(cs_device_t * dev, cs_space_t * i2c, u_int32_t size,
	u_int8_t *array);


/*! \} */

#endif /* _LIBEROUTER_I2C_H_ */
