/*!
 * \file flstat.c
 * \brief Functions for controlling FLSTAT component.
 * \author Peter Stranak <stranak@liberouter.org>
 * \date 2010-2011
 *
 * Copyright (C) 2010-2011 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <errno.h>

#include <commlbr.h>

#include "flstat.h"

/*! Control register */
#define FLSTAT_CTRL_REG		0x0

/*! Clock cycles counter */
#define FLSTAT32_CNT_REG	0x4
/*! Source ready counter */
#define FLSTAT32_SRC_OFF	0x0
/*! Destination ready counter */
#define FLSTAT32_DST_OFF	0x4
/*! Both ready counter */
#define FLSTAT32_BOTH_OFF	0x8
/*! Interface offset for 32b FLSTAT */
#define FLSTAT32_IFC_OFF	0x10

/*! Clock cycles counter */
#define FLSTAT64_CNT_REG	0x8
/*! Source ready counter */
#define FLSTAT64_SRC_OFF	0x10
/*! Destination ready counter */
#define FLSTAT64_DST_OFF	0x20
/*! Both ready counter */
#define FLSTAT64_BOTH_OFF	0x30
/*! Interface offset for 32b FLSTAT */
#define FLSTAT64_IFC_OFF	0x40

/*! Enable bit */
#define FLSTAT_BIT_EN		0x0001
/*! Reset bit */
#define FLSTAT_BIT_RESET	0x0002
/*! Overflow bit */
#define FLSTAT_BIT_OVERFLOW	0x0004
/*! Sample bit */
#define FLSTAT_BIT_SAMPLE	0x0008
/*! Sampling available bit */
#define FLSTAT_BIT_CAN_SAMPLE	0x0010
/*! Stop counting bit */
#define FLSTAT_BIT_STOP		0x0020
/*! 64b flag bit */
#define FLSTAT_BIT_64		0x0040
/*! Number of watched interfaces */
#define FLSTAT_BIT_IFCS		0xFF00

/*!
 * \brief Return current value of the control register
 *
 * \param dev	Device
 * \param space	Space
 *
 * \returns FLSTAT control register
 */
u_int32_t cs_flstat_read_control(cs_device_t *dev, cs_space_t *space)
{
	return cs_space_read_4(dev, space, FLSTAT_CTRL_REG);
}

/*!
 * \brief Enable/disable FLSTAT
 *
 * \param dev		Device
 * \param space		Space
 * \param enable	Flag (1 - enable, 0 - disable)
 */
void cs_flstat_enable(cs_device_t *dev, cs_space_t *space, int enable)
{
	u_int32_t control;

	control = cs_space_read_4(dev, space, FLSTAT_CTRL_REG);
	if (enable) {
		control |= FLSTAT_BIT_EN;
	} else {
		if (control & FLSTAT_BIT_CAN_SAMPLE) {
			control |= FLSTAT_BIT_STOP;
		} else {
			control &= ~FLSTAT_BIT_EN;
		}
	}
	cs_space_write_4(dev, space, FLSTAT_CTRL_REG, control);
}

/*!
 * \brief Reset FLSTAT counters
 *
 * \param dev		Device
 * \param space		Space
 */
void cs_flstat_reset(cs_device_t *dev, cs_space_t *space)
{
	u_int32_t control;

	control = cs_space_read_4(dev, space, FLSTAT_CTRL_REG);
	control |= FLSTAT_BIT_RESET;
	cs_space_write_4(dev, space, FLSTAT_CTRL_REG, control);
}

/*!
 * \brief Sample current values of counters
 *
 * \param dev		Device
 * \param space		Space
 */
void cs_flstat_sample(cs_device_t *dev, cs_space_t *space)
{
	u_int32_t control;

	control = cs_space_read_4(dev, space, FLSTAT_CTRL_REG);
	control |= FLSTAT_BIT_SAMPLE;
	cs_space_write_4(dev, space, FLSTAT_CTRL_REG, control);
}

/*!
 * \brief Check if FLSTAT is enabled
 *
 * \param control	FLSTAT control register
 *
 * \retval 1	Enabled
 * \retval 0	Disabled
 */
int cs_flstat_is_enabled(u_int32_t control)
{
	return (control & FLSTAT_BIT_EN) ? 1 : 0;
}

/*!
 * \brief Check if sampling is available
 *
 * \param control	FLSTAT control register
 *
 * \retval 1	Available
 * \retval 0	Not available
 */
int cs_flstat_is_sampling_available(u_int32_t control)
{
	return (control & FLSTAT_BIT_CAN_SAMPLE) ? 1 : 0;
}

/*!
 * \brief Fill a structure with data read from FLSTAT

 * \param dev			Device
 * \param space			Space
 * \param fl_comp		Structure with FLSTAT component data (#cs_flstat_t)
 *
 * \retval 0		OK
 * \retval negative	Error
 */
int cs_flstat_read(cs_device_t *dev, cs_space_t *space, cs_flstat_t *fl_comp)
{
	u_int32_t *ptr;
	int i, j;

	fl_comp->control_reg = cs_flstat_read_control(dev, space);
	fl_comp->cntr_width = (fl_comp->control_reg & FLSTAT_BIT_64) ? 64 : 32;
	fl_comp->interfaces = (fl_comp->control_reg & FLSTAT_BIT_IFCS) >> 8;
	fl_comp->overflow = (fl_comp->control_reg & FLSTAT_BIT_OVERFLOW) ? 1 : 0;

	fl_comp->flstat_ifc = malloc(fl_comp->interfaces * sizeof(struct flstat_ifc));
	if (fl_comp->flstat_ifc == NULL) {
		VERBOSE(CL_VERBOSE_LIBRARY, "failed to allocate memory");
		return -ENOMEM;
	}

	if (fl_comp->cntr_width == 32) {
		fl_comp->clock = cs_space_read_4(dev, space, FLSTAT32_CNT_REG);
		for (i = 0; i < fl_comp->interfaces; i++) {
			j = FLSTAT32_IFC_OFF;
			fl_comp->flstat_ifc[i].src_ready = cs_space_read_4(dev, space, j + FLSTAT32_SRC_OFF);
			fl_comp->flstat_ifc[i].dst_ready = cs_space_read_4(dev, space, j + FLSTAT32_DST_OFF);
			fl_comp->flstat_ifc[i].both_ready = cs_space_read_4(dev, space, j + FLSTAT32_BOTH_OFF);
			j += FLSTAT32_IFC_OFF;
		}
	} else {
		ptr = (void *)&fl_comp->clock;
		cs_space_read_multi_4(dev, space, FLSTAT64_CNT_REG, 2, ptr);
		for (i = 0; i < fl_comp->interfaces; i++) {
			j = FLSTAT64_IFC_OFF;
			ptr = (void *)&fl_comp->flstat_ifc[i].src_ready;
			cs_space_read_multi_4(dev, space, j + FLSTAT64_SRC_OFF, 2, ptr);
			ptr = (void *)&fl_comp->flstat_ifc[i].dst_ready;
			cs_space_read_multi_4(dev, space, j + FLSTAT64_DST_OFF, 2, ptr);
			ptr = (void *)&fl_comp->flstat_ifc[i].both_ready;
			cs_space_read_multi_4(dev, space, j + FLSTAT64_BOTH_OFF, 2, ptr);
			j += FLSTAT64_IFC_OFF;
		}
	}

	return 0;
}

/*!
 * \brief Release allocated resources
 *
 * \param fl_comp	Structure with FLSTAT component data (#cs_flstat_t)
 */
void cs_flstat_free(cs_flstat_t *fl_comp)
{
	free(fl_comp->flstat_ifc);
	fl_comp->flstat_ifc = NULL;
}
