/*!
 * \file crypto.c
 * \brief Basic communication with cryptochips.
 * \author Jakub Bezak <xbezak00@liberouter.org>
 * \author Matus Holec <xholec00@liberouter.org>
 * \date 2009
 */

/*
 * Copyright (C) 2007-2009  CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
1* 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

/* local include */
#include "i2c.h"
#include "crypto.h"

/* liberouter include */
#include <commlbr.h>

/* std include */
#include <unistd.h>

__RCSID("$Id$");

/*! Size of command - cmd addr1 addr2 N */
#define CRYPTO_CMD_SIZE				0x04

/*! Cmd for setting the user zone */
#define CRYPTO_CMD_USER_ZONE		0xB4
/*! addr1 for setting user zone */
#define CRYPTO_ADDR1_USER_ZONE		0x03
/*! Amount of data sent, to set the user zone */
#define CRYPTO_N_USER_ZONE			0x00


/*! Cmd for writing int the user zone */
#define CRYPTO_CMD_WRITE_USER_ZONE	0xB0


/*! Cmd for reading from the user zone */
#define CRYPTO_CMD_READ_USER_ZONE	0xB2

/*! time in usec used to wait between commands */
#define CRYPTO_USLEEP_VALUE		100000

/*! Cmd for unlocking configuration memory */
#define CRYPTO_CMD_UNLOCK_CONF_MEM 0xBA
/*! Addr1 for unlocking configuration memory */
#define CRYPTO_ADDR1_UNLOCK_CONF_MEM 0x11
/*! Addr2 for unlocking configuration memory */
#define CRYPTO_ADDR2_UNLOCK_CONF_MEM 0x00
/*! Amount of data for unlocking configuration memory */
#define CRYPTO_N_UNLOCK_CONF_MEM 0x03
/*! First data item for unlocking configuration memory */
#define CRYPTO_ITEM_ONE_UNLOCK_CONF_MEM 0xFF
/*! Second data item for unlocking configuration memory */
#define CRYPTO_ITEM_TWO_UNLOCK_CONF_MEM 0xFF
/*! THIRD data item for unlocking configuration memory */
#define CRYPTO_ITEM_THREE_UNLOCK_CONF_MEM 0xFF


/*! CMD for reading from configuration memory */
#define CRYPTO_CMD_READ_CONF_MEM 0xB6
/*! ADDR1 for reading/writing from/to configuration memory */
#define CRYPTO_ADDR1_READ_WRITE_CONF_MEM 0x00
/*! ADDR1 for reading/writing the fuses */
#define CRYPTO_ADDR1_READ_WRITE_FUSES 0x01
/*! ADDR2 for reading/writing the fuses */
#define CRYPTO_ADDR2_READ_WRITE_FUSES 0x00
/*! Amount of data to be read while reading fuse byte*/
#define CRYPTO_N_READ_FUSES 0x01


/*! CMD for writing to configuration memory */
#define CRYPTO_CMD_WRITE_CONFIG_ZONE 0xB4

/*! Defines for memory pages*/
#define CONFIG_MEMORY_MAX_RANGE 0xFF
#define USER_ZONE_MAX_RANGE 0x20
#define MEMORY_PAGE_SIZE 0x10


/*! Memory ranges for crypto configuration memory */
#define CRYPTO_CONFIG_MEMORY_CODES_ADDR_LOW 0x08
#define CRYPTO_CONFIG_MEMORY_CODES_ADDR_HIGH 0x0F
#define CRYPTO_CONFIG_MEMORY_ID_ADDR_LOW 0x19
#define CRYPTO_CONFIG_MEMORY_ID_ADDR_HIGH 0x1F
#define CRYPTO_CONFIG_MEMORY_REGISTERS_ADDR_LOW 0x20
#define CRYPTO_CONFIG_MEMORY_REGISTERS_ADDR_HIGH 0x27
#define CRYPTO_CONFIG_MEMORY_ISSUER_CODE_ADDR_LOW 0x40
#define CRYPTO_CONFIG_MEMORY_ISSUER_CODE_ADDR_HIGH 0x4F

#define CRYPTO_CONFIG_MEMORY_C0_ADDR_LOW 0x51
#define CRYPTO_CONFIG_MEMORY_C0_ADDR_HIGH 0x57
#define CRYPTO_CONFIG_MEMORY_C1_ADDR_LOW 0x61
#define CRYPTO_CONFIG_MEMORY_C1_ADDR_HIGH 0x67
#define CRYPTO_CONFIG_MEMORY_C2_ADDR_LOW 0x71
#define CRYPTO_CONFIG_MEMORY_C2_ADDR_HIGH 0x77
#define CRYPTO_CONFIG_MEMORY_C3_ADDR_LOW 0x81
#define CRYPTO_CONFIG_MEMORY_C3_ADDR_HIGH 0x87

/*SEK = Session Encryption Key*/
#define CRYPTO_CONFIG_MEMORY_SEK0_ADDR_LOW 0x58
#define CRYPTO_CONFIG_MEMORY_SEK0_ADDR_HIGH 0x5F
#define CRYPTO_CONFIG_MEMORY_SEK1_ADDR_LOW 0x68
#define CRYPTO_CONFIG_MEMORY_SEK1_ADDR_HIGH 0x6F
#define CRYPTO_CONFIG_MEMORY_SEK2_ADDR_LOW 0x78
#define CRYPTO_CONFIG_MEMORY_SEK2_ADDR_HIGH 0x7F
#define CRYPTO_CONFIG_MEMORY_SEK3_ADDR_LOW 0x88
#define CRYPTO_CONFIG_MEMORY_SEK3_ADDR_HIGH 0x8F


#define CRYPTO_CONFIG_MEMORY_SECRET_SEED0_ADDR_LOW 0x90
#define CRYPTO_CONFIG_MEMORY_SECRET_SEED0_ADDR_HIGH 0x97
#define CRYPTO_CONFIG_MEMORY_SECRET_SEED1_ADDR_LOW 0x98
#define CRYPTO_CONFIG_MEMORY_SECRET_SEED1_ADDR_HIGH 0x9F
#define CRYPTO_CONFIG_MEMORY_SECRET_SEED2_ADDR_LOW 0xA0
#define CRYPTO_CONFIG_MEMORY_SECRET_SEED2_ADDR_HIGH 0xA7
#define CRYPTO_CONFIG_MEMORY_SECRET_SEED3_ADDR_LOW 0xA8
#define CRYPTO_CONFIG_MEMORY_SECRET_SEED3_ADDR_HIGH 0xAF

/*! forbidden segments for writing of configuration memory*/
#define CRYPTO_RESERVED_ADDRES1_LOW 0x28
#define CRYPTO_RESERVED_ADDRES1_HIGH 0x3F
#define CRYPTO_RESERVED_ADDRES2_LOW 0xC8
#define CRYPTO_RESERVED_ADDRES2_HIGH 0xE7
#define CRYPTO_RESERVED_ADDRES3_LOW 0xF0
#define CRYPTO_RESERVED_ADDRES3_HIGH 0xFF


/*! ADDR1 for blowing the fuse byte*/
#define CRYPTO_ADDR1_BLOW_FUSE_BYTE 0x01

/*! ADDR2 for blowing the FAB fuse byte*/
#define CRYPTO_ADDR2_BLOW_FAB_FUSE_BYTE 0x06

/*! ADDR2 for blowing the CMA fuse byte*/
#define CRYPTO_ADDR2_BLOW_CMA_FUSE_BYTE 0x04

/*! ADDR2 for blowing the PER fuse byte*/
#define CRYPTO_ADDR2_BLOW_PER_FUSE_BYTE 0x00

/*! N for blowing the fuse byte*/
#define CRYPTO_N_BLOW_FUSE_BYTE 0x00

enum e_fuse
{
  FAB,
  CMA,
  PER,
  ALL
};

/* LOCAL FUNCTIONS DECLARATION */
int 
cs_crypto_set_user_zone(cs_device_t * dev, cs_space_t * i2c, int zone);

int
cs_crypto_write(cs_device_t * dev, cs_space_t * i2c, u_int8_t cmd, u_int8_t addr1, u_int8_t addr2, u_int8_t data_size, u_int8_t *data);


/*!
 * \brief 	Sets user zone.
 * \param dev	Device
 * \param i2c	Space
 * \param zone	Number of zone to set
 * \return 0 on succes, -1 otherwise
 */
int
cs_crypto_set_user_zone(cs_device_t * dev, cs_space_t * i2c, int zone)
{
	u_int8_t cmd, addr1, addr2, data_size;

	cmd = CRYPTO_CMD_USER_ZONE;
	addr1 = CRYPTO_ADDR1_USER_ZONE;
	addr2 = zone & 0xFF;
	data_size = CRYPTO_N_USER_ZONE;
	
	if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, data_size, NULL)) != 0)
		return -1;
	
	return 0;
}

/*!
 * \brief 	Writes into the user zone, assembles data to write.
 * \param dev	Device
 * \param i2c	Space
 * \param cmd	Command
 * \param addr1 Upper byte of address
 * \param addr2 Lower byte of address
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -1 otherwise
 */
int 
cs_crypto_write(cs_device_t * dev, cs_space_t * i2c, u_int8_t cmd, u_int8_t addr1, u_int8_t addr2, u_int8_t data_size, u_int8_t * data)
{
	u_int8_t total_size;
	u_int8_t * cmd_write;
	total_size = data_size + CRYPTO_CMD_SIZE;

	cmd_write = (u_int8_t *)malloc(sizeof(u_int8_t) * total_size);
	if  (cmd_write == NULL)
		return -1;

	cmd_write[0] = cmd;
	cmd_write[1] = addr1;
	cmd_write[2] = addr2;
	cmd_write[3] = data_size;

	memcpy(cmd_write + 4, data, sizeof(u_int8_t) * data_size);

	if (cs_i2c_write_array(dev, i2c, total_size, cmd_write)) {
		free(cmd_write);
		return -1;
	}

	usleep(CRYPTO_USLEEP_VALUE);	/* need to wait - TODO: make it correct according cryptochip datasheet */

	free(cmd_write);
	return 0;
}

/*!
 * \brief 	Assembles command part of data to be written into the user memory.
 * \param dev	Device
 * \param i2c	Space
 * \param zone	Number of user zone
 * \param address Address for writing
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -1 otherwise
 */
int
cs_crypto_write_data(cs_device_t * dev, cs_space_t * i2c, int zone, u_int16_t address, u_int8_t data_size, u_int8_t *data)
{
	u_int8_t cmd, addr1, addr2;
	u_int8_t one_page_data[16] = {0x00};
	int i, written_data_amount = 0, written_data_counter = 0, tmp, tmp_data_amount = 0, reached_border = 0;
	cs_crypto_set_user_zone(dev, i2c, zone);
	
	MSG(1, "write address: %08X, write data size: %08X\n", address, data_size);


	if ((address + data_size) > USER_ZONE_MAX_RANGE) {
		//errx(1, "Cryptochip address space is only from 0x0 to 0x1F, but %08X", address + data_size);
		return -1;
	}
	
	cmd = CRYPTO_CMD_WRITE_USER_ZONE;
	addr2 = address & 0xFF;
	
	address >>= 8;
	
	addr1 = address & 0xFF;

	if(((addr2/MEMORY_PAGE_SIZE) < ((addr2+data_size)/MEMORY_PAGE_SIZE)) && (addr2 % MEMORY_PAGE_SIZE) && ((int) (addr2 + data_size) < USER_ZONE_MAX_RANGE)){
	/* data goes via break of a page */
		tmp = address/MEMORY_PAGE_SIZE; 
		tmp++;
		tmp = (tmp*MEMORY_PAGE_SIZE);
		tmp_data_amount = tmp - address;
		if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, tmp_data_amount, data)) != 0) 
			return -1;
		/*shift data*/
		for(i = 0; i < data_size; i++){
			data[i] = data[i+tmp_data_amount];
		}
		data_size -= tmp_data_amount;
		addr2 = tmp;
	}
	
	if(data_size > MEMORY_PAGE_SIZE){
	/* data needs to be split into 16 BYTES blocks */
		written_data_amount = 0;
		tmp = addr2;
		while(written_data_amount < data_size){
			if(reached_border)
				break;
			for(i = 0; i < MEMORY_PAGE_SIZE; i++, written_data_counter++){
				if(written_data_counter + 1 > data_size)
					break;
				one_page_data[i] = data[written_data_counter];
			}
			
			tmp += i;
			if(tmp > USER_ZONE_MAX_RANGE){
				i = USER_ZONE_MAX_RANGE - addr2;
				reached_border = 1;
			}
			written_data_amount += i;
			
			if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, i, one_page_data)) != 0) 
				return -1;
			addr2 += i;
		}
	
	}
	
	else{
		if(addr2 + data_size > USER_ZONE_MAX_RANGE)
			return -1;
		if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, data_size, data)) != 0) 
			return -1;
	}

	return 0;

}

/*!
 * \brief 	Reads data from user zone.
 * \param dev 	Device
 * \param i2c 	Space
 * \param address Address to be read from
 * \param data_size Size of data to be read
 * \param data 	Array to store the data
 * \return 0
 */
int
cs_crypto_read_data(cs_device_t * dev, cs_space_t * i2c, int zone, u_int16_t address, u_int8_t data_size, u_int8_t *data)
{
	u_int8_t cmd_read[4] = {0x00};
	
	/* set user zone */
	cs_crypto_set_user_zone(dev, i2c, zone);

	/* cmd - command code */
	cmd_read[0] = CRYPTO_CMD_READ_USER_ZONE;
	
	/* addr2 - address low byte */
	cmd_read[2] = address & 0xFF;
	
	/* move to high byte */
	address >>= 8;
	
	/* addr1 - address high byte */
	cmd_read[1] = address & 0xFF;
	
	/* N */
	cmd_read[3] = data_size;

	cs_i2c_read_array(dev, i2c, cmd_read, data);
		
	usleep(CRYPTO_USLEEP_VALUE);	/* need to wait - TODO: make it correct according cryptochip datasheet */

	return 0;
	
}

/*!
 * \brief 	Unlocks configuration memory
 * \param dev 	Device
 * \param i2c 	Space
 * \return 0 on success -1 otherwise
 */
int
cs_crypto_unlock_conf_memory(cs_device_t * dev, cs_space_t * i2c, u_int8_t address1, u_int8_t *data)
{
  u_int8_t cmd, addr1, addr2, data_size;

	cmd = CRYPTO_CMD_UNLOCK_CONF_MEM;
	addr1 = address1;
	addr2 = CRYPTO_ADDR2_UNLOCK_CONF_MEM;
	data_size =  CRYPTO_N_UNLOCK_CONF_MEM;
	

	if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, data_size, data)) != 0)
		return -1;
	usleep(CRYPTO_USLEEP_VALUE);
  return 0;
}


/*!
 * \brief 	Assembles command part of data to be written into the configuration memory.
 * \param dev	Device
 * \param i2c	Space
 * \param address Address for writing
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -1 on failure of writing, -2 on attempt to write into reserved memory area
 */
int
cs_crypto_write_config_memory_data(cs_device_t * dev, cs_space_t * i2c, u_int8_t address, u_int8_t data_size, u_int8_t *data)
{
	u_int8_t cmd, addr1, addr2;
	u_int8_t one_page_data[16] = {0x00};
	int i, written_data_amount = 0, written_data_counter = 0, tmp, tmp_data_amount = 0, reached_border = 0;
	
	cmd = CRYPTO_CMD_WRITE_CONFIG_ZONE;
	addr1 =  CRYPTO_ADDR1_READ_WRITE_CONF_MEM;
	addr2 = address;
	tmp = addr2;
	/*check whether is not attempting to write into reserved memory area*/
	
	if( ((tmp >= CRYPTO_RESERVED_ADDRES1_LOW) && (tmp <= CRYPTO_RESERVED_ADDRES1_HIGH)) ||
	    ((tmp >= CRYPTO_RESERVED_ADDRES2_LOW) && (tmp <= CRYPTO_RESERVED_ADDRES2_HIGH)) ||
	    ((tmp >= CRYPTO_RESERVED_ADDRES3_LOW) && (tmp <= CRYPTO_RESERVED_ADDRES3_HIGH)) ||
      ((tmp < CRYPTO_RESERVED_ADDRES1_LOW) && ((int)(tmp + data_size - 1) >= CRYPTO_RESERVED_ADDRES1_LOW) ) || 
      ((tmp < CRYPTO_RESERVED_ADDRES2_LOW) && ((int)(tmp + data_size - 1) >= CRYPTO_RESERVED_ADDRES2_LOW) ) ||
      ((tmp < CRYPTO_RESERVED_ADDRES3_LOW) && ((int)(tmp + data_size - 1) >= CRYPTO_RESERVED_ADDRES3_LOW) )
       )
	    	return -2;

  /* config memory size = 256 B, unable to write across more than one page */
		if(((address/16) < ((address+data_size)/16)) && (address % 16) && ((int) (address + data_size) < CONFIG_MEMORY_MAX_RANGE)){
	/* data goes via break of a page */
		tmp = address/16; 
		tmp++;
		tmp = (tmp*16);
		tmp_data_amount = tmp - address;
		if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, tmp_data_amount, data)) != 0) 
			return -1;
		/*shift data*/
		for(i = 0; i < data_size; i++){
			data[i] = data[i+tmp_data_amount];
		}
		data_size -= tmp_data_amount;
		addr2 = tmp;
	}
	
	if(data_size > MEMORY_PAGE_SIZE){
	/* data needs to be split into 16 BYTES blocks */
		written_data_amount = 0;
		tmp = addr2;
		while(written_data_amount < data_size){
			if(reached_border)
				break;
			for(i = 0; i < MEMORY_PAGE_SIZE; i++, written_data_counter++){
				if(written_data_counter + 1 > data_size)
					break;
				one_page_data[i] = data[written_data_counter];
			}			
			tmp += i;
			VERBOSE(2,"tmp: %d max: %d", tmp, CONFIG_MEMORY_MAX_RANGE);
			if(tmp >= CONFIG_MEMORY_MAX_RANGE){
				i = CONFIG_MEMORY_MAX_RANGE - addr2;
				reached_border = 1;
			}
			written_data_amount += i;
			
			VERBOSE(2,"tmp: %d" ,i);
			if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, i, one_page_data)) != 0) 
				return -1;
			addr2 += i;
		}
	
	}
	
	else{
		if((int) (addr2 + data_size - 1) > CONFIG_MEMORY_MAX_RANGE)
			return -1;
		if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, data_size, data)) != 0) 
			return -1;
	}
	
	usleep(CRYPTO_USLEEP_VALUE);
	
	return 0;
}

/*!
 * \brief 	Assembles command part of data to be written into the segment for codes in the configuration memory.
 * \param dev	Device
 * \param i2c	Space
 * \param address Address for writing
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -3 on writing out of borders of segment
 */
int
cs_crypto_write_config_memory_codes(cs_device_t * dev, cs_space_t * i2c, u_int8_t address, u_int8_t data_size, u_int8_t *data)
{	
	/* check address range */
	if((address < CRYPTO_CONFIG_MEMORY_CODES_ADDR_LOW) || (address > CRYPTO_CONFIG_MEMORY_CODES_ADDR_HIGH)){
		return -3;
	}
	
	if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_CODES_ADDR_HIGH){
		return -3;
	}
	
	return cs_crypto_write_config_memory_data(dev, i2c, address, data_size, data);

}



/*!
 * \brief 	Assembles command part of data to be written into the segment for identification number in the configuration memory.
 * \param dev	Device
 * \param i2c	Space
 * \param address Address for writing
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -3 on writing out of borders of segment
 */
int
cs_crypto_write_config_memory_id_number(cs_device_t * dev, cs_space_t * i2c, u_int8_t address, u_int8_t data_size, u_int8_t *data)
{	
	/* check address range */
	if((address < CRYPTO_CONFIG_MEMORY_ID_ADDR_LOW) || (address > CRYPTO_CONFIG_MEMORY_ID_ADDR_HIGH)){
		return -3;
	}
	
	if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_ID_ADDR_HIGH){
		return -3;
	}
	
	return cs_crypto_write_config_memory_data(dev, i2c, address, data_size, data);
}


/*!
 * \brief 	Assembles command part of data to be written into the segment for registers in the configuration memory.
 * \param dev	Device
 * \param i2c	Space
 * \param address Address for writing
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -3 on writing out of borders of segment
 */
int
cs_crypto_write_config_memory_registers(cs_device_t * dev, cs_space_t * i2c, u_int8_t address, u_int8_t data_size, u_int8_t *data)
{
	/* check address range */
	if((address < CRYPTO_CONFIG_MEMORY_REGISTERS_ADDR_LOW) || (address > CRYPTO_CONFIG_MEMORY_REGISTERS_ADDR_HIGH)){
		return -3;
	}
	
	if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_REGISTERS_ADDR_HIGH){
		return -3;
	}
	
	return cs_crypto_write_config_memory_data(dev, i2c, address, data_size, data);
}


/*!
 * \brief 	Assembles command part of data to be written into the segment for issuer code in the configuration memory.
 * \param dev	Device
 * \param i2c	Space
 * \param address Address for writing
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -3 on writing out of borders of segment
 */
int
cs_crypto_write_config_memory_issuer_code(cs_device_t * dev, cs_space_t * i2c, u_int8_t address, u_int8_t data_size, u_int8_t *data)
{	
	/* check address range */
	if((address < CRYPTO_CONFIG_MEMORY_ISSUER_CODE_ADDR_LOW) || (address > CRYPTO_CONFIG_MEMORY_ISSUER_CODE_ADDR_HIGH)){
		VERBOSE(2,"address fail");
    return -3;
	}
	
	if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_ISSUER_CODE_ADDR_HIGH){
		VERBOSE(2,"address + data size fail, value: 0x%02X + 0x%02x = 0x%02x",address, data_size, address + data_size);
    return -3;
	}
	
	return cs_crypto_write_config_memory_data(dev, i2c, address, data_size, data);
}


/*!
 * \brief 	Assembles command part of data to be written into the segment for cryptograms in the configuration memory.
 * \param dev	Device
 * \param i2c	Space
 * \param address Address for writing
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -3 on writing out of borders of segment
 */
int
cs_crypto_write_config_memory_cryptogram_N(cs_device_t * dev, cs_space_t * i2c, u_int8_t address, u_int8_t data_size, u_int8_t *data)
{
	int cryptogram_no = -1;
	
	/* check address range */
	if((address >= CRYPTO_CONFIG_MEMORY_C0_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_C0_ADDR_HIGH))
		cryptogram_no=0;
	else if((address >= CRYPTO_CONFIG_MEMORY_C1_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_C1_ADDR_HIGH))
		cryptogram_no=1;
	else if((address >= CRYPTO_CONFIG_MEMORY_C2_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_C2_ADDR_HIGH))
		cryptogram_no=2;
	else if((address >= CRYPTO_CONFIG_MEMORY_C3_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_C3_ADDR_HIGH))
		cryptogram_no=3;
	
	if(cryptogram_no == -1)
		return -3;
	
	switch(cryptogram_no){
		case 0: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_C0_ADDR_HIGH)
			return -3;
		case 1: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_C1_ADDR_HIGH)
			return -3;
		case 2: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_C2_ADDR_HIGH)
			return -3;
		case 3: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_C3_ADDR_HIGH)
			return -3;
	}
	
	return cs_crypto_write_config_memory_data(dev, i2c, address, data_size, data);
}


/*!
 * \brief 	Assembles command part of data to be written into the segment for session encryption keys in the configuration memory.
 * \param dev	Device
 * \param i2c	Space
 * \param address Address for writing
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -3 on writing out of borders of segment
 */
int
cs_crypto_write_config_memory_session_encryption_key(cs_device_t * dev, cs_space_t * i2c, u_int8_t address, u_int8_t data_size, u_int8_t *data)
{
	int session_encryption_key_no = -1;
	
	/* check address range */
	if((address >= CRYPTO_CONFIG_MEMORY_SEK0_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_SEK0_ADDR_HIGH))
		session_encryption_key_no=0;
	else if((address >= CRYPTO_CONFIG_MEMORY_SEK1_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_SEK1_ADDR_HIGH))
		session_encryption_key_no=1;
	else if((address >= CRYPTO_CONFIG_MEMORY_SEK2_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_SEK2_ADDR_HIGH))
		session_encryption_key_no=2;
	else if((address >= CRYPTO_CONFIG_MEMORY_SEK3_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_SEK3_ADDR_HIGH))
		session_encryption_key_no=3;
	
	if(session_encryption_key_no == -1)
		return -3;
	
	switch(session_encryption_key_no){
		case 0: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_SEK0_ADDR_HIGH)
			return -3;
		case 1: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_SEK1_ADDR_HIGH)
			return -3;
		case 2: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_SEK2_ADDR_HIGH)
			return -3;
		case 3: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_SEK3_ADDR_HIGH)
			return -3;
	}
	
	return cs_crypto_write_config_memory_data(dev, i2c, address, data_size, data);
}


/*!
 * \brief 	Assembles command part of data to be written into the segment for secret seeds in the configuration memory.
 * \param dev	Device
 * \param i2c	Space
 * \param address Address for writing
 * \param data_size Size of data to be written
 * \param data	Data to be written.
 * \return 0 on succes, -3 on writing out of borders of segment
 */
int
cs_crypto_write_config_memory_secret_seeds(cs_device_t * dev, cs_space_t * i2c, u_int8_t address, u_int8_t data_size, u_int8_t *data)
{
	int secret_seed_no = -1;
	
	/* check address range */
	if((address >= CRYPTO_CONFIG_MEMORY_SECRET_SEED0_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_SECRET_SEED0_ADDR_HIGH))
		secret_seed_no=0;
	else if((address >= CRYPTO_CONFIG_MEMORY_SECRET_SEED1_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_SECRET_SEED1_ADDR_HIGH))
		secret_seed_no=1;
	else if((address >= CRYPTO_CONFIG_MEMORY_SECRET_SEED2_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_SECRET_SEED2_ADDR_HIGH))
		secret_seed_no=2;
	else if((address >= CRYPTO_CONFIG_MEMORY_SECRET_SEED3_ADDR_LOW) && (address <= CRYPTO_CONFIG_MEMORY_SECRET_SEED3_ADDR_HIGH))
		secret_seed_no=3;
	
	if(secret_seed_no == -1)
		return -3;
	
	switch(secret_seed_no){
		case 0: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_SECRET_SEED0_ADDR_HIGH)
			return -3;
		case 1: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_SECRET_SEED1_ADDR_HIGH)
			return -3;
		case 2: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_SECRET_SEED2_ADDR_HIGH)
			return -3;
		case 3: if((address + data_size - 1) > CRYPTO_CONFIG_MEMORY_SECRET_SEED3_ADDR_HIGH)
			return -3;
	}
	
	return cs_crypto_write_config_memory_data(dev, i2c, address, data_size, data);
}


/*!
 * \brief 	Reads data from configuration memory.
 * \param dev 	Device
 * \param i2c 	Space
 * \param address Address to be read from
 * \param data_size Size of data to be read
 * \param data 	Array to store the data
 * \return 0
 */
int
cs_crypto_read_config_memory(cs_device_t * dev, cs_space_t * i2c, u_int8_t address, u_int8_t data_size, u_int8_t *data)
{
	u_int8_t cmd_read[4] = {0x00};
	/* cmd - command code */
	cmd_read[0] = CRYPTO_CMD_READ_CONF_MEM;
	
	/* addr2 - address low byte */
	cmd_read[1] = CRYPTO_ADDR1_READ_WRITE_CONF_MEM;
	
	
	cmd_read[2] = address;
	
	/* N */
	cmd_read[3] = data_size;
  
	cs_i2c_read_array(dev, i2c, cmd_read, data);
		
	usleep(CRYPTO_USLEEP_VALUE);	/* need to wait - TODO: make it correct according cryptochip datasheet */

	return 0;
	
}

/*!
 * \brief 	Reads fuse byte.
 * \param dev	Device
 * \param i2c	Space
 * \return 0
 */
int
cs_crypto_read_fuse_byte(cs_device_t * dev, cs_space_t * i2c, u_int8_t *data)
{
  u_int8_t cmd_read[4] = {0x00};
  
	/* cmd - command code */
	cmd_read[0] = CRYPTO_CMD_READ_CONF_MEM;
	cmd_read[1] = CRYPTO_ADDR1_READ_WRITE_FUSES;
	
	cmd_read[2] = CRYPTO_ADDR2_READ_WRITE_FUSES;  
	
  cmd_read[3] = CRYPTO_N_READ_FUSES;	
  
	cs_i2c_read_array(dev, i2c, cmd_read, data);
		
	usleep(CRYPTO_USLEEP_VALUE);
  
  return 0;	
}

/*!
 * \brief 	Sets given fuse byte to zero, cusing blowing the fuse, controls necessary order of blowing the fuses. 
 * \param dev	Device
 * \param i2c	Space
 * \param fuse Determines which fuse should be blown. 
 * \return 0
 */
int
cs_crypto_blow_fuse_byte(cs_device_t * dev, cs_space_t * i2c, enum e_fuse fuse){
  
  u_int8_t read_fuse_byte[1] = {0x00};
  u_int8_t cmd, addr1, addr2, data_size;  
  int i = 0;
  u_int8_t crypto_addr2_fuse_byte_array [] = {
  CRYPTO_ADDR2_BLOW_FAB_FUSE_BYTE,
  CRYPTO_ADDR2_BLOW_CMA_FUSE_BYTE,
  CRYPTO_ADDR2_BLOW_PER_FUSE_BYTE
  };
  
  switch(fuse){
    case FAB:
      cmd = CRYPTO_CMD_WRITE_CONFIG_ZONE;
      addr1 = CRYPTO_ADDR1_BLOW_FUSE_BYTE;
      addr2 = CRYPTO_ADDR2_BLOW_FAB_FUSE_BYTE;
      data_size = CRYPTO_N_BLOW_FUSE_BYTE;
      if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, i, NULL)) != 0) 
				return -1;
      break;
    case CMA:
      /*check if FAB fuse is blown*/
      cs_crypto_read_fuse_byte(dev, i2c, read_fuse_byte);
      if(read_fuse_byte[0] & 0x01){
        return -2;
      }
      else{
        cmd = CRYPTO_CMD_WRITE_CONFIG_ZONE;
        addr1 = CRYPTO_ADDR1_BLOW_FUSE_BYTE;
        addr2 = CRYPTO_ADDR2_BLOW_CMA_FUSE_BYTE;
        data_size = CRYPTO_N_BLOW_FUSE_BYTE;
        if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, i, NULL)) != 0) 
				return -1;
      }
      break;
    case PER:
      /*check if FAB and CMA fuses are blown*/
      cs_crypto_read_fuse_byte(dev, i2c, read_fuse_byte);
      if((read_fuse_byte[0] & 0x01) || (read_fuse_byte[0] & 0x02)){
        return -2;
      }
      else{
        cmd = CRYPTO_CMD_WRITE_CONFIG_ZONE;
        addr1 = CRYPTO_ADDR1_BLOW_FUSE_BYTE;
        addr2 = CRYPTO_ADDR2_BLOW_PER_FUSE_BYTE;
        data_size = CRYPTO_N_BLOW_FUSE_BYTE;
        if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, i, NULL)) != 0) 
				  return -1;
      }
      break;
    case ALL:
      cmd = CRYPTO_CMD_WRITE_CONFIG_ZONE;
      addr1 = CRYPTO_ADDR1_BLOW_FUSE_BYTE;
      data_size = CRYPTO_N_BLOW_FUSE_BYTE;
      for(i = 0; i < 3; i++){
        addr2 = crypto_addr2_fuse_byte_array[i]; 
        if((cs_crypto_write(dev, i2c, cmd, addr1, addr2, i, NULL)) != 0) 
				  return -1;
      }
      
      break;
  }
  return 0;
}
